/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


//INCLUDES
#include "fsplatform.h"
#include "fslog.h"
#include "fsconstants.h"
#include "fsworker.h"

// --------------------------------------------------------------------
// FileSystem::FileSystem()
// Constructor.
// --------------------------------------------------------------------
FileSystem::FileSystem()
{
    m_watcher = NULL;
}

// --------------------------------------------------------------------
// FileSystem::FileSystem(FileSystemWorker* aParent)
// aParent - to send the data back
// Constructor.
// --------------------------------------------------------------------
FileSystem::FileSystem(FileSystemWorker* aParent)
{
    m_FSWorker = aParent;
    m_watcher = NULL;
}

// --------------------------------------------------------------------
// FileSystem::~FileSystem()
// Destructor.
// --------------------------------------------------------------------
FileSystem::~FileSystem()
{
    LOG("Enter ~FileSystem\n");
    if (m_watcher)
        {
        LOG("memory card notifier is active, so delete it\n");
        delete m_watcher;
        m_watcher = NULL;
        }
    LOG("Exit ~FileSystem\n");
}

// --------------------------------------------------------------------
// FileSystem::getMountPoints()
// Gets aviailable drives on device
// --------------------------------------------------------------------
QList<QVariant> FileSystem::getMountPoints()
{
    LOG("Enter getMountPoints\n");
    QList<QVariant> mountPointArray;

    QFileInfoList entries;
    QStringList newEntList = readMountPoints();
    for (int child = 0; child < newEntList.count(); child++) {
        QFileInfo fi(newEntList.at(child));
        entries.append(fi);
    }
    QMap<QString,QVariant> entryInfoMap;

    if (entries.size() > 0) {
        foreach(QFileInfo info, entries) {
            // From fssymbianfilesystem.cpp: //content name is always "Root"
            // for drives
            QString name(ROOT_TYPE);
            // Get the uri of the mount point
            QString uri("file://");
            uri += info.filePath();
            int mountType = INTERNAL_MEMORY;
            // From spec: isDefault always false
            int isDefault = false;

            if (uri.contains("/media")) {
                mountType = REMOVABLE_MEMORY;
            }

            entryInfoMap.clear();
            entryInfoMap.insert(MOUNTPOINT_URI, uri);
            entryInfoMap.insert(MOUNTPOINT_SIZE, fullSpaceInfo(info.filePath()));
            entryInfoMap.insert(MOUNTPOINT_AVAILABLESIZE, freeSpaceInfo(info.filePath()));
            entryInfoMap.insert(MOUNTPOINT_NAME, name);
            entryInfoMap.insert(MOUNTPOINT_ISWRITABLE, isMpRW(info.filePath()));
            entryInfoMap.insert(MOUNTPOINT_TYPE, mountType);
            entryInfoMap.insert(MOUNTPOINT_ISDEFAULT, isDefault);
            //qDebug() << "entryInfoMap: " << entryInfoMap;
            mountPointArray.append(entryInfoMap);
        }//foreach
    }
    LOG("Exit getMountPoints\n");
    return mountPointArray;
}

// --------------------------------------------------------------------
// FileSystem::getDefaultPath(contentType)
// Gets aviailable default paths on device based on content type
// --------------------------------------------------------------------
QList<QVariant> FileSystem::getDefaultPath(QString contentType)
{
    LOG("Enter getDefaultPath\n");
    QList<QVariant> defaultPathArray;

    QDir p(QDir::homePath() += "/MyDocs");
    QStringList finalList;
    QFileInfoList entries = p.entryInfoList(QDir::Dirs | QDir::Hidden);
    QMap<QString,QVariant> entryInfoMap;
    QString name;

    if (entries.size() > 0) {
        foreach(QFileInfo info, entries) {
            QString uri = QString("file://%1").arg(info.absoluteFilePath());
            bool proceed = false;

            if (contentType.compare(ALL_CONTENT_TYPE) == 0) {
                proceed = true;
            } else if (uri.contains("sounds") && contentType.compare(AUDIO_TYPE) == 0) {
                proceed = true;
            } else if (uri.contains("videos") && contentType.compare(VIDEO_TYPE) == 0) {
                proceed = true;
            } else if (uri.contains("images") && contentType.compare(IMAGE_TYPE) == 0) {
                proceed = true;
            }

            if (proceed) {
                name.clear();
                if (uri.contains("sounds")) {
                    name.append(AUDIO_TYPE);
                } else if (uri.contains("videos")) {
                    name.append(VIDEO_TYPE);
                } else if (uri.contains("images")) {
                    name.append(IMAGE_TYPE);
                }
                entryInfoMap.clear();
                entryInfoMap.insert(MOUNTPOINT_URI, uri);
                entryInfoMap.insert(MOUNTPOINT_SIZE, info.size());
                entryInfoMap.insert(MOUNTPOINT_AVAILABLESIZE, info.size()); //FIXME
                entryInfoMap.insert(MOUNTPOINT_NAME, name);
                entryInfoMap.insert(MOUNTPOINT_ISWRITABLE, info.isWritable());
                entryInfoMap.insert(MOUNTPOINT_TYPE, INTERNAL_MEMORY);
                entryInfoMap.insert(MOUNTPOINT_ISDEFAULT, true);

                defaultPathArray.append(entryInfoMap);
            }
        }//foreach
    }

    LOG("Exit getDefaultPath\n");
    return defaultPathArray;
}

// --------------------------------------------------------------------
// FileSystem::notifyMountEvents()
// creates a request for getting notification of mount events
// --------------------------------------------------------------------
bool FileSystem::notifyMountEvents()
{
    LOG("Enter FileSystem::notifyMountEvents\n");
    m_watcher = new QFileSystemWatcher(m_FSWorker);
    m_watcher->addPath("/media/");
    m_watcher->addPath("/media/mmc1/");
    m_watcher->addPath("/etc/mtab");
    m_watcher->addPath("/etc/fstab");
    connect(m_watcher,SIGNAL(directoryChanged(const QString&)),this,SLOT(dirChangedEvent(const QString&)));
    connect(m_watcher,SIGNAL(fileChanged(const QString&)),this,SLOT(fileChangedEvent(const QString&)));

    m_currentInfo.clear();
    m_currentInfo << readMountPoints();
    return true;
}

// --------------------------------------------------------------------
// FileSystem::cancelNotify()
// cancels the notification request
// --------------------------------------------------------------------
void FileSystem::cancelNotify()
{
    LOG("Enter FileSystem::cancelNotify\n");
    if (m_watcher) {
        delete m_watcher;
        m_watcher = NULL;
    }
    LOG("Exit FileSystem::cancelNotify\n");
}

void FileSystem::dirChangedEvent(const QString& /*path*/)
{
    notifyMountEvent();
}

void FileSystem::fileChangedEvent(const QString& /*path*/)
{
    notifyMountEvent();
}

// --------------------------------------------------------------------
// FileSystem::notifyMountEvent()
// To notify the file system worker that there is a change in drives
// --------------------------------------------------------------------
void FileSystem::notifyMountEvent()
{
    //get the drive details and return
    LOG("Enter FileSystem::notifyMountEvent\n");

    QMap<QString,QVariant> mountEventInfoMap;
    QMap<QString,QVariant> mountPointMap;
    QVariant mountPointMapVar;

    QStringList newEntList = readMountPoints();
    if (newEntList.size() > m_currentInfo.size()) {
        //New mount point was added
        foreach(QString old, m_currentInfo) {
            newEntList.removeAll(old);
        }
        QString newDir = newEntList.takeFirst();
        QFileInfo fi(newDir);
        //if (fi.exists()) {
            //if (fi.isDir()) {
                mountPointMap.clear();
                mountPointMap.insert(MOUNTPOINT_URI, QString("file://%1").arg(fi.filePath()));
                mountPointMap.insert(MOUNTPOINT_SIZE, fullSpaceInfo(newDir));
                mountPointMap.insert(MOUNTPOINT_AVAILABLESIZE, freeSpaceInfo(newDir));
                mountPointMap.insert(MOUNTPOINT_NAME, newDir);
                mountPointMap.insert(MOUNTPOINT_ISWRITABLE, isMpRW(newDir));
                mountPointMap.insert(MOUNTPOINT_TYPE, REMOVABLE_MEMORY);
                mountPointMap.insert(MOUNTPOINT_ISDEFAULT, false);

                mountEventInfoMap.insert(MOUNTEVENTINFO_STATUS_STR, CONNECTED);
                mountPointMapVar.setValue(mountPointMap);

                mountEventInfoMap.insert(MOUNTEVENTINFO_POINT_STR, mountPointMapVar);
                m_FSWorker->newMountPointEvent(mountEventInfoMap);

                m_currentInfo << newDir;
            //}
        //}
    } else {
        //something was removed
        foreach(QString old, m_currentInfo) {
            if (!newEntList.contains(old)) {
                m_currentInfo.removeAll(old);

                mountPointMap.clear();
                mountPointMap.insert(MOUNTPOINT_URI, QString("file://%1").arg(old));

                mountEventInfoMap.insert(MOUNTEVENTINFO_STATUS_STR, DISCONNECTED);
                mountPointMapVar.setValue(mountPointMap);

                mountEventInfoMap.insert(MOUNTEVENTINFO_POINT_STR, mountPointMapVar);
                m_FSWorker->newMountPointEvent(mountEventInfoMap);
            }
        }
    }
    LOG("Exit FileSystem::notifyMountEvent\n");
}

// --------------------------------------------------------------------
// FileSystem::checkPath(QString &Uri)
// checks the validity of the Uri
// --------------------------------------------------------------------
bool FileSystem::checkPath(QString &Uri)
{
    LOG("Enter FileSystem::checkPath\n");
    if (Uri.length() == 0 || Uri.indexOf("/") != 0) {
        LOG("only protocol is specified, no other info\n");
        return false;
    }

    LOG("Exit FileSystem::checkPath\n");
    return true;
}

QStringList FileSystem::readMountPoints()
{
    QStringList mountPoints;
    struct mntent *mnt;
    char *table = _PATH_MOUNTED;
    FILE *fp;

    fp = setmntent (table, "r");
    if (fp == NULL) {
        qDebug()<<"Error while open "<<_PATH_MOUNTED;
        return mountPoints;
    }

    while ((mnt = getmntent (fp)))
    {
        //qDebug()<<"fsname: "<<mnt->mnt_fsname;
        //qDebug()<<"mnt_dir: "<<mnt->mnt_dir;

        QString tmp(mnt->mnt_dir);
        int fstype=fsType(tmp);

        if (fstype==SYSFS_MAGIC || fstype==DEVFS_SUPER_MAGIC || fstype==PROC_SUPER_MAGIC
           || fstype==DEVPTS_SUPER_MAGIC || fstype==S_MAGIC_SECURITYFS
           ) {
            continue;
        }

        if (QString(mnt->mnt_fsname)==QString("none")) {
            continue;
        }

        if (mountPoints.contains(tmp))
        {
            continue;
        }

        mountPoints.push_back(tmp);
    }

    if (endmntent (fp) == 0) {
        qDebug()<<"Error while close "<<_PATH_MOUNTED;
    }

    return mountPoints;
}

qint32 FileSystem::fsType(const QString& mountPoint)
{
    struct statfs stat;
    if (statfs(mountPoint.toAscii().data(), &stat)!=0)
    {
        return 0;
    }

    //qDebug()<<QString("%1").arg(stat.f_type,10,16,QChar('0'));
    return stat.f_type;
}

qulonglong FileSystem::freeSpaceInfo(const QString& mountPoint)
{
    struct statfs stat;
    if (statfs(mountPoint.toAscii().data(), &stat)!=0) {
        return 0;
    }

    qlonglong fullSize(stat.f_bfree);
    qlonglong blocksize(stat.f_bsize);
    return fullSize*blocksize;
}

qulonglong FileSystem::fullSpaceInfo(const QString& mountPoint)
{
    struct statfs stat;
    if (statfs(mountPoint.toAscii().data(), &stat)!=0) {
        return 0;
    }

    qlonglong fullSize(stat.f_blocks);
    qlonglong blocksize(stat.f_bsize);
    return fullSize*blocksize;
}

bool FileSystem::isMpRW(const QString& mountPoint)
{
    struct mntent *mnt;
    char *table = _PATH_MOUNTED;
    FILE *fp;
    bool isRW(false);

    fp = setmntent (table, "r");
    if (fp == NULL) {
        qDebug()<<"Error while open "<<_PATH_MOUNTED;
    } else {
        while ((mnt = getmntent (fp)))
        {
            QString tmp(mnt->mnt_dir);
            if (tmp.compare(mountPoint) == 0) {
                //qDebug()<<"mnt_opts: "<<mnt->mnt_opts;
                QString tmp_opts(mnt->mnt_opts);
                if (tmp_opts.compare("ro") != 0) {
                    isRW = true;
                }
            break;
            }
        }

        if (endmntent (fp) == 0) {
            qDebug()<<"Error while close "<<_PATH_MOUNTED;
        }
    }

    return isRW;
}

