/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QDir>
#include <QString>
#include <QFileInfo>
#include <QDateTime>
#include <QDirIterator>

#include "fsworker.h"
#include "fserrorcodes.h"
#include "fslog.h"
#include "fsutilities.h"
#include "fsplatform.h"

//for createDir
FileSystemWorker::FileSystemWorker(int task,
        QString srcUri, int callbackId, \
        QHash<QString, QString> mountTable ) \
        :m_task(task), m_srcUri(srcUri), \
        m_cbId(callbackId), m_PlatformNotifier(NULL), \
        m_mountTable(mountTable)
                {
                }

//for getDirContents & search
FileSystemWorker::FileSystemWorker(int task, QString srcUri,
        QString matchPattern, int callbackId , \
        QHash<QString, QString> mountTable ) \
        :m_task(task), m_srcUri(srcUri), \
        m_matchPattern(matchPattern), \
        m_cbId(callbackId), m_PlatformNotifier(NULL) ,\
        m_mountTable(mountTable)

                {
                }

//for remove
FileSystemWorker::FileSystemWorker(int task, QString srcUri,
        bool option, int callbackId, \
        QHash<QString, QString> mountTable) \
        :m_task(task), m_srcUri(srcUri), \
        m_option(option), \
        m_cbId(callbackId), m_PlatformNotifier(NULL) , \
        m_mountTable(mountTable)
                {
                }

//for openFile
FileSystemWorker::FileSystemWorker(int task, QString srcUri,\
        QString mode, QString encoding, QList<FileInfo>* fileHandleList,\
        int callbackId, QHash<QString, QString> mountTable) \
        :m_task(task), m_srcUri(srcUri), \
        m_matchPattern(mode), m_encoding(encoding), \
        m_cbId(callbackId), m_openFileList(fileHandleList), \
        m_PlatformNotifier(NULL), m_mountTable(mountTable)
                {
                }

//for copy & move
FileSystemWorker::FileSystemWorker(int task, QString srcUri, QString destUri,
        QString newName, bool overWrite, int callbackId, QHash<QString, \
                QString> mountTable) \
        :m_task(task), m_srcUri(srcUri), \
        m_matchPattern(newName), m_option(overWrite), \
        m_destUri(destUri), m_cbId(callbackId), \
        m_PlatformNotifier(NULL),m_mountTable(mountTable)
                {
                }

FileSystemWorker::FileSystemWorker(int task, const QMap<QString,QVariant>& matchObject,
        int callbackId, QHash<QString, QString> mountTable) : m_task(task), m_cbId(callbackId),
        m_PlatformNotifier(NULL), m_MatchObject(matchObject),m_mountTable(mountTable)

            {
            LOG("Initialized match object for search\n");
            }


FileSystemWorker::~FileSystemWorker()
    {
    LOG("Enter ~FileSystemWorker\n");
    if (m_PlatformNotifier)
        {
        LOG("~FileSystemWorker, Platform notifier is active, deleting it\n");
        delete m_PlatformNotifier;
        }
    LOG("Exit ~FileSystemWorker\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::validatePath()  
// Validates if the given path is one of the allowed paths
// depending on the caller's credentials
// If its not valid returns NULL.
// --------------------------------------------------------------------
QString FileSystemWorker::validatePath(QString Uri)
    {
    LOG("Enter validatePath\n");
    QString returnUri = Uri;

    if (m_mountTable.value(FILE_PROTOCOL + NOKIA_USER) == "false")
        {
        //3rd party user hence check if the supplied path is supported
        LOG("Third party user\n");
        returnUri = chkIfAllowed(Uri);
        }

    return returnUri;
    }
// --------------------------------------------------------------------
// FileSystemWorker::chkIfAllowed()
// Checks if the specified Uri is one of the allowed paths depending on
// caller's credentials
// --------------------------------------------------------------------
QString FileSystemWorker::chkIfAllowed(QString Uri)
    {
    LOG("Enter chkIfAllowed\n");   
    QDir dirPath(Uri.remove(0, FILE_PROTOCOL.length())); //To remove the . and .. from the actual path
    Uri = dirPath.path();

    Uri = (FILE_PROTOCOL + Uri);

    Uri = Uri.toLower();
    QHashIterator<QString, QString> iter(m_mountTable);
    while (iter.hasNext())
        {
        iter.next();
        QString virtPath = iter.key();
        QString actualPath = iter.value();
        LOG(QString("Uri prefix: %1 \n").arg(actualPath));
        if (Uri.contains(actualPath) && Uri.indexOf(actualPath) == 0)
            {
            //Supported path is supplied 
            LOG("Allowed path supplied\n");
            return Uri;
            }
        }

    LOG("Error, Uri prefix not found in table\n");
    return NULL;

    }


// --------------------------------------------------------------------
// FileSystemWorker::ExtractPathFromUri()
// To extract the file path from the uri specified by the user.
// --------------------------------------------------------------------
bool FileSystemWorker::ExtractPathFromUri(QString &Uri)
    {
    LOG("ENTER ExtractPathFromUri\n");

    if (Uri.contains('\\') == true)
        {
        LOG("Uri contains back slash\n");
        return false;
        }

    // Begin Extract & strip protocol...
    if (Uri.contains(FILE_PROTOCOL,Qt::CaseInsensitive) == false)
        {
        LOG("protocol not specified\n");
        return false;
        }

    int protocolPos = Uri.indexOf(FILE_PROTOCOL);
    if (protocolPos != 0)
        {
        LOG("protocol pos is not at the begining\n");
        return false;
        }

    Uri = Uri.remove(0,FILE_PROTOCOL.length());

    bool isValid(false);
    IMountPoint* fs = new FileSystem();
    isValid = fs->checkPath(Uri);
    delete fs;

    LOG("EXIT ExtractPathFromUri\n");
    return isValid;
    }

// --------------------------------------------------------------------
// FileSystemWorker::defaultSearchPaths()
// Sets the default search path depending on the user's credentials
// --------------------------------------------------------------------
QStringList FileSystemWorker::defaultSearchPaths()
    {
    LOG("Enter defaultSearchPaths\n");
    QStringList searchPathList;
    if (m_mountTable.value(FILE_PROTOCOL + NOKIA_USER) == "false")
        {
        //3rd party user widget, search only in default paths
        LOG("Untrusted widget\n");

        QHashIterator<QString, QString> iter(m_mountTable);
        while (iter.hasNext())
            {
            iter.next();
            QString actPath = iter.value();
            LOG(QString("Actual path from mount table: %1 \n").arg(actPath));
            if (iter.key() != (FILE_PROTOCOL + NOKIA_USER))
                {
                actPath.remove(0, FILE_PROTOCOL.length());
                searchPathList.append(actPath);
                }
            }
        }

    else
        {
        LOG("Trusted widget, so search in all the drives\n");
        QFileInfoList driveList = QDir::drives();

        foreach (QFileInfo drive, driveList)
                {
                LOG(QString("driveName: %1\n").arg(drive.absoluteFilePath()));
                searchPathList.append(drive.absoluteFilePath());
                }
        }
    return searchPathList;
    }
// --------------------------------------------------------------------
// FileSystemWorker::run()
// To extract the file path from the uri specified by the user.
// --------------------------------------------------------------------
void FileSystemWorker::run()
    {
    LOG("Inside FileSystemWorker Run\n");
    switch (m_task)
        {
        case CREATEDIR_METHOD:
            run_createDir();
            break;
        case REMOVE_METHOD:
            run_remove();
            break;
        case RENAME_METHOD:
            run_rename();
            break;
        case GETDIRCONTENTS_METHOD:
            run_getDirContents();
            break;
        case COPY_METHOD:
            run_copy(false);
            break;
        case MOVE_METHOD:
            run_copy(true);
            break;
        case SEARCH_METHOD:
            run_search();
            break;
        case OPENFILE_METHOD:
            run_openFile();
            break;
        case GETMOUNTPOINTS_METHOD:
            run_getMountPoints();
            break;
        case GETDEFAULTPATH_METHOD:
            run_getDefaultPath();
            break;
        case NOTIFYMOUNTEVENTS_METHOD:
            run_notifyMountEvents();
            break;
        case GETELEMENTINFO_METHOD:
            run_getElementInfo();
            break;
        case SEARCHMATCHOBJECT_METHOD:
            run_searchMatchObject();
            break;
        default:
            {
            //it should never come here
            LOG("NO api matched\n");
            }
        }
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_createDir()
// To create user specefied directory
// --------------------------------------------------------------------
void FileSystemWorker::run_createDir()
    {
    LOG("Enter run_createDir\n");

    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_ALLOWED_ERR error\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
        return;
        }

    LOG(QString("Extracted Path : %1\n").arg(m_srcUri));

    int errCode = SUCCESS;
    QString errMessage = SUCCESS_STR;
    bool status = false;
    QDir dir;
    status = dir.mkpath(m_srcUri); // or we can use QFSFileEngine mkdir(path,true)
    LOG(QString("createDir %1\n").arg(status? "Success":"Failed"));

    if (status == false)
        {
        LOG("NOT_SUPPORTED_ERR\n");
        errMessage = NOT_SUPPORTED_ERR_STR;
        errCode = NOT_SUPPORTED_ERR;
        }

    emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(errCode, errMessage));
    LOG("Exit run_createDir\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_remove()
// deletes the user supplied file/directory
// --------------------------------------------------------------------
void FileSystemWorker::run_remove()
    {
    LOG("ENTER run_remove\n");
    LOG(QString("srcUri : %1\n").arg(m_srcUri));
    LOG(QString("recursive : %1\n").arg(m_option));
    //Check the validity of URI only if it is not NULL,
    //Else pass it on for further validation directly
    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR, INVALID_URI_ERR_STR));
        }

    LOG(QString("Extracted Path : %1\n").arg(m_srcUri));

    int errCode = SUCCESS;
    QString errMessage = SUCCESS_STR;

    //find whether its a file or directory
    QFileInfo element(m_srcUri);
    if (element.exists() == true)
        {
        LOG("element exists\n");
        if (element.isFile() == true)
            {
            LOG("element is File\n");
            QFile file(m_srcUri);
            if (file.remove() == true)
                {
                LOG("Remove file Success\n");
                LOG("SUCCESS\n");
                errCode = SUCCESS;
                errMessage = SUCCESS_STR;
                }
            else
                {
                LOG("Remove file Failure\n");
                LOG("FILE_ALREADY_OPEN_ERR\n");
                errCode = FILE_ALREADY_OPEN_ERR;
                errMessage = FILE_ALREADY_OPEN_STR;
                }
            }
        else if (element.isDir() == true)
            {
            if (m_option == false)
                {
                QDir dir(m_srcUri);
                QFileInfoList fileInfoList = dir.entryInfoList(QDir::AllDirs|QDir::Files|QDir::NoDotAndDotDot);

                if (fileInfoList.length() > 0)
                    {
                    LOG("Dir is not empty\n");
                    LOG("DIR_NOT_EMPTY_ERR\n");
                    errCode = DIR_NOT_EMPTY_ERR;
                    errMessage = DIR_NOT_EMPTY_STR;
                    }
                else
                    {
                    if (dir.rmdir(m_srcUri) == true)
                        {
                        LOG("Removing dir Success\n");
                        errCode = SUCCESS;
                        errMessage = SUCCESS_STR;
                        }
                    else
                        {
                        LOG("Removing dir Failure\n");
                        LOG("FILE_ALREADY_OPEN_ERR\n");
                        errCode = FILE_ALREADY_OPEN_ERR;
                        errMessage = FILE_ALREADY_OPEN_STR;
                        }
                    }
                }
            else//do recursive delete
                {
                QDir dir;
                if ( remove_Dir_Recursive(m_srcUri) == true )
                    {
                    LOG("Recursive delete Success\n");
                    //removing user supplied directory
                    if ( dir.rmdir(m_srcUri) == true )
                        {
                        LOG("User supplied path deletion Success\n");
                        errCode = SUCCESS;
                        errMessage = SUCCESS_STR;
                        }
                    else
                        {
                        LOG("User supplied path deletion Failed\n");
                        LOG("FILE_ALREADY_OPEN_ERR\n");
                        errCode = FILE_ALREADY_OPEN_ERR;
                        errMessage = FILE_ALREADY_OPEN_STR;
                        }
                    }
                else
                    {
                    LOG("Recursive delete Failure\n");
                    LOG("FILE_ALREADY_OPEN_ERR\n");
                    errCode = FILE_ALREADY_OPEN_ERR;
                    errMessage = FILE_ALREADY_OPEN_STR;
                    }
                }
            }
        else
            {
            //neither file nor directory
            //donno how to delete this...
            }
        }
    else
        {
        LOG("element not exists\n");
        LOG("URI_NOT_FOUND_ERR\n");
        errCode = URI_NOT_FOUND_ERR;
        errMessage = SOURCE_NOT_EXISTS_STR;
        }

    emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(errCode, errMessage));
    LOG("EXIT run_remove\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::remove_Dir_Recursive()
// deletes a directory recursively
// --------------------------------------------------------------------
bool FileSystemWorker::remove_Dir_Recursive(QString path)
    {
    LOG("**Enter RmDir_Recursive**\n");
    LOG(QString("curr path: %1\n").arg(path));

    QDir curr_dir(path);

    QFileInfoList dirInfoList = curr_dir.entryInfoList(QDir::AllDirs|QDir::NoDotAndDotDot);
    QString subDir = "";

    //recursively go through every subdir of current directory
    foreach (QFileInfo dirInfo, dirInfoList)
        {
        subDir = dirInfo.absoluteFilePath();
        LOG(QString("sub dir path : %1\n").arg(subDir));

        //go till the leaf node is reached
        if (remove_Dir_Recursive(subDir) == false)
            {
            //some file deletion has failed dont go any further
            return false;
            }

        //delete what was just cleared
        if (curr_dir.rmdir(subDir) == false)
            {
            //if we can't delete this folder dont go any further
            LOG(QString("Removing dir, FIRST FAILURE at : %1 Failed\n").arg(subDir));
            return false;
            }
        else
            {
            LOG(QString("Removing dir : %1 Success\n").arg(subDir));
            }
        }

    //all sub dirs of this dir is cleaned
    //now delete all files in the current directory
    bool isdirCleaned = true;
    QFileInfoList fileInfoList = curr_dir.entryInfoList(QDir::Files);
    QString fileName;
    foreach(QFileInfo fileInfo, fileInfoList)
        {
        fileName = fileInfo.absoluteFilePath();

        if (curr_dir.remove(fileName) == false)
            {
            LOG(QString("Deleting file, FIRST FAILURE at: %1 \n").arg(fileName));
            isdirCleaned = false;
            break;
            }
        else
            {
            LOG(QString("deleting file : %1 success \n").arg(fileName));
            isdirCleaned = true;
            }
        }

    LOG("**Exit RmDir_Recursive**\n");
    return isdirCleaned; //is the current directory successfull cleaned?
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_getDirContents()
// To get the directory contents based on match pattern
// --------------------------------------------------------------------
void FileSystemWorker::run_getDirContents()
    {
    LOG("***ENTER run_getDirContents***\n");
    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
        return;
        }

    LOG(QString("Extracted Path : %1\n").arg(m_srcUri));

    QFileInfo elementInfo(m_srcUri);
    if (elementInfo.exists() == false)
        {
        LOG("URI_NOT_FOUND_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(URI_NOT_FOUND_ERR,SOURCE_NOT_EXISTS_STR));
        return;
        }

    if (elementInfo.isDir() == false)
        {
        LOG("NOT_SUPPORTED_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,ELEMENT_NOT_DIR_STR));
        return;
        }

    QDir sourceDir(m_srcUri);
    QStringList matchPatternList(m_matchPattern);

    QFileInfoList fileList = sourceDir.entryInfoList(matchPatternList,
                                QDir::AllEntries | QDir::NoDotAndDotDot);

    QList<QVariant> fileAttribArray;
    fileAttribArray.clear();

    int len = fileList.length();
    if ( len > 0 )
        {
        LOG(QString("Matching content found %1\n").arg(len));
        int elementType = -1;
        FileSystemMap fileInfoMap;
        QVariant fileInfoMapVar;
        for (int i =0; i < len; i++)
            {
            QFileInfo file = fileList.at(i);

            if (file.isFile() == true)
                elementType = FILE_TYPE;
            else if (file.isDir() == true)
                elementType = DIRECTORY_TYPE;
            else if (file.isSymLink() == true)
                elementType = LINK_TYPE;

            fileInfoMap.clear();

            fileInfoMap.insert(ELEMENT_URI, FILE_PROTOCOL + file.absoluteFilePath());
            fileInfoMap.insert(ELEMENT_TYPE,elementType);
            fileInfoMap.insert(ELEMENT_READONLY, file.isWritable() ? "false":"true");
            fileInfoMap.insert(ELEMENT_SIZE, (file.size() / (double) KILO_BYTE));
            fileInfoMap.insert(ELEMENT_MODIFIEDDATE,file.lastModified().toString());
			fileInfoMap.insert(ELEMENT_CREATEDATE, file.created().toString());
            LOG(QString("URI: %1 TYPE: %2 READONLY: %3 SIZE: %4 MODIFIEDDATE: %5 CREATEDATE: %6\n")
                    .arg(FILE_PROTOCOL + file.absoluteFilePath())
                    .arg(elementType)
                    .arg(file.isWritable()?"false":"true")
                    .arg(file.size()/(double)KILO_BYTE)
                    .arg(file.lastModified().toString())
                    .arg(file.created().toString())
            );

            fileInfoMapVar.setValue(fileInfoMap);
            fileAttribArray.append(fileInfoMapVar);
            }
        }
    else
        {
        LOG("No matching items found\n");
        }

    LOG("*************EXIT getDirContents******************\n");
    emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR,fileAttribArray));
    return;
    }

// --------------------------------------------------------------------
// FileSystemWorker::rename()
// renames a file/directory with the user sepecified name
// --------------------------------------------------------------------
void FileSystemWorker::run_rename()
    {
    LOG("ENTER run_rename\n");
    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    QString newName = m_matchPattern;

    if (newName.isEmpty() == true || newName.length() == 0)
        {
        LOG("INVALID_ARG_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_ARG_ERR,NEWNAME_INVALID_STR));
        return;
        }

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
        return;
        }

    LOG(QString("Extracted Path : %1\n").arg(m_srcUri));
    int errCode = SUCCESS;
    QString errMessage = SUCCESS_STR;

    QFileInfo elementInfo(m_srcUri);

    if (elementInfo.exists() == true)
        {
        LOG("src exists\n");
        QDir parent = elementInfo.dir();
        QString parentPath = parent.absolutePath();
        LOG("parentPath :" + parentPath + "\n");
        QString newElement = parentPath + "/" + newName;
        LOG("newElement :" + newElement + "\n");

        QFileInfo newElementInfo(newElement);
        if (newElementInfo.exists() == true)
            {
            //dest already exists
            LOG("dest already exists, URI_ALREADY_EXISTS_ERR\n");
            errCode = URI_ALREADY_EXISTS_ERR;
            errMessage = DESTINATION_EXISTS_STR;
            }
        else
            {
            LOG("no element with newName exists\n");
            if (elementInfo.isFile() == true)
                {
                LOG("element is a file\n");
                QFile file(m_srcUri);
                if (file.rename(newElement) == true)
                    {
                    errCode = SUCCESS;
                    errMessage = SUCCESS_STR;
                    LOG("file rename success, SUCCESS\n");
                    }
                else
                    {
                    errCode = FILE_ALREADY_OPEN_ERR;
                    errMessage = FILE_ALREADY_OPEN_STR;
                    LOG("file rename failure, in use, FILE_ALREADY_OPEN_ERR\n");
                    }
                }
            else if (elementInfo.isDir())
                {
                LOG("element is a directory\n");
                QDir dir;
                if (dir.rename(m_srcUri,newElement))
                    {
                    errCode = SUCCESS;
                    errMessage = SUCCESS_STR;
                    LOG("dir rename success, SUCCESS\n");
                    }
                else
                    {
                    //i am not sure about this, even if it cant rename with the
                    //newName proposed, then also it will throw this error
                    //which is not valid
                    errCode = FILE_ALREADY_OPEN_ERR;
                    errMessage = FILE_ALREADY_OPEN_STR;
                    LOG("dir rename failure, in use, FILE_ALREADY_OPEN_ERR\n");
                    }
                }

            }
        }
    else
        {
        //src uri not exists
        errCode = URI_NOT_FOUND_ERR;
        errMessage = SOURCE_NOT_EXISTS_STR;
        LOG("src uri not exists, URI_NOT_FOUND_ERR\n");
        }

    LOG(QString("return Value: %1\n").arg(errCode));
    LOG("EXIT run_rename\n");
    emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(errCode,errMessage));
    return;
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_openFile()
// (creates &) opens  an existing file
// --------------------------------------------------------------------
void FileSystemWorker::run_openFile()
    {
    LOG("ENTER run_openFile\n");
    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
        return;
        }

    QString path = m_srcUri;
    LOG(QString("Extracted Path : %1\n").arg(path));

    //checking mode
    if (m_matchPattern.length() == 0 || m_matchPattern.isEmpty() == true)
        {
        LOG("INVALID_ARG_ERR, mode not specified\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_ARG_ERR,INVALID_ARG_ERR_STR));
        return;
        }

    if (m_encoding.length() == 0 || m_encoding.isEmpty() == true)
        {
        LOG("INVALID_ARG_ERR, encoding not specified\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_ARG_ERR,INVALID_ARG_ERR_STR));
        return;
        }

    if (m_encoding != "UTF-8")
        {
        LOG("NOT_SUPPORTED_ERR, encoding is other than UTF-8, currently not supported\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,ENCODING_FORMAT_NOT_SUPPORTED_STR));
        return;
        }

    QFileInfo element(path);
    if (element.exists() == true && element.isFile() == false)
        {
        LOG("NOT_SUPPORTED_ERR, uri exists but not a file.\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,SOURCE_NOT_FILE_STR));
        return;
        }

    QIODevice::OpenMode openMode = QIODevice::NotOpen;
    if ( getFileOpenMode(m_matchPattern, element.exists(), openMode) == false )
        {
        LOG("NOT_SUPPORTED_ERR, mode is invalid\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,MODE_IS_INVALID_STR));
        return;
        }

    int errCode = -1;
    QVariant data = 0;
    QString errMessage = SUCCESS_STR;
    QFile* file = new QFile(path);

    bool fileOpenStatus = false;
    fileOpenStatus = file->open(openMode);

    if (fileOpenStatus == true)
        {
        LOG("file open Success\n");
        if (file->handle() == -1)//handle will be -1 if file is not open, qt docs
            {
            LOG("file handle is -1\n");
            if (file) {delete file;}
            errCode = FILE_ALREADY_OPEN_ERR;
            errMessage = FILE_ALREADY_OPEN_ERR_STR;
            }
        else
            {
            FileInfo fileInfo;
            fileInfo.fileHandle = file;
            fileInfo.encoding = m_encoding;
            m_openFileList->append(fileInfo);
            data = m_openFileList->count() - 1;
            LOG(QString("id returned for file handle: %1\n").arg(data.toInt()));
            errCode = SUCCESS;
            errMessage = SUCCESS_STR;
            }
        }
    else
        {
        if (element.exists() == true)
            {
            LOG("File exists but open failed, FILE_ALREADY_OPEN_ERR\n");
            errCode = FILE_ALREADY_OPEN_ERR;
            errMessage = FILE_ALREADY_OPEN_ERR_STR;
            }
        else
            {
            //this could happen bcoz of invalid file name as well
            LOG("File doesn't exists and not created or opened, URI_NOT_FOUND_ERR\n");
            errCode = URI_NOT_FOUND_ERR;
            errMessage = SOURCE_NOT_EXISTS_STR;
            }
        }

    emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(errCode, errMessage, data));
    LOG("EXIT run_openFile\n");
    return;
    }

// --------------------------------------------------------------------
// FileSystemWorker::getFileOpenMode()
// converts the user supplied mode string to value
// --------------------------------------------------------------------
bool FileSystemWorker::getFileOpenMode(QString mode,bool fileExists, QIODevice::OpenMode &modeVal)
    {
    LOG("***ENTER getFileOpenMode***\n");

    if (mode.compare(READ_MODE) == 0)
        {
        modeVal = QIODevice::ReadOnly;
        LOG("mode is r, so ReadOnly\n");
        }
    else if (mode.compare(READ_PLUS_MODE) == 0)
        {
        //if file doesn't exist, it should not be created in read mode, so just assign
        //read capability to make open method fail
        //otherwise if we assign read write , it will create file if doesn't exists..
        //else it should be opened in read write mode

        LOG(QString("mode is r+, file %1\n")
                .arg(fileExists ? "exists, so ReadWrite" : "not exists, so ReadOnly for not to create file")
        );
        modeVal = fileExists ? QIODevice::ReadWrite : QIODevice::ReadOnly;
        }
    else if (mode.compare(WRITE_MODE) == 0)
        {
        modeVal = QIODevice::WriteOnly | QIODevice::Truncate;
        LOG("mode is w, so WriteOnly + Truncate\n");
        }
    else if (mode.compare(WRITE_PLUS_MODE) == 0)
        {
        modeVal = QIODevice::ReadWrite | QIODevice::Truncate;
        LOG("mode is w+, so ReadWrite + Truncate\n");
        }
    else if (mode.compare(APPEND_MODE) == 0)
        {
        modeVal = QIODevice::Append | QIODevice::WriteOnly;
        LOG("mode is a, so Append & WriteOnly\n");
        }
    else if (mode.compare(APPEND_PLUS_MODE) == 0)
        {
        modeVal = QIODevice::Append | QIODevice::ReadWrite;
        LOG("mode is a+, so Append & ReadWrite\n");
        }
    else if (mode.compare(READ_TEXT_MODE) == 0)
        {
        modeVal = QIODevice::ReadOnly | QIODevice::Text;
        LOG("mode is rt, so ReadOnly & Text\n");
        }
    else if (mode.compare(READ_PLUS_TEXT_MODE) == 0)
        {
        //if file doesn't exist, it should not be created in read mode, so just assign
        //read capability to make open method fail
        //otherwise if we assign read write , it will create file if doesn't exists..
        //else it should be opened in read write mode
        LOG(QString("mode is r+t file %1\n")
                .arg(fileExists ? "exists, so ReadWrite & Text" : " doesn't exists, so ReadOnly & Text")
        );
        modeVal = fileExists ? QIODevice::ReadWrite | QIODevice::Text :
        QIODevice::ReadOnly | QIODevice::Text;
        }
    else if (mode.compare(WRITE_TEXT_MODE) == 0)
        {
        modeVal = QIODevice::WriteOnly | QIODevice::Text;
        LOG("mode is wt, so WriteOnly & Text\n");
        }
    else if (mode.compare(WRITE_PLUS_TEXT_MODE) == 0)
        {
        modeVal = QIODevice::ReadWrite | QIODevice::Text;
        LOG("mode is w+t, so ReadWrite & Text\n");
        }
    else if (mode.compare(APPEND_TEXT_MODE) == 0)
        {
        modeVal = QIODevice::Append | QIODevice::WriteOnly | QIODevice::Text;
        LOG("mode is at, so Append & WriteOnly & Text\n");
        }
    else if (mode.compare(APPEND_PLUS_TEXT_MODE) == 0)
        {
        modeVal = QIODevice::Append | QIODevice::ReadWrite | QIODevice::Text;
        LOG("mode is a+t, so Append & ReadWrite & Text\n");
        }
    else
        {
        //mode wrongly specified
        return false;
        }
    LOG("***EXIT getFileOpenMode***\n");
    return true;
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_copy()
// copies a file or directory
// --------------------------------------------------------------------
void FileSystemWorker::run_copy(bool removeAfterCopy)
    {
    LOG("***ENTER run_copy***\n");
    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    if (m_destUri != NULL)
        {
        m_destUri = validatePath(m_destUri);
        if (m_destUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
        return;
        }

    if (ExtractPathFromUri(m_destUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
        return;
        }

    LOG(QString("Extracted Src Path : %1\n").arg(m_srcUri));
    LOG(QString("Extracted Dest Path : %1\n").arg(m_destUri));

    bool OverWrite = m_option;
    QString newName;
    if (m_matchPattern == NULL)
        {
        newName = "";
        }
    else
        {
        newName = m_matchPattern;
        }

    QFileInfo srcElementInfo(m_srcUri);
    if (srcElementInfo.exists() == false)
        {
        LOG("URI_NOT_FOUND_ERR, Src not exists\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(URI_NOT_FOUND_ERR, SOURCE_NOT_EXISTS_STR));
        return;
        }

    QFileInfo destElementInfo(m_destUri);
    if (destElementInfo.exists() == true)
        {
        LOG("DestUri exists\n");
        if (destElementInfo.isDir() == false)
            {
            LOG("NOT_SUPPORTED_ERR, Dest is not a directory\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, ELEMENT_NOT_DIR_STR));
            return;
            }
        }
    else
        {
        LOG("DestUri not exists, so create it\n");
        QDir d;
        if (destElementInfo.exists() == false && d.mkpath(m_destUri) == false)
            {
            LOG("Creating destUri directory Failed\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, NOT_SUPPORTED_ERR_STR));
            return;
            }
        }

    m_srcUri = srcElementInfo.absoluteFilePath();
    m_destUri = destElementInfo.absoluteFilePath();

    LOG(QString("File sys returned Src Path : %1\n").arg(m_srcUri));
    LOG(QString("File sys returned Dest Path : %1\n").arg(m_destUri));

    int errCode = SUCCESS;
    QString errMessage = SUCCESS_STR;

    if (srcElementInfo.isFile() == true)
        {
        if (newName.isEmpty() == true)//if new name is not empty
            {
            newName = srcElementInfo.fileName();
            LOG(QString("new file name is empty, so copy file with same name: %1\n").arg(newName));
            }

        QString newDestElementStr(QString("%1/%2").arg(m_destUri).arg(newName));
        LOG(QString("new element Path: %1 \n").arg(newDestElementStr));
        m_destUri = newDestElementStr;

        if ( copy_file(m_srcUri, m_destUri, OverWrite, removeAfterCopy, errCode, errMessage) == true )
            {
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS, SUCCESS_STR));
            }
        else
            {
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(errCode, errMessage));
            }
        }
    else if (srcElementInfo.isDir())
        {
        if (newName.isEmpty() == false)
            {
            LOG("new name is not empty, so create a directory with this name\n");
            QString newDir = QString("%1/%2").arg(m_destUri).arg(newName);
            LOG(QString("newDir name: %1 \n").arg(newDir));

            QDir d;
            if (d.mkpath(newDir) == false)
                {
                LOG("Creating destUri directory with newName Failed\n");
                emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, NOT_SUPPORTED_ERR_STR));
                return;
                }
            m_destUri = newDir;
            }

        if ( copy_dir(m_srcUri, m_destUri, OverWrite, removeAfterCopy, m_srcUri, errCode, errMessage) == true)
            {
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS, SUCCESS_STR));
            }
        else
            {
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(errCode, errMessage));
            }

        }
    else
        {
        //its neither a file nor a directory
        //dont know wat to do at present
        }

    LOG("***EXIT run_copy***\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::copy_file()
// copies a file
// --------------------------------------------------------------------
bool FileSystemWorker::copy_file(QString SrcUri, QString DestUri,
        bool OverWrite, bool removeAfterCopy,
        int& errCode, QString& errMessage)
    {
    LOG("Enter copy_file\n");
    QFile srcfile(SrcUri);
    QFileInfo newElementInfo(DestUri);

    if (newElementInfo.exists() == true && newElementInfo.isFile() == true)
        {
        LOG("new element exists & its a file\n");
        if (OverWrite == true)
            {
            LOG("overWrite is set, so remove the file\n");
            if (srcfile.remove(DestUri) == false)
                {
                LOG("FILE_ALREADY_OPEN_ERR, dest file deletion failed\n");
                errCode = FILE_ALREADY_OPEN_ERR;
                errMessage = FILE_ALREADY_OPEN_STR;
                return false;
                }
            }
        else
            {
            LOG("URI_ALREADY_EXISTS_ERR, dest already exists, but overWrite is not set\n");
            errCode = URI_ALREADY_EXISTS_ERR;
            errMessage = DESTINATION_EXISTS_STR;
            return false;
            }
        }

    LOG("b4 copy\n");
    bool ret = srcfile.copy(DestUri);
    LOG("after copy\n");

    if ( ret == false )
        {
        LOG("FILE_ALREADY_OPEN_ERR, src file copy failed\n");
        errCode = FILE_ALREADY_OPEN_ERR;
        errMessage = FILE_ALREADY_OPEN_STR;
        return false;
        }
    else
        {
        LOG("src file copied successfully\n");
        if (removeAfterCopy == true)
            {
            LOG("Removing src file after copying\n");
            if ( srcfile.remove() == false)
                {
                LOG("FILE_ALREADY_OPEN_ERR, During move operation, src file copy success, but remove failed\n");
                errCode = FILE_ALREADY_OPEN_ERR;
                errMessage = FILE_ALREADY_OPEN_STR;
                return false;
                }
            }
        }

    errCode = SUCCESS;
    errMessage = SUCCESS_STR;
    LOG("Exit copy_file\n");
    return true;
    }

// --------------------------------------------------------------------
// FileSystemWorker::copy_dir()
// copies a directory
// --------------------------------------------------------------------
bool FileSystemWorker::copy_dir(QString SrcUri, QString DestUri,\
        bool OverWrite, bool removeAfterCopy,\
        QString currDir,\
        int& errCode, QString& errMessage)
    {
    LOG(QString("**Enter copy_dir**, curr path: %1\n").arg(currDir));

    QDir curr_dir(currDir);
    QFileInfoList dirInfoList = curr_dir.entryInfoList(QDir::AllDirs|QDir::NoDotAndDotDot);
    QString subDir = "";

    //recursively go through every subdir of current directory
    foreach (QFileInfo dirInfo, dirInfoList)
        {
        subDir = dirInfo.absoluteFilePath();
        LOG(QString("sub dir path : %1\n").arg(subDir));

        //go till the leaf node is reached
        if (copy_dir(SrcUri, DestUri, OverWrite, removeAfterCopy, subDir, errCode, errMessage) == false)
            {
            //some file copy or removal failed dont go any further
            return false;
            }

        //During move we need to remove after copying the directory
        if (removeAfterCopy == true && curr_dir.rmdir(subDir) == false)
            {
            //if we can't delete this folder dont go any further
            LOG(QString("Removing dir, FIRST FAILURE at : %1 Failed\n").arg(subDir));
            errCode = FILE_ALREADY_OPEN_ERR;
            errMessage = FILE_ALREADY_OPEN_STR;
            return false;
            }
        }

    //create the destination folder first
    QString newFolderPath = currDir;
    newFolderPath.remove(SrcUri);
    LOG(QString("path extracted to be created at destination: %1\n").arg(newFolderPath));
    newFolderPath = QString("%1%2").arg(DestUri).arg(newFolderPath);
    LOG(QString("new dest path to be created: %1\n").arg(newFolderPath));
    QDir d;
    if (d.mkpath(newFolderPath) == false)
        {
        LOG("new path creation at destination failed\n");
        errCode = NOT_SUPPORTED_ERR;
        errMessage = NOT_SUPPORTED_ERR_STR;
        return false;
        }
    else
        {
        LOG("new path creation at destination success\n");
        }

    //copy files from source to destination
    bool isfileCopied = true;
    QFileInfoList fileInfoList = curr_dir.entryInfoList(QDir::Files);
    QString destFileName = "";
    foreach(QFileInfo fileInfo, fileInfoList)
        {
        destFileName = QString("%1/%2").arg(newFolderPath).arg(fileInfo.fileName());
        LOG(QString("src file name: %1\ndest file name: %2\n").arg(fileInfo.absoluteFilePath()).arg(destFileName));
        if (copy_file(fileInfo.absoluteFilePath(), destFileName , OverWrite, removeAfterCopy, errCode, errMessage) == false)
            {
            isfileCopied = false;
            break;
            }
        }

    LOG(QString("**Exit copy_dir**, curr path: %1\n").arg(currDir));
    return isfileCopied; //is the current directory successfully cleaned?
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_search()
// Performs a search
// --------------------------------------------------------------------
void FileSystemWorker::run_search()
    {
    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    LOG("***ENTER run_search***\n");

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR, INVALID_URI_ERR_STR));
        return;
        }

    LOG(QString("Extracted Path : %1\n").arg(m_srcUri));

    QFileInfo elementInfo(m_srcUri);
    if (elementInfo.exists() == false)
        {
        LOG("URI_NOT_FOUND_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(URI_NOT_FOUND_ERR,SOURCE_NOT_EXISTS_STR));
        return;
        }

    if (elementInfo.isDir() == false)
        {
        LOG("NOT_SUPPORTED_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,ELEMENT_NOT_DIR_STR));
        return;
        }

    //for all the subdirectories
    QDirIterator dirIterator(m_srcUri, QStringList("*"), QDir::Dirs|QDir::NoDotAndDotDot, QDirIterator::Subdirectories);

    bool iAmLastDir = !dirIterator.hasNext();
    LOG(QString("iAmLastDir: %1\n").arg(iAmLastDir));

    QStringList matchPatternList(m_matchPattern);
    //for the user supplied directory
    FindInDir(m_srcUri, matchPatternList, iAmLastDir);

    //for all the sub directories
    QFileInfo file;
    while (iAmLastDir == false)
        {
        dirIterator.next();
        iAmLastDir = !dirIterator.hasNext();
        file = dirIterator.fileInfo();
        FindInDir(file.absoluteFilePath(), matchPatternList, iAmLastDir);
        }

    return;
    }

// --------------------------------------------------------------------
// FileSystemWorker::FindInDir()
// Finds the matching contents in a directory
// --------------------------------------------------------------------
void FileSystemWorker::FindInDir(QString currDir,QStringList matchPatternList, bool lastDir)
    {
    LOG(QString("Enter FindInDir, currDir: %1, iAmLastDir: %2\n").arg(currDir).arg(lastDir));
    QDir dir(currDir);

    QFileInfoList fileList = dir.entryInfoList(matchPatternList);

    QMap<QString,QVariant> searchResult;
    searchResult.clear();
    QList<QVariant> fileAttribArray;
    fileAttribArray.clear();

    int len = fileList.length();
    if ( len > 0 )
        {
        LOG(QString("Matching content found %1\n").arg(len));
        int elementType = -1;
        FileSystemMap fileInfoMap;
        QVariant fileInfoMapVar;
        for (int i =0; i < len; i++)
            {
            QFileInfo file = fileList.at(i);

            if (file.isFile() == true)
                elementType = FILE_TYPE;
            else if (file.isDir() == true)
                elementType = DIRECTORY_TYPE;
            else if (file.isSymLink() == true)
                elementType = LINK_TYPE;

            fileInfoMap.clear();
      
            fileInfoMap.insert(ELEMENT_URI, FILE_PROTOCOL + file.absoluteFilePath());
            fileInfoMap.insert(ELEMENT_TYPE, elementType);
            fileInfoMap.insert(ELEMENT_READONLY, file.isWritable() ? "false":"true");
            fileInfoMap.insert(ELEMENT_SIZE, file.size() / (double) KILO_BYTE);
            fileInfoMap.insert(ELEMENT_MODIFIEDDATE, file.lastModified().toString());
   			fileInfoMap.insert(ELEMENT_CREATEDATE, file.created().toString());
            LOG(QString("URI: %1 TYPE: %2 READONLY: %3 SIZE: %4 MODIFIEDDATE: %5 CREATEDATE: %6\n")
                    .arg(FILE_PROTOCOL + file.absoluteFilePath())
                    .arg(elementType)
                    .arg(file.isWritable()?"false":"true")
                    .arg(file.size()/(double)KILO_BYTE)
                    .arg(file.lastModified().toString())
                    .arg(file.created().toString())
            );

            fileInfoMapVar.setValue(fileInfoMap);
            fileAttribArray.append(fileInfoMapVar);
            }
        }
    else
        {
        LOG("No matching items found\n");
        }

    searchResult.insert(ELEMENT_STR,fileAttribArray);

    if (len > 0 || lastDir == true)
        {
        searchResult.insert( STATE_STR, ( (lastDir == true) ? COMPLETE : PARTIAL) );
        LOG("emitting FileSystemWorkerSignal\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS, SUCCESS_STR, searchResult));
        }

    LOG("Exit FindInDir\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_getMountPoints()
// Gets the available mount points on device
// --------------------------------------------------------------------
void FileSystemWorker::run_getMountPoints()
    {
    LOG("Enter run_getMountPoints\n");
    IMountPoint* fs = new FileSystem();
    QList<QVariant> mountPointArray = fs->getMountPoints();

    if (mountPointArray.length() > 0)
        {
        LOG("getMountPoints success\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS, SUCCESS_STR, mountPointArray));
        }
    else
        {
        LOG("getMountPoints failed\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, GETMOUNTPOINTS_OP_FAILED_STR));
        }
    delete fs;
    LOG("Exit run_getMountPoints\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_getDefaultPath()
// Gets the available default paths on device
// --------------------------------------------------------------------
void FileSystemWorker::run_getDefaultPath()
    {
    LOG("Enter run_getDefaultPath\n");
    QString contentType = m_srcUri;

    if (contentType.length() == 0)
        {
        LOG("No content Type specified, so setting it to All\n");
        contentType = ALL_CONTENT_TYPE;
        }

    if (contentType.compare(AUDIO_TYPE,Qt::CaseInsensitive) != 0 &&
            contentType.compare(VIDEO_TYPE,Qt::CaseInsensitive) != 0 &&
            contentType.compare(IMAGE_TYPE,Qt::CaseInsensitive) != 0 &&
            contentType.compare(ALL_CONTENT_TYPE,Qt::CaseInsensitive) != 0
    )
        {
        LOG("content type is not supported at present\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, GETDEFAULTPATH_OP_FAILED_STR));
        return;
        }

    IMountPoint* fs = new FileSystem();
    QList<QVariant> defaultPathArray = fs->getDefaultPath(contentType);

    if (defaultPathArray.length() > 0)
        {
        LOG("getDefaultPath success\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS, SUCCESS_STR, defaultPathArray));
        }
    else
        {
        LOG("getDefaultPath failed\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, GETDEFAULTPATH_OP_FAILED_STR));
        }

    delete fs;
    LOG("Exit run_getDefaultPath\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_notifyMountEvents()
// requests notification of mount events
// --------------------------------------------------------------------
void FileSystemWorker::run_notifyMountEvents()
    {
    LOG("Enter run_notifyMountEvents\n");
    m_PlatformNotifier = new FileSystem(this);
    if (m_PlatformNotifier)
        {
        LOG("m_PlatformNotifier is allocated\n");
        if ( !m_PlatformNotifier->notifyMountEvents() )
            {
            LOG("FileSystem::notifyMountEvents returned false\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, SYSTEM_ERROR_STR));
            }
        else
            {
            LOG("FileSystem::notifyMountEvents returned true, notification is active\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS, SUCCESS_STR, NOTIFICATION_DUMMY_ID));
            }
        }
    else
        {
        LOG("m_PlatformNotifier is NOT allocated\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, SYSTEM_ERROR_STR));
        }
    LOG("Exit run_notifyMountEvents\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_cancelNotify()
// cancels the notifcation request
// --------------------------------------------------------------------
void FileSystemWorker::run_cancelNotify()
    {
    LOG("Enter run_cancelNotify\n");
    if (m_PlatformNotifier)
        {
        LOG("m_PlatformNotifier is active, so cancel it\n");
        m_PlatformNotifier->cancelNotify();
        }
    LOG("Exit run_cancelNotify\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::newMountPointEvent()
// new point availability notifcation
// --------------------------------------------------------------------
void FileSystemWorker::newMountPointEvent(QMap<QString,QVariant> mountPoint)
    {
    LOG("Enter FileSystemWorker::newMountPointEvent\n");
    //some positive number to make it go as async call
    emit FileSystemWorkerSignal(m_task, 1, Utilities::ReturnMap(SUCCESS, SUCCESS_STR, mountPoint));
    LOG("Exit FileSystemWorker::newMountPointEvent\n");
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_getElementInfo()
// To get the properties of an element
// --------------------------------------------------------------------
void FileSystemWorker::run_getElementInfo()
    {
    LOG("***ENTER run_getElementInfo***\n");
    if (m_srcUri != NULL)
        {
        m_srcUri = validatePath(m_srcUri);
        if (m_srcUri == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        }

    if (ExtractPathFromUri(m_srcUri) == false)
        {
        LOG("INVALID_URI_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
        return;
        }

    LOG(QString("Extracted Path : %1\n").arg(m_srcUri));

    QFileInfo elementInfo(m_srcUri);
    if (elementInfo.exists() == false)
        {
        LOG("URI_NOT_FOUND_ERR\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(URI_NOT_FOUND_ERR, SOURCE_NOT_EXISTS_STR));
        return;
        }

    int elementType = -1;
    FileSystemMap fileInfoMap;

    if (elementInfo.isFile() == true)
        elementType = FILE_TYPE;
    else if (elementInfo.isDir() == true)
        elementType = DIRECTORY_TYPE;
    else if (elementInfo.isSymLink() == true)
        elementType = LINK_TYPE;

    fileInfoMap.insert(ELEMENT_URI, FILE_PROTOCOL + elementInfo.absolutePath() + "/" + elementInfo.fileName());
    fileInfoMap.insert(ELEMENT_TYPE,elementType);
    fileInfoMap.insert(ELEMENT_READONLY, elementInfo.isWritable() ? "false":"true");
    fileInfoMap.insert(ELEMENT_SIZE,(elementInfo.size() / (double) KILO_BYTE));
    fileInfoMap.insert(ELEMENT_MODIFIEDDATE,elementInfo.lastModified().toString());
    fileInfoMap.insert(ELEMENT_CREATEDATE, elementInfo.created().toString());

    LOG(QString("URI: %1 TYPE: %2 READONLY: %3 SIZE: %4 MODIFIEDDATE: %5 CREATEDATE: %6\n")
            .arg(FILE_PROTOCOL + elementInfo.absolutePath() + "/" + elementInfo.fileName())
            .arg(elementType)
            .arg(elementInfo.isWritable()?"false":"true")
            .arg(elementInfo.size()/(double)KILO_BYTE)
            .arg(elementInfo.lastModified().toString())
            .arg(elementInfo.created().toString())
    );

    LOG("*************EXIT run_getElementInfo******************\n");
    emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR,fileInfoMap));
    return;
    }

// --------------------------------------------------------------------
// FileSystemWorker::run_searchMatchObject()
// To search for matching items in the entire filesystem
// --------------------------------------------------------------------
void FileSystemWorker::run_searchMatchObject()
    {
    LOG("*************ENTER run_searchMatchObject******************\n");
    LOG(QString("search Start time: %1\n").arg(QDateTime::currentDateTime().toString()));

    QDateTime createDate;
    bool createDateSet = false;
    if (!m_MatchObject.value(CREATE_DATE_STR).isNull())
        {
        LOG("createDate is specified\n");
        createDateSet = true;
        createDate = m_MatchObject.value(CREATE_DATE_STR).toDateTime();
        LOG(QString("createDate: %1\n").arg(createDate.toString()));
        }
    else
        {
        createDateSet = false;
        LOG("createDate is NOT specified\n");
        }

    QDateTime modifyDate;
    bool modifyDateSet = false;
    if (!m_MatchObject.value(MODIFY_DATE_STR).isNull())
        {
        LOG("modifyDate is specified\n");
        modifyDateSet = true;
        modifyDate = m_MatchObject.value(MODIFY_DATE_STR).toDateTime();
        LOG(QString("modifyDate: %1\n").arg(modifyDate.toString()));
        }
    else
        {
        modifyDateSet = false;
        LOG("modifyDate is NOT specified\n");
        }

    qlonglong filesize = 0;
    bool filesizeSet = false;
    if (!m_MatchObject.value(FILE_SIZE_STR).isNull())
        {
        LOG("fileSize is specified\n");
        filesizeSet = true;
        filesize = m_MatchObject.value(FILE_SIZE_STR).toLongLong();
        LOG(QString("filesize: %1\n").arg(filesize));

        if (filesize<0)
            {
            LOG("DATA_OUT_OF_RANGE_ERR, filesize is -ve\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_OUT_OF_RANGE_ERR,INVALID_ARG_ERR_STR));
            return;
            }
        }
    else
        {
        filesizeSet = false;
        LOG("fileSize is NOT specified\n");
        }


    //Begin Identify all the paths we need to search
    QStringList searchPathList;
    searchPathList.clear();
    bool filePathSet = false;
    // file path will also be conforming to URI standards
    QString filePath = m_MatchObject.value(FILE_PATH_STR).toString();
    LOG(QString("File path specified: %1\n").arg(filePath));
    if(filePath != NULL)
        {
        LOG("filePath is not NULL\n");
        filePath = validatePath(filePath);
        
        if (filePath == NULL)
            {
            LOG("NOT_SUPPORTED_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(
                    NOT_ALLOWED_ERR, NOT_ALLOWED_ERR_STR));
            return;
            }
        
        if(ExtractPathFromUri(filePath) == false)
            {
            LOG("INVALID_URI_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_URI_ERR,INVALID_URI_ERR_STR));
            return;
            }
        }
    if (filePath.isNull())
        {
        LOG("filePath is NULL\n");
        filePathSet = false;
        //Set the default search paths
        searchPathList = defaultSearchPaths();
        }
    else
        {
        filePathSet = true;
        LOG("Path is set, so search only in the user supplied path\n");

        LOG(QString("filePath: %1\n").arg(filePath));

        QFileInfo elementInfo(filePath);
        if (elementInfo.exists() == false)
            {
            LOG("URI_NOT_FOUND_ERR\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(URI_NOT_FOUND_ERR,SOURCE_NOT_EXISTS_STR));
            return;
            }

        if (elementInfo.isDir() == false)
            {
            LOG("NOT_SUPPORTED_ERR, its not a folder\n");
            emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,ELEMENT_NOT_DIR_STR));
            return;
            }
        searchPathList.append(filePath);
        }
    //End Identify all the paths we need to search

    QDir searchdir;
    bool fileNameSet = false;
    if (!m_MatchObject.value(FILE_NAME_STR).isNull())
        {
        fileNameSet = true;
        LOG("match pattern filter is set\n");
        QString matchpattern = m_MatchObject.value(FILE_NAME_STR).toString();
        LOG(QString("matchpattern: %1\n").arg(matchpattern));
        QStringList matchPatternList(matchpattern);
        searchdir.setNameFilters(matchPatternList);
        }
    else
        {
        fileNameSet = false;
        LOG("filename is NOT specified\n");
        QStringList matchPatternList("*");
        searchdir.setNameFilters(matchPatternList);
        }

    bool isDirSet = false;
    if (!m_MatchObject.value(ISDIRECTORY_STR).isNull())
        {
        isDirSet = true;
        LOG("isDirectory filter is set\n");
        bool isDir = m_MatchObject.value(ISDIRECTORY_STR).toBool();
        if (true == isDir)
            {
            searchdir.setFilter(QDir::Dirs);
            }
        else
            {
            searchdir.setFilter(QDir::Files);
            }
        }
    else
        {
        isDirSet = false;
        LOG("isDirectory is NOT specified\n");
        }

    QList<QVariant> fileAttribArray;
    fileAttribArray.clear();
    QMap<QString, QVariant> searchResult;
    searchResult.clear();

    if ( !(createDateSet || modifyDateSet || filesizeSet ||
            filePathSet   || fileNameSet   || isDirSet)
    )
        {
        LOG("None of the keys in FILE map is set\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR, fileAttribArray));
        }

    try
    {
    foreach(QString path, searchPathList)
                {
                //for all the subdirectories
                LOG(QString("Searching in path from path list: %1\n").arg(path));
                QDirIterator dirIterator(path, QStringList("*"), QDir::Dirs|QDir::NoDotAndDotDot, QDirIterator::Subdirectories);

                QFileInfoList elementInfoList;
                elementInfoList.clear();

                //for the parent directory
                searchdir.setPath(path);
                bool lastdirtraversed = false;
                do
                    {
                    LOG(QString("Current search directory: %1\n").arg(searchdir.path()));

                    elementInfoList = searchdir.entryInfoList();

                    int len = elementInfoList.length();
                    if ( len > 0 )
                        {
                        LOG(QString("Matching content found %1\n").arg(len));
                        FileSystemMap fileInfoMap;
                        QVariant fileInfoMapVar;

                        for (int i =0; i < len; i++)
                            {
                            fileInfoMap.clear();
                            QFileInfo file = elementInfoList.at(i);


                            LOG(QString("Before filtering FILE_NAME: %1 FILE_PATH: %2 FILE_SIZE: %3 ISDIRECTORY: %4 MODIFIEDDATE: %5 CREATEDATE: %6\n")
                                    .arg(file.fileName())
                                    .arg(file.absolutePath())
                                    .arg(file.size()/(double)KILO_BYTE)
                                    .arg(file.isDir()? "True" : "False")
                                    .arg(file.lastModified().toString())
                                    .arg(file.created().toString())
                            );

                            if (createDateSet == true)
                                {
                                if ( createDate != file.created() )
                                    {
                                    LOG("createDate is not matching\n");
                                    continue;
                                    }
                                }

                            if (modifyDateSet == true)
                                {
                                if ( modifyDate != file.lastModified() )
                                    {
                                    LOG("modifyDate is not matching\n");
                                    continue;
                                    }
                                }

                            if (filesizeSet == true)
                                {
                                if (file.size() != filesize)
                                    {
                                    LOG("filesize is not matching\n");
                                    continue;
                                    }
                                }

                            //ultimately we got the result here
                            LOG(QString("After filtering FILE_NAME: %1 FILE_PATH: %2 FILE_SIZE: %3 ISDIRECTORY: %4 MODIFIEDDATE: %5 CREATEDATE: %6\n")
                                    .arg(file.fileName())
                                    .arg(file.absolutePath())
                                    .arg(file.size()/(double)KILO_BYTE)
                                    .arg(file.isDir()? "True" : "False")
                                    .arg(file.lastModified().toString())
                                    .arg(file.created().toString())
                            );

                            int elementType = -1;
                            //check for type of the element found
                            if (file.isFile() == true)
                                elementType = FILE_TYPE;
                            else if (file.isDir() == true)
                                elementType = DIRECTORY_TYPE;
                            else if (file.isSymLink() == true)
                                elementType = LINK_TYPE;

                            fileInfoMap.insert(ELEMENT_URI, FILE_PROTOCOL + file.absolutePath() + "/" + file.fileName());
                            fileInfoMap.insert(ELEMENT_TYPE, elementType);
                            fileInfoMap.insert(ELEMENT_READONLY, file.isWritable() ? "false" : "true");
                            fileInfoMap.insert(ELEMENT_SIZE, file.size() / (double) KILO_BYTE);
                            fileInfoMap.insert(ELEMENT_CREATEDATE, file.created().toString());
                            fileInfoMap.insert(ELEMENT_MODIFIEDDATE, file.lastModified().toString());

                            fileInfoMapVar.setValue(fileInfoMap);
                            fileAttribArray.append(fileInfoMapVar);
                            }
                        }
                    else
                        {
                        LOG("No matching items found in this directory\n");
                        }


                    if (!dirIterator.hasNext())
                        {
                        LOG("Last directory is reached.\n");
                        lastdirtraversed = true;
                        }
                    else
                        {
                        dirIterator.next();//get next folder
                        searchdir.setPath(dirIterator.fileInfo().absoluteFilePath());
                        lastdirtraversed = false;
                        }

                    }while( !lastdirtraversed ); //till the last directory gets traversed

                }//end of for search paths loop

        searchResult.insert(ELEMENT_STR, fileAttribArray);
        searchResult.insert(STATE_STR, COMPLETE);
    LOG(QString("Total matching items found: %1\n").arg(fileAttribArray.length()));
    LOG(QString("search End time: %1\n").arg(QDateTime::currentDateTime().toString()));
    emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR, searchResult));
    LOG("*************EXIT run_searchMatchObject******************\n");
    }
    catch(...)
        {
        LOG("Some exception has occured while searching, so send too many results exception\n");
        emit FileSystemWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_OUT_OF_RANGE_ERR,TOO_MANY_RESULTS_STR));

        }
    }

//end of file
