/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "fsioworker.h"
#include "fslog.h"
#include "fsutilities.h"
#include "fsconstants.h"
#include "fserrorcodes.h"

#include <QFile>

// --------------------------------------------------------------------
// FileIOWorker::FileIOWorker()
// constructor
// --------------------------------------------------------------------
FileIOWorker::FileIOWorker()
    {
    }

// --------------------------------------------------------------------
// FileIOWorker::FileIOWorker()
// Destructor
// --------------------------------------------------------------------
FileIOWorker::~FileIOWorker()
    {
    }

// --------------------------------------------------------------------
// FileIOWorker::FileIOWorker()
// constructor
// --------------------------------------------------------------------
FileIOWorker::FileIOWorker(int task, QFile** file, int cbId , \
        int maxLength, int pos) \
        :m_task(task), m_file(file), m_cbId(cbId),\
        m_maxLength(maxLength), m_pos(pos), m_data(""), m_encoding("")
                {
                }

// --------------------------------------------------------------------
// FileIOWorker::run()
// To run the actual method to perform a file system task.
// --------------------------------------------------------------------
void FileIOWorker::run()
    {
    LOG("Inside FileIOWorker Run\n");
    switch (m_task)
        {
        case CLOSE_METHOD:
            run_close();
            break;
        case READ_METHOD:
            run_read();
            break;
        case READLINE_METHOD:
            run_readLine();
            break;
        case READBASE64_METHOD:
            run_readBase64();
            break;
        case WRITE_METHOD:
            run_write();
            break;
        case WRITELINE_METHOD:
            run_writeLine();
            break;
        case WRITEBASE64_METHOD:
            run_writeBase64();
            break;
        case FLUSH_METHOD:
            run_flush();
            break;
        case SEEK_METHOD:
            run_seek();
            break;
        default:
            {
            //it should never come here
            LOG("NO api matched\n");
            }
        }
    }

// --------------------------------------------------------------------
// FileIOWorker::run_close()
// To close an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_close()
    {
    LOG("ENTER run_close\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    LOG("handle is valid, closing the file\n");

    (*m_file)->close();
    *m_file = NULL;
    emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR));

    LOG("EXIT run_close\n");
    }

// --------------------------------------------------------------------
// FileIOWorker::run_read()
// To read from an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_read()
    {
    LOG("ENTER run_read\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    if (m_maxLength < 0)
        {
        LOG("DATA_OUT_OF_RANGE_ERR, passed length is negative\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_OUT_OF_RANGE_ERR,INVALID_ARG_ERR_STR));
        return;
        }
    /* assuming it will never be directly used from qt
    if (m_pos < 0)
        {
        LOG("DATA_OUT_OF_RANGE_ERR, passed postion is negative\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_OUT_OF_RANGE_ERR,INVALID_ARG_ERR_STR));
        return;
        }*/

    LOG("handle, maxlength, pos are valid, performing read\n");

    QString value("");
    if ((*m_file)->isReadable())
        {
        if (m_pos != -1)
            {
            int currPos = (*m_file)->pos();
            LOG(QString("Curre pos in file before seek: %1\n").arg(currPos));
            //bytes available will give the remaining bytes from current position
            int endPos = (*m_file)->bytesAvailable() + currPos;
            LOG(QString("Total bytes Available in file: %1\n").arg(endPos));
            if (m_pos > endPos)
                m_pos = endPos;
            if ( !( (*m_file)->seek(m_pos) ) )//if seek fails
                {
                LOG("NOT_SUPPORTED_ERR, Seek operation failed\n");
                emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,SEEK_OP_FAILED_STR));
                return;
                }
            }

        int dataSize = (*m_file)->bytesAvailable();
        if (m_maxLength > dataSize)
            m_maxLength = dataSize;

        QByteArray buf = (*m_file)->read(m_maxLength);
        value = QString(buf);
        LOG(QString("string read : %1\n").arg(value));
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR,value));
        }
    else
        {
        LOG("NOT_SUPPORTED_ERR, File is not readable\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,MODE_IS_INVALID_STR));
        }
    LOG("EXIT run_read\n");
    }

// --------------------------------------------------------------------
// FileIOWorker::run_readLine()
// To read a line from an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_readLine()
    {
    LOG("ENTER run_readLine\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    if (m_maxLength < 0)
        {
        LOG("DATA_OUT_OF_RANGE_ERR, passed length is negative\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_OUT_OF_RANGE_ERR,INVALID_ARG_ERR_STR));
        return;
        }

    LOG("handle is valid, performing readLine\n");

    QString value("");
    if ((*m_file)->isReadable())
        {
        int dataSize = (*m_file)->bytesAvailable();
        if (m_maxLength > dataSize)
            m_maxLength = dataSize;
        QByteArray buf = (*m_file)->readLine(m_maxLength + 1);
        value = QString(buf);
        LOG(QString("string read : %1\n").arg(value));
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR,value));
        LOG("EXIT run_readLine\n");
        }
    else
        {
        LOG("NOT_SUPPORTED_ERR, File is not readable\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,MODE_IS_INVALID_STR));
        }

    }

// --------------------------------------------------------------------
// FileIOWorker::run_readBase64()
// To read data from an open file in base64 format.
// --------------------------------------------------------------------
void FileIOWorker::run_readBase64()
    {
    LOG("ENTER run_readBase64\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    if (m_maxLength < 0)
        {
        LOG("DATA_OUT_OF_RANGE_ERR, passed length is negative\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_OUT_OF_RANGE_ERR,INVALID_ARG_ERR_STR));
        return;
        }

    LOG("handle, maxlength, pos are valid, performing read\n");

    QString value("");
    if ((*m_file)->isReadable())
        {
        int dataSize = (*m_file)->bytesAvailable();
        if (m_maxLength > dataSize)
            m_maxLength = dataSize;
        QByteArray buf = (*m_file)->read(m_maxLength);
        value = QString(buf.toBase64());
        LOG(QString("string read : %1\n").arg(value));
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR,value));
        }
    else
        {
        LOG("NOT_SUPPORTED_ERR, File is not readable\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,MODE_IS_INVALID_STR));
        }

    }

// --------------------------------------------------------------------
// FileIOWorker::run_write()
// To write data into an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_write()
    {
    LOG("ENTER run_write\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    LOG("handle is valid, performing write\n");

    /*if(m_pos < 0) //assuming it will never be directly used from qt
        {
        LOG("DATA_OUT_OF_RANGE_ERR, passed postion is negative\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_OUT_OF_RANGE_ERR,INVALID_ARG_ERR_STR));
        return;
        }*/

    if ((*m_file)->isWritable())
        {
        if (m_pos != -1)
            {
            int currPos = (*m_file)->pos();
            LOG(QString("Curre pos in file before seek: %1\n").arg(currPos));
            //bytes available will give the remaining bytes from current position
            int endPos = (*m_file)->bytesAvailable() + currPos;
            LOG(QString("Total bytes Available in file: %1\n").arg(endPos));
            if (m_pos > endPos)
                m_pos = endPos;
            if ( !( (*m_file)->seek(m_pos) ) )//if seek fails
                {
                LOG("NOT_SUPPORTED_ERR, Seek operation failed\n");
                emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,SEEK_OP_FAILED_STR));
                return;
                }
            }

        int ret = (*m_file)->write(m_data.toAscii());
        if (ret != -1)
            {
            emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR));
            LOG("write successful\n");
            }
        else
            {
            LOG("NOT_SUPPORTED_ERR, Write operation failed\n");
            emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,WRITE_OP_FAILED_STR));
            }
        }
    else
        {
        LOG("NOT_SUPPORTED_ERR, File is not writable\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,MODE_IS_INVALID_STR));
        }

    }

// --------------------------------------------------------------------
// FileIOWorker::run_writeLine()
// To write data with a new line into an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_writeLine()
    {
    LOG("ENTER run_writeLine\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    LOG("handle is valid, performing writeLine\n");

    if ((*m_file)->isWritable())
        {
        int ret = (*m_file)->write(m_data.toAscii());
        if (ret != -1)
            {
            int newLineRet = (*m_file)->write("\n");
            if (newLineRet != -1)
                {
                emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR));
                LOG("write successful\n");
                }
            else
                {
                LOG("NOT_SUPPORTED_ERR, Write operation failed for new line\n");
                emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,WRITE_OP_FAILED_STR));
                }

            }
        else
            {
            LOG("NOT_SUPPORTED_ERR, Write operation failed\n");
            emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,WRITE_OP_FAILED_STR));
            }
        }
    else
        {
        LOG("NOT_SUPPORTED_ERR, File is not writable\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,MODE_IS_INVALID_STR));
        }

    }
// --------------------------------------------------------------------
// FileIOWorker::run_writeBase64()
// convert base64 data into bytes and then write to an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_writeBase64()
    {
    LOG("ENTER run_writeBase64\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    LOG("handle is valid, performing write\n");

    if ((*m_file)->isWritable())
        {
        int ret = (*m_file)->write(QByteArray::fromBase64(m_data.toAscii()));
        if (ret != -1)
            {
            emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR));
            LOG("writeBase64 successful\n");
            }
        else
            {
            LOG("WriteBase64 operation failed\n");
            emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,WRITE_OP_FAILED_STR));
            }
        }

    else
        {
        LOG("DATA_NOT_FOUND_ERR, File is not writable\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR,MODE_IS_INVALID_STR));
        }

    }
// --------------------------------------------------------------------
// FileIOWorker::run_flush()
// To perform flush on an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_flush()
    {
    LOG("ENTER run_flush\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    LOG("handle is valid, flushing the file\n");

    if ( (*m_file)->flush() == false )
        {
        LOG("flush FAILED\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, FLUSH_OP_FAILED_STR));
        return;
        }

    LOG("flush SUCCESS\n");
    emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR));
    LOG("EXIT run_flush\n");
    return;
    }


// --------------------------------------------------------------------
// FileIOWorker::run_seek()
// To perform seek on an open file.
// --------------------------------------------------------------------
void FileIOWorker::run_seek()
    {
    LOG("ENTER run_seek\n");

    if ( (!m_file) || (*m_file == NULL) )
        {
        LOG("DATA_NOT_FOUND_ERR, File already closed\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(DATA_NOT_FOUND_ERR,HANDLE_NOT_FOUND_ERR_STR));
        return;
        }

    LOG("handle is valid, performing seek\n");
    int seekOption = m_maxLength;

    if ( (seekOption != SEEKSTART) && (seekOption != SEEKCURRENT) && (seekOption != SEEKEND) )
        {
        LOG("INVALID_ARG_ERR, seekoption is not valid\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(INVALID_ARG_ERR, INVALID_ARG_ERR_STR));
        return;
        }

    int currPos = (*m_file)->pos();
    LOG(QString("Curre pos in file before seek: %1\n").arg(currPos));

    //bytes available will give the remaining bytes from current position
    int endPos = (*m_file)->bytesAvailable() + currPos;
    LOG(QString("Total bytes Available in file: %1\n").arg(endPos));

    int newPos = 0;
    if (seekOption == SEEKSTART)
        {
        LOG("From Begining\n");
        newPos = 0 + m_pos;
        }
    else if (seekOption == SEEKCURRENT)
        {
        LOG("From Current\n");
        if (m_pos == 0)
            {
            //no need of any seek, user is just querying for current position
            LOG("User is interseted only in current position");
            emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR, currPos));
            return;
            }
        newPos = currPos + m_pos;
        }
    else if (seekOption == SEEKEND)
        {
        LOG("From End\n");
        newPos = endPos + m_pos;
        }

    LOG(QString("computed new position : %1\n").arg(newPos));

    if (newPos < 0)
        {
        LOG("New position falls before begining of the file\n");
        newPos = 0;
        }
    else if (newPos > endPos)
        {
        LOG("New position falls after the end of the file\n");
        newPos = endPos;
        }
    else
        {
        LOG("New position is within the file range\n");
        }

    if (newPos == currPos)
        {
        LOG("new computed position is equal to current position, no need of seek\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS, SUCCESS_STR, currPos));
        return;
        }


    LOG(QString("final new position : %1\n").arg(newPos));
    if ( (*m_file)->seek(newPos) == false )
        {
        LOG("DATA_OUT_OF_RANGE_ERR, seekoption is not valid\n");
        emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(NOT_SUPPORTED_ERR, SEEK_OP_FAILED_STR));
        return;
        }

    LOG("seek SUCCESS\n");
    currPos = (*m_file)->pos();
    LOG(QString("Current pos in file after seek: %1\n").arg(currPos));

    emit FileIOWorkerSignal(m_task, m_cbId, Utilities::ReturnMap(SUCCESS,SUCCESS_STR, currPos));
    LOG("EXIT run_seek\n");
    }
