/*
* ============================================================================
*  Name        : referenceService.cpp
*  Part of     : serviceframework / WRT
*  Description : Qt class for reference service
*  Version     : %version: 1 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QtCore>
#include <QDebug>
#include <QMap>
#include "referenceservice.h"
#include "secsession.h"

/*!
    \class ReferenceService

    \brief An example servie provider for Service Framework

    This reference service provider is meant to be an example service provider for people who want to develop their own service
    providers. It has 3 public slots/methods that web app calls to demonstrate how synchronous/asynchronous mothod call work.
    The private function  printQVariant() demonstrates how to decode a QVarint to determine its actual type.
*/

/*!
    Default constructor
 */
ReferenceService::ReferenceService()
{
}

/*!
    Destructor
 */
ReferenceService::~ReferenceService()
{
}

/*!
    \see IServiceSecurity::setSecuritySession(WRT::SecSession *secSession);
 */
void ReferenceService::setSecuritySession(WRT::SecSession *aSecSession)
{
    iSecSession = aSecSession;
}

/*!
    This method receives a parameter of type QVariant from client applications.
    The content of the QVarinat object is decoded and printed to the standard output.
    Please see ReferenceService::printQVariant() to see how a QVariant object can
    be decoded into its actual type.

    @param (JavaScript) aQVariantMap represents a JSObject passed from web app
    @return (JavaScript) aQVariantMap that will be converted to a JSObject by Service Framework before being\n
                         passed back to web app
 */
QVariant ReferenceService::testMapSync(const QVariant& aQVariantMap)
{
    QByteArray out;
    printQVariant(aQVariantMap, out);
    qDebug() << out;
    return aQVariantMap;
}

/*!
    This method receives a parameter of type QVariant from client applications.
    The content of the QVarinat object is decoded and printed to the standard output.
    Please see ReferenceService::printQVariant() to see how a QVariant object can
    be decoded into its actual type.

    @param (JavaScript) aQVariantList represents an array passed from web app
    @return (JavaScript) aQVariantList that will be converted to a JS array by Service Framework before being\n
                         passed back to web app
 */
QVariant ReferenceService::testListSync(const QVariant& aQVariantList)
{
    QByteArray out;
    printQVariant(aQVariantList, out);
    qDebug() << out;
    return aQVariantList;
}

/*!
    This method receives an integer parameter from client applications and returns it asynchronously.
    This is to show how asynchronous callback works by emitting a Qt signal along with the value(s) to be
    passed back.

    @param (JavaScript) aInt represents an integer in JS
 */
void ReferenceService::testIntAsync(int aInt)
{
    emit testIntAsyncCallback(aInt);
}

/*!
    This method receives an integer parameter from client applications and returns it synchronously.
    This method is introduced in com.nokia.IRefTypeTest version 2.0 to show how a service provider can implement multiple versions of an interface

    @param (JavaScript) aInt represents an integer in JS
    @return (JavaScript) aInt input parameter
 */
int ReferenceService::testIntSync(int aInt)
{
    return aInt;
}


/*!
    Serialize the content of a QVariant to a QByteArray.
    The content of QByteArray can later be printed out to show the data stored inside the
    QVariant object. The main purpose of this method is to show how a QVariant,
    coming from client applications, can be decoded into its actual type.

    @param aQVariant Object to be decoded
    @param aOut a byte array to store the content of the decoded object
    @param aLevel indicates how deep aQVariant is embedded in another QVariantMap object
 */
void ReferenceService::printQVariant(const QVariant& aQVariant, QByteArray& aOut, int aLevel)
{
    switch (aQVariant.type()) {
    case QVariant::Bool:
        {
        if (aQVariant.toBool())
            aOut.append("True");
        else
            aOut.append("False");
        }
        break;
    case QVariant::Int:
        aOut.append(QString::number(aQVariant.toInt()).toAscii());
        break;
    case QVariant::Double:
        aOut.append(QString::number(aQVariant.toDouble()).toAscii());
        break;
    case QVariant::String:
        aOut.append(aQVariant.toString().toAscii());
        break;
    // In JavaScript, this is equivalent to a JSObject
    case QVariant::Map:
        {
            aLevel++;
            QVariantMap map = aQVariant.toMap();
            QVariantMap::const_iterator i = map.begin();
            while (i!=map.end())
            {
                aOut.append("\n");
                for (int j=0; j<aLevel; j++)
                    aOut.append("    ");
                aOut.append("<"+i.key().toAscii()+"> ");
                printQVariant(i.value(), aOut, aLevel);
                ++i;
            }
            aLevel--;
        }
        break;
    // In JavaScript, this is equivalent to an array
    case QVariant::List:
        {
            QList<QVariant> list = aQVariant.toList();
            for (int k=0; k<list.count(); k++)
            {
                aOut.append("\n");
                for (int j=0; j<aLevel; j++)
                {
                    aOut.append("    ");
                }
                aOut.append("List["+QString::number(k).toAscii()+"]=");
                printQVariant(list[k], aOut, aLevel);
            }
        }
        break;
    default:
        break;
    }
}

/*!
    \fn ReferenceService::testIntAsyncCallback(int ret)

    Signal emitted when the testIntAsync() asynchronous request is completed
    @param (JavaScript) ret will be converted to a JavaScript integer value before being passed back to web app
    @return void
*/
