/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "qjsonizer.h"

#include <QStringList>
#include <QMap>
#include <QtDebug>

JSonizer::JSonizer()
{
    m_contactMap.clear();
}

JSonizer::~JSonizer() {
}

QVariant JSonizer::contact2json(const QContact *contact)
{
    QVariant var = NULL;

    nameFields2json(contact);
    telFields2json(contact);
    addressFields2json(contact);
    companyFields2json(contact);
    miscFields2json(contact);

    var = m_contactMap;

    return var;
}

void JSonizer::nameFields2json(const QContact * contact)
{
    QContactName name = contact->detail(QContactName::DefinitionName);
    // Insert to Contact map
    QMap<QString,QVariant> valueMap;
    valueMap.insert("last",QVariant(name.lastName()));
    valueMap.insert("first",QVariant(name.firstName()));
    valueMap.insert("middle",QVariant(name.middleName()));
    valueMap.insert("prefix",QVariant(name.prefix()));
    valueMap.insert("suffix",QVariant(name.suffix()));

    cleanContactMap(&valueMap);
    if (!valueMap.isEmpty()) {
        m_contactMap.insert(QString("name"),QVariant(valueMap));
    }
}

void JSonizer::telFields2json(const QContact * contact)
{
    QContactPhoneNumber tel = contact->detail(QContactPhoneNumber::DefinitionName);

    QMap<QString,QVariant> valueMap;
    // Insert to Contact map
    tel.setSubTypes(QContactPhoneNumber::SubTypeLandline);
    valueMap.insert("land",QVariant(tel.number()));

    tel.setSubTypes(QContactPhoneNumber::SubTypeMobile);
    valueMap.insert("mobile",QVariant(tel.number()));

    tel.setSubTypes(QContactPhoneNumber::SubTypeVideo);
    valueMap.insert("video",QVariant(tel.number()));

    tel.setSubTypes(QContactPhoneNumber::SubTypeModem);
    valueMap.insert("fax",QVariant(tel.number()));

    tel.setSubTypes(QContactPhoneNumber::SubTypeVoice);
    valueMap.insert("voip",QVariant(tel.number()));

    // Insert to contact map
    cleanContactMap(&valueMap);
    if (!valueMap.isEmpty()) {
        m_contactMap.insert(QString("tel"),QVariant(valueMap));
    }
    QMap<QString,QVariant> homeMap;
    QMap<QString,QVariant> tmpMap;
    QContactPhoneNumber tel2 = contact->detail(QContactPhoneNumber::ContextHome);

    tel2.setSubTypes(QContactPhoneNumber::SubTypeLandline);
    tmpMap.insert("land",QVariant(tel2.number()));
    homeMap.insert("home",QVariant(tmpMap));

    tel2.setSubTypes(QContactPhoneNumber::SubTypeMobile);
    tmpMap.insert("mobile",QVariant(tel2.number()));
    homeMap.insert("home",QVariant(tmpMap));

    tel2.setSubTypes(QContactPhoneNumber::SubTypeVideo);
    tmpMap.insert("video",QVariant(tel2.number()));
    homeMap.insert("home",QVariant(tmpMap));

    tel2.setSubTypes(QContactPhoneNumber::SubTypeModem);
    tmpMap.insert("fax",QVariant(tel2.number()));
    homeMap.insert("home",QVariant(tmpMap));

    tel2.setSubTypes(QContactPhoneNumber::SubTypeVoice);
    tmpMap.insert("voip",QVariant(tel2.number()));
    homeMap.insert("home",QVariant(tmpMap));

    cleanContactMap(&tmpMap);
    if (!tmpMap.isEmpty()) {
        m_contactMap.insertMulti(QString("tel.home"),QVariant(tmpMap));
    }
    tmpMap.clear();

    QContactPhoneNumber tel3 = contact->detail(QContactPhoneNumber::ContextWork);

    tel3.setSubTypes(QContactPhoneNumber::SubTypeLandline);
    tmpMap.insert("land",QVariant(tel3.number()));
    homeMap.insert("work",QVariant(tmpMap));

    tel3.setSubTypes(QContactPhoneNumber::SubTypeMobile);
    tmpMap.insert("mobile",QVariant(tel3.number()));
    homeMap.insert("work",QVariant(tmpMap));

    tel3.setSubTypes(QContactPhoneNumber::SubTypeVideo);
    tmpMap.insert("video",QVariant(tel3.number()));
    homeMap.insert("work",QVariant(tmpMap));

    tel3.setSubTypes(QContactPhoneNumber::SubTypeModem);
    tmpMap.insert("fax",QVariant(tel3.number()));
    homeMap.insert("work",QVariant(tmpMap));

    tel3.setSubTypes(QContactPhoneNumber::SubTypeVoice);
    tmpMap.insert("voip",QVariant(tel3.number()));
    homeMap.insert("work",QVariant(tmpMap));

    // Insert numbers
    cleanContactMap(&tmpMap);
    if (!tmpMap.isEmpty()) {
        m_contactMap.insertMulti(QString("tel.work"),QVariant(tmpMap));
    }
}

void JSonizer::addressFields2json(const QContact * contact)
{
    QContactAddress address = contact->detail(QContactAddress::DefinitionName);
    QMap<QString,QVariant> valueMap;
    valueMap.insert("street",QVariant(address.street()));
    valueMap.insert("local",QVariant(address.locality()));
    valueMap.insert("region",QVariant(address.region()));
    valueMap.insert("code",QVariant(address.postcode()));
    valueMap.insert("country",QVariant(address.country()));

    QContactEmailAddress email = contact->detail(QContactEmailAddress::FieldContext);
    valueMap.insert("email",QVariant(email.emailAddress()));

    QContactUrl url = contact->detail(QContactUrl::FieldContext);
    valueMap.insert("uri",QVariant(url.url()));

    // Insert to contact map
    cleanContactMap(&valueMap);
    if (!valueMap.isEmpty()) {
        m_contactMap.insert(QString("address"),QVariant(valueMap));
    }

    QMap<QString,QVariant> addressMap;
    QMap<QString,QVariant> tmpMap;
    // Home
    QContactAddress address2 = contact->detail(QContactAddress::ContextHome);

    tmpMap.insert("street",QVariant(address2.street()));
    addressMap.insert("home",QVariant(tmpMap));
    tmpMap.insert("local",QVariant(address2.locality()));
    addressMap.insert("home",QVariant(tmpMap));
    tmpMap.insert("region",QVariant(address2.region()));
    addressMap.insert("home",QVariant(tmpMap));
    tmpMap.insert("code",QVariant(address2.postcode()));
    addressMap.insert("home",QVariant(tmpMap));
    tmpMap.insert("country",QVariant(address2.country()));
    addressMap.insert("home",QVariant(tmpMap));

    QContactEmailAddress email2 = contact->detail(QContactEmailAddress::ContextHome);
    tmpMap.insert("email",QVariant(email2.emailAddress()));
    addressMap.insert("home",QVariant(tmpMap));

    QContactUrl url2 = contact->detail(QContactUrl::ContextHome);
    tmpMap.insert("uri",QVariant(url2.url()));
    addressMap.insert("home",QVariant(tmpMap));

    cleanContactMap(&tmpMap);
    if (!tmpMap.isEmpty()) {
        m_contactMap.insert(QString("address.home"),QVariant(tmpMap));
    }
    tmpMap.clear();

    // Work
    QContactAddress address3 = contact->detail(QContactAddress::ContextWork);

    tmpMap.insert("street",QVariant(address3.street()));
    addressMap.insert("work",QVariant(tmpMap));
    tmpMap.insert("local",QVariant(address3.locality()));
    addressMap.insert("work",QVariant(tmpMap));
    tmpMap.insert("region",QVariant(address3.region()));
    addressMap.insert("work",QVariant(tmpMap));
    tmpMap.insert("code",QVariant(address3.postcode()));
    addressMap.insert("work",QVariant(tmpMap));
    tmpMap.insert("country",QVariant(address3.country()));
    addressMap.insert("work",QVariant(tmpMap));

    QContactEmailAddress email3 = contact->detail(QContactEmailAddress::ContextWork);
    tmpMap.insert("email",QVariant(email3.emailAddress()));
    addressMap.insert("work",QVariant(tmpMap));

    QContactUrl url3 = contact->detail(QContactUrl::ContextWork);
    tmpMap.insert("uri",QVariant(url3.url()));
    addressMap.insert("work",QVariant(tmpMap));

    // Insert addresses
    cleanContactMap(&tmpMap);
    if (!tmpMap.isEmpty()) {
        m_contactMap.insert(QString("address.work"),QVariant(tmpMap));
    }
}

void JSonizer::companyFields2json(const QContact * contact)
{
    QContactOrganization organization = contact->detail(QContactOrganization::DefinitionName);
    QMap<QString,QVariant> valueMap;
    valueMap.insert("name",QVariant(organization.name()));
    valueMap.insert("title",QVariant(organization.logoUrl()));

    // Insert company information
    cleanContactMap(&valueMap);
    if (!valueMap.isEmpty()) {
        m_contactMap.insert(QString("company"),QVariant(valueMap));
    }
}

void JSonizer::miscFields2json(const QContact * contact)
{
    m_contactMap.insert(QString("id"),QVariant(contact->id().localId()));

    QContactNote note = contact->detail(QContactNote::DefinitionName);
    if (!isValueEmptyOrNull(QVariant(note.note()))) {
        m_contactMap.insert(QString("notes"),QVariant(note.note()));
    }
    QContactAnniversary anniversary = contact->detail(QContactAnniversary::DefinitionName);
    if (!isValueEmptyOrNull(QVariant(anniversary.originalDate()))) {
        m_contactMap.insert(QString("anniversary"),QVariant(anniversary.originalDate()));
    }
    QContactBirthday birthday = contact->detail(QContactBirthday::DefinitionName);
    if (!isValueEmptyOrNull(QVariant(birthday.date()))) {
        m_contactMap.insert(QString("birthday"),QVariant(birthday.date().toString("ddd MMMM d yyyy")));
    }
    QContactNickname nickname = contact->detail(QContactNickname::DefinitionName);
    if (!isValueEmptyOrNull(QVariant(nickname.nickname()))) {
        m_contactMap.insert(QString("nickname"),QVariant(nickname.nickname()));
    }
    QContactAvatar avatar = contact->detail(QContactAvatar::DefinitionName);
    if (!isValueEmptyOrNull(QVariant(avatar.imageUrl()))) {
        m_contactMap.insert(QString("photo"),QVariant(avatar.imageUrl()));
    }
    QContactDisplayLabel displayName = contact->detail(QContactDisplayLabel::DefinitionName);
    if (!isValueEmptyOrNull(QVariant(displayName.label()))) {
        m_contactMap.insert(QString("displayName"),QVariant(displayName.label()));
    }
    QList<QVariant> xspidList;
    QContactGeoLocation position = contact->detail(QContactGeoLocation::DefinitionName);
    if (!isValueEmptyOrNull(QVariant(position.label()))) {
        xspidList.append(position.label());
        xspidList.append(QString::number(position.latitude()));
        xspidList.append(QString::number(position.longitude()));
        xspidList.append(QString::number(position.altitude()));
        xspidList.append(QString::number(position.accuracy()));
        xspidList.append(QString::number(position.altitudeAccuracy()));
        xspidList.append(QString::number(position.heading()));
        xspidList.append(QString::number(position.speed()));
        xspidList.append(position.timestamp().toString());
        // Append strings to xspid key
        m_contactMap.insert(QString("xspid"),QVariant(xspidList));
    }
}

void JSonizer::cleanContactMap(QMap<QString,QVariant>* dataMap)
{
    QMap<QString,QVariant>::iterator begin = dataMap->begin();
    while (begin != dataMap->end()) {
        //qDebug() << "key: " << QString(begin.key()) << " value: " << dataMap->value(begin.key()).toString();
        if (dataMap->value(begin.key()).toString().isEmpty() ||
                    dataMap->value(begin.key()).isNull()) {
            dataMap->remove(begin.key());
        }
        ++begin;
    }
}

bool JSonizer::isValueEmptyOrNull(QVariant value)
{
    bool isEmptyOrNull = false;
    //qDebug() << "isValueEmptyOrNull: " << value;
    if (value.isNull() || value.toString().isEmpty()) {
        isEmptyOrNull = true;
    }
    return isEmptyOrNull;
}

