/*
 * ============================================================================
 *  Name        : qcommlogmodel.cpp
 *  Part of     : serviceframework / WRT
 *  Description : Model Event class for CommLog service provider
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QTime>
#include <QDebug>

#include "qcommlogmodel.h"
#include "qcommlogerrorcodes.h"
#include "qcommlogkeyconstants.h"



uint CommHistoryModel::m_modelSerial = 0;
//---------------------------CommHistoryModel----------------------------------------------------
CommHistoryModel::CommHistoryModel()
{}

QString CommHistoryModel::newObjectPath()
{
    return QString("/CommHistoryModel") + QString::number(m_modelSerial++);
}

CommHistoryModel::~CommHistoryModel()
{
}

void CommHistoryModel::fillEventFromModel(RTComElEvent *model, Event &event)
{

    double duration = difftime(RTCOM_EL_EVENT_GET_FIELD(model,storage_time),RTCOM_EL_EVENT_GET_FIELD(model,start_time));

    event.setId(RTCOM_EL_EVENT_GET_FIELD(model, id));
    event.setServiceId((Event::EventServiceId)RTCOM_EL_EVENT_GET_FIELD(model,service_id));
    event.setTypeId((Event::EventTypeId)RTCOM_EL_EVENT_GET_FIELD(model, event_type_id));

    event.setStorageTime(QTime(0,0,duration));
    event.setStartTime(QDateTime::fromTime_t(RTCOM_EL_EVENT_GET_FIELD(model,start_time)));
    event.setEndTime(QDateTime::fromTime_t(RTCOM_EL_EVENT_GET_FIELD(model,end_time)));
    event.setIsRead(RTCOM_EL_EVENT_GET_FIELD(model,is_read));
    event.setFlags((Event::EventFlags)RTCOM_EL_EVENT_GET_FIELD(model,flags));
    event.setBytesSent(RTCOM_EL_EVENT_GET_FIELD(model,bytes_sent));
    event.setBytesReceived(RTCOM_EL_EVENT_GET_FIELD(model,bytes_received));
    event.setLocalUid(RTCOM_EL_EVENT_GET_FIELD(model,local_uid));
    event.setLocalName(RTCOM_EL_EVENT_GET_FIELD(model,local_name));
    event.setRemoteUid(RTCOM_EL_EVENT_GET_FIELD(model,remote_uid));
    event.setRemoteName(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,remote_name)));
    event.setRemoteEbookUid(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,remote_ebook_uid)));
    event.setChannel(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,channel)));
    event.setFreeText(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,free_text)));
    event.setGroupUid(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,group_uid)));
    event.setService(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,service)));
    event.setEventType(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,event_type)));
    event.setAdditionalText(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,additional_text)));
    event.setIconName(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,icon_name)));
    event.setPangoMarkup(QString::fromUtf8(RTCOM_EL_EVENT_GET_FIELD(model,pango_markup)));
#ifndef RTCOM_OLD
    event.setDirection(Event::EventDirection(RTCOM_EL_EVENT_GET_FIELD(model,outgoing)));
#else
    //Using librtcom-eventlogger_0.8
    switch (event.typeId() )
    {
       case Event::CallInbound:
       case Event::CallMissed:
       case Event::SmsInbound:
       case Event::ChatInbound:
       //Voicemail is inbound always in this case
       case Event::CallVoicemail:
          event.setDirection(Event::Inbound);
          break;
       case Event::CallOutbound:
       case Event::SmsOutbound:
       case Event::ChatOutbound:
          event.setDirection(Event::Outbound);
          break;
       default:
          event.setDirection(Event::UnknownDirection);
          break;
    }
#endif

}

QVariantMap CommHistoryModel::fillMessageData(const Event &aEvent)
{

    QVariantMap msg;
    //logId
    msg.insert(KLLogId, aEvent.getId());
    //type and summary
    switch ( aEvent.getServiceId())
    {
        case Event::CallService:
            msg.insert(KLType, KLCall);
            msg.insert(KLSummary, KLSummaryCall);
            //duration
            msg.insert(KLDuration,aEvent.getStorageTime());
            break;
        case Event::IMService:
            break;
        case Event::SMSService:
            msg.insert(KLType, KLSms);
            msg.insert(KLSummary, KLSummarySMS);
            //duration for SMS isn't applicable
            msg.insert(KLDuration,QTime(0,0,0));
            break;
        default:
            qDebug() << "Set type" << msg.value(KLType, "Unknown");
            break;
    }
    //startTime
    msg.insert(KLTime, aEvent.getStartTime());
    if ( aEvent.getEndTime() != QDateTime::fromTime_t(0))
    {
        msg.insert(KLTime, aEvent.getEndTime());
    }
    //Take phone number // QString phoneNumber = aEvent.remoteUid();
    msg.insert(KLPhone, aEvent.getRemoteUid());
    //Take contact name
    msg.insert(KLContactName,aEvent.getRemoteName());
    //flag (received, outgoing, missed)
    switch (aEvent.getDirection())
    {
        case Event::Inbound:
            msg.insert(KLFlag, KLReceived);
            if (aEvent.getTypeId() == Event::CallMissed)
            {
                msg.insert(KLFlag,KLMissed);
            }
            break;
        case Event::Outbound:
            msg.insert(KLFlag, KLOutgoing);
            break;
        default:
            qDebug() << "Set flag" << msg.value(KLFlag, "Unknown");
            break;
    }

    return msg;
}

//TODO check function
 bool CommHistoryModel::getEvent(RTComEl *dbElement, int aEventId, Event &aEvent)
 {
     qDebug()<<__PRETTY_FUNCTION__<<"new event was happened. aEventId = "<<aEventId;
     bool res = true;
     RTComElEvent *event = rtcom_el_event_new();
     RTComElQuery *query = rtcom_el_query_new(dbElement);
     RTComElIter *iterator;

     //Take event with id = aEventId
     if ( !rtcom_el_query_prepare(query,kid.toAscii().data(),aEventId,RTCOM_EL_OP_EQUAL,NULL))
     {
          qDebug()<<__FUNCTION__<<": Query isn't created";
          //Release memory
          rtcom_el_event_free_contents(event);
          g_object_unref(query);
          rtcom_el_event_free(event);
          return false;
     }
     iterator = rtcom_el_get_events(dbElement,query);
     if (!iterator)
     {
         qDebug() <<__FUNCTION__<< ": Failed to get iterator";
          //Release memory
          rtcom_el_event_free_contents(event);
          g_object_unref(query);
          rtcom_el_event_free(event);
          return true;//check it
     }

     if (!rtcom_el_iter_first(iterator))
     {
         qDebug() << "CommHistorySmsModel::getEvents. Failed to start iterator.";
          //Release memory
          rtcom_el_event_free_contents(event);
          g_object_unref(query);
          g_object_unref(iterator);
          rtcom_el_event_free(event);
          return false;

     }
 #ifndef RTCOM_OLD
     res = rtcom_el_iter_get_full(iterator,event);
 #else
     res = rtcom_el_iter_get(iterator,event);
 #endif

     if (res)
     {
         qDebug()<<"Event found. Mapping it";
         fillEventFromModel(event,aEvent);
     }
     qDebug() << "CommHistoryModel::getEvent " << aEventId <<". res = " << res ;
     //Release memory
     rtcom_el_event_free_contents(event);
     rtcom_el_event_free(event);
     g_object_unref(iterator);
     g_object_unref(query);
     return true;

 }

//---------------------------CommHistorySmsModel----------------------------------------------------
CommHistorySmsModel::CommHistorySmsModel()
{
}

CommHistorySmsModel::~CommHistorySmsModel()
{
}
bool CommHistorySmsModel::getEvents(RTComEl *dbElement)
{
    RTComElEvent *event= rtcom_el_event_new();
    RTComElQuery *query = rtcom_el_query_new(dbElement);
    RTComElIter * iterator = NULL;
    //Clear
    m_eventTree.clear();

    if ( !rtcom_el_query_prepare(query,kserviceid.toAscii().data(),rtcom_el_get_service_id(dbElement,ksmsservice.toAscii().data()) , RTCOM_EL_OP_EQUAL, NULL))
    {
         qDebug()<<__FUNCTION__<<": Query isn't created";
         //Release memory
         rtcom_el_event_free_contents(event);
         rtcom_el_event_free(event);
         g_object_unref(query);
         return false;
    }
    iterator = rtcom_el_get_events(dbElement,query);
    if (!iterator)
    {
        qDebug() <<__FUNCTION__<< ": Failed to get iterator. Event count = 0. Return true";
        rtcom_el_event_free_contents(event);
        rtcom_el_event_free(event);
        g_object_unref(query);
        return true;
    }

    if (!rtcom_el_iter_first(iterator))
    {
        qDebug() << "CommHistorySmsModel::getEvents. Failed to start iterator.";
        //Release memory
        rtcom_el_event_free_contents(event);
        rtcom_el_event_free(event);
        g_object_unref(iterator);
        g_object_unref(query);
        return false;
    }
#ifndef RTCOM_OLD
    while (rtcom_el_iter_get_full(iterator,event))
#else
    while (rtcom_el_iter_get(iterator,event))
#endif
    {
        Event aEvent;
        //Map from RTComElEvent to Event
        fillEventFromModel(event,aEvent);
        m_eventTree.append(fillMessageData(aEvent));
        rtcom_el_iter_next(iterator);
    }

    //Release memory
    rtcom_el_event_free_contents(event);
    rtcom_el_event_free(event);
    g_object_unref(iterator);
    g_object_unref(query);
    return true;
}
bool CommHistorySmsModel::acceptsEvent(const Event &event) const
{
    if (event.getServiceId() == Event::SMSService ) return true;

    return false;
}

//-----------------CommHistoryCallModel---------------------------------------------------
CommHistoryCallModel::CommHistoryCallModel()
{
}

CommHistoryCallModel::~CommHistoryCallModel()
{
}
bool CommHistoryCallModel::getEvents(RTComEl *dbElement)
{

    RTComElEvent *event= rtcom_el_event_new();
    RTComElQuery *query = rtcom_el_query_new(dbElement);
    RTComElIter * iterator = NULL;
    //Clear
    m_eventTree.clear();
    if (! rtcom_el_query_prepare(query,kserviceid.toAscii().data(),rtcom_el_get_service_id(dbElement,kcallservice.toAscii().data()) , RTCOM_EL_OP_EQUAL, NULL) )
    {
         qDebug()<<__FUNCTION__<<": Query isn't created";
         //Release memory
         rtcom_el_event_free_contents(event);
         rtcom_el_event_free(event);
         g_object_unref(query);
         return false;
    }
    iterator = rtcom_el_get_events(dbElement,query);
    if (!iterator)
    {
        qDebug() <<__FUNCTION__<< ": Failed to get iterator. Event count = 0. Return true";
        rtcom_el_event_free_contents(event);
        rtcom_el_event_free(event);
        g_object_unref(query);
        return true;
    }

    if (!rtcom_el_iter_first(iterator))
    {
        qDebug() << "CommHistorySmsModel::getEvents. Failed to start iterator.";
        //Release memory
        rtcom_el_event_free_contents(event);
        rtcom_el_event_free(event);
        g_object_unref(iterator);
        g_object_unref(query);
        return false;
    }

#ifndef RTCOM_OLD
    while (rtcom_el_iter_get_full(iterator,event))
#else
    while (rtcom_el_iter_get(iterator,event))
#endif
    {
        Event aEvent ;
        //Map from RTComElEvent to Event
        fillEventFromModel(event,aEvent);
        m_eventTree.append(fillMessageData(aEvent));
        rtcom_el_iter_next(iterator);
    }

    //Release memory
    rtcom_el_event_free_contents(event);
    rtcom_el_event_free(event);
    g_object_unref(iterator);
    g_object_unref(query);
    return true;

}
bool CommHistoryCallModel::acceptsEvent(const Event &event) const
{
   if (event.getServiceId() == Event::CallService) return true;

    return false;
}
