/*
 * ============================================================================
 *  Name        : qgetlisttask.cpp
 *  Part of     : serviceframework / WRT
 *  Description : getlist runnable object class for Calendar service
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QtDebug>

#include "qcalendarconstants.h"
#include "qcalendarhelper.h"
#include "qgetlisttask.h"

GetListTask::GetListTask(CMulticalendar* m_Calendar, const QMap<QString,QVariant>& matchCriteria, qint32 transId):
    m_matchCriteria(matchCriteria), m_trId(transId)
{
    qDebug() << "GetListTask() constructor, received multiCalendar=" << m_Calendar;
    multiCalendar = m_Calendar;
    calendar = multiCalendar->getSynchronizedCalendar();
    qDebug() << "GetListTask(): done";
}

GetListTask::~GetListTask()
{
    //do not delete calendar! it'll be deleted on deletion of multicalendar.
    calendar = NULL;
    qDebug() << "GetListTask() destructor";
}

void GetListTask::run()
{
    qDebug() << "\n\nrun()";
    QList<QVariantMap>* mapList = new QList<QVariantMap>();
    qDebug() << "* run(): call for GetListTask::getList";
    qint32 status = getList(mapList);
    qDebug() << "* run(): status:" << status << ", trId:" << m_trId;
    qDebug() << "* run(): mapList:" << mapList;
    qDebug() << "* run(): emitting listOfCalendarEntries";
    emit listOfCalendarEntries(mapList, m_trId, status);
    qDebug() << "\n\nrun()";
}

qint32 GetListTask::getList(QList<QVariantMap>* mapList)
{
    QString uid;
    QDateTime rangeBegin;
    QDateTime rangeEnd;
    QString type;
    QString text;
    QVariantMap qRange;
    bool isId = FALSE;
    bool isType = FALSE;
    bool isRange = FALSE;
    bool isText = FALSE;

    qint32 status = JSEStatusCodes::SUCCESS;

    qDebug() << "GetListTask::getList()";
    qDebug() << "* GetListTask::getList(): m_matchCriteria=" << m_matchCriteria;
    //Validate matchCriteria---------------------------------------------------------
    if (!m_matchCriteria.isEmpty())
    {
        if (m_matchCriteria.contains(KMatchPatternId))
        {
            if (m_matchCriteria.value(KMatchPatternId).isNull() || m_matchCriteria.value(KMatchPatternId) == KUndefined)
            {
                return JSEStatusCodes::SUCCESS;
            }
            else
            {
                uid = m_matchCriteria.value(KMatchPatternId).toString();
                qDebug() << "* GetListTask::getList(): entry UID:" << uid;
                isId = TRUE;
            }
        }
        if (!isId)
        {
            if (m_matchCriteria.contains(KMatchPatternRange) &&
                !m_matchCriteria.value(KMatchPatternRange).isNull() && (m_matchCriteria.value(KMatchPatternRange) != KUndefined))
            {
                qRange = m_matchCriteria.value(KMatchPatternRange).toMap();
                if ((qRange.contains(KRangeDataBegin) && !qRange.value(KRangeDataBegin).toDateTime().isValid())
                    || (qRange.contains(KRangeDataEnd) && !qRange.value(KRangeDataEnd).toDateTime().isValid()))
                {
                    qDebug() << "* GetListTask::getList(): range dates are invalid:" << qRange;
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
                if (qRange.contains(KRangeDataBegin) &&
                    (qRange.value(KRangeDataBegin) != NULL) &&
                    qRange.value(KRangeDataBegin) != KUndefined)
                {
                    rangeBegin = qRange.value(KRangeDataBegin).toDateTime();
                }
                else
                {
                    rangeBegin = MIN_DATE_TIME;
                }
                qDebug() << "* GetListTask::getList(): range.begin:" << rangeBegin;
                if (qRange.contains(KRangeDataEnd) &&
                    (qRange.value(KRangeDataEnd) != NULL) &&
                    qRange.value(KRangeDataEnd) != KUndefined)
                {
                    rangeEnd = qRange.value(KRangeDataEnd).toDateTime();
                }
                else
                {
                    rangeEnd = MAX_DATE_TIME;
                }
                if (rangeBegin > rangeEnd)
                {
                    qDebug() << "* GetListTask::getList(): invalid range:" << qRange;
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
                isRange = TRUE;
                qDebug() << "* GetListTask::getList(): range.end:" << rangeEnd;
            }
            if (m_matchCriteria.contains(KMatchPatternType) &&
                !m_matchCriteria.value(KMatchPatternType).isNull() && (m_matchCriteria.value(KMatchPatternType) != KUndefined))
            {
                type = m_matchCriteria.value(KMatchPatternType).toString();
                isType = TRUE;
            }

            if (m_matchCriteria.contains(KMatchPatternText) &&
                !m_matchCriteria.value(KMatchPatternText).isNull() && (m_matchCriteria.value(KMatchPatternText) != KUndefined))
            {
                text = m_matchCriteria.value(KMatchPatternText).toString();
                qDebug() << "* GetListTask::getList(): text:" << text;
                isText = TRUE;
            }
        }
    }
//Fetching----------------------------------------------------------------------------
//If match pattern is empty - fetching all entries
    if (!isId && !isRange && !isType && !isText)
    {
        qDebug() << "* GetListTask::getList(): match pattern is empty, fetching all entries...";
        fetchEventsByRange(mapList, MIN_DATE_TIME, MAX_DATE_TIME);
        fetchTodosByRange(mapList, MIN_DATE_TIME, MAX_DATE_TIME);
    }
//Get Entry by Id---------------------------------------------------------------------
    else if (isId)
    {
        qDebug() << "* GetListTask::getList(): fetching entry by uid:" << uid;
        QVariantMap entry;
        if (!CalendarHelpers::getCalendarEntryById(multiCalendar, entry, uid))
        {
            qDebug() << "* GetListTask::getList(): Entry not found!";
            status = JSEStatusCodes::DATA_NOT_FOUND_ERR;
            return status;
        }
        mapList->append(entry);
    }
    else if (isText)
    {
        qDebug() << "* GetListTask::getList(): fetching entries by text";
        if (isType && !isRange)
        {
            qDebug() << "* GetListTask::getList(): type is present, fetching by type first";
            fetchEntriesByType(type, mapList);
        }
        else if (isType && isRange)
        {
            qDebug() << "* GetListTask::getList(): type and range is present, fetching by range";
            if (!type.compare(KTypeToDo, Qt::CaseInsensitive))
            {
                fetchTodosByRange(mapList, rangeBegin, rangeEnd);
            }
            else if (!type.compare(KTypeMeeting, Qt::CaseInsensitive) ||
                    !type.compare(KTypeAnniversary, Qt::CaseInsensitive) ||
                    !type.compare(KTypeReminder, Qt::CaseInsensitive) ||
                    !type.compare(KTypeDayEvent, Qt::CaseInsensitive))
            {
                fetchEventsByRange(mapList, rangeBegin, rangeEnd);
                //fetched all events, searching for needed type
                for (int entriesCount = 0; entriesCount < mapList->count(); entriesCount++)
                {
                    if (mapList->at(entriesCount).value(KCalendarEntryType) != type)
                    {
                        mapList->removeAt(entriesCount);
                        entriesCount--;
                    }
                }
            }
        }
        else if (!isType && isRange)
        {
            qDebug() << "* GetListTask::getList(): range is present, fetching by range first";
            fetchEventsByRange(mapList, rangeBegin, rangeEnd);
            fetchTodosByRange(mapList, rangeBegin, rangeEnd);
        }
        else if (!isType && !isRange)
        {
            qDebug() << "* GetListTask::getList(): no more patterns, getting all events and todos";
            fetchEventsByRange(mapList, MIN_DATE_TIME, MAX_DATE_TIME);
            fetchTodosByRange(mapList, MIN_DATE_TIME, MAX_DATE_TIME);
        }
        qDebug() << "* GetListTask::getList(): all entries count:" << mapList->count();
        qDebug() << "* GetListTask::getList(): searching for text:" << text;
        for (int entriesCount = 0; entriesCount < mapList->count(); entriesCount++)
        {
            QVariant summary = mapList->at(entriesCount).value(KCalendarEntrySummary);
            qDebug() << "* GetListTask::getList(): current summary:" << summary;
            if (!summary.toString().contains(text, Qt::CaseInsensitive))
            {
                qDebug() << "* GetListTask::getList(): summary doesn't contain needed text, removing from list";
                mapList->removeAt(entriesCount);
                entriesCount--;
            }
        }
        qDebug() << "* GetListTask::getList(): founded entries count:" << mapList->count();
    }
    else if (isType && !isRange)
    {
//Get Entry by Type---------------------------------------------------------------------
        qDebug() << "* GetListTask::getList(): fetching entries by type...";
        fetchEntriesByType(type, mapList);
    }
//Get Entry by Range---------------------------------------------------------------------
/*BLOCKED until Bug#156938 fixed
    if (!qRange.isEmpty() && type.isEmpty())
    {
        qDebug() << "* GetListTask::getList(): fetching entry by Range";
        vector< CComponent * > components;
        int limit = 100;
        while (status != 220)
        {
            vector<CComponent *> currentQuery;
            int status = multiCalendar->getComponentsAllCalendars(rangeBegin.toTime_t(),
                                                                  rangeEnd.toTime_t(),
                                                                  limit, (limit - 100),
                                                                  currentQuery, 1);
            qDebug() << "* GetListTask::getList(): status:" << status;
            qDebug() << "* GetListTask::getList(): rangeBegin:" << rangeBegin;
            qDebug() << "* GetListTask::getList(): rangeEnd:" << rangeEnd;
            qDebug() << "* GetListTask::getList(): limit:" << limit;
            qDebug() << "* GetListTask::getList(): currentQuery.size():" << currentQuery.size();
            for (uint size = 0; size < currentQuery.size(); size++)
            {
                components.push_back(currentQuery.at(size));
            }
            limit += 100;
        }
        vector<time_t> times;
        for (uint i = 0; i < components.size(); i++)
        {
            CComponent * component = components.at(i);
            component->generateInstanceTimes(rangeBegin.toTime_t(), rangeEnd.toTime_t(), times);
            for (uint j = 0; j < times.size(); j++)
            {
                if (times.at(j) > rangeBegin.toTime_t() && times.at(j) < rangeEnd.toTime_t())
                {
                    QDateTime instanceTime = QDateTime::fromTime_t(times.at(j));
                    mapList->append(CalendarHelpers::createCalendarEntry(component, instanceTime));
                }
            }
            delete component;
        }
        qDebug() << "* GetListTask::getList(): done";
    }*/

//workaround for Bug#156938
    else if (!isType && isRange)
    {
        qDebug() << "* GetListTask::getList(): fetching entries by qRange...";
        fetchEventsByRange(mapList, rangeBegin, rangeEnd);
        fetchTodosByRange(mapList, rangeBegin, rangeEnd);
    }
    else if (isType && isRange)
    {
        qDebug() << "* GetListTask::getList(): fetching entries by Type && qRange...";
        if (!type.compare(KTypeToDo, Qt::CaseInsensitive))
        {
            fetchTodosByRange(mapList, rangeBegin, rangeEnd);
        }
        else if (!type.compare(KTypeMeeting, Qt::CaseInsensitive) ||
                !type.compare(KTypeAnniversary, Qt::CaseInsensitive) ||
                !type.compare(KTypeReminder, Qt::CaseInsensitive) ||
                !type.compare(KTypeDayEvent, Qt::CaseInsensitive))
        {
            fetchEventsByRange(mapList, rangeBegin, rangeEnd);
            for (int entriesCount = 0; entriesCount < mapList->count(); entriesCount++)
            {
                if (mapList->at(entriesCount).value(KCalendarEntryType) != type)
                {
                    mapList->removeAt(entriesCount);
                    entriesCount--;
                }
            }
        }
    }

    qDebug() << "* GetListTask::getList(): fetching complete, mapList:" << mapList;
    qDebug() << "* GetListTask::getList(): returning status:" << status;
    return status;
}

void GetListTask::fetchEventsByRange(QList<QVariantMap>* mapList, QDateTime rangeBegin, QDateTime rangeEnd)
{
    vector< CEvent *> events;
    vector<time_t> times;
    int errorCode;
    events = calendar->getAllAddedEvents(0, errorCode);
    if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
    {
        qDebug() << "getAllAddedEvents failed with error:" << errorCode;
        return;
    }
    qDebug() << "* fetchEventsByRange(): events count:" << events.size();
    CComponent * component = NULL;
    for (uint i = 0; i < events.size(); i++)
    {
        CRecurrence* recur = NULL;
        CEvent * event = NULL;
        event = events.at(i);
        if (event == NULL)
        {
            qDebug() << "event = NULL!";
            return;
        }
        component = static_cast<CComponent*>(event);
        qDebug() << "* fetchEventsByRange(): created component:" << component;
        if (component != NULL)
        {
            qDebug() << "* fetchEventsByRange(): getting recurrence for the" << (i + 1) << " event";
            //do not delete recur! it'll be deleted on deletion of component.
            recur = component->getRecurrence();
            qDebug() << "* fetchEventsByRange(): recur:" << recur;
        }
        if (recur != NULL)
        {
            QVariantList exDatesList;
//getting x_properties-------------------------------------------------------------------------------------------
            QString entryType;
            vector <CProperties *> pProp;
            pProp = event->getXProperties();
            PropType propType;
            bool isXPropPresent = FALSE;
            qDebug() << "* createCalendarEntry(): x_properties count: " << pProp.size();
            for (uint i = 0; i < pProp.size(); i++)
            {
                CProperties * property = NULL;
                property = pProp.at(i);
                if (property)
                {
                    QString propName = QString::fromStdString(property->getPropName());
                    qDebug() << "* createCalendarEntry(): x_property number:" << (i + 1);
                    qDebug() << "* createCalendarEntry(): x_property name:" << propName;
                    if (!propName.compare(X_PROPERTY, Qt::CaseSensitive))
                    {
                        qDebug() << "* createCalendarEntry(): it is needed x_property, getting value";
                        propType = property->getPropValue();
                        isXPropPresent = TRUE;
                        break;
                    }
                }
                delete property;
            }
            if (isXPropPresent)
            {
                entryType = QString::fromStdString(propType.szString);
                qDebug() << "* createCalendarEntry(): x_property type:" << entryType;
            }
            else
            {
                entryType = KTypeMeeting;
            }
//---------------------------------------------------------------------------------------------------------------
            if (!entryType.compare(KTypeMeeting, Qt::CaseInsensitive))
            {
                exDatesList = CalendarHelpers::createExceptionDatesFromRecurrence(recur);
                qDebug() << "* fetchEventsByRange(): this is not allDay event, exDatesList:" << exDatesList;
            }
            component->generateInstanceTimes(rangeBegin.toTime_t(), rangeEnd.toTime_t(), times);
            qDebug() << "* fetchEventsByRange(): generated instance times count:" << times.size();
            for (uint j = 0; j < times.size(); j++)
            {
                qDebug() << "* fetchEventsByRange(): validating instance times..";
                if (times.at(j) >= rangeBegin.toTime_t() && times.at(j) <= rangeEnd.toTime_t())
                {
                    QDateTime instanceTime = QDateTime::fromTime_t(times.at(j));
                    qDebug() << "\n\n\n* fetchEventsByRange(): instanceTime:" << instanceTime;
                    qDebug() << "* fetchEventsByRange(): exDatesList:" << exDatesList;
                    if (exDatesList.count())
                    {
                        qDebug() << "* fetchEventsByRange(): check if exDates == instance times..";
                        bool isEven = FALSE;
                        for (int exDatesCount = 0; exDatesCount < exDatesList.count(); exDatesCount++)
                        {
                            qDebug() << "* fetchEventsByRange():   instanceTime:" << instanceTime;
                            qDebug() << "* fetchEventsByRange(): exceptionDates:" << exDatesList.at(exDatesCount);
                            if (exDatesList.at(exDatesCount) == instanceTime)
                            {
                                qDebug() << "\n\n\n* fetchEventsByRange(): they are even, instance skipped!";
                                isEven = TRUE;
                                break;
                            }
                        }
                        if (!isEven)
                        {
                            qDebug() << "* fetchEventsByRange(): not even, appending mapList..";
                            mapList->append(CalendarHelpers::createCalendarEntry(component, instanceTime));
                            qDebug() << "\n\n\n* fetchEventsByRange(): instance created!";
                        }
                    }
                    else
                    {
                        mapList->append(CalendarHelpers::createCalendarEntry(component, instanceTime));
                        qDebug() << "\n\n\n* fetchEventsByRange(): instance created!";
                    }
                }
                else
                {
                    qDebug() << "* fetchEventsByRange():instance is not in needed range!";
                }

            }
        }
        else
        {
            qDebug() << "* fetchEventsByRange(): no recurrence";
            QDateTime dateStart = QDateTime::fromTime_t(component->getDateStart());
            QDateTime dateEnd = QDateTime::fromTime_t(component->getDateEnd());
            if ((dateStart >= rangeBegin && dateStart <= rangeEnd) ||
                (dateEnd >= rangeBegin && dateEnd <= rangeEnd))
            {
                qDebug() << "* fetchEventsByRange(): entry falls into given range";
                mapList->append(CalendarHelpers::createCalendarEntry(component));
            }
            else
            {
                qDebug() << "* fetchEventsByRange(): entry doesn't fall into given range";
            }
        }
        qDebug() << "* fetchEventsByRange(): deleting component:" << component;
        delete component;
        component = NULL;
    }
    qDebug() << "* fetchTodosByRange(): done, mapList updated with founded events";
}

void GetListTask::fetchTodosByRange(QList<QVariantMap>* mapList, QDateTime rangeBegin, QDateTime rangeEnd)
{
    vector< CTodo *> todos;
    vector<time_t> times;
    int errorCode;
    qDebug() << "* fetchTodosByRange(): getting all Todos from calendar";
    todos = calendar->getAllAddedTodos(0, errorCode);
    if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
    {
        qDebug() << "* fetchTodosByRange(): error getting todos list, errorCode:" << errorCode;
        return;
    }
    qDebug() << "* fetchTodosByRange(): todos count:" << todos.size();
    qDebug() << "* fetchTodosByRange(): validating Todos date";
    for (uint i = 0; i < todos.size(); i++)
    {
        CTodo * todo = NULL;
        QDateTime dueDate;
        todo = todos.at(i);
        if (todo)
            dueDate = QDateTime::fromTime_t(todo->getDue());
        if (dueDate >= rangeBegin && dueDate <= rangeEnd)
        {
            CComponent * component = static_cast<CComponent *>(todos.at(i));
            mapList->append(CalendarHelpers::createCalendarEntry(component));
        }
        delete todo;
    }
    qDebug() << "* fetchTodosByRange(): done, mapList updated with founded todos";
}

void GetListTask::fetchEntriesByType(QString type, QList<QVariantMap>* mapList)
{
    int errorCode;
    qDebug() << "fetchEntriesByType()";
    qDebug() << "type:" << type;
    if (!type.compare(KTypeToDo, Qt::CaseInsensitive))
    {
        qDebug() << "* fetchEntriesByType(): fetching todos";
        vector< CTodo *> todosList = calendar->getAllAddedTodos(0, errorCode);
        if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
        {
            qDebug() << "* fetchEntriesByType(): error getting todos list, errorCode:" << errorCode;
            qDebug() << "* fetchEntriesByType(): no entries found, return";
            return;
        }
        qDebug() << "* fetchEntriesByType(): todos list.count:" << todosList.size();
        for (uint i = 0; i < todosList.size(); i++)
        {
            mapList->append(CalendarHelpers::createCalendarEntry(todosList.at(i)));
            qDebug() << "* fetchEntriesByType(): current entry id:" << mapList->at(i).value(KCalendarEntryId);
            delete todosList.at(i);
        }
    }
    else
    {
        fetchEventsByRange(mapList, MIN_DATE_TIME, MAX_DATE_TIME);
        for (int entriesCount = 0; entriesCount < mapList->count(); entriesCount++)
        {
            if (mapList->at(entriesCount).value(KCalendarEntryType) != type)
            {
                mapList->removeAt(entriesCount);
                entriesCount--;
            }
        }
    }
    qDebug() << "* fetchEntriesByType(): founded entries count:" << mapList->count();
    qDebug() << "fetchEntriesByType()";
}
