/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
/* resolver test module requires serviceresolver be running
(otherwise 2 instances of resolver are launched and that fails some test cases)
*/

#include <QtTest/QtTest>
#include "serviceresolver.h"
#include "serviceipc.h"
#include "serviceplugininterface.h"
#include "serviceresolutioncriteria.h"
#include "serviceinfolist.h"
#include "servicebase.h"
#include "serviceframeworkdefs.h"
#include "testhelpers.h"

namespace WRT{

static const char KTESTXMLFILE[]          = "ServiceTest5.xml";
static const char KTESTXMLFILE2[]         = "ServiceTest6.xml";
static const char KTESTXMLFILE3[]         = "ServiceTest7.xml";
static const char KTESTXMLFILE4[]         = "Test13.xml";
static const int  KWAITINTERVAL           = 2000;     // 1 second

static const char KIMPORTFOLDER[]         = "import/";
static const char KREGISTEREDFOLDER[]     = "registered/";
static const char KDATABASEFILE[]         = "services.db";

#ifdef __SYMBIAN32__
    static const char RESOLVER_DATABASE_PATH[] = "c:/private/102829B8/";
    static const char KTESTDATAFOLDER[]         = "c:/testdata/";
    static const char KTESTPLUGINFOLDER[]         = "C:\\resource\\qt\\sfwplugin\\";
#else
    static const char KTESTDATAFOLDER[]         = "testdata/";
#endif

class ServiceResolverModuleTest : public QObject
{
    Q_OBJECT

    private:
        void copyFileToImport( const QString aFile,
                               const QString aFolder=
                                 (QString(KCwrtSvcProvInstallFolder)+QString(KIMPORTFOLDER)) );
        void deleteFiles( const QString aFolder );
        void testListOfServicesRegistered(int listSize);
        IServiceBase* loadPlugin(const QString&);
        ServiceFwIPC*         iSesesion;


    private slots:  // test cases
        void initTestCase();
        void testRegisterService();
        void testDBRecovery();
        void testGetServiceVersion();
        void testGetService();
        //void testServiceUnistallation();
        void testInvalidRequest();
        void cleanupTestCase();
};

void ServiceResolverModuleTest::copyFileToImport(
    const QString aFile,
    const QString aFolder /*=KCwrtSvcProvInstallFolder+KIMPORTFOLDER*/ )
{
   // Copy a file into the import folder
#ifndef __SYMBIAN32__
   QDir dir;
#else
   QDir dir(RESOLVER_DATABASE_PATH);
#endif
   QVERIFY( dir.mkpath( aFolder ));
   QVERIFY(dir.cd(aFolder));
   QVERIFY(QFile::copy(KTESTDATAFOLDER + aFile,dir.absoluteFilePath(aFile)));

   // Check not read-only so that it can be deleted
   QFile file( aFolder + aFile );
   QFile::Permissions perm = file.permissions();
   if ( !perm.testFlag( QFile::WriteOwner ))
   {
       perm |= QFile::WriteOwner;
   }
   file.setPermissions( perm );
   file.close();
}


void ServiceResolverModuleTest::initTestCase()
{
    __countTestsHelper__(this);
    // Cleanup files in import and registered folder
    deleteFiles(QString(KCwrtSvcProvInstallFolder)+QString(KIMPORTFOLDER));
    deleteFiles(QString(KCwrtSvcProvInstallFolder)+QString(KREGISTEREDFOLDER));
#ifndef __SYMBIAN32__
    QDir dir;
    //QFile::copy(TESTSERVICE2_DLL_SOURCE,  QDir::currentPath()+TESTSERVICE2_DLL_NAME);
#else
    QDir dir(RESOLVER_DATABASE_PATH);
#endif
    if (dir.exists(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE))){
        dir.remove(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE));
    }
    // clean up database
    QTest::qWait(KWAITINTERVAL*2);

    iSesesion = new ServiceFwIPC(this);

    bool connected = false;
    if (iSesesion)
    {
        int retry(2);
        for (;;)
        {
            connected = iSesesion->connect(SERVICE_RESOLVER_NAME);
            if (connected)
            {
                break;
            }
            else
            {
                QString serviceResolverSvr(KCwrtSvcProvInstallFolder);
                serviceResolverSvr.append(SERVICE_RESOLVER_SERVER);
                if ( !iSesesion->startServer(
                    SERVICE_RESOLVER_NAME, serviceResolverSvr))
                { // start server failed.
                    break;
                }
            }
            if (0==--retry)
            {
                break;
            }
        }
    }
    QCOMPARE(connected, true);
    if (connected)
        qDebug() << "session created";
}


void ServiceResolverModuleTest::cleanupTestCase()
{
    // Cleanup files in import and registered folder
    deleteFiles(QString(KCwrtSvcProvInstallFolder)+QString(KIMPORTFOLDER));
    deleteFiles(QString(KCwrtSvcProvInstallFolder)+QString(KREGISTEREDFOLDER));
    #ifndef __SYMBIAN32__
    QDir dir;
    #else
    QDir dir(RESOLVER_DATABASE_PATH);
    #endif
    if (dir.exists(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE))){
       dir.remove(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE));
    }
    iSesesion->disconnect();
    delete iSesesion;
}

void ServiceResolverModuleTest::deleteFiles( const QString aFolder )
{
#ifndef __SYMBIAN32__
    QDir dir(QDir::currentPath());
#else
    QDir dir(RESOLVER_DATABASE_PATH);
#endif

    if ( dir.exists( aFolder )) {
        QVERIFY(dir.cd(aFolder));
        QDirIterator it(dir, QDirIterator::Subdirectories);
        while (it.hasNext()) {
            it.next();
            dir.remove(it.filePath());
        }
        QVERIFY(dir.cdUp());
        dir.rmdir(aFolder);
    }

}


void ServiceResolverModuleTest::testListOfServicesRegistered(int listSize)
{
    QString null;
    bool ret = iSesesion->sendSync(LISTSERVICEREQ, null.toAscii());
    if (ret){
        ServiceInfoList services;
        services.deserialize(iSesesion->readAll());
        QCOMPARE(services.count(), listSize);

        if (listSize >= 1) {
            ServiceInfo service = services.getService(0);
            QCOMPARE(service.serviceName(),QString("TestService"));
#ifdef __SYMBIAN32__
            QCOMPARE(service.serviceDllFileName(),QString(KTESTPLUGINFOLDER) + "testservice5.qtplugin");
#else
            QCOMPARE(service.serviceDllFileName(),QString(KTESTDATAFOLDER) + "testservice5.dll");
#endif
            QCOMPARE(service.serviceVersion(),QString("1.4.56"));
        }
        if (listSize >= 2) {
            ServiceInfo service = services.getService(1);
            QCOMPARE(service.serviceName(),QString("TestService1"));
#ifdef __SYMBIAN32__
            QCOMPARE(service.serviceDllFileName(),QString(KTESTPLUGINFOLDER) + "testservice6.qtplugin");
#else
            QCOMPARE(service.serviceDllFileName(),QString(KTESTDATAFOLDER) + "testservice6.dll");
#endif
            QCOMPARE(service.serviceVersion(),QString("1.4.57"));
        }
    }
    else
    {
        QFAIL("Service registeration failed");
    }
}


void ServiceResolverModuleTest::testRegisterService()
{
    //register a service
    copyFileToImport(KTESTXMLFILE );
    QTest::qWait( KWAITINTERVAL );
    testListOfServicesRegistered(1);
    //try to register the same service again
    copyFileToImport(KTESTXMLFILE);
    QTest::qWait( KWAITINTERVAL );
    testListOfServicesRegistered(1);
    //register another vesion of the same service
    copyFileToImport(KTESTXMLFILE2);
    QTest::qWait( KWAITINTERVAL );
    testListOfServicesRegistered(2);

    //register without .dll
    copyFileToImport(KTESTXMLFILE3);
    QTest::qWait( KWAITINTERVAL );
    testListOfServicesRegistered(2);

    //invalid xml file
    copyFileToImport(KTESTXMLFILE4);
    QTest::qWait( KWAITINTERVAL );
    testListOfServicesRegistered(2);
    QVERIFY(!QFile::exists(QString(KCwrtSvcProvInstallFolder)
        + QString(KIMPORTFOLDER)
        + QString(KTESTXMLFILE4)));
    qDebug() << "testRegisterService passed";
}

void ServiceResolverModuleTest::testDBRecovery()
{
    //delete DB
#ifndef __SYMBIAN32__
    QDir dir;
#else
    QDir dir(RESOLVER_DATABASE_PATH);
#endif
    if (dir.exists(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE))){
       dir.remove(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE));
    }
    //get list of services, db should be recovered from registered folder
    testListOfServicesRegistered(2);

    //delete DB
    if (dir.exists(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE)))
    {
       dir.remove(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE));
    }

    QTest::qWait( KWAITINTERVAL);
    //get a service version, db should be recovered from registered folder
    QVERIFY(iSesesion->sendSync(GETSERVICEVERSIONREQ, "TestService"));
    QString serviceVersion(iSesesion->readAll());
    QCOMPARE(serviceVersion,QString("1.4.56"));

    //delete DB
    if (dir.exists(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE)))
    {
       dir.remove(QString(KCwrtSvcProvInstallFolder)+QString(KDATABASEFILE));
    }
    //get a service, db should be recovered from registered folder
    ServiceResolutionCriteria criteria;
    criteria.setServiceName("TestService");
    criteria.setInterfaceName("com.nokia.IDownloader");
    criteria.setInterfaceVersion(KDefaultVersion);
    if (iSesesion->sendSync(GETSERVICEREQ, criteria.serialize()))
    {
        ServiceInfo service;
        if (service.deserialize(iSesesion->readAll()))
        {
            QVERIFY(service.isValid());
        }
    }
    qDebug() << "testDBRecovery passed";
}

void ServiceResolverModuleTest::testGetServiceVersion()
{
    QVERIFY(iSesesion->sendSync(GETSERVICEVERSIONREQ, "TestService"));
    QString serviceVersion(iSesesion->readAll());
    QCOMPARE(serviceVersion,QString("1.4.56"));

    QVERIFY(iSesesion->sendSync(GETSERVICEVERSIONREQ, "NotRegisteredServiceName"));
    QString serviceVersion2(iSesesion->readAll());
    QCOMPARE(serviceVersion2,QString("ServiceNotSupport"));

}


IServiceBase* ServiceResolverModuleTest::loadPlugin(const QString& aDllName)
{
    IServiceBase* base(NULL);
    QPluginLoader* loader = new QPluginLoader(aDllName);
    QObject* plugin( loader->instance() );
    if ( plugin )
    {
        IServicePlugin* service( qobject_cast<IServicePlugin*>(plugin) );
        if ( service )
        {
            base = service->getServiceBase(true, NULL);
            base->addRef();
        }
        delete plugin;
        plugin = NULL;
    }
    return base;
}


void ServiceResolverModuleTest::testGetService()
{
    //invalid criteria, not all the conditions are specified
    ServiceResolutionCriteria criteria2;
    criteria2.setServiceName("TestService");

    if (iSesesion->sendSync(GETSERVICEREQ, criteria2.serialize()))
    {
        ServiceInfo service;
        service.deserialize(iSesesion->readAll());
        QVERIFY(!service.isValid());
    }


    //get service info of an unregistered service, valid criteria
    criteria2.setServiceName("TestService");
    criteria2.setInterfaceName("com.nokia.IDownloader1");
    criteria2.setInterfaceVersion("1.0");

    if (iSesesion->sendSync(GETSERVICEREQ, criteria2.serialize()))
    {
        ServiceInfo service;
        service.deserialize(iSesesion->readAll());
        QVERIFY(!service.isValid());
    }

    //get service with default service version
    criteria2.setServiceName("TestService");
    criteria2.setInterfaceName("com.nokia.IDownloader");
    criteria2.setInterfaceVersion(KDefaultVersion);
    if (iSesesion->sendSync(GETSERVICEREQ, criteria2.serialize()))
    {
        // data is available
        ServiceInfo service;
        if (service.deserialize(iSesesion->readAll()))
        {
            QVERIFY(service.isValid());
#ifdef __SYMBIAN32__
            QCOMPARE(service.serviceDllFileName(),QString(KTESTPLUGINFOLDER) + "testservice5.qtplugin");
#else
            QCOMPARE(service.serviceDllFileName(),QString(KTESTDATAFOLDER) + "testservice5.dll");
#endif
            QCOMPARE(service.serviceName(),QString("TestService"));
            QCOMPARE(service.interfaceCapabilities().count(), 2);
            QCOMPARE(service.serviceVersion(),QString("1.4.56"));
        }
        else
        {
            QFAIL("unable to resolve the service");
        }
    }
    qDebug() << "testGetService passed";
}

/*void ServiceResolverModuleTest::testServiceUnistallation()
{
    testListOfServicesRegistered(2);
    //remove service dll and check if service will be unisntalled upon GETSERVICE request
    QFile dllFile(QString(KTESTDATAFOLDER).append("testservice6.dll"));
    dllFile.remove();
    QDir dir(KTESTDATAFOLDER);
    QVERIFY(!dir.exists("testservice6.dll"));

    ServiceResolutionCriteria criteria;
    criteria.setServiceName("TestService1");
    criteria.setInterfaceName("com.nokia.IDownloader");
    criteria.setInterfaceVersion(KDefaultVersion);
    if (iSesesion->sendSync(GETSERVICEREQ, criteria.serialize()))
    {
        ServiceInfo service;
        service.deserialize(iSesesion->readAll());
        QVERIFY(!service.isValid());
        //verify number of services
        testListOfServicesRegistered(1);
    } else {
        QFAIL("Failed sending request!");
    }

    //remove service dll and check if service will be unisntalled upon LISTSERVICEREQ
    dllFile.setFileName(QString(KTESTDATAFOLDER).append("testservice5.dll"));
    dllFile.remove();
    QVERIFY(!dir.exists("testservice5.dll"));
    //remove xml file from registered folder. it should not affect uninstallation of service
    QFile xmlFile(QString(KCwrtSvcProvInstallFolder)
        + QString(KREGISTEREDFOLDER) + QString(KTESTXMLFILE));
    QFile::Permissions perm = xmlFile.permissions();
    perm |= QFile::Permission(QFile::ReadOnly);
    perm |= QFile::Permission(!QFile::ReadWrite);
    xmlFile.setPermissions(perm);
    testListOfServicesRegistered(0);
}*/

void ServiceResolverModuleTest::testInvalidRequest()
{
    iSesesion->sendSync("INVALIDREQ", "");
    ServiceInfo service;
    QVERIFY(!service.deserialize(iSesesion->readAll()));
}

}

QTEST_MAIN(WRT::ServiceResolverModuleTest);
#include "serviceresolvermoduletest.moc"
