/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QStringList>
#include "serviceresolutioncriteria.h"

#define SERVICE_CRITERIA_SEP ";"


namespace WRT {

enum TCriteriaIndexes
    {
        ECriteriaIndex=0,
        ECriteriaIndex1,
        ECriteriaIndex2,
        ECriteriaIndexCount
    };

/*!
    \class ServiceResolutionCriteria
    Utility class used to define the service search criteria \n
    Class is used by service resolver and service handler
*/

/*!
    Constructor
*/
ServiceResolutionCriteria::ServiceResolutionCriteria()
{
}

/*!
    Destructor
*/
ServiceResolutionCriteria::~ServiceResolutionCriteria(void)
{

}

/*!
    Gets the service name
    @return service name
*/
QString ServiceResolutionCriteria::serviceName()
{
    return m_criteriaServiceName;
}

/*!
    Gets the interface name
    @return interface name
*/
QString ServiceResolutionCriteria::interfaceName()
{
    return m_criteriaInterfaceName;
}

/*!
    Gets the interface version
    @return interface version
*/
QString ServiceResolutionCriteria::interfaceVersion()
{
    return m_criteriaInterfaceVersion;
}

/*!
    Sets the service name
    @param aServiceName service name
    @return void
*/
void ServiceResolutionCriteria::setServiceName(const QString& aServiceName)
{
    m_criteriaServiceName = aServiceName;
}


/*!
    Sets the interface name
    @param anInterfaceName interface name
    @return void
*/
void ServiceResolutionCriteria::setInterfaceName(const QString& anInterfaceName)
{
    m_criteriaInterfaceName = anInterfaceName;
}

/*!
    Sets the service version
    @param aInterfaceVersion interface version
    @return void
*/
void ServiceResolutionCriteria::setInterfaceVersion(const QString& aInterfaceVersion)
{
    m_criteriaInterfaceVersion = aInterfaceVersion;
}

/*!
    Checks if the criteria is valid \n
    For a criteria to be valid, all properties must be defined
    @return true if criteria is valid, false otherwise
*/
bool ServiceResolutionCriteria::isValid()
{
    return ((m_criteriaServiceName.length()>0) && (m_criteriaInterfaceName.length()>0) && (m_criteriaInterfaceVersion.length()>0));

}

/*!
    Creates data stream from criteria propertires
    @return serialized data stream or empty stream if criteria is invalid
*/
QByteArray ServiceResolutionCriteria::serialize()
{
    QByteArray stream;
    if (isValid()) {
        stream.append(m_criteriaServiceName.toAscii());
        stream.append(SERVICE_CRITERIA_SEP);
        stream.append(m_criteriaInterfaceName.toAscii());
        stream.append(SERVICE_CRITERIA_SEP);
        stream.append(m_criteriaInterfaceVersion.toAscii());
    }
    return stream;
}

/*!
    Extracts criteria metadata from input stream
    @return true if operation successful, false otherwise
*/
bool ServiceResolutionCriteria::deserialize(const QByteArray& aDataStream)
{
    bool bSuccess(false);
    QString strCriteria(aDataStream);
    QStringList criteria = strCriteria.split(SERVICE_CRITERIA_SEP);
    if (criteria.count() == ECriteriaIndexCount) {
        m_criteriaServiceName = criteria[ECriteriaIndex];
        m_criteriaInterfaceName = criteria[ECriteriaIndex1];
        m_criteriaInterfaceVersion = criteria[ECriteriaIndex2];
        bSuccess = true;
    }
    return bSuccess;
}

}
