/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "serviceinfo.h"

#define SERVICE_ITEM_SEP "<$>"
#define CAPABILITY_SEP ","

namespace WRT {

enum TMetadataIndexes
    {
        EMetadataIndex=0,
        EMetadataIndex1,
        EMetadataIndex2,
        EMetadataIndex3,
        EMetadataIndex4,
        EMetadataIndex5,
        EMetadataCount
    };

/*!
    \class ServiceInfo
    Utility class that encapsulates the service metadata and interface capabilities info for a getServiceObject request a\n
    Class is used by service resolver and service handler
*/

/*!
    Default Constructor
*/
ServiceInfo::ServiceInfo()
{

}

/*!
    Constructor
    @param aName service name
    @param aDllFileName service file name
    @param aInterfaceVersion interafce version
    @param aServiceVersion service version
    @param aXmlFileName xml file name
*/
ServiceInfo::ServiceInfo(const QString& aName,const QString& aDllFileName, const QString& aServiceVersion,
                         const QString& aInterfaceVersion, const QString& aXmlFileName)
{
    m_serviceName = aName;
    m_serviceDllFileName = aDllFileName;
    m_serviceXMLFileName = aXmlFileName;
    m_interfaceVersion = aInterfaceVersion;
    m_serviceVersion = aServiceVersion;
}

/*!
    Destructor
*/
ServiceInfo::~ServiceInfo(void)
{
}

/*!
    Gets the service name
    @return service name
*/
QString ServiceInfo::serviceName()
{
    return m_serviceName;
}

/*!
    Gets the service file name
    @return service file name
*/
QString ServiceInfo::serviceDllFileName()
{
    return m_serviceDllFileName;
}

/*!
    Gets the service version
    @return service version
*/
QString ServiceInfo::serviceVersion()
{
    return m_serviceVersion;
}

/*!
    Gets the service version
    @return service version
*/
QString ServiceInfo::interfaceVersion()
{
    return m_interfaceVersion;
}

/*!
    Gets the xml file name
    @return xml file name
*/
QString ServiceInfo::xmlFileName()
{
    return m_serviceXMLFileName;
}

/*!
    Gets the interface capabilities
    @return interface capabilities
*/
QStringList ServiceInfo::interfaceCapabilities()
{
    return m_interfaceCapabilities.split(CAPABILITY_SEP,QString::SkipEmptyParts);
}

/*!
    Sets the interface capabilities
    @param capabilities interface capabilities metadata
    @return void
*/
void ServiceInfo::setInterfaceCapabilities(const QString& interfaceCapabilities)
{
    m_interfaceCapabilities = interfaceCapabilities;
}

/*!
    Sets the xml file name
    @param aServiceXMLFileName xml file name
    @return void
*/
void ServiceInfo::setXmlFileName(const QString& aServiceXMLFileName)
{
    m_serviceXMLFileName = aServiceXMLFileName;
}

/*!
    Sets the service file name
    @param aDllFileName service file name
    @return void
*/
void ServiceInfo::setServiceDllFileName(const QString& aDllFileName)
{
    m_serviceDllFileName = aDllFileName;
}

/*!
    Sets the service name
    @param aName service name
    @return void
*/
void ServiceInfo::setServiceName(const QString& aName)
{
    m_serviceName = aName;
}

/*!
    Sets the interface version
    @param aVersion service version
    @return void
*/
void ServiceInfo::setInterfaceVersion(const QString& aVersion)
{
    m_interfaceVersion = aVersion;
}

/*!
    Sets the service version
    @param aVersion service version
    @return void
*/
void ServiceInfo::setServiceVersion(const QString& aVersion)
{
    m_serviceVersion = aVersion;
}

/*!
    Creates data stream from service propertires
    @return serialized data stream or empty stream if service is invalid
*/
QByteArray ServiceInfo::serialize()
{
    QByteArray stream;
    if (isValid()) {
        stream.append(m_serviceName.toAscii());
        stream.append(SERVICE_ITEM_SEP);
        stream.append(m_serviceDllFileName.toAscii());
        stream.append(SERVICE_ITEM_SEP);
        stream.append(m_serviceXMLFileName.toAscii());
        stream.append(SERVICE_ITEM_SEP);
        stream.append(m_serviceVersion.toAscii());
        stream.append(SERVICE_ITEM_SEP);
        stream.append(m_interfaceVersion.toAscii());
        stream.append(SERVICE_ITEM_SEP);
        stream.append(m_interfaceCapabilities.toAscii());
    }
    return stream;
}

/*!
    Extracts service metadata from input stream
    @return true if operation successful, false otherwise
*/
bool ServiceInfo::deserialize(const QByteArray& aDataStream)
{
    bool bSuccess(false);
    QString strServices(aDataStream);
    QStringList serviceData = strServices.split(SERVICE_ITEM_SEP);
    if (serviceData.count() == EMetadataCount) {
        m_serviceName = serviceData[EMetadataIndex];
        m_serviceDllFileName = serviceData[EMetadataIndex1];
        m_serviceXMLFileName = serviceData[EMetadataIndex2];
        m_serviceVersion = serviceData[EMetadataIndex3];
        m_interfaceVersion = serviceData[EMetadataIndex4];
        m_interfaceCapabilities = serviceData[EMetadataIndex5];
        bSuccess = true;
    }

    return bSuccess;
}

/*!
    Checks if the service is valid \n
    For a service to be valid, name, dll filename and xml file name properties must be defined
    @return true if service is valid, false otherwise
*/
bool ServiceInfo::isValid()
{
    return (!m_serviceName.isEmpty())
            && (!m_serviceXMLFileName.isEmpty())
            && (!m_serviceDllFileName.isEmpty());
}

}
