/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifdef __SYMBIAN32__
#include <e32std.h>
#include <f32file.h>
#endif // __SYMBIAN32__

#include <QDir>
#include <QUrl>
#include <QString>
#include <QStringList>
#include <QFileInfo>

#include "sfwlog.h"
#include "serviceobjectbinding.h"
#include "bindingutility.h"
#include "secmgrdefs.h"
#include "trustsession.h"
#include "secsession.h"
#include "servicebase.h"
#include "serviceframeworkdefs.h"
#include "servicefactory.h"
#include "wrtsettings.h"

#if defined(QT_MOBILITY_SFW)
    #include "qservicecontext.h"
    #include "qabstractsecuritysession.h"
#endif

// enable secsession for all except in 3.1, 3.2 emulator environment
#if !defined(__WINSCW__) || defined(__S60_50__)
#if !defined(__SYMBIAN32__) || defined(CWRT_S60_WEBKIT)
// for all except Avkon browser
#include "WidgetManager.h"
#include "WidgetProperties.h"
#endif
#endif

namespace WRT
{
static const char KGetServiceObject[]       = "getServiceObject";
static const char KGetServiceProviders[]    = "getServiceProviders";
static const char KPropertyToken[]          = "securityToken";
static const char KGetLastErrCode[]         = "getLastErrCode";
static const char KGetLastErrDescription[]  = "getLastErrDescription";
static const char KLocationHREF[]           = "location.href";
static const char KGetServiceVersion[]       = "getServiceVersion";
static const char KSetJSObjectProt[]                = "SFW_CreateJSObject";
// enable secsession for all except in 3.1, 3.2 emulator environment
#if !defined(__WINSCW__) || defined(__S60_50__)
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)

// Variables used only for Avkon browser
static const TUint32 KSecureWidgetId = 0x102829A0;
static const TUint32 KNormalWidgetId = 0x10282822;

// Have to parse url manually because NPPlugin returns a url with 3 slashes which
// confuses the current parsers.
static const char KNormalWidgetFileScheme[] = "file:///[C-Z]:/private/10282822/";
static const char KSecureWidgetFileScheme[] = "file:///[C-Z]:/private/102829A0/";

#else

// Variables for all except Avkon browser
static const char KCWRTWidgetHostScheme[] = "widget://";
static const char KCWRTSharedLibScheme[] = "lib://";
static const char KCWRTSandboxScheme[] = "sandbox://";
static const char KCWRTDriveLetter[] = "\\w:/";

#endif
#endif

#if defined(QT_MOBILITY_SFW)
class ServiceContext : public QServiceContext
{
    public:
    ServiceContext() {}
    ~ServiceContext() {}
    virtual void notify( ContextType /*type*/, const QVariant& /*variant*/) {}
};
#endif

/*!
    \class ServiceFactory
    \brief The ServiceFactory class is used to create a service object.

   It exports APIs to script list service providers and create a service object
*/

/*!
    Constructs an object of the ServiceFactory.

    It initialize service handler object and create identifiers for scriptable APIs.
    getServiceObject and getServiceProviders

    @param npp represents a single instance of a plugin
*/
ServiceFactory::ServiceFactory(NPP npp) :
    NPScriptableObjectBase(npp),
    m_bindingUtility(npp),
#if defined(QT_MOBILITY_SFW)
    m_serviceManager(NULL),
#else
    m_ServiceHandler(NULL),
#endif
    m_TrustSession(NULL),
    m_SecSession(NULL),
    m_content(NULL)
{
    SFW_FUNC_EX("ServiceFactory::ServiceFactory");
    m_npidGetServiceObject = NPN_GetStringIdentifier(KGetServiceObject);
    m_npidListServices = NPN_GetStringIdentifier(KGetServiceProviders);
    m_getLastErrCode = NPN_GetStringIdentifier(KGetLastErrCode);
    m_getLastErrDescription = NPN_GetStringIdentifier(KGetLastErrDescription);
#if !defined(QT_MOBILITY_SFW)
    m_npidGetServiceVersion = NPN_GetStringIdentifier(KGetServiceVersion);
#endif
    m_setJSObjectProt = NPN_GetStringIdentifier(KSetJSObjectProt);

#if defined(QT_MOBILITY_SFW)
    m_serviceManager = new QServiceManager(NULL);
    if (m_serviceManager->error() != QServiceManager::NoError ) {
        NPN_SETEXCEPTION(this, m_serviceManager->error(), getErrorText(m_serviceManager->error()).toUtf8());
    }
#else
    m_ServiceHandler = new ServiceHandler(NULL);
    if (m_ServiceHandler->getLastErrCode() != 0 ) {
        NPN_SETEXCEPTION(this, m_ServiceHandler->getLastErrCode(),
                         m_ServiceHandler->getLastErrDescription().toLatin1());
    }
#endif

// enable secsession for all except in 3.1, 3.2 emulator environment
#if !defined(__WINSCW__) || defined(__S60_50__)
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
    // only needed for Avkon browser
    createTrustSession();
#endif
#endif

    m_PropertyToken = NPN_GetStringIdentifier(KPropertyToken);
}

/*!
    Destructor, free service handler object
*/
ServiceFactory::~ServiceFactory()
{
    SFW_FUNC_EX("ServiceFactory::~ServiceFactory");
    ObjectBinding* objectBinding;
    foreach(objectBinding, m_bindingObjectList) {
        objectBinding->setFactory(NULL);
    }
    m_bindingObjectList.clear();
#if defined(QT_MOBILITY_SFW)
    delete m_serviceManager;
#else
    delete m_ServiceHandler;
#endif
    delete m_SecSession;
    delete m_TrustSession;
}

#if defined(QT_MOBILITY_SFW)
/*!
 *  Returns the error description
 */
QString ServiceFactory::getErrorText(QServiceManager::Error error) const
{
    QString m_errorMessage;

    switch (error)
    {
        case QServiceManager::NoError:
            m_errorMessage = "NoError";
            break;
        case QServiceManager::StorageAccessError:
            m_errorMessage = "StorageAccessError";
            break;
        case QServiceManager::InvalidServiceLocation:
            m_errorMessage = "InvalidServiceLocation";
            break;
        case QServiceManager::InvalidServiceXml:
            m_errorMessage = "InvalidServiceXml";
            break;
        case QServiceManager::InvalidServiceInterfaceDescriptor:
            m_errorMessage = "InvalidServiceInterfaceDescriptor";
            break;
        case QServiceManager::ServiceAlreadyExists:
            m_errorMessage = "ServiceAlreadyExists";
            break;
        case QServiceManager::ImplementationAlreadyExists:
            m_errorMessage = "ImplementationAlreadyExists";
            break;
        case QServiceManager::PluginLoadingFailed:
            m_errorMessage = "PluginLoadingFailed";
            break;
        case QServiceManager::ComponentNotFound:
            m_errorMessage = "ComponentNotFound";
            break;
        case QServiceManager::ServiceCapabilityDenied:
            m_errorMessage = "ServiceCapabilityDenied";
            break;
        case QServiceManager::UnknownError:
            m_errorMessage = "UnknownError";
            break;
        default:
            m_errorMessage = "";
    }

    return m_errorMessage;
}
#endif

/*!
    Check if a method exists in the service factory object given \a methodId

    It has two scriptable APIs:
    (1) getServiceObject create a service object
    (2) getServiceProviders list all of service providers

    @param methodId method identifier
    @return true if the method exists, otherwise false

    @see Invoke
*/
bool ServiceFactory::HasMethod(NPIdentifier methodId)
{
    bool bRet = (methodId == m_npidGetServiceObject ||
                 methodId == m_npidListServices ||
                 methodId == m_getLastErrCode ||
                 methodId == m_setJSObjectProt ||
                 methodId == m_getLastErrDescription
#if !defined(QT_MOBILITY_SFW)
                 || methodId == m_npidGetServiceVersion);
#else
                 );
#endif
    return bRet;
}

/*!
    Dispatch a script call to the public method in the the this object

    @param methodId the identifier of the method
    @param args the list of arguments for the method
    @param argCount number of arguments.
    @param[out] result  return value of the method
    @return true if the method call succeeded, otherwise false

    @see HasMethod
*/
bool ServiceFactory::Invoke(NPIdentifier methodId, const NPVariant *args,
                            uint32_t argCount, NPVariant *result)
{
    bool bRet = false;

    if (methodId == m_npidGetServiceObject ) {
        if (argCount >= 2){
            if (NPVARIANT_IS_STRING(args[0]) && NPVARIANT_IS_STRING(args[1]) &&
                            (argCount != 3 || NPVARIANT_IS_STRING(args[2]))) {
                const NPString *s = &NPVARIANT_TO_STRING(args[0]);
                const NPString *s2 = &NPVARIANT_TO_STRING(args[1]);
                const NPString *s3;
                if (argCount == 3) {
                    s3 = &NPVARIANT_TO_STRING(args[2]);
                }
                else {
                    NPVariant v;
                    STRINGZ_TO_NPVARIANT (KDefaultVersion, v); //set default value to get latest version
                    s3 = &NPVARIANT_TO_STRING(v);
                }
                bRet = getServiceObject(s,s2,s3,result);
            } else {
                NPN_SETEXCEPTION(this, ServiceFactory::SFW_ERROR_PARAMETER_TYPE_MISMATCH,
                        ServiceFactoryErrDesc[ServiceFactory::SFW_ERROR_PARAMETER_TYPE_MISMATCH]);
            }
       } else {
           NPN_SETEXCEPTION(this, ServiceFactory::SFW_ERROR_NUMBER_OF_PARAMETERS,
                   ServiceFactoryErrDesc[ServiceFactory::SFW_ERROR_NUMBER_OF_PARAMETERS]);
       }
    }
    else if (methodId == m_npidListServices) {
        bRet = getServiceProviders(result);
    }
#if !defined(QT_MOBILITY_SFW)
    else if (methodId == m_npidGetServiceVersion) {
        if (argCount >= 1){
            if (NPVARIANT_IS_STRING(args[0])) {
                const NPString *s = &NPVARIANT_TO_STRING(args[0]);
                bRet = getServiceVersion(s,result);
            } else {
                NPN_SETEXCEPTION(this, ServiceFactory::SFW_ERROR_PARAMETER_TYPE_MISMATCH,
                        ServiceFactoryErrDesc[ServiceFactory::SFW_ERROR_PARAMETER_TYPE_MISMATCH]);
            }
       } else {
           NPN_SETEXCEPTION(this, ServiceFactory::SFW_ERROR_NUMBER_OF_PARAMETERS,
                   ServiceFactoryErrDesc[ServiceFactory::SFW_ERROR_NUMBER_OF_PARAMETERS]);
       }
    }
#endif
    else if (methodId == m_getLastErrCode) {
        int errCode = this->getLastErrCode();
        INT32_TO_NPVARIANT(errCode, *result );
        bRet = true;
    }
    else if (methodId == m_getLastErrDescription) {
        QByteArray ba = this->getLastErrDescription().toUtf8();
        int size = ba.size();
        m_content = (char*) NPN_MemAlloc(size + 1);
        strcpy(m_content, ba.data());
        m_content[size] = '\0';
        STRINGZ_TO_NPVARIANT(m_content, *result);
        bRet = true;
    }
    else if (methodId == m_setJSObjectProt ) {
#ifdef __SYMBIAN32__
        if ( argCount == 3 &&
            NPVARIANT_IS_OBJECT(args[0]) &&
            NPVARIANT_IS_OBJECT(args[1]) &&
            NPVARIANT_IS_OBJECT(args[2]) ) {
            NPObject* objectFunc = NPVARIANT_TO_OBJECT(args[0]);
            NPObject* arrayFunc = NPVARIANT_TO_OBJECT(args[1]);
            NPObject* dateFunc = NPVARIANT_TO_OBJECT(args[2]);
            m_bindingUtility.initNPFunctionPrototypes(objectFunc,arrayFunc,dateFunc);
            bRet = true;
        }
#else
        // STUB out for Windows, Maemo and other implementations.
        // TBD: Maemo fix
        bRet = true;
#endif
    }
    return bRet;
}

/*!
    Check if a property exists or not in service factory object

    @param propertyId the id of property
    @return true if the property exist, otherwise false

    @see GetProperty
*/
bool ServiceFactory::HasProperty(NPIdentifier propertyId)
{
    bool bRet = false;
    if (propertyId == m_PropertyToken) {
        bRet = true;
    }
    return bRet;
}

/*!
    Get property

    @param propertyId the id of property
    @param[out] result property value
    @return true if succeeded, otherwise false

    @see HasProperty
*/
bool ServiceFactory::GetProperty(NPIdentifier propertyId, NPVariant *result)
{
    bool bRet = false;
    if (propertyId == m_PropertyToken) {
#if !defined(__SERIES60_31__) && defined(__SYMBIAN32__)
        //TODO:
        //createSecSession();
        //QVariant value(m_SecSession->token());
        //*result = m_bindingUtility.fromQVariant(value);
        //if (!NPVARIANT_IS_VOID(*result)) {
        //    bRet = true;
        //}
#else
        (void)result;
        bRet = true;
#endif //__SERIES60_31__
    }
    return bRet;
}

/*!
    Set property

    @param propertyId the id of property
    @param value property value
    @return true if succeeded, otherwise false

    @see GetProperty
*/
bool ServiceFactory::SetProperty(NPIdentifier propertyId, const NPVariant *value)
{
    bool bRet = false;
    if (propertyId == m_PropertyToken) {
#if !defined(__SERIES60_31__) && defined(__SYMBIAN32__)
        // TODO:
        //createSecSession();
        //QVariant token = m_bindingUtility.toQVariant(*value);
        //m_SecSession->setToken(token.toString());
#else
        (void)value;
        bRet = true;
#endif //__SERIES60_31__
    }
    return bRet;
}

/*!
    create a binding object

    @param npClass pointer to NPClass structure
    @return constructed npobject
*/
NPObject* ServiceFactory::createBindingObject(NPClass* npClass)
{
    NPObject* pObj = NPN_CreateObject(m_Npp, npClass);
    ObjectBinding* bindingObj = static_cast<ObjectBinding*>(pObj);
    if (bindingObj) {
        Q_ASSERT(!m_bindingObjectList.contains(bindingObj));
        bindingObj->setFactory(this);
        m_bindingObjectList.append(bindingObj);
    }
    return pObj;
}

/*!

    @param npClass pointer to NPClass structure
*/
void ServiceFactory::removeBindingObject(ObjectBinding* bindingObject)
{
    m_bindingObjectList.removeOne(bindingObject);
}

/*!
    Create a service object

    @param providerName provider name
    @param moduleName module name in the provider
    @param interfaceName interface name in the module
    @param version version of the provider
    @param[out] result returned service object
    @return true if the method call succeeded, otherwise false

    @see getServiceProviders
*/
bool ServiceFactory::getServiceObject(const NPString *providerName,
                                      const NPString *interfaceName,
                                      const NPString *interfaceVersion,
                                      NPVariant *result )
{
    CLEAR_ERRORS(this);
    bool bRet(true);

// enable secsession for all except in 3.1, 3.2 emulator environment
#if !defined(__WINSCW__) || defined(__S60_50__)
    if (!createSecSession()) {
        // terminate if secsession is not created successfully
        NULL_TO_NPVARIANT(*result);
        NPN_SETEXCEPTION(this, ServiceFactory::SFW_ERROR_SECSESSION_ERROR,
                           ServiceFactoryErrDesc[ServiceFactory::SFW_ERROR_SECSESSION_ERROR]);
        return false;
    }
#endif

    NPObject* pObj = createBindingObject(GET_NPOBJECT_CLASS(ServiceObjectBinding));

    if (!pObj) {
        bRet = false;
        NULL_TO_NPVARIANT(*result);
    }
    else {
        // create service object, service handler own the service object
#if defined(QT_MOBILITY_SFW)
        QObject* base(NULL);
#else
        IServiceBase* base(NULL);
#endif
        ServiceInterfaceInfo interfaceInfo = { NULL, 0 };
        const char* namePtr(NULL);
        const char* versionPtr(NULL);

#if defined(QT_MOBILITY_SFW)
        QByteArray name = BindingUtility::NPStringToQByteArray(providerName);
#else
        QByteArray name = BindingUtility::NPStringToQByteArray(interfaceName);
#endif
        namePtr = name.constData();
        QByteArray version = BindingUtility::NPStringToQByteArray(interfaceVersion);
        versionPtr = version.constData();
        QByteArray name_ver(namePtr);
        name_ver.append(KSlash);
        name_ver.append(versionPtr);
        interfaceInfo.m_iid = name_ver.constData();

#if defined(QT_MOBILITY_SFW)
        ServiceContext context;
        context.setClientName(name);
        context.setClientId(QString(name_ver.constData()));
        // Get a list of the interfaces provided by the service.
        QList<QServiceInterfaceDescriptor> desc = m_serviceManager->findInterfaces(name);

        // Get interface with certain name and version.
        for (int i = 0; i < desc.size(); i++)
        {
            QString versionInfo = QString::number(desc.at(i).majorVersion()) + "." + QString::number(desc.at(i).minorVersion());
            if (versionInfo == version &&
                desc.at(i).interfaceName() == BindingUtility::NPStringToQByteArray(interfaceName))
            {
                base = m_serviceManager->loadLocalTypedInterface<QObject>(desc.at(i),
                                         qobject_cast<QServiceContext*>(&context),
                                         qobject_cast<QAbstractSecuritySession*>(m_SecSession));
                break; // Service found, stop looping.
            }
        }
#else
        m_ServiceHandler->getInterface(&base,
            BindingUtility::NPStringToQString(providerName),
            interfaceInfo,
            true,
            m_SecSession);
#endif

        if (base) { // service object created successfully
            static_cast<ServiceObjectBinding*>(pObj)->setInterface(
                    base,
                    m_bindingUtility);
            OBJECT_TO_NPVARIANT(pObj, *result);
#if !defined(QT_MOBILITY_SFW)
            base->release(); // ownership passed to service object binding
#endif
        } else { // failed to create service object
            NPN_ReleaseObject(pObj);
            //return null to javascript if service object is not found.
            NULL_TO_NPVARIANT(*result);
            bRet = false;
#if defined(QT_MOBILITY_SFW)
            NPN_SETEXCEPTION(this, m_serviceManager->error(), getErrorText(m_serviceManager->error()).toUtf8());
#else
            NPN_SETEXCEPTION(this, m_ServiceHandler->getLastErrCode(),
                                   m_ServiceHandler->getLastErrDescription().toLatin1());
#endif
        }
    }
    return bRet;
}

#if !defined(QT_MOBILITY_SFW)
/*!
    Get service provider version
    @param providerName provider name
    @param[out] result version string
    @return true if the method call succeeded, otherwise false

    @see getServiceObject
*/
bool ServiceFactory::getServiceVersion(const NPString *providerName,NPVariant* result)
{
    CLEAR_ERRORS(this);

// enable secsession for all except in 3.1, 3.2 emulator environment
#if !defined(__WINSCW__) || defined(__S60_50__)
    if (!createSecSession()) {
        NPN_SETEXCEPTION(this, ServiceFactory::SFW_ERROR_SECSESSION_ERROR,
                               ServiceFactoryErrDesc[ServiceFactory::SFW_ERROR_SECSESSION_ERROR]);
        return false;
    }
#endif
    bool bRet(true);

    QString version = m_ServiceHandler->getServiceVersion(BindingUtility::NPStringToQString(providerName));
    if (m_ServiceHandler->getLastErrCode() != 0) {
            NPN_SETEXCEPTION(this, m_ServiceHandler->getLastErrCode(),
                             m_ServiceHandler->getLastErrDescription().toLatin1());
            bRet = false;
    } else {
        QVariant var(version);
        *result = m_bindingUtility.fromQVariant(var);
    }
    return bRet;
}
#endif

/*!
    Get all of service providers

    @param[out] result array of service providers
    @return true if the method call succeeded, otherwise false

    @see getServiceObject
*/
bool ServiceFactory::getServiceProviders(NPVariant* result)
{
    CLEAR_ERRORS(this);

// enable secsession for all except in 3.1, 3.2 emulator environment
#if !defined(__WINSCW__) || defined(__S60_50__)
    if (!createSecSession()) {
        NPN_SETEXCEPTION(this, ServiceFactory::SFW_ERROR_SECSESSION_ERROR,
                               ServiceFactoryErrDesc[ServiceFactory::SFW_ERROR_SECSESSION_ERROR]);
        return false;
    }
#endif

    // Get the list of providers and convert to NP Variant
#if defined(QT_MOBILITY_SFW)
    QStringList providers = m_serviceManager->findServices();
    if (m_serviceManager->error() != QServiceManager::NoError) {
        NPN_SETEXCEPTION(this, m_serviceManager->error(), getErrorText(m_serviceManager->error()).toUtf8());
#else
    QStringList providers = m_ServiceHandler->getServiceProviders(m_SecSession);
    if (m_ServiceHandler->getLastErrCode() != 0) {
            NPN_SETEXCEPTION(this, m_ServiceHandler->getLastErrCode(),
                             m_ServiceHandler->getLastErrDescription().toLatin1());
#endif
    } else {
        QVariant var(providers);
        *result = m_bindingUtility.fromQVariant(var);
    }
    return true;
}

// enable secsession for all except in 3.1, 3.2 emulator environment
#if !defined(__WINSCW__) || defined(__S60_50__)

/*!
 * Gets the host URL of the client
 * \return Host URL
 */
QString ServiceFactory::getHostUrl()
{
    QString hostUrl;
    NPVariant strVar;
    STRINGZ_TO_NPVARIANT(KLocationHREF, strVar);

    NPVariant resultVariant;
    VOID_TO_NPVARIANT(resultVariant);
    bool success = NPN_Evaluate(m_Npp, m_bindingUtility.windowObject(),
                              &NPVARIANT_TO_STRING(strVar),
                              &resultVariant);
    if (success && resultVariant.type == NPVariantType_String) {
        int length = NPSTRING_LENGTH(resultVariant.value.stringValue);
        if (length > 0) {
            const NPString* tempStr = &NPVARIANT_TO_STRING(resultVariant);
            hostUrl = BindingUtility::NPStringToQString(tempStr);
        }
    }
    BindingUtility::freeNPVariant(resultVariant);
    return hostUrl;
}

// secsession is needed for both Avkon & CWRT
/*!
 * Creates a new Trust session and save it to internal member variable
 * \return void
 */
void ServiceFactory::createTrustSession()
{
    if (!m_TrustSession) {
        // Get location of the trust policy file
        QString fileLocation = WrtSettings::createWrtSettings()->valueAsString("SecurityTrustPolicyFile");
        m_TrustSession = new TrustSession(fileLocation.toLower());
    }
}


/*!
 * Creates a new security manager session if not created already,
 * and save it to internal member variable.
 * \return true if succeeded, otherwise false
 */

#if !defined(__SYMBIAN32__) || defined(CWRT_S60_WEBKIT)

bool ServiceFactory::createSecSession()
{
    bool rtVal(false);

    // find widget's pkgPath
    QString pkgPath(QDir::fromNativeSeparators(getHostUrl()));

#ifdef __MAEMO__
    int index = pkgPath.lastIndexOf(QString(KSlash));
    pkgPath.truncate(index+1);
#endif

    if (pkgPath.startsWith(KCWRTWidgetHostScheme) ||
        pkgPath.startsWith(KCWRTSandboxScheme)) {
        QString widgetSchemeBase(DEFAULT_ROOT_DIRECTORY +
                                 QDir::separator()+
                                 WIDGET_FOLDER +
                                 QDir::separator());
#ifdef __SYMBIAN32__

        int pos = pkgPath.indexOf(PRIVATE_DIRECTORY);
        QString ProcessUid = pkgPath.mid(pos).section('/',2,2);
        widgetSchemeBase.replace(DEFAULT_DIRECTORY,ProcessUid);

#endif
        widgetSchemeBase = QDir::fromNativeSeparators(widgetSchemeBase);

        // Remove drive letter if it exists
        widgetSchemeBase.replace(
                QRegExp(KCWRTDriveLetter, Qt::CaseInsensitive), KCWRTDriveLetter);

        QRegExp widgetLibRegExp(widgetSchemeBase +
                                SHARED_LIBRARY_FOLDER +
                                "/\\w+/",
                                Qt::CaseInsensitive);
        QRegExp widgetRegExp(widgetSchemeBase +
                             "\\w+/",
                             Qt::CaseInsensitive);
        if (widgetLibRegExp.indexIn(pkgPath) != -1) {
            pkgPath = widgetLibRegExp.capturedTexts().at(0);
        } else if (widgetRegExp.indexIn(pkgPath) != -1) {
            pkgPath = widgetRegExp.capturedTexts().at(0);
        }
    } else if (pkgPath.startsWith(KCWRTSharedLibScheme)) {
        QRegExp libRegExp(QString(KCWRTSharedLibScheme) +
                          "\\w+/",
                          Qt::CaseInsensitive);
        if (libRegExp.indexIn(pkgPath) != -1) {
            pkgPath = libRegExp.capturedTexts().at(0);
            pkgPath.replace(KCWRTSharedLibScheme,
                            QDir::fromNativeSeparators(SHARED_LIBRARY_PATH));
        }
    } else {
        // possibly a local file
        QUrl url(pkgPath);
        QString localFile = url.toLocalFile();
        if (!localFile.isNull() && !localFile.isEmpty()) {
            QFileInfo info(localFile);
            pkgPath = info.absolutePath();
        }
    }

    // Get widget properties
    WidgetManager* widgetManager = new WidgetManager(NULL);
    widgetManager->launcherPath(pkgPath);
    WidgetProperties* props = widgetManager->getProperties(pkgPath);

    // create secsession & setup attributes
    if (props) {
        // First check if m_SecSession object exists already.
        // If it does, then double check that the attributes and
        // capabilities are correct.  This is to protect against the
        // case of a widget hijacking this object from another widget.
        bool newSession(true);
        if (m_SecSession) {
            QHash<QString, QString> clientInfo = m_SecSession->clientInfo();
            if (clientInfo[KID] == props->id() &&
                clientInfo[KNAME] == props->id() &&
                clientInfo[KCLIENTNAME] == props->id() &&
                clientInfo[KWIDGETPATH] == props->installPath() &&
                clientInfo[KWIDGETDATAPATH] == props->resourcePath() &&
                clientInfo[KDISPLAYTEXT] == props->title()) {
                    newSession = false;
                    rtVal = true;
                }
        }

        // Create a new session if none existed before
        // (m_SecSession is NULL), or if the previous SecSession
        // does not match this widget
        if (newSession) {
            QFile secSessionFile(props->resourcePath() + QString("secsession"));
            if (secSessionFile.open(QIODevice::ReadOnly)) {
                delete m_SecSession;
                m_SecSession = NULL;
                m_SecSession = new SecSession(secSessionFile.readAll());
                m_SecSession->setClientInfo(KID, props->id());
                m_SecSession->setClientInfo(KNAME, props->id());
                m_SecSession->setClientInfo(KCLIENTNAME, props->id());
                m_SecSession->setClientInfo(KWIDGETPATH, props->installPath());
                m_SecSession->setClientInfo(KWIDGETDATAPATH, props->resourcePath());
                m_SecSession->setClientInfo(KDISPLAYTEXT, props->title());
                secSessionFile.close();
                rtVal = true;
            }
        }
    }

    // If cannot find widget, then create default SecSession based on the origin
    if (!rtVal) {
        createTrustSession();
        QString domain = m_TrustSession->domainFor(ETrustCriteriaOrigin, pkgPath);
        QString fileLocation = WrtSettings::createWrtSettings()->valueAsString("SecurityAccessPolicyFile");
        delete m_SecSession;
        m_SecSession = NULL;
        m_SecSession = new SecSession(fileLocation.toLower(),
                                      domain,
                                      QString());
        m_SecSession->setClientInfo(KCLIENTNAME, pkgPath);
        m_SecSession->setClientInfo(KWIDGETPATH, pkgPath);
        m_SecSession->setClientInfo(KWIDGETDATAPATH, pkgPath);
        rtVal = true;
    }

    delete widgetManager;
    return rtVal;
}

#else

// Avkon browser secsession creation
bool ServiceFactory::createSecSession()
{
    TrustCriteriaType type;
    QString domainParam;
    QString hostUrl;

    bool rtValue(false);

    // Check which process it belongs to
    // If part of a widget runtime, use the process ID as criteria for checking domain
    TUint32 secureId = RProcess().SecureId().iId;
    if (secureId == KSecureWidgetId || secureId == KNormalWidgetId) {
        type = ETrustCriteriaProcessId;
        domainParam.setNum(secureId);
    }
    else
    {
        // Otherwise, use the URL value to check domain.
        // We make a second copy of this because the hostURL will
        // be used separately later
        type = ETrustCriteriaOrigin;
        domainParam = hostUrl = getHostUrl();
    }

    if (!domainParam.isNull() && !domainParam.isEmpty() && m_TrustSession) {
        // Get Domain from Trust manager
        QString domain = m_TrustSession->domainFor(type, domainParam);

        // Compare this domain to the current domain.  Only create a new
        // SecSession if it's the first time, or if the domain doesn't match
        // what was created last time.  If the domains match, then we can
        // re-use the same SecSession.
        if (!m_SecSession || domain != m_CurrentDomain) {
            delete m_SecSession;
            m_SecSession = NULL;
            m_CurrentDomain = domain;

            // Get location of the access policy file
            QString fileLocation = WrtSettings::createWrtSettings()->valueAsString("SecurityAccessPolicyFile");
            m_SecSession = new SecSession(fileLocation.toLower(),
                                          m_CurrentDomain,
                                          QString());
            if (hostUrl.isNull() || hostUrl.isEmpty()) {
                hostUrl = getHostUrl();
            }

            // setup clientname and clientpath attributes in SecSession
            QString clientName = getClientName(hostUrl);
            m_SecSession->setClientInfo(KCLIENTNAME, clientName);
            QString clientPath = getClientPath(hostUrl, clientName);
            m_SecSession->setClientInfo(KWIDGETPATH, clientPath);
            m_SecSession->setClientInfo(KWIDGETDATAPATH, clientPath);
        }
        rtValue = true;  // return true even in the case of recycling SecSesion
    }

    return rtValue;
}

// additional variables & functions needed only for Avkon browser

/*!
 * Gets the client name given the host url.
 * Parses the URL and if it belongs to a widget then gets the name
 * of the widget.  Otherwise the client name is the same as the hosturl.
 * \return Client Name
 */
QString ServiceFactory::getClientName(const QString &aHostUrl)
{
    QString clientName = aHostUrl;

    // Check which process it belongs to
    // If part of a widget runtime, use the process ID as criteria for checking domain
    TUint32 secureId = RProcess().SecureId().iId;
    QString fileScheme;
    if (secureId == KSecureWidgetId) {
        fileScheme.append(KSecureWidgetFileScheme);
    } else if (secureId == KNormalWidgetId) {
        fileScheme.append(KNormalWidgetFileScheme);
    }
    if (!fileScheme.isNull() && !fileScheme.isEmpty()) {
        // Get the widget name by looking up the main HTML
        // Use regular expression to ignore drive letter
        if (clientName.contains(QRegExp(fileScheme, Qt::CaseInsensitive))) {
            clientName.remove(QRegExp(fileScheme, Qt::CaseInsensitive));
            int index = clientName.indexOf(QString(KSlash));
            if (index >= 0) {
                clientName.truncate(index);
            }
        }
    }

    return clientName;
}

/*!
 * Gets the client path given the host url and client name.
 * Parses the URL and if it belongs to a widget then gets the path
 * of the widget.  Otherwise the client path is set to the system temp directory.
 * \return Client Path
 */
QString ServiceFactory::getClientPath(const QString &aHostUrl, const QString &aClientName)
{
    // return system temp directory if this is not a widget
    // (ie. host url == client name)
    if (aHostUrl == aClientName) {
        return QDir::tempPath();
    }

    QString clientPath = aHostUrl;

    // no trailing "/" to align with Qt convention
    int truncateIndex = clientPath.lastIndexOf(aClientName) + aClientName.length();
    clientPath.truncate(truncateIndex);
    clientPath.remove(QString("file:///"));

    return clientPath;
}

#endif
#endif


#ifdef __cplusplus
extern "C" {
#endif

/*!
    Static helper function to create factory object

*/
Q_DECL_EXPORT NPObject* createFactoryObject(NPP npp)
{
    NPObject* obj = NPN_CreateObject(npp, GET_NPOBJECT_CLASS(ServiceFactory));
    return obj;
}

#ifdef __cplusplus
}
#endif

} // namespace
// END OF FILE
