/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QMetaObject>
#include <QMetaClassInfo>
#include <QVector>
#include <QByteArray>
#include <QMap>
#include <QDateTime>

#ifdef __SYMBIAN32__
#include <string.h>

#if !defined(__S60_50__)
#include <npn_enumerate.h>
#include <javascriptcore/npruntime.h>
#endif //__S60_50__

#endif // __SYMBIAN32__

#include "bindingutility.h"
#include "sfwexception.h"

namespace WRT
{
static const int MAX_UTC_YEAR_SUPPORTED = 2105;
static const int MIN_UTC_YEAR_SUPPORTED = 1970;
static const char KGetTime[] = "getTime";
static const char KValueOf[] = "valueOf";
static const char KObjectScript[] = "new Object()";
static const char KArrayScript[] = "new Array();";
static const char KDateScript[] = "new Date();";
static const char KPush[] = "push";
static const char KSetTime[] = "setTime";
#ifdef  __SYMBIAN32__
static const char KNumberScript[] = "new Number();";
static const char KBooleanScript[] = "new Boolean();";
static const char KStringScript[] = "new String();";
static const TInt KContructorCompareLength = 12;
#else
static const char KConstructor[] = "constructor";
static const char KToString[] = "toString";
static const char KArrayConstructor[] = "function Array()";
static const char KDateConstructor[] = "function Date()";
static const char KNumberConstructor[] = "function Number()";
static const char KStringConstructor[] = "function String()";
static const char KBooleanConstructor[] = "function Boolean()";
#endif

/*!
    \class BindingUtility
    \brief The BindingUtility class helps conversion between Javascript data types and Qt data types
*/

/*!
    Constructor
    @param npp NPP instance handle

*/
BindingUtility::BindingUtility(NPP npp)
: m_npp(npp),
m_windowObject(NULL),
m_objectFunc(NULL),
m_dateFunc(NULL),
m_arrayFunc(NULL)
{
    NPN_GetValue( npp, NPNVWindowNPObject, &m_windowObject );
    NPN_RetainObject(m_windowObject);
#ifdef __SYMBIAN32__
    // Initialize NPIdentifiers
    m_Ids[Constructor] = NPN_GetStringIdentifier("constructor");
    m_Ids[ToString] = NPN_GetStringIdentifier("toString");
    NPIdentifier idNull = NPN_GetStringIdentifier("");
    NPIdentifier valueOf = NPN_GetStringIdentifier("valueOf");
    m_npidCall = NPN_GetStringIdentifier("call");
    // Initialize NPStrings
    STRINGZ_TO_NPVARIANT(KDateScript, m_scripts[DateObject]);
    STRINGZ_TO_NPVARIANT(KArrayScript, m_scripts[ArrayObject]);
    STRINGZ_TO_NPVARIANT(KNumberScript, m_scripts[NumberObject]);
    STRINGZ_TO_NPVARIANT(KBooleanScript, m_scripts[BooleanObject]);
    STRINGZ_TO_NPVARIANT(KStringScript, m_scripts[StringObject]);

    // Instantiate the reference objects' constructor object
    NPVariant tempConstructor;
    NPVariant tempVariant;
    for (int i=0; i<JSObjectTypesCount; i++) {
        VOID_TO_NPVARIANT(tempVariant);
        VOID_TO_NPVARIANT(tempConstructor);
        NPN_Evaluate(npp, m_windowObject, &NPVARIANT_TO_STRING(m_scripts[i]), &tempVariant);
        NPN_GetProperty(npp, NPVARIANT_TO_OBJECT(tempVariant), m_Ids[Constructor], &tempConstructor);
        m_constructor[i] = reinterpret_cast<TUint8*>(NPVARIANT_TO_OBJECT(tempConstructor));
        NPN_ReleaseObject(NPVARIANT_TO_OBJECT(tempVariant));
    }
#endif
}

/*!
    Destructor
*/
BindingUtility::~BindingUtility()
{
#ifdef __SYMBIAN32__
    //release resource
    for (int i=0; i<JSObjectTypesCount; i++) {
        NPN_ReleaseObject(reinterpret_cast<NPObject*>(m_constructor[i]));
    }
#endif
    NPN_ReleaseObject(m_windowObject);
    if ( m_objectFunc ) {
        NPN_ReleaseObject(m_objectFunc);
    }
    if ( m_arrayFunc ) {
        NPN_ReleaseObject(m_arrayFunc);
    }
    if ( m_dateFunc ) {
        NPN_ReleaseObject(m_dateFunc);
    }
}

/*!
    Utility function to convert from NPVariant to QVariant
    @param var Source NPVariant
    @return QVariant

    @see fromQVariant
*/
QVariant BindingUtility::toQVariant(const NPVariant& var)
{
    QVariant retVar;
    switch (var.type) {
    case NPVariantType_String: {
        const NPString *str = &NPVARIANT_TO_STRING(var);
        retVar.setValue(NPStringToQString(str));
        break;
    }
    case NPVariantType_Int32: {
        int i32 = NPVARIANT_TO_INT32(var);
        retVar.setValue(i32);
        break;
    }
    case NPVariantType_Bool: {
        bool bl = NPVARIANT_TO_BOOLEAN(var);
        retVar.setValue(bl);
        break;
    }
    case NPVariantType_Double: {
        double dbl = NPVARIANT_TO_DOUBLE(var);
        retVar.setValue(dbl);
        break;
    }
    case NPVariantType_Object: {
        NPObject *pNpobj = NPVARIANT_TO_OBJECT(var);
        // get all of properties
        NPIdentifier* propids(NULL);
        uint32_t propCount(0);
#ifndef  __SYMBIAN32__
        // check constructor, may not work on S60, a bug logged.
        NPIdentifier cstorId = NPN_GetStringIdentifier(KConstructor);
        NPIdentifier toStringId = NPN_GetStringIdentifier(KToString);
        NPVariant cstorVar;
        VOID_TO_NPVARIANT(cstorVar);
        NPVariant cstorStringVar;
        VOID_TO_NPVARIANT(cstorStringVar);
        NPN_GetProperty(m_npp, pNpobj, cstorId, &cstorVar);
        NPN_Invoke(m_npp, NPVARIANT_TO_OBJECT(cstorVar),
                   toStringId, NULL, 0, &cstorStringVar);
        NPN_ReleaseObject(NPVARIANT_TO_OBJECT(cstorVar));
        QString cstorStr = toQVariant(cstorStringVar).toString();
        freeNPString(NPVARIANT_TO_STRING(cstorStringVar));

        if (cstorStr.contains(QString(KArrayConstructor))) { // Array Object
            NPN_Enumerate(m_npp, pNpobj, &propids, &propCount);
            retVar = fromArray(pNpobj, propids, propCount);
        }
        else if (cstorStr.contains(KDateConstructor)) { //Date Object
            retVar = fromDate(pNpobj);
        }
        else if (cstorStr.contains(KNumberConstructor)) { // Number object
            retVar = fromNumber(pNpobj);
        }
        else if (cstorStr.contains(KStringConstructor)) { // String object
            retVar = fromString(pNpobj);
        }
        else if (cstorStr.contains(KBooleanConstructor)) { // Boolean object
            retVar = fromBool(pNpobj);
        }
        else { // Handle JS Object
            NPN_Enumerate(m_npp, pNpobj, &propids, &propCount);
            retVar = fromObject(pNpobj, propids, propCount);
        }
// Workaround for S60 v3.1, v3.2
// TODO, could be removed for S60 v5.0 when the TSW error BYAG-7LU2FH fixed.
#else
        // Instantiate the input object constructor object
        NPVariant tempConstructor;
        VOID_TO_NPVARIANT(tempConstructor);
        NPN_GetProperty(m_npp, pNpobj, m_Ids[Constructor], &tempConstructor);
        const TUint8* pNpobjConstructorPtr =
                        (const TUint8*)NPVARIANT_TO_OBJECT(tempConstructor);

        // Handle Date object
        if (Mem::Compare(pNpobjConstructorPtr, KContructorCompareLength,
                m_constructor[DateObject], KContructorCompareLength) == 0) {
            retVar = fromDate(pNpobj);
        }
        // Handle Boolean object
        else if (Mem::Compare(pNpobjConstructorPtr, KContructorCompareLength,
                m_constructor[BooleanObject], KContructorCompareLength) == 0) {
            retVar = fromBool(pNpobj);
        }
        // Handle Number object
        else if (Mem::Compare(pNpobjConstructorPtr, KContructorCompareLength,
                m_constructor[NumberObject], KContructorCompareLength) == 0) {
            retVar = fromNumber(pNpobj);
        }
        // Handle String object
        else if (Mem::Compare(pNpobjConstructorPtr, KContructorCompareLength,
                m_constructor[StringObject], KContructorCompareLength) == 0) {
            retVar = fromString(pNpobj);
        }
        // Handle Array object
        else if (Mem::Compare(pNpobjConstructorPtr, KContructorCompareLength,
                m_constructor[ArrayObject], KContructorCompareLength) == 0) {
            NPN_Enumerate(m_npp, pNpobj, &propids, &propCount);
            retVar = fromArray(pNpobj, propids, propCount);
        }
        // Handle JS Object
        else {
            NPN_Enumerate(m_npp, pNpobj, &propids, &propCount);
            retVar = fromObject(pNpobj, propids, propCount);
        }
        NPN_ReleaseObject(NPVARIANT_TO_OBJECT(tempConstructor));
#endif //  __SYMBIAN32__
        if (propids) {
            NPN_MemFree(propids);
        }
        break;
    }
    case NPVariantType_Void:
    case NPVariantType_Null:
    default: {
        break;
    }
    }   // end of switch

    return retVar;
}

/*!
    Utility function to convert from QVariant to NPVariant
    @param qvariant source QVariant
    @return result NPVariant

    @see toQVariant
*/
NPVariant BindingUtility::fromQVariant(const QVariant& qvariant)
{
    NPVariant npvar;
    NULL_TO_NPVARIANT(npvar);

    switch (qvariant.type()) {
    case QVariant::Bool:
        BOOLEAN_TO_NPVARIANT(qvariant.toBool(),npvar);
        break;
    case QVariant::Int:
        INT32_TO_NPVARIANT(qvariant.toInt(),npvar);
        break;
    case QVariant::Double:
        DOUBLE_TO_NPVARIANT(qvariant.toDouble(),npvar);
        break;
    case QVariant::Map: {
        NPObject* jsObject;
        if ( !m_objectFunc ) {
            jsObject = Evaluate(KObjectScript);
        }
        else {
            NPVariant result;
            bool success = NPN_Invoke(m_npp, m_objectFunc, m_npidCall, NULL, 0, &result);
            if ( NPVARIANT_IS_OBJECT(result) && success) {
                jsObject = NPVARIANT_TO_OBJECT(result);
            }
            else {
                jsObject = Evaluate(KObjectScript);
            }
        }
        QVariantMap map = qvariant.toMap();
        QVariantMap::const_iterator i = map.constBegin();
        while (i!=map.constEnd()) {
            QString key = i.key();
            NPVariant variant = fromQVariant(i.value());
            NPN_SetProperty(m_npp,jsObject,
                            NPN_GetStringIdentifier(key.toUtf8().data()),&variant);
            ++i;
        }
        OBJECT_TO_NPVARIANT(jsObject,npvar);
        break;
    }
    case QVariant::List:
    case QVariant::StringList: {
        NPObject* jsObject;
        if ( !m_arrayFunc ) {
            jsObject = Evaluate(KArrayScript);
        }
        else {
            NPVariant result;
            bool success = NPN_Invoke(m_npp, m_arrayFunc, m_npidCall, NULL, 0, &result);
            if ( NPVARIANT_IS_OBJECT(result) && success) {
                jsObject = NPVARIANT_TO_OBJECT(result);
            }
            else {
                jsObject = Evaluate(KArrayScript);
            }
        }
        NPVariant result;
        NPIdentifier pushId = NPN_GetStringIdentifier(KPush);
        QVariantList list = qvariant.toList();
        foreach (const QVariant& item, list) {
            NPVariant variant = fromQVariant(item);
            NPN_Invoke(m_npp,jsObject,pushId,&variant,1,&result);
        }
        OBJECT_TO_NPVARIANT(jsObject,npvar);
        break;
    }
    case QVariant::DateTime: {
        double secs=qvariant.toDateTime().toUTC().toTime_t();
        double MSecs = 1000*secs;
        MSecs += qvariant.toDateTime().time().msec();
        NPVariant variant;
        DOUBLE_TO_NPVARIANT(MSecs, variant);

        NPObject* jsObject;
        if ( !m_dateFunc ) {
            jsObject = Evaluate(KDateScript);
        }
        else {
            NPVariant result;
            bool success = NPN_Invoke(m_npp, m_dateFunc, m_npidCall, NULL, 0, &result);
            if ( NPVARIANT_IS_OBJECT(result) && success) {
                jsObject = NPVARIANT_TO_OBJECT(result);
            }
            else {
                jsObject = Evaluate(KDateScript);
            }
        }

        NPVariant result;
        NPIdentifier setTimeId = NPN_GetStringIdentifier(KSetTime);
        NPN_Invoke(m_npp,jsObject,setTimeId,&variant,1,&result);
        OBJECT_TO_NPVARIANT(jsObject,npvar);
        break;
    }
    default: { // including QVariant::String
        // make a copy so that not touch original one
        QVariant qvar(qvariant);
        if (!qvar.convert(QVariant::String)) {
            break;
        }
        QString qs = qvar.toString();
        QByteArray ba = qs.toUtf8();
        int size = ba.size();
        char* content = (char*) NPN_MemAlloc(size);
        memset(content, 0, size);
        memcpy(content, ba, size);
        STRINGN_TO_NPVARIANT(content, size, npvar);
        break;
    }
    }
    return npvar;
}

/*!
    Utility function to evaluate a script
    @param script the script to be evaluated.
    @return NPObject or NULL if failed.
*/
NPObject* BindingUtility::Evaluate(const QByteArray& script)
{
    NPVariant jsObjVar;
    NPVariant strVar;
    NPObject* jsObject(NULL);
    STRINGZ_TO_NPVARIANT(script.data(), strVar);

    if (NPN_Evaluate(m_npp, m_windowObject, &NPVARIANT_TO_STRING(strVar), &jsObjVar)) {
        Q_ASSERT(NPVARIANT_IS_OBJECT(jsObjVar));
        jsObject = NPVARIANT_TO_OBJECT(jsObjVar);
    }
    return jsObject;
}

/*!
    Return window object
    @return window NPObject
*/

NPObject* BindingUtility::windowObject()
{
    return m_windowObject;
}

/*!
    Initialize function prototypes
    @param objProto object function prototype
    @param arrayProto array function prototype
    @param dateProto date function prototype
*/
void BindingUtility::initNPFunctionPrototypes(NPObject* objProto,
                                              NPObject* arrayProto,
                                              NPObject* dateProto)
{
    if ( !m_objectFunc ) {
        m_objectFunc = objProto;
        NPN_RetainObject(objProto);
    }
    if ( !m_arrayFunc ) {
        m_arrayFunc = arrayProto;
        NPN_RetainObject(arrayProto);
    }
    if ( !m_dateFunc ) {
        m_dateFunc = dateProto;
        NPN_RetainObject(dateProto);
    }
}

/**
    Utility function to convert from NPString to QString
    @param aString NPString value
    @return QString value
*/
QString BindingUtility::NPStringToQString(const NPString* aString)
{
    return QString::fromUtf8(NPSTRING_UTF8(*aString),NPSTRING_LENGTH(*aString));
}

/**
    Utility function to convert from NPString to QByteArray
    @param aString NPString value
    @return QByteArray
*/
QByteArray BindingUtility::NPStringToQByteArray(const NPString* aString)
{
    const char* data = NPSTRING_UTF8(*aString);
    int size = NPSTRING_LENGTH(*aString);
    return QByteArray(data, size);
}

/**
    Utility function to release NPString content
    @param aString NPString value
*/
void BindingUtility::freeNPString(NPString& aString)
{
    const NPUTF8* utf8 = NPSTRING_UTF8(aString);
    NPN_MemFree((void*)utf8);
}

/**
    Utility function to release NPVariant content
    @param aVariant NPVariant
    @see BindingUtility::freeNPString NPN_ReleaseObject
*/
void BindingUtility::freeNPVariant(NPVariant& aVariant)
{
    switch (aVariant.type) {
    case NPVariantType_Object:
        NPN_ReleaseObject(NPVARIANT_TO_OBJECT(aVariant));
        break;
    case NPVariantType_String:
        freeNPString(NPVARIANT_TO_STRING(aVariant));
        break;
    default:
        break;
        //do nothing
    }
}

/**
    Convert JS array object into a QVariant

    @param npObject the NPObject
    @param propIds properties
    @param propCount the number of properties
    @return Qvariant object
*/
QVariant BindingUtility::fromArray(NPObject* npObject,
                                   NPIdentifier* propIds,
                                   uint32_t propCount)
{
    QVariant retVar;
    NPVariant propValue;
    QVariantList list;
    for (uint32_t i=0; i<propCount; ++i) {
        VOID_TO_NPVARIANT(propValue);
        if (NPN_GetProperty(m_npp, npObject, propIds[i], &propValue)) {
            QVariant qvar = toQVariant(propValue);
            list.append(qvar);
            freeNPVariant(propValue);
        }
    }
    retVar.setValue(list);
    return retVar;
}

/**
    Convert JS date object into a QVariant

    @param npObject the NPObject
    @return Qvariant object
*/
QVariant BindingUtility::fromDate(NPObject* npObject)
{
    QVariant retVar;
    NPIdentifier getTimeId = NPN_GetStringIdentifier(KGetTime);
    NPVariant result;
    VOID_TO_NPVARIANT(result);
    NPN_Invoke(m_npp,npObject,getTimeId,NULL,0,&result);
    qint64 MSecs = NPVARIANT_TO_DOUBLE(result);
    QDateTime dt = QDateTime::fromTime_t(0).addMSecs(MSecs);
    QDate qdate = dt.date();
    if (qdate.year() < MIN_UTC_YEAR_SUPPORTED || qdate.year() > MAX_UTC_YEAR_SUPPORTED) {
#ifdef __MAEMO__
        // In Maemo, no exception thrown, but invalid date object returned
        dt = QDateTime();
#else
        SFWException *e = new SFWException(BindingUtility::SFW_ERROR_INAVLID_DATE,
                                           BindingUtilityErrDesc[BindingUtility::SFW_ERROR_INAVLID_DATE]);
        throw (e);
#endif
    }
    retVar.setValue(dt);
    return retVar;
}

/**
    Convert JS number object into a QVariant

    @param npObject the NPObject
    @return Qvariant object
*/
QVariant BindingUtility::fromNumber(NPObject* npObject)
{
    QVariant retVar;
    NPIdentifier valueOfId = NPN_GetStringIdentifier(KValueOf);
    NPVariant result;
    VOID_TO_NPVARIANT(result);
    NPN_Invoke(m_npp,npObject,valueOfId,NULL,0,&result);
    if (result.type == NPVariantType_Double) {
        double number = NPVARIANT_TO_DOUBLE(result);
        retVar.setValue(number);
    }else if (result.type == NPVariantType_Int32) {
        qint32 number = NPVARIANT_TO_INT32(result);
        retVar.setValue(number);
    }
    return retVar;
}

/**
    Convert JS number object into a QVariant

    @param npObject the NPObject
    @return Qvariant object
*/
QVariant BindingUtility::fromString(NPObject* npObject)
{
    QVariant retVar;
    NPIdentifier valueOfId = NPN_GetStringIdentifier(KValueOf);
    NPVariant result;
    VOID_TO_NPVARIANT(result);
    NPN_Invoke(m_npp,npObject,valueOfId,NULL,0,&result);
    NPString str= NPVARIANT_TO_STRING(result);
    retVar.setValue(NPStringToQString(&str));
    freeNPString(str);
    return retVar;
}

/**
    Convert JS boolean object into a QVariant

    @param npObject the NPObject
    @return Qvariant object
*/
QVariant BindingUtility::fromBool(NPObject* npObject)
{
    QVariant retVar;
    NPIdentifier valueOfId = NPN_GetStringIdentifier(KValueOf);
    NPVariant result;
    VOID_TO_NPVARIANT(result);
    NPN_Invoke(m_npp,npObject,valueOfId,NULL,0,&result);
    bool boolVal = NPVARIANT_TO_BOOLEAN(result);
    retVar.setValue(boolVal);
    return retVar;
}

/**
    Convert JS object into a QVariant

    @param npObject the NPObject
    @param propIds properties
    @param propCount the number of properties
    @return Qvariant object
*/
QVariant BindingUtility::fromObject(NPObject* npObject,
                                    NPIdentifier* propIds,
                                    uint32_t propCount)
{
    QVariant retVar;
    QVariantMap qvarMap;
    NPVariant propValue;
    for (uint32_t i=0; i<propCount; ++i) {
        VOID_TO_NPVARIANT(propValue);
        NPUTF8 *propName=NPN_UTF8FromIdentifier(propIds[i]);
        if (NPN_GetProperty(m_npp, npObject, propIds[i], &propValue)) {
            QVariant qvar = toQVariant(propValue);
            qvarMap.insert(QString(propName), qvar);
            freeNPVariant(propValue);
        }
        NPN_MemFree(propName);
    }
    retVar.setValue(qvarMap);
    return retVar;
}

} // namespace
// END OF FILE
