/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */



#include <QFile>
#include <QBuffer>
#include <QSet>
#include <QXmlStreamWriter>
#include "xmlpolicyserializer.h"
#include "secmgrdefs.h"
#include "storage.h"
#include "secure_defs.h"

static const char KCODEC[] = "ISO-8859-1";
static const char KVERSION[] = "1.0";
static const char KACCESSPOLICY[] = "accesspolicy";

namespace WRT {

/*!
 * \class XMLPolicySerializer
 * Provides serialization implementation into an XML file
 *
 * \note Security Note: XML Policy Serializer provides a simple and efficient serialization
 *  mechanism for session state, but serialized sessions MUST be stored with the same
 * security as the original policy file (since they essentially duplicate portions of the
 * policy file).
 */

/*!
 * Constructor
 */
XMLPolicySerializer::XMLPolicySerializer() :
    m_xmlWriter(NULL) {
}

XMLPolicySerializer::~XMLPolicySerializer() {
    //Temporary fix:
    //TO DO: allocation and de-allocation of iXmlWrtier should be done in a better way.
    if (m_xmlWriter)
        delete m_xmlWriter;
}

/*!
 * Persist the policy
 * \param[in] key - Key used
 * \param[in] store - Storage directory
 * \param[in] domain - Domain to save for
 * \param[in] capabilities - Capabilities to save
 * \param[in] aliases - Aliases to save
 * \return TRUE if successful, else FALSE
 */
bool XMLPolicySerializer::persist(const QString &key, const QString &store,
        const QString &domain, const CapabilitySet *capabilities,
        const AliasSet *aliases)
{
    bool result = false;
    Storage* storage = Storage::createInstance(FILE_STORAGE_NAME,
            WRT::Storage::vis_shared, WRT::Storage::prot_signed);

    QString secKey(store + key);
    QIODevice * ptr = 0;
    bool saveToStorage = false;
    if (storage->getFileList().contains(secKey)) {
        if (!storage->verify(secKey)) {
            //File has been altered. bad boy
            //TODO WHAT TO DO HERE IF THE FILE HAS BEEN ALTERED!!!
            delete storage;
            return false;
        }
        // Get file from storage.
        ptr = storage->getFile(secKey);
    } else {
        // File not in storage
        ptr = new QBuffer();
        saveToStorage = true;
    }

    if (ptr && ptr->open(QIODevice::WriteOnly)) {
        if (!m_xmlWriter) {
            m_xmlWriter = new QXmlStreamWriter(ptr);
        } else {
            m_xmlWriter->setDevice(ptr);
        }
        m_xmlWriter->setAutoFormatting(true);
        m_xmlWriter->setCodec(KCODEC);
        m_xmlWriter->writeStartDocument(KVERSION);

        m_xmlWriter->writeStartElement(KACCESSPOLICY);

        // Write aliases
        QHashIterator<QString, QStringList> aliasIterator(*aliases);
        while (aliasIterator.hasNext()) {
            aliasIterator.next();
            m_xmlWriter->writeStartElement(KALIAS);
            m_xmlWriter->writeAttribute(KNAME, aliasIterator.key());
            CapabilityList caps = aliasIterator.value();
            for (int i = 0; i < caps.size(); i++) {
                m_xmlWriter->writeStartElement(KCAPABILITY);
                m_xmlWriter->writeAttribute(KNAME, caps.at(i));
                m_xmlWriter->writeEndElement(); // KCAPABILITY
            }
            m_xmlWriter->writeEndElement(); // KALIAS
        }

        // Write domain
        m_xmlWriter->writeStartElement(KDOMAIN);
        m_xmlWriter->writeAttribute(KNAME, domain);
        if (capabilities) {
            QHashIterator<QString, QPointer<Condition> >
                    capIterator(*capabilities);
            QSet<Condition*> conditionSet;
            Condition *cond;
            while (capIterator.hasNext()) {
                capIterator.next();
                cond = capIterator.value();
                if (cond) {
                    if (!conditionSet.contains(cond)) {
                        conditionSet.insert(cond);
                    }
                } else {
                    m_xmlWriter->writeStartElement(KCAPABILITY);
                    m_xmlWriter->writeAttribute(KNAME, capIterator.key());
                    m_xmlWriter->writeEndElement(); // KCAPABILITY
                }
            }

            // Write conditions
            QSetIterator<Condition*> condIterator(conditionSet);
            while (condIterator.hasNext()) {
                cond = condIterator.next();
                cond->persist();
            }
        }
        m_xmlWriter->writeEndElement(); // KDOMAIN;

        m_xmlWriter->writeEndElement(); // KACCESSPOLICY
        m_xmlWriter->writeEndDocument();

        ptr->close();

        if (saveToStorage) {
            storage->add(secKey, *(static_cast<QBuffer*>(ptr)));
        }
        //this is not needed anymore
        delete storage;
        delete ptr;

        result = true;
    }
    return result;
}

bool XMLPolicySerializer::persist(QByteArray &blob,  const QString &domain,
                          const CapabilitySet *capabilities, const AliasSet *aliases)
{
    bool result = false;
    QIODevice* ptr = new QBuffer(&blob);
    if (ptr && ptr->open(QIODevice::WriteOnly)) {
        if (!m_xmlWriter) {
            m_xmlWriter = new QXmlStreamWriter(ptr);
        } else {
            m_xmlWriter->setDevice(ptr);
        }
        m_xmlWriter->setAutoFormatting(true);
        m_xmlWriter->setCodec(KCODEC);
        m_xmlWriter->writeStartDocument(KVERSION);

        m_xmlWriter->writeStartElement(KACCESSPOLICY);

        // Write aliases
        QHashIterator<QString, QStringList> aliasIterator(*aliases);
        while (aliasIterator.hasNext()) {
            aliasIterator.next();
            m_xmlWriter->writeStartElement(KALIAS);
            m_xmlWriter->writeAttribute(KNAME, aliasIterator.key());
            CapabilityList caps = aliasIterator.value();
            for (int i = 0; i < caps.size(); i++) {
                m_xmlWriter->writeStartElement(KCAPABILITY);
                m_xmlWriter->writeAttribute(KNAME, caps.at(i));
                m_xmlWriter->writeEndElement(); // KCAPABILITY
            }
            m_xmlWriter->writeEndElement(); // KALIAS
        }

        // Write domain
        m_xmlWriter->writeStartElement(KDOMAIN);
        m_xmlWriter->writeAttribute(KNAME, domain);
        if (capabilities) {
            QHashIterator<QString, QPointer<Condition> >
                    capIterator(*capabilities);
            QSet<Condition*> conditionSet;
            Condition *cond;
            while (capIterator.hasNext()) {
                capIterator.next();
                cond = capIterator.value();
                if (cond) {
                    if (!conditionSet.contains(cond)) {
                        conditionSet.insert(cond);
                    }
                } else {
                    m_xmlWriter->writeStartElement(KCAPABILITY);
                    m_xmlWriter->writeAttribute(KNAME, capIterator.key());
                    m_xmlWriter->writeEndElement(); // KCAPABILITY
                }
            }

            // Write conditions
            QSetIterator<Condition*> condIterator(conditionSet);
            while (condIterator.hasNext()) {
                cond = condIterator.next();
                cond->persist();
            }
        }
        m_xmlWriter->writeEndElement(); // KDOMAIN;

        m_xmlWriter->writeEndElement(); // KACCESSPOLICY
        m_xmlWriter->writeEndDocument();

        ptr->close();
        delete ptr;
        result = true;
    }
    return result;
}

/*!
 * Persist the policy
 * \param[in] capabilities - Capabilities to save
 * \param[in] flags - Flags to save
 * \param[in] grants - User grant value to save
 * \return TRUE if successful, else FALSE
 */
bool XMLPolicySerializer::persist(const CapabilityList &capabilities,
        unsigned int flags, unsigned int grants) {
    bool result = false;

    if (m_xmlWriter) {
        QHash<unsigned int, QString> conditionTable;
        conditionTable[EUserGrantOneShot] = KONESHOT;
        conditionTable[EUserGrantSession] = KSESSION;
        conditionTable[EUserGrantBlanket] = KBLANKET;

        m_xmlWriter->writeStartElement(KUSER);

        // Write scopes
        if (flags & EUserGrantDefaultMask) {
            m_xmlWriter->writeStartElement(KDEFAULTSCOPE);
            m_xmlWriter->writeAttribute(KTYPE, (conditionTable[(flags
                    & EUserGrantDefaultMask) >> 3]));
            m_xmlWriter->writeEndElement(); // KDEFAULTSCOPE
        }
        if (flags & EUserGrantCapsMask) {
            if ((flags & EUserGrantBlanket) == EUserGrantBlanket) {
                m_xmlWriter->writeStartElement(KSCOPE);
                m_xmlWriter->writeAttribute(KTYPE,
                        conditionTable[EUserGrantBlanket]);
                m_xmlWriter->writeEndElement(); // KSCOPE
            }
            if ((flags & EUserGrantSession) == EUserGrantSession) {
                m_xmlWriter->writeStartElement(KSCOPE);
                m_xmlWriter->writeAttribute(KTYPE,
                        conditionTable[EUserGrantSession]);
                m_xmlWriter->writeEndElement(); // KSCOPE
            }
            if ((flags & EUserGrantOneShot) == EUserGrantOneShot) {
                m_xmlWriter->writeStartElement(KSCOPE);
                m_xmlWriter->writeAttribute(KTYPE,
                        conditionTable[EUserGrantOneShot]);
                m_xmlWriter->writeEndElement(); // KSCOPE
            }
        }

        // Write capabilities
        for (int i = 0; i < capabilities.size(); i++) {
            m_xmlWriter->writeStartElement(KCAPABILITY);
            m_xmlWriter->writeAttribute(KNAME, capabilities[i]);
            m_xmlWriter->writeEndElement(); // KCAPABILITY
        }

        // Write grant value
        m_xmlWriter->writeStartElement(KGRANT);
        m_xmlWriter->writeAttribute(KTYPE, QString::number(grants));
        m_xmlWriter->writeEndElement(); // KGRANT

        m_xmlWriter->writeEndElement(); // KUSER

        result = true;
    }

    return result;
}

}
