/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QProcess>
#include <QBuffer>
#include <QDataStream>
#include <QDirIterator>
#include <clientinfo.h>
#include "storage.h"
#include "securestorageserversession.h"
#include "securestoragemetadata.h"
#include "securestorageserverentry.h"
#include "sslog.h"

namespace WRT
{
// CONSTANTS
static const char KSEPARATOR            = '/';
static const char KSHAREDFOLDER[]       = "shared";
static const char KPRIVATEFOLDER[]      = "private";
static const char KMETALISTFILE[]       = "metalist";
static const int  KFLUSHTIMERINTERVAL   = 1000;     // 1 seconds


/*!
 * \class WRT::SecureStorageServerSession
 * Session class for a client of SecureStorage
 */

/*!
 * Constructor
 * \param[in] aParent - QObject owning this object
 */
SecureStorageServerSession::SecureStorageServerSession(QObject *aParent) :
    QObject(aParent),
    m_fileListInitialized(false)
{
    connect(&m_flushTimer,
            SIGNAL(timeout()),
            this,
            SLOT(flushTimeout()));
}

/*!
 * Destructor
 */
SecureStorageServerSession::~SecureStorageServerSession()
{
    m_listFileHandle.close();
    if (m_fileList.isEmpty()) {
        m_listFileHandle.remove();
        QDir dir;
        dir.rmpath(m_storagePath);
    }
    if (!m_entryList.isEmpty()) {
        qDeleteAll(m_entryList);
    }
    if (m_flushTimer.isActive()) {
        m_flushTimer.stop();
    }
}

/*!
 * Sets the name of the storage
 * \param[in] aId - ID of this storage
 */
void SecureStorageServerSession::setName(const QString &aName)
{
    SS_FUNC("SecureStorageServerSession::setName()");
    m_name = aName;
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
}

/*!
 * Gets the name of the storage
 * \return ID of this storage
 */
QString SecureStorageServerSession::name() const
{
    SS_FUNC("SecureStorageServerSession::name()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    return m_name;
}

/*!
 * Sets the visibility and protection mode of the storage
 * \param[in] aVisibility - Visibility of this storage
 * \param[in] aProtection - Protection mode of this storage
 */
void SecureStorageServerSession::setMode(Storage::visibility aVisibility,
             Storage::protection aProtection)
{
    SS_FUNC("SecureStorageServerSession::setMode()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    m_visibility = aVisibility;
    m_protection = aProtection;
}

/*!
 * Sets the client info object
 * \param[in] aClientInfo - Client info structure
 */
void SecureStorageServerSession::setClientInfo(ClientInfo *aClientInfo)
{
    SS_FUNC("SecureStorageServerSession::setClientInfo()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    m_clientInfo = aClientInfo;
}

/*!
 * Initializes the storage session
 */
void SecureStorageServerSession::init()
{
    SS_FUNC("SecureStorageServerSession::init()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");

    // Setup storage path
    m_storagePath = QDir::fromNativeSeparators(getStorageDir());
    if (!m_storagePath.endsWith(KSEPARATOR)) {
        m_storagePath += KSEPARATOR;
    }
    if (m_visibility == Storage::vis_shared) {
        m_storagePath += KSHAREDFOLDER;
    } else if (m_visibility == Storage::vis_private) {
        m_storagePath += KPRIVATEFOLDER;
        m_storagePath += KSEPARATOR;
        m_storagePath += QString::number(m_clientInfo->processId());
    } else {
        // TODO: Exception?
        m_storagePath.clear();
    }

    m_storagePath += KSEPARATOR;
    m_storagePath += m_name;
    m_storagePath += KSEPARATOR;
    QDir dir;
    dir.mkpath(m_storagePath);

    m_listFileHandle.setFileName(QString(m_storagePath) + QString(KMETALISTFILE));
    m_listFileHandle.open(QIODevice::ReadWrite);
}

/*!
 * Uninitializes the storage session
 */
void SecureStorageServerSession::uninit()
{
    SS_FUNC("SecureStorageServerSession::uninit()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    writeFileList();
}

/*!
 * Gets list of files associated with this storage
 * \return List of files in this storage
 */
QByteArray SecureStorageServerSession::getFileList()
{
    SS_FUNC("SecureStorageServerSession::getFileList()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    QByteArray ret;

    if (!m_fileListInitialized) {
        readFileList();
        m_fileListInitialized = true;
    } else {
        verifyFileList();
    }

    // Convert file list to QByteArray
    SS_DEBUG("SecureStorageServerSession::getFileList(): m_fileList.count() = " << m_fileList.count());
    if (m_fileList.count() > 0) {
        QBuffer buffer;
        if (buffer.open(QIODevice::ReadWrite)) {
            QDataStream bufferStream(&buffer);
            bufferStream << m_fileList;
            ret = buffer.data();
            buffer.close();
        }
    }
    SS_DEBUG("SecureStorageServerSession::getFileList(): ret.size() = " << ret.size());
    return ret;
}

/*!
 * Saves list of files associated with this storage
 * \param[in] aFileList - List of files
 */
void SecureStorageServerSession::saveFileList(const QByteArray &aFileList)
{
    SS_FUNC("SecureStorageServerSession::saveFileList()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    QDataStream bufferStream(aFileList);
    bufferStream >> m_fileList;
    if (m_fileList.isEmpty()) {
        qDeleteAll(m_entryList);
        m_entryList.clear();
    }

    //Do integrity check to verify file list
    verifyFileList();

    //Start timer to flush to file system
    startFlushTimer();
}

/*!
 * Saves the metadata for a file
 * \param[in] aMetadata - Metadata info
 */
void SecureStorageServerSession::saveMetadata(const QByteArray &aMetadata)
{
    SS_FUNC("SecureStorageServerSession::saveMetadata()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    QBuffer buffer;
    buffer.setData(aMetadata);
    SecureStorageMetadata *meta = new SecureStorageMetadata();;
    meta->internalize(buffer);

    QString sourceFileName(meta->sourceFile());
    if (!m_fileList.contains(sourceFileName)) {
        m_fileList.insert(sourceFileName, QString::number(m_clientInfo->processId()));
    }

    SecureStorageServerEntry *entry(m_entryList.value(sourceFileName, NULL));
    if (entry) {
        delete entry->metadata;
        entry->metadata = meta;
    } else {
        entry = new SecureStorageServerEntry();
        entry->metadata = meta;
        m_entryList.insert(sourceFileName, entry);
    }

    //Start timer to flush to file system
    startFlushTimer();
}

/*!
 * Removes the metadata for a file
 * \param[in] aFileName - Name of file
 */
void SecureStorageServerSession::removeMetadata(const QString &aFileName)
{
    SS_FUNC("SecureStorageServerSession::removeMetadata()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    if (m_fileList.contains(aFileName)) {
        // Verify only the owner can remove
        if (m_fileList.value(aFileName) == QString::number(m_clientInfo->processId())) {
            SecureStorageServerEntry *entry(m_entryList.value(aFileName, NULL));
            if (entry) {
                delete entry;
                m_entryList.remove(aFileName);
            }
        }

        //Start timer to flush to file system
        startFlushTimer();
    }
}

/*!
 * Gets the metadata for a file
 * \param[in] aFileName - Name of file
 */
QByteArray SecureStorageServerSession::getMetadata(const QString &aFileName)
{
    SS_FUNC("SecureStorageServerSession::getMetadata()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    QBuffer buffer;
    SecureStorageServerEntry *entry(m_entryList.value(aFileName, NULL));
    if (entry) {
        (entry->metadata)->externalize(buffer);
    }
    return buffer.data();
}

/*!
 * Saves the key for a file
 * \param[in] aFileName - Name of file
 * \param[in] aKey - Key to save
 */
void SecureStorageServerSession::saveKey(const QString &aFileName, const QByteArray &aKey)
{
    SS_FUNC("SecureStorageServerSession::saveKey()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    if (m_fileList.contains(aFileName)) {
        // Verify only the owner can save
        if (m_fileList.value(aFileName) == QString::number(m_clientInfo->processId())) {
            SecureStorageServerEntry *entry(m_entryList.value(aFileName, NULL));
            if (entry) {
                entry->key = aKey;
            }
        }

        //Start timer to flush to file system
        startFlushTimer();
    }
}

/*!
 * Gets the key for a file
 * \param[in] aFileName - Name of file
 * \return Key for the file
 */
QByteArray SecureStorageServerSession::getKey(const QString &aFileName)
{
    SS_FUNC("SecureStorageServerSession::getKey()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    SS_DEBUG("SecureStorageServerSession::getKey(): aFileName = " << aFileName);
    QByteArray key;
    if (m_fileList.contains(aFileName)) {
        SecureStorageServerEntry *entry(m_entryList.value(aFileName, NULL));
        if (entry) {
            key = entry->key;
            SS_DEBUG("SecureStorageServerSession::getKey(): found key.size() = " << key.size());
        }
    }
    return key;
}

/*!
 * Gets a file in the storage
 * \param[in] aFileName - Name of file to get
 * Default behaviour does nothing
 */
QByteArray SecureStorageServerSession::getFile(const QString &aFileName)
{
    SS_FUNC("SecureStorageServerSession::getFile()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    return QByteArray();
}

//------------------------------------------------------------------------
// PROTECTED METHODS
//------------------------------------------------------------------------

/*!
 * Returns top level directory where the storage should be
 * NOTE: This should be overridden
 * \return Directory of storage
 */
QString SecureStorageServerSession::getStorageDir()
{
    return QDir::tempPath();
}

/*!
 * Verifies and does integrety check of file list
 */
bool SecureStorageServerSession::verifyFileList()
{
    SS_FUNC("SecureStorageServerSession::verifyFileList()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    bool result(true);
    // TODO
    return result;
}

/*!
 * Reads the list of entries from the file
 */
void SecureStorageServerSession::readFileList()
{
    SS_FUNC("SecureStorageServerSession::readFileList()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    internalizeFileList(m_listFileHandle);
}

/*!
 * Writes the list of entries into a file
 */
void SecureStorageServerSession::writeFileList()
{
    SS_FUNC("SecureStorageServerSession::writeFileList()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    externalizeFileList(m_listFileHandle);
    m_listFileHandle.flush();
}

/*!
 * Takes in a QIODevice and does the actual internalizing of the entry list
 * Can be overriden to provide platform specific behaviour.
 * \param[in] aDevice - QIODevice for internalizing
 */
void SecureStorageServerSession::internalizeFileList(QIODevice &aDevice)
{
    SS_FUNC("SecureStorageServerSession::internalizeFileList()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    m_fileList.clear();
    m_entryList.clear();
    aDevice.seek(0);
    QDataStream dataStream(&aDevice);
    int numEntries(0);
    if (!dataStream.atEnd()) {
        dataStream >> numEntries;
    }
    for (int i = 0; i < numEntries; ++i) {
        QString sourceFileName;
        QByteArray hash;
        QString owner;
        QByteArray key;
        dataStream >> sourceFileName;
        dataStream >> owner;
        dataStream >> hash;
        dataStream >> key;
        SecureStorageMetadata *metadata = new SecureStorageMetadata;
        metadata->setSourceFile(sourceFileName);
        metadata->setHash(hash);
        m_fileList.insert(sourceFileName, owner);
        SecureStorageServerEntry *entry = new SecureStorageServerEntry();
        entry->metadata = metadata;
        entry->key = key;
        m_entryList.insert(sourceFileName, entry);
    }
}

/*!
 * Takes in a QIODevice and does the actual externalizing of the entry list
 * Can be overriden to provide platform specific behaviour.
 * \param[in] aDevice - QIODevice for externalizing
 */
void SecureStorageServerSession::externalizeFileList(QIODevice &aDevice)
{
    SS_FUNC("SecureStorageServerSession::externalizeFileList()");
    SS_DEBUG("   SecureStorageServerSession[" << m_name << "]");
    aDevice.seek(0);
    QDataStream dataStream(&aDevice);
    QList<QString> fileList(m_fileList.keys());
    int numEntries(fileList.count());
    dataStream << numEntries;
    foreach (QString fileName, fileList) {
        dataStream << fileName;
        dataStream << m_fileList.value(fileName);
        SecureStorageServerEntry *entry(m_entryList.value(fileName));
        dataStream << (entry->metadata)->hash();
        dataStream << entry->key;
    }
}

/*!
 * Starts the flush timer
 */
void SecureStorageServerSession::startFlushTimer()
{
    SS_FUNC("SecureStorageServerSession::startFlushTimer()");
    if (m_flushTimer.isActive()) {
        m_flushTimer.stop();
    }
    m_flushTimer.setSingleShot(true);
    m_flushTimer.start(KFLUSHTIMERINTERVAL);
}

/*!
 * Flush timer timeout
 * Writes the file list to file system
 */
void SecureStorageServerSession::flushTimeout()
{
    SS_FUNC("SecureStorageServerSession::flushTimeout()");
    writeFileList();
}

} // end namespace WRT
