/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QFile>
#include <QFileInfo>
#include <QDir>
#include <QBuffer>
#include <QCryptographicHash>
#include "securestorage.h"
#include "securestoragemetadata.h"
#include "sslog.h"

namespace WRT
{
/*!
 * \class WRT::SecureStorage
 * Common Secure Storage Implementation
 */

/*!
 * Constructor
 * \param[in] aParent - QObject parent
 */
SecureStorage::SecureStorage(const QString &aStorageName,
                             Storage::visibility aVisibility /* = vis_shared */,
                             Storage::protection aProtection /* = prot_signed */,
                             QObject* aParent /* = 0 */) :
    Storage(aStorageName, aParent),
    m_visibility(aVisibility),
    m_protection(aProtection)
{
    // NOTE: It is up to the subclasses to initialize the file list
}

/*!
 * Destructor
 */
SecureStorage::~SecureStorage()
{
    m_fileList.clear();
}

/*!
 * From Storage
 * \see Storage::getFileList();
 */
QStringList SecureStorage::getFileList() const
{
    SS_FUNC("SecureStorage::getFileList()");
    QHash<QString, QString> fileList(updateFileList());
    return fileList.keys();
}

/*!
 * From Storage
 * \see Storage::add(const QFile &aFile);
 */
void SecureStorage::add(const QFile &aFile)
{
    SS_FUNC("SecureStorage::add()");
    SS_DEBUG("SecureStorage::add(): aFile = " << aFile.fileName());
    // Check if file is in storage first
    m_currentFile = QFileInfo(aFile.fileName()).absoluteFilePath().toLower();
    m_fileList = updateFileList();
    if (m_fileList.contains(m_currentFile)) {
        // TODO:
        // Already exists in storage
    }
    else {
        QFile file(m_currentFile);
        if (file.open(QIODevice::ReadWrite)) {
            // Encrypt file first if needed
            QByteArray data(file.readAll());
            if (m_protection == Storage::prot_encrypted) {
                data = encryptData(data);
                file.seek(0);
                file.write(data);
            }
            updateMetadataInfo(m_currentFile, data);
            file.close();
        }
    }
}

/*!
 * From Storage
 * \see Storage::add(const QString &aFilePath, const QBuffer &aData);
 */
void SecureStorage::add(const QString &aFilePath, const QBuffer &aData)
{
    SS_FUNC("SecureStorage::add()");
    SS_DEBUG("SecureStorage::add(): aFilePath = " << aFilePath);
    m_currentFile = QFileInfo(QDir::fromNativeSeparators(aFilePath)).absoluteFilePath().toLower();
    m_fileList = updateFileList();
    // Check if file is in storage first
    if (!m_fileList.contains(m_currentFile)) {
        SS_DEBUG("SecureStorage::add(): OK to add")
        QDir dir(QFileInfo(m_currentFile).dir());
        dir.mkpath(dir.path());
        QFile file(m_currentFile);
        if (file.open(QIODevice::WriteOnly | QIODevice::Truncate)) {
            SS_DEBUG("SecureStorage::add(): File opened successfully")
            // Encrypt file if needed
            QByteArray data(aData.data());
            if (m_protection == Storage::prot_encrypted) {
                data = encryptData(data);
            }

            // Write data to the file
            file.write(data);
            file.close();

            // Then add the file to storage
            updateMetadataInfo(m_currentFile, data);
        }
    }
}

/*!
 * From Storage
 * \see Storage::remove(const QFile &aFile);
 */
void SecureStorage::remove(const QFile &aFile)
{
    SS_FUNC("SecureStorage::remove()");
    SS_DEBUG("SecureStorage::remove(): aFile = " << aFile.fileName());
    m_currentFile = QFileInfo(aFile.fileName()).absoluteFilePath().toLower();
    m_fileList = updateFileList();
    if (m_fileList.contains(m_currentFile)) {

        // Check that only the owner can remove a file
        if (m_fileList.value(m_currentFile) == getOwner()) {
            // Remove from file list and save
            m_fileList.remove(m_currentFile);
            saveFileList();

            // Remove metadata
            removeMetadata(m_currentFile);
        }
    }
}

/*!
 * From Storage
 * \see Storage::verify(const QFile &aFile);
 */
bool SecureStorage::verify(const QFile &aFile) const
{
    SS_FUNC("SecureStorage::verify()");
    SS_DEBUG("SecureStorage::verify(): aFile = " << aFile.fileName());
    bool result(false);
    const QString fileName(QFileInfo(aFile.fileName()).absoluteFilePath().toLower());
    QHash<QString, QString> fileList(updateFileList());
    if (fileList.contains(fileName)) {
        SS_DEBUG("SecureStorage::verify(): file is in storage");
        QFile file(fileName);
        if (file.open(QIODevice::ReadOnly)) {
            // Get metadata for file from storage
            SS_DEBUG("SecureStorage::verify(): getting metadata");
            QScopedPointer<SecureStorageMetadata> metaFile(getMetadata(fileName));
            if (!metaFile.isNull()) {
                SS_DEBUG("SecureStorage::verify(): metafile is: " << metaFile);
                // Create hash for current file and compare with the value
                // from the metadata in storage
                QByteArray hash(createHash(file.readAll()));
                if (hash == metaFile->hash()) {
                    result = true;
                }
            }
            file.close();
        }
    }
    SS_DEBUG("SecureStorage::verify(): returning: " << result)
    return result;
}

/*!
 * From Storage
 * \see Storage::getFile(const QString &aPathFile);
 */
QIODevice * SecureStorage::getFile(const QString &aPathFile)
{
    SS_FUNC("SecureStorage::getFile()");
    SS_DEBUG("SecureStorage::getFile(): aPathFile = " << aPathFile);
    QIODevice *ret(NULL);
    m_currentFile = QFileInfo(QDir::fromNativeSeparators(aPathFile)).absoluteFilePath().toLower();
    m_fileList = updateFileList();
    if (m_fileList.contains(m_currentFile)) {
        SS_DEBUG("SecureStorage::getFile(): file is in storage");
        if (QFileInfo(m_currentFile).exists()) {
            SS_DEBUG("SecureStorage::getFile(): file exists");
            if (m_visibility == vis_shared || m_fileList.value(m_currentFile) == getOwner()) {
                SS_DEBUG("SecureStorage::getFile(): owner matches");
                if (m_protection == Storage::prot_encrypted) {
                    // If it's encrypted then decrypt the file first
                    QFile file(m_currentFile);
                    if (file.open(QIODevice::ReadOnly)) {
                        QByteArray data = file.readAll();
                        file.close();
                        if (data.size() > 0) {
                            ret = decryptData(data);
                            if (ret) {
                                ret->close();
                            }
                        } else {
                            ret = new QFile(m_currentFile);
                        }
                    }
                } else {
                    // Otherwise not encrypted so just return the file
                    ret = new QFile(m_currentFile);
                }
            }
        } else {
            SS_DEBUG("SecureStorage::getFile(): file does not exist");
            remove(aPathFile);
        }
    }
    return ret;
}

/*!
 * From Storage
 * \see Storage::count();
 */
qint32 SecureStorage::count() const
{
    SS_FUNC("SecureStorage::count()");
    QHash<QString, QString> fileList(updateFileList());
    return fileList.count();
}

//------------------------------------------------------------------------
// Protected Methods
//------------------------------------------------------------------------

/*!
 * Computes a hash value given the input
 * \param[in] aData - Input value to compute the hash against
 * \return Hash value as a QByteArray
 */
QByteArray SecureStorage::createHash(const QByteArray &aData) const
{
    SS_FUNC("SecureStorage::createHash()");
    return QCryptographicHash::hash(aData, QCryptographicHash::Md5);
}

/*!
 * Updates file list.
 * Default behaviour only returns internal variable
 */
QHash<QString, QString> SecureStorage::updateFileList() const
{
    SS_FUNC("SecureStorage::updateFileList()");
    return m_fileList;
}

/*!
 * Updates metadata info given the filename and data contained in the file
 * Creates a hash from the given data.
 * \param[in] aFileName - Name of the file
 * \param[in] aData - Data of the file
 */
void SecureStorage::updateMetadataInfo(const QString &aFileName,
                                       const QByteArray &aData)
{
    SS_FUNC("SecureStorage::updateMetadataInfo()");
    // Create hash and save metadata info
    SecureStorageMetadata metaData;
    metaData.setSourceFile(aFileName);
    metaData.setHash(createHash(aData));
    saveMetadata(metaData);

    // Add file to filelist
    m_fileList.insert(aFileName, getOwner());
    saveFileList();
}

} // end namespace WRT
