/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include "webcursornavigation.h"

#include <QCursor>
#include <QGraphicsView>
#include <QWidget>

#include <qgraphicswebview.h>
#include <qwebelement.h>
#include <qwebframe.h>
#include <qwebpage.h>
#include <qwebview.h>

namespace WRT {

const int KNormalScrollRange = 40;
const int KFullStep = 14;
const int KMinimumParagraphHeight = 30;
const int KHitTestDistance = 100;

/*!
    \class WebCursorNavigation
    \since cwrt 1.0
    \brief cwrt navigation.

    \sa WebNavigation, WebTouchNavigation, WebDirectionalNavigation, WebHtmlTabIndexedNavigation
*/
WebCursorNavigation::WebCursorNavigation(QObject* object, QWebPage* webPage) :
m_viewObject(object)
, m_webPage(webPage)
, m_flipcounter(1)
, m_direction(0)
, m_lastdirection(0)
#ifdef QT_KEYPAD_NAVIGATION
, m_navigationMode(static_cast<Qt::NavigationMode>(-1))
#endif
{
    Q_ASSERT(m_viewObject != 0);
#ifdef QT_KEYPAD_NAVIGATION
    Q_ASSERT(m_navigationMode == static_cast<Qt::NavigationMode>(-1));
    m_navigationMode = QApplication::navigationMode();
    QApplication::setNavigationMode(Qt::NavigationModeCursorForceVisible);
#endif
    m_viewObject->installEventFilter(this);
    connect(&m_scrollTimer, SIGNAL(timeout()), this, SLOT(scrollTimeout()));
    connect(&m_keypressTimer, SIGNAL(timeout()), this, SLOT(keypressTimeout()));
}

WebCursorNavigation::~WebCursorNavigation()
{
#ifdef QT_KEYPAD_NAVIGATION
    Q_ASSERT(m_navigationMode == Qt::NavigationModeCursorForceVisible);
    QApplication::setNavigationMode(m_navigationMode);
    m_navigationMode = static_cast<Qt::NavigationMode>(-1);
#endif
    m_viewObject->removeEventFilter(this);
}

bool WebCursorNavigation::eventFilter(QObject *object, QEvent *event)
{
    if (object == m_viewObject) {
        switch (event->type()) {
        case QEvent::KeyPress: {
            QKeyEvent* ev = static_cast<QKeyEvent*>(event);
            keyPressEvent(ev);
            return true;
        }
        case QEvent::KeyRelease: {
            QKeyEvent* ev = static_cast<QKeyEvent*>(event);
            keyReleaseEvent(ev);
            return true;
        }
#ifdef QT_KEYPAD_NAVIGATION
        case QEvent::MouseButtonPress:
        case QEvent::GraphicsSceneMousePress:
            QApplication::setNavigationMode(Qt::NavigationModeNone);
        return false;
        case QEvent::MouseButtonRelease:
        case QEvent::GraphicsSceneMouseRelease:
            QApplication::setNavigationMode(Qt::NavigationModeCursorForceVisible);
        return false;
#endif
        }
    }
    return false;
}

/*!
    Timeout for long keypress.
    \sa scrollTimeout()
*/
void WebCursorNavigation::keypressTimeout()
{
    if (!m_scrollTimer.isActive())
      scrollTimeout();
}

/*!
    Timeout for scroller. Scrolls the page on a timer every 50 miliseconds
    \sa keypressTimeout()
*/
void WebCursorNavigation::scrollTimeout()
{
    if (!scroll(m_direction))
        moveCursor(m_direction);

    if (!m_scrollTimer.isActive())
        m_scrollTimer.start(50);
}

/*!
    If the key is directional, starts the keyPress timer. The cursor navigation is processed on keyReleaseEvent.
    If the key is <Select> or <Return>, eat the event; we'll do that action once on keyRelease.
    Otherwise, send the keyPress event onwards to QWebPage.
*/
void WebCursorNavigation::keyPressEvent(QKeyEvent* ev)
{
    if (ev->key() == Qt::Key_Up
        || ev->key() == Qt::Key_Down
        || ev->key() == Qt::Key_Left
        || ev->key() == Qt::Key_Right) {

        if (!m_keypressTimer.isActive())
            m_keypressTimer.start(300);

        m_direction = ev->key();
    }

    if (ev->key() == Qt::Key_Return
        || ev->key() == Qt::Key_Enter
        || ev->key() == Qt::Key_Select) {
            QPoint cursorPosition = getCursorPosition();

            QWebFrame* webFrame = m_webPage->frameAt(cursorPosition);
            if (!webFrame)
                webFrame = m_webPage->currentFrame();

            Qt::KeyboardModifier modifier = Qt::NoModifier;
            QWebHitTestResult htr = webFrame->hitTestContent(cursorPosition);

            if (!htr.element().tagName().toLower().compare("select")
                && !htr.element().hasAttribute("multiple")
                && htr.element().hasFocus()) // focused single select elements handle the key events.
                return;

            if (!htr.element().tagName().toLower().compare("select")  && htr.element().hasAttribute("multiple"))
                modifier = Qt::ControlModifier;

            QMouseEvent evpress(QEvent::MouseButtonPress, cursorPosition, Qt::LeftButton, Qt::NoButton, modifier);
            m_webPage->event(&evpress);
    }

}

/*!
    If the key is directional the cursor navigation is processed.
    If the key is a <Select> or <Return>, send a left button mouse press and release to QWebPage.
    Otherwise just send the keyRelease event onwards to QWebPage.
*/
void WebCursorNavigation::keyReleaseEvent(QKeyEvent* ev)
{
    //stop fast scrolling timers
    m_keypressTimer.stop();
    m_scrollTimer.stop();

    if (ev->key() == Qt::Key_Up
        || ev->key() == Qt::Key_Down
        || ev->key() == Qt::Key_Left
        || ev->key() == Qt::Key_Right) {
        if (!scroll(ev->key()))
            moveCursor(ev->key());
    }

    if (ev->key() == Qt::Key_Return
        || ev->key() == Qt::Key_Enter
        || ev->key() == Qt::Key_Select) {

            QPoint cursorPosition = getCursorPosition();
            QWebFrame* webFrame = m_webPage->frameAt(cursorPosition);
            if (!webFrame)
                webFrame = m_webPage->currentFrame();

            Qt::KeyboardModifier modifier = Qt::NoModifier;
            QWebHitTestResult htr = webFrame->hitTestContent(cursorPosition);

            if (!htr.element().tagName().toLower().compare("select")
                && !htr.element().hasAttribute("multiple")
                && htr.element().hasFocus()) // focused single select elements handle the key events.
                return;

            if (!htr.element().tagName().toLower().compare("select")  && htr.element().hasAttribute("multiple"))
                modifier = Qt::ControlModifier;

            QMouseEvent evrel(QEvent::MouseButtonRelease, cursorPosition, Qt::LeftButton, Qt::NoButton, modifier);
            m_webPage->event(&evrel);
    }
}


/*!
    Moves the cursor a fixed interval in the given direction
*/
void WebCursorNavigation::moveCursor(int direction)
{
    QPoint cursorPosition = getCursorPosition();

    QRect rect(0, 0,
               m_webPage->viewportSize().width(),
               m_webPage->viewportSize().height());

    switch (direction) {
    case Qt::Key_Left: {
        m_flipcounter = (m_lastdirection == Qt::Key_Right) ? ++m_flipcounter : 1;
        int dx = cursorPosition.x() - (KFullStep / m_flipcounter);
        cursorPosition.setX((dx > 0) ? dx : 0);

        break;
    }
    case Qt::Key_Right: {
        m_flipcounter = (m_lastdirection == Qt::Key_Left) ? ++m_flipcounter : 1;
        int dx = cursorPosition.x() + (KFullStep / m_flipcounter);
        cursorPosition.setX((dx < rect.width() - KFullStep) ? dx : rect.width() - KFullStep);
        break;
    }
    case Qt::Key_Up: {
        m_flipcounter = (m_lastdirection == Qt::Key_Down) ? ++m_flipcounter : 1;
        int dy = cursorPosition.y() - (KFullStep / m_flipcounter);
        cursorPosition.setY((dy > 0) ? dy : 0);
        break;
    }
    case Qt::Key_Down : {
        m_flipcounter = (m_lastdirection == Qt::Key_Up) ? ++m_flipcounter : 1;
        int dy = cursorPosition.y() + (KFullStep / m_flipcounter);
        cursorPosition.setY((dy < rect.height() - KFullStep) ? dy : rect.height() - KFullStep);
        break;
    }
    }

    QMouseEvent evmm(QEvent::MouseMove, cursorPosition, Qt::NoButton, Qt::NoButton, Qt::NoModifier);
    m_webPage->event(&evmm);
    m_lastdirection = direction;
    setCursorPosition(cursorPosition);
}

/*!
    Scrolls QWebFrame a fixed interval in a given direction.
*/
bool WebCursorNavigation::scroll(int direction)
{
    QPoint cursorPosition = getCursorPosition();
    QWebFrame* webFrame = m_webPage->frameAt(cursorPosition);
    if (!webFrame)
        webFrame = m_webPage->currentFrame();

    QPoint scrollPosition = webFrame->scrollPosition();
    QRect frameRect = webFrame->geometry();
    int xmargin = 2 * frameRect.width() / 5;
    int ymargin = 2 * frameRect.height() / 5;

    switch (direction) {
    case Qt::Key_Left :
        if (cursorPosition.x() < frameRect.x() + xmargin)
            webFrame->scroll(-getNearestEdge(KNormalScrollRange, Qt::Key_Left), 0);
        break;
    case Qt::Key_Right:
        if (cursorPosition.x() > (frameRect.right() - xmargin))
            webFrame->scroll(getNearestEdge(KNormalScrollRange, Qt::Key_Right), 0);
        break;
    case Qt::Key_Up:
        if (cursorPosition.y() < frameRect.y() + ymargin)
            webFrame->scroll(0, -KNormalScrollRange);
        break;
    case Qt::Key_Down:
        if (cursorPosition.y() > (frameRect.bottom() - ymargin))
            webFrame->scroll(0, KNormalScrollRange);
        break;
    }

    return scrollPosition != webFrame->scrollPosition();
}

QPoint WebCursorNavigation::getCursorPosition()
{
    QPoint cursorPosition = QCursor::pos();
    if (m_viewObject->isWidgetType()) {
        cursorPosition = static_cast<QWidget*>(m_viewObject)->mapFromGlobal(QCursor::pos());
    } else {
        QGraphicsWebView* graphicsWebView = static_cast<QGraphicsWebView*>(m_viewObject);
        QGraphicsScene* graphicsScene = graphicsWebView->scene();
        QList<QGraphicsView*> gvList = graphicsScene->views();
        QList<QGraphicsView*>::iterator it;
        for (it = gvList.begin(); it != gvList.end(); it++) {
            if (static_cast<QGraphicsView*>(*it)->hasFocus()) {
                QWidget* viewport = static_cast<QGraphicsView*>(*it)->viewport();
                cursorPosition = viewport->mapFromGlobal(QCursor::pos());
                break;
            }
        }
    }
    return cursorPosition;
}

void WebCursorNavigation::setCursorPosition(const QPoint& point)
{
    if (m_viewObject->isWidgetType()) {
        QCursor::setPos(static_cast<QWidget*>(m_viewObject)->mapToGlobal(point));
    } else {
        QGraphicsWebView* graphicsWebView = static_cast<QGraphicsWebView*>(m_viewObject);
        QGraphicsScene* graphicsScene = graphicsWebView->scene();
        QList<QGraphicsView*> gvList = graphicsScene->views();
        QList<QGraphicsView*>::iterator it;
        for (it = gvList.begin(); it != gvList.end(); it++) {
            if (static_cast<QGraphicsView*>(*it)->hasFocus()) {
                QWidget* viewport = static_cast<QGraphicsView*>(*it)->viewport();
                QCursor::setPos(viewport->mapToGlobal(point));
                break;
            }
        }
    }
}

/*!
    Returns the distance to scroll to the nearest edge of a text paragraph.
*/
int WebCursorNavigation::getNearestEdge(int scrollRange, int direction)
{
    QSize size = m_webPage->viewportSize();

    //Identify the number of hit tests needed
    int hitTestCount = size.height() / KHitTestDistance;
    int x = (direction == Qt::Key_Right) ? scrollRange : 0;

    QPoint pos(x, KHitTestDistance);
    for (int i = 0; i < hitTestCount; i++) {
        QWebHitTestResult htr = m_webPage->mainFrame()->hitTestContent(pos);
        QRect rect = htr.boundingRect();
        QPoint contentPosition = m_webPage->mainFrame()->scrollPosition();
        int scrollDistance = (direction == Qt::Key_Right) ? rect.x() - contentPosition.x() : pos.x() - rect.x();
        if (scrollDistance > 0 && scrollRange > scrollDistance && rect.height() > KMinimumParagraphHeight)
           scrollRange = scrollDistance;
        pos = QPoint(pos.x(), pos.y() + (i + 1) * KHitTestDistance);
    }

    return scrollRange;
}

}

