/**
 * A wrapper JS class for objects implementing IFileSystem interface of FileSystem Plugin
 * @class FileSystem
 */

/**
 * @class
 */ 
nokia.device.FileSystem = function PSFileSystemInterface(){
	
	var DeviceException = nokia.device.DeviceException;
    var error = new DeviceException(0,'dummy');

	/*
	 * Internal
	 * qt FileSystem interface object.
	 */
	try {
		var fileSystemService = nokia._services.load("nokia.device.filesystem", "com.nokia.IFileSystem", "1.0");
	} 
	catch (er) {
		alert("Load Error:" + er);
		return;
	}
	
	/**
	 * Checks for the mandatory string type
	 */
	function _chkFS_MandatoryString(str, paramName)
	{
		if( (str == null) || (typeof str == 'undefined') )
			throw new DeviceException(error.MISSING_ARG_ERR, paramName);
	
		if(typeof str != 'string' )
			throw new DeviceException(error.INVALID_ARG_ERR, paramName);	
	}
	
	/**
	 * Checks for the optional string type
	 */
	function _chkFS_OptionalString(str, paramName)
	{
		if( (str != 'undefined') && (str != null)&& (typeof str != 'string') )
			throw new DeviceException(error.INVALID_ARG_ERR,paramName);
	}
	
	/**
	 * Checks for the optional function type
	 */
	function _chkFS_Optionalcallback(callback, paramName)
	{
		//if call back defined, not null and not a fn
		if( typeof callback != 'undefined' )
		{
			if( ( callback != null ) && (typeof callback != 'function') )
				throw new DeviceException(error.INVALID_ARG_ERR, paramName);
		}
	}
	
	/**
	 * Checks for the mandatory function type
	 */
	function _chkFS_Mandatorycallback(callback, paramName)
	{
		if( (typeof callback == 'undefined') || (callback == 'null') )
			throw new DeviceException(error.MISSING_ARG_ERR,paramName);
			
		if(typeof callback != 'function')
			throw new DeviceException(error.INVALID_ARG_ERR,paramName);
	}
	
	/**
	 * Checks for the optional boolean type
	 */
	function _chkFS_OptionalBool(boolVal, paramName)
	{
		if( (boolVal != 'undefined') && (boolVal != null) && (typeof boolVal != 'boolean') )
			throw new DeviceException(error.INVALID_ARG_ERR, paramName);
	}
	
	/**
	 * Checks for the mandatory number type
	 */
	function _chkFS_MandatoryNumber(numberVal, paramName)
	{
		if( (numberVal == null) || (typeof numberVal == 'undefined') )
			throw new DeviceException(error.MISSING_ARG_ERR, paramName);
	
		if(typeof numberVal != 'number' )
			throw new DeviceException(error.INVALID_ARG_ERR, paramName);
	}
	
	/**
	 * Checks for the optional number type
	 */
	function _chkFS_OptionalNumber(numberVal, paramName)
	{
		if( (numberVal != 'undefined') && (numberVal != null)&& (typeof numberVal != 'number') )
			throw new DeviceException(error.INVALID_ARG_ERR,paramName);
	}
	
	/**
	 * Checks for the mandatory object type
	 */
	function _chkFS_MandatoryObject(matchObj, paramName)
	{
		if( (matchObj == null) || (typeof matchObj == 'undefined') )
			throw new DeviceException(error.MISSING_ARG_ERR, paramName);
	
		if(typeof matchObj != 'object' )
			throw new DeviceException(error.INVALID_ARG_ERR, paramName);	
	}	
	
/**
 * @class
 */
 	function FileHandle(id){
	    var handleId = id;
		
		return{
		/**
		 * Calls close method of fileSystemService that implements IFileSystem interface 
		 * of FileSystem Plugin
		 *
		 * @function close
		 * @param {function} successCB - success call back function
		 * @param {function} errorCB - error call back function
		 * @returns trans Id in async mode
		 */
		close : function(successCB, errorCB){
			
			_chkFS_Optionalcallback(successCB, 'success callback');
			_chkFS_Optionalcallback(errorCB, 'error Callback');
		
			var connectId = -1;	
			if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
			{
				connectId = fileSystemService.addEventListener('closeSignal(int , QVariant)', closeCB );
			}
		
		    var retValue = fileSystemService.close(handleId, connectId);
					
			if (connectId != -1) 
			{//async call
				return retValue;//returning trans id
			}
			else
			{//sync call
				if (retValue.ErrorCode)
					throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);			
			}
			
			function closeCB(tid, output)
			{			
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB();
						}
				}
			}	
		},//end of close
			/**
			 * Calls read method of fileSystemService that implements IFileSystem interface 
			 * of FileSystem Plugin
			 *
			 * @function read
			 * @param {int} count - length of data to be read
			 * @param {int} position - position in file from where to read from
			 * @param {function} successCB - success call back function
			 * @param {function} errorCB - error call back function 
			 * @returns Data read for sync call and transactin id for async calls
			 */
			read : function(count, position, successCB, errorCB){
			
				_chkFS_MandatoryNumber(count,'length to read is invalid');
				_chkFS_OptionalNumber(position, 'position is invalid');
				_chkFS_Optionalcallback(successCB,'success Callback');
				_chkFS_Optionalcallback(errorCB,'error Callback');
							  
				if( (position) && position < 0 )
				{
					throw new DeviceException(error.DATA_OUT_OF_RANGE_ERR,'position');
				}
				
			  if( (position == 'undefined') || (position == null) )
					position = -1;
			
				var connectId = -1;	
				if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
				{
					connectId = fileSystemService.addEventListener('readSignal(int , QVariant)', readCB );
				}			
			
				var retValue = fileSystemService.read(handleId, count, position, connectId); 
			
				if (connectId != -1) 
				{//async call
					return retValue;//returning trans id
				}
				else
				{//sync call
					if (retValue.ErrorCode)
						throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);	
					else
						return retValue.data;		
				}
				
				function readCB(tid, output)
				{
					if(tid == connectId)
					{
						fileSystemService.removeEventListener(tid);
						if( output.ErrorCode )
						{
							//if error callback is defined call it, else ignore the result
							if( typeof errorCB == 'function' )
								{
									errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
								}
						}
						else
						{
							//if success callback is defined call it, else ignore the result
							if( typeof successCB == 'function' )
								{
									successCB(output.data);
								}
						}
					}		
				}
			},//end of read
			
			/**
			 * Calls readLine method of fileSystemService that implements IFileSystem interface 
			 * of FileSystem Plugin
			 *
			 * @function readLine
			 * @param {int} count - length of data to be read
			 * @param {function} successCB - success call back function
			 * @param {function} errorCB - error call back function 
			 * @returns Data read for sync call and transactin id for async calls
			 */
			readLine : function(count, successCB, errorCB){
				
				_chkFS_OptionalNumber(count,'data length invalid');
				_chkFS_Optionalcallback(successCB,'success Callback');
				_chkFS_Optionalcallback(errorCB,'error Callback');
				
				
			  if(count == 'undefined' || count == null)
					count = 0;
			  
				var connectId = -1;	
				if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
				{
					connectId = fileSystemService.addEventListener('readLineSignal(int , QVariant)', readLineCB );
				}			
			
				var retValue = fileSystemService.readLine(handleId, count, connectId); 
			
				if (connectId != -1) 
				{//async call
					return retValue;//returning trans id
				}
				else
				{//sync call
					if (retValue.ErrorCode)
						throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);	
					else
						return retValue.data;		
				}
				
				function readLineCB(tid, output)
				{
					if(tid == connectId)
					{
						fileSystemService.removeEventListener(tid);
						if( output.ErrorCode )
						{
							//if error callback is defined call it, else ignore the result
							if( typeof errorCB == 'function' )
								{
									errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
								}
						}
						else
						{
							//if success callback is defined call it, else ignore the result
							if( typeof successCB == 'function' )
								{
									successCB(output.data);
								}
						}
					}
				}
			},//end of readline
			
			/**
			 * Calls readBase64 method of fileSystemService that implements IFileSystem interface 
			 * of FileSystem Plugin
			 *
			 * @function readBase64
			 * @param {int} count - length of data to be read
			 * @param {function} successCB - success call back function
			 * @param {function} errorCB - error call back function 
			 * @returns Data read for sync call and transactin id for asycn calls
			 */
			readBase64 : function(count, successCB, errorCB){
				
				_chkFS_MandatoryNumber(count,'byte count invalid');
				_chkFS_Optionalcallback(successCB,'success Callback');
				_chkFS_Optionalcallback(errorCB,'error Callback');
				
			
				var connectId = -1;	
				if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
				{
					connectId = fileSystemService.addEventListener('readBase64Signal(int , QVariant)', readBase64CB );
				}			
			
				var retValue = fileSystemService.readBase64(handleId, count, connectId); 
			
				if (connectId != -1) 
				{//async call
					return retValue;//returning trans id
				}
				else
				{//sync call
					if (retValue.ErrorCode)
						throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);	
					else
						return retValue.data;		
				}
				
				function readBase64CB(tid, output)
				{			
					if(tid == connectId)
					{
						fileSystemService.removeEventListener(tid);
						if( output.ErrorCode )
						{
							//if error callback is defined call it, else ignore the result
							if( typeof errorCB == 'function' )
								{
									errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
								}
						}
						else
						{
							//if success callback is defined call it, else ignore the result
							if( typeof successCB == 'function' )
								{
									successCB(output.data);
								}
						}
					}		
				}
			},//end of readase64
			
			/**
			 * Calls write method of fileSystemService that implements IFileSystem interface 
			 * of FileSystem Plugin
			 *
			 * @function write
			 * @param {string} data - data to be written in the file
			 * @param {pos} data - postion in which data has to be written
			 * @param {function} successCB - success call back function
			 * @param {function} errorCB - error call back function 
			 * @returns transaction id for async calls
			 */
			write : function(data, position, successCB, errorCB){
				
				_chkFS_MandatoryString(data,'data not valid');
				_chkFS_OptionalNumber(position, 'position invalid');
				_chkFS_Optionalcallback(successCB,'success Callback');
				_chkFS_Optionalcallback(errorCB,'error Callback');
				
			  
				if( position < 0 )
				{
					throw new DeviceException(error.DATA_OUT_OF_RANGE_ERR,'position');
				}
				
				if(position == 'undefined' || position == null)
					position = -1;	
			  
				var connectId = -1;	
				if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
				{
					connectId = fileSystemService.addEventListener('writeSignal(int , QVariant)', writeCB );
				}			
			
				var retValue = fileSystemService.write(handleId, data, position, connectId); 
				if (connectId != -1) 
				{//async call
					return retValue;//returning trans id
				}
				else
				{//sync call
					if (retValue.ErrorCode)
						throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);	
				}
				
				function writeCB(tid, output)
				{	
					if(tid == connectId)
					{
						fileSystemService.removeEventListener(tid);
						if( output.ErrorCode )
						{
							//if error callback is defined call it, else ignore the result
							if( typeof errorCB == 'function' )
								{
									errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
								}
						}
						else
						{
							//if success callback is defined call it, else ignore the result
							if( typeof successCB == 'function' )
								{
									successCB();
								}
						}
					}		
				}
			},//end of write
						
			/**
			 * Calls writeLine method of fileSystemService that implements IFileSystem interface 
			 * of FileSystem Plugin
			 *
			 * @function writeLine
			 * @param {string} data - data to be written in the file
			 * @param {function} successCB - success call back function
			 * @param {function} errorCB - error call back function 
			 * @returns transaction id for async call
			 */
			writeLine : function(data, successCB, errorCB){
	
				_chkFS_MandatoryString(data,'data not valid');
				_chkFS_Optionalcallback(successCB,'success Callback');
				_chkFS_Optionalcallback(errorCB,'error Callback');
				
				
				var connectId = -1;	
				if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
				{
					connectId = fileSystemService.addEventListener('writeLineSignal(int , QVariant)', writeLineCB );
				}			
			
			 	var retValue = fileSystemService.writeLine(handleId, data, connectId); 
			
				if (connectId != -1) 
				{//async call
					return retValue;//returning trans id
				}
				else
				{//sync call
					if (retValue.ErrorCode)
						throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);	
				}
				
				function writeLineCB(tid, output)
				{
					if(tid == connectId)
					{
						fileSystemService.removeEventListener(tid);
						if( output.ErrorCode )
						{
							//if error callback is defined call it, else ignore the result
							if( typeof errorCB == 'function' )
								{
									errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
								}
						}
						else
						{
							//if success callback is defined call it, else ignore the result
							if( typeof successCB == 'function' )
								{
									successCB();
								}
						}
					}		
				}
			},//end of writeline
			
			/**
			 * Calls writeBase64 method of fileSystemService that implements IFileSystem interface 
			 * of FileSystem Plugin
			 *
			 * @function writeBase64
			 * @param {string} data - data to be written in the file
			 * @param {function} successCB - success call back function
			 * @param {function} errorCB - error call back function 
			 * @returns transaction id for async calls
			 */
			writeBase64 : function(data, successCB, errorCB){
			
				_chkFS_MandatoryString(data,'data not valid');
				_chkFS_Optionalcallback(successCB,'success Callback');
				_chkFS_Optionalcallback(errorCB,'error Callback');
				
				 	
				var connectId = -1;	
				if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
				{
					connectId = fileSystemService.addEventListener('writeBase64Signal(int , QVariant)', writeBase64CB );
				}
			
				var retValue = fileSystemService.writeBase64(handleId, data, connectId); 
			
				if (connectId != -1) 
				{//async call
					return retValue;//returning trans id			
				}
				else
				{//sync call
					if (retValue.ErrorCode)
						throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);	
				}
				
				function writeBase64CB(tid, output)
				{
					if(tid == connectId)
					{
						fileSystemService.removeEventListener(tid);
						if( output.ErrorCode )
						{
							//if error callback is defined call it, else ignore the result
							if( typeof errorCB == 'function' )
								{
									errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
								}
						}
						else
						{
							//if success callback is defined call it, else ignore the result
							if( typeof successCB == 'function' )
								{
									successCB();
								}
						}
					}		
				}
			}, // end of writebase64			
			
		/**
		 * Calls flush method of fileSystemService that implements IFileSystem interface 
		 * of FileSystem Plugin
		 *
		 * @function flush
		 * @param {function} successCB - success call back function
		 * @param {function} errorCB - error call back function
		 * @returns trans Id in async mode
		 */
		
		flush : function(successCB, errorCB){

			_chkFS_Optionalcallback(successCB, 'success callback');
			_chkFS_Optionalcallback(errorCB, 'error Callback');
			
		
			var connectId = -1;
			if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
			{
				connectId = fileSystemService.addEventListener('flushSignal(int, QVariant)', flushCB );
			}
		
		    var retValue = fileSystemService.flush(handleId, connectId);
			if (connectId != -1) {//async call
				return retValue;//returning trans id
			}
			else{//sync call
				if (retValue.ErrorCode)
					throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);			
			}
		
			function flushCB(tid, output)
			{				
				if(tid == connectId)
				{
					fileSystemService.removeEventListener(tid);
					if( output.ErrorCode )
					{
						//if error callback is defined call it, else ignore the result
						if( typeof errorCB == 'function' )
							{
								errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
							}
					}
					else
					{
						//if success callback is defined call it, else ignore the result
						if( typeof successCB == 'function' )
							{
								successCB();
							}
					}
				}		
			}
		},//end of flush
		
		/**
		 * Calls seek method of fileSystemService that implements IFileSystem interface 
		 * of FileSystem Plugin
		 *
		 * @function seek
		 * @param {int} seekOption - seekOption should SEEKSTART, SEEKCURRENT, SEEKEND
		 * @param {int} position - postion to be moved
		 * @returns trans Id in async mode
		 */
		
		seek : function(seekOption, position){			
			_chkFS_MandatoryNumber(seekOption, 'seek Option invalid');
			_chkFS_OptionalNumber(position, 'position invalid');
		
			if( (position == null) || (typeof position == 'undefined') )
				position = 0;
		
		    var retValue = fileSystemService.seek(handleId, seekOption, position);
		
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);			
			else
				return retValue.data;
		} //end of seek	
			
				
		};//end of file handle return
	}//end of file handle class
	
	return{
    /**
    * @constant {String} 
    * interfaceName.
    */	
    interfaceName: "filesystem",
    /**
    * @constant {Number} 
    * version.
    */      
    version: nokia.device._interfaces['filesystem'],
	/**
	 * 
	 * @constant {Number} file seek constants
	*/
	SEEKSTART	: 0,
	/**
	 * 
	 * @constant {Number} file seek constants
	*/	
	SEEKCURRENT : 1,
	/**
	 * 
	 * @constant {Number} file seek constants
	*/		
	SEEKEND	    : 2,
	/**
	 * 
	 * @constant {Number} memory type constants
	*/			
	INTERNAL : 0,
	/**
	 * 
	 * @constant {Number} memory type constants
	*/	
	REMOVABLE : 1,
	/**
	 * 
	 * @constant {Number} memory type constants
	*/	
	REMOTE : 2,
	/**
	 * 
	 * @constant {Number} memory type constants
	*/	
	UNKNOWN : 3,	
	/**
	 * 
	 * @constant {Number} connect status constants
	*/	
	DISCONNECTED : 0,
	/**
	 * 
	 * @constant {Number} connect status constants
	*/	
	CONNECTED : 1,
	
	/**
	 * 
	 * @constant {Number} search constants
	*/	
	PARTIAL : 0,
	/**
	 * 
	 * @constant {Number} search constants
	*/	
	COMPLETE : 1,
	/**
	 * 
	 * @constant {Number} element type constants
	*/		
	FILE : 0,
	/**
	 * 
	 * @constant {Number} element type constants
	*/		
	DIRECTORY : 1,
	/**
	 * 
	 * @constant {Number} element type constants
	*/		
	LINK : 2,

	
	
	/**
	 * Calls createDir method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function createDir
	 * @param {string} Uri - Uri path to be created in the file system
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */		
	createDir : function(SrcUri, successCB, errorCB){
		_chkFS_MandatoryString(SrcUri,'Uri');
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');
		
	
		var connectId = -1;
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('createDirSignal(int, QVariant)', createDirCB );
		}
		
	  var retValue = fileSystemService.createDir(SrcUri,connectId);
	    
		if (connectId != -1) {//async call
			return retValue;//returning trans id
		}
		else{//sync call		
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);			
		}
	
		function createDirCB(tid, output)
		{		
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}				
				}
				else
				{
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB();
						}
				}
			}
		}
	},//end of createDir
	
	/**
	 * Calls rename method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function rename
	 * @param {string} SrcUri - Source Uri path to be renamed in the file system
	 * @param {string} NewName - new name for the Source.
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	rename : function(SrcUri, NewName, successCB, errorCB){

		_chkFS_MandatoryString(SrcUri,'Source Uri');
		_chkFS_MandatoryString(NewName,'New Name');//new name is a mandatory param here
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');
		
	
		var connectId = -1;
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('renameSignal(int, QVariant)', renameCB );
		}
	
	    var retValue = fileSystemService.rename(SrcUri, NewName, connectId);
		
		if (connectId != -1) {//async call
			return retValue;//returning trans id
		}
		else{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);			
		}		
	
		function renameCB(tid, output)
		{
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}		
				}
				else
				{								
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB();
						}
				}
			}
		}
 	},//end of rename
	
	/**
	 * Calls getDirContents method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function getDirContents
	 * @param {string} SrcUri - Source direcorty Uri
	 * @param {string} matchPattern - items to be matched.
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	getDirContents : function(SrcUri, matchPattern, successCB, errorCB){
			
		_chkFS_MandatoryString(SrcUri,'Source Uri');
		_chkFS_OptionalString(matchPattern, 'match Pattern');	
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');	
		
	
		if( (matchPattern == 'undefined') || (matchPattern == null) )
			matchPattern = '*';
	
		var connectId = -1;
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('getDirContentsSignal(int , QVariant)', getDirContentsCB );
		}		
	
	    var retValue = fileSystemService.getDirContents(SrcUri, matchPattern, connectId);
		
		if (connectId != -1) 
		{//async call
			return retValue;//returning trans id
		}
		else
		{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);
			else
				return retValue.data;	
		}
		
		function getDirContentsCB(tid, output)
		{	
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB(output.data);
						}
				}
			}
		}
	},//end of getDirContents

	/**
	 * Calls getElementInfo method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function getElementInfo
	 * @param {string} SrcUri - Source direcorty Uri
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	getElementInfo : function(SrcUri, successCB, errorCB){
			
		_chkFS_MandatoryString(SrcUri,'Source Uri');	
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');	
	
		var connectId = -1;
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('getElementInfoSignal(int , QVariant)', getElementInfoCB );
		}

	    var retValue = fileSystemService.getElementInfo(SrcUri, connectId);
		
		if (connectId != -1) 
		{//async call
			return retValue;//returning trans id
		}
		else
		{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);
			else
				return retValue.data;	
		}
		
		function getElementInfoCB(tid, output)
		{
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB(output.data);
						}
				}
			}
		}
	},//end of getElementInfo
	
	/**
	 * Calls remove method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function remove
	 * @param {string} SrcUri - Source Uri path to be renamed in the file system
	 * @param {boolean} recursive - new name for the Source.
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	remove : function(SrcUri, recursive, successCB, errorCB){
	
		_chkFS_MandatoryString(SrcUri,'Source Uri');
		_chkFS_OptionalBool(recursive,'recursive');
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');
		
			
		if( (typeof recursive == 'undefined') || (recursive == null) )
			recursive = false;
	
		var connectId = -1;	
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('removeSignal(int , QVariant)', removeCB );
		}		
		
	    var retValue = fileSystemService.remove(SrcUri, recursive, connectId);
		
		if (connectId != -1) 
		{//async call
			return retValue;//returning trans id
		}
		else
		{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);		
		}
		
		function removeCB(tid, output)
		{	
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{									
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB();
						}
				}
			}
		}	
	},//end of rename
	
	/**
	 * Calls cancel method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function cancel
	 * @param {string} tid - transaction id
	 * @returns None
	 */
	cancel : function(tid){
		
		_chkFS_MandatoryNumber(tid, 'Transaction Id');
	
	    var retValue = fileSystemService.cancel(tid);
	
		if (retValue.ErrorCode)
		{
			throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);
		}		
		else
		{
			//if cancel is success remove the event listener based on tid
			fileSystemService.removeEventListener(tid);
		}
		
	},//end of cancel
	
	/**
	 * Calls openFile method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function openFile
	 * @param {string} uri - uri of the file
	 * @param {string} mode - mode in which the file should be opened.
	 * @param {string} encoding - encoding format.
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode or file handle in sync mode
	 */
	openFile : function(fileUri, openMode, encoding, successCB, errorCB){
	
		_chkFS_MandatoryString(fileUri,'uri');
		_chkFS_MandatoryString(openMode,'mode');
		_chkFS_MandatoryString(encoding,'encoding');	
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');	
		
	
		var connectId = -1;	
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('openFileSignal(int , QVariant)', openFileCB );
		}					
	
	    var retValue = fileSystemService.openFile(fileUri, openMode, encoding, connectId);
			
		if (connectId != -1) 
		{//async call
			return retValue;//returning trans id
		}
		else
		{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);
			else
				return new FileHandle(retValue.data);				
		}	
		
		function openFileCB(tid, output)
		{
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB(new FileHandle(output.data, fileSystemService));
						}
				}
			}				
		}			
	},//end of openFile
	
	/**
	 * Calls copy method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function copy
	 * @param {string} SrcUri - Source path
	 * @param {string} DestUri - Destination path
	 * @param {string} newName - new name
	 * @param {boolean} overWrite - overwrite option
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns None
	 */
	copy : function(SrcUri, DestUri, newName, overWrite, successCB, errorCB){
	
		_chkFS_MandatoryString(SrcUri,'SrcUri');
		_chkFS_MandatoryString(DestUri,'DestUri');	
		_chkFS_OptionalString(newName,'new Name');
		_chkFS_OptionalBool(overWrite, 'overWrite');
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');
		
		
		if( (typeof newName == 'undefined') || (newName == null))
			newName = "";			
	
		if( (typeof overWrite == 'undefined') || (overWrite == null))
			overWrite = false;	
	
		var connectId = -1;	
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('copySignal(int , QVariant )', copyCB );
		}
	
		var retValue = fileSystemService.copy(SrcUri, DestUri, newName, overWrite, connectId);
	
		if (connectId != -1) 
		{//async call
			return retValue;//returning trans id
		}
		else
		{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);			
		}
	
		function copyCB(tid, output)
		{
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
								errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));						
						}
				}
				else
				{									
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB();
						}
				}
			}
		}
	},//end of copy
	
	/**
	 * Calls move method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function move
	 * @param {string} SrcUri - Source path
	 * @param {string} DestUri - Destination path
	 * @param {string} newName - new name
	 * @param {boolean} overWrite - overwrite option
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns None
	 */
	move : function(SrcUri, DestUri, newName, overWrite, successCB, errorCB){
	
		_chkFS_MandatoryString(SrcUri,'SrcUri');
		_chkFS_MandatoryString(DestUri,'DestUri');	
		_chkFS_OptionalString(newName,'new Name');
		_chkFS_OptionalBool(overWrite, 'overWrite');
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');	
			
		
		if( (typeof newName == 'undefined') || (newName == null))
			newName = "";
	
		if( (typeof overWrite == 'undefined') || (overWrite == null))
			overWrite = false;
	
		var connectId = -1;	
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{	
			connectId = fileSystemService.addEventListener('moveSignal(int , QVariant )', moveCB );
		}
			
		var retValue = fileSystemService.move(SrcUri, DestUri, newName, overWrite, connectId);
	
		if (connectId != -1) 
		{//async call
			return retValue;//returning trans id
		}
		else
		{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);	
		}
	
		function moveCB(tid, output)
		{
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{									
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB();
						}
				}
			}
		}
	},//end of move
	
	/**
	 * Calls search method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function search
	 * @param {function} successCB - success call back function
	 * @param {string} matchPattern - items to be matched.
	 * @param {string} SrcUri - Source directory Uri
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	search : function(successCB, matchPattern, SrcUri, errorCB){
	
		_chkFS_Mandatorycallback(successCB,'callback');
		_chkFS_MandatoryString(matchPattern, 'matchPattern');
		_chkFS_MandatoryString(SrcUri, 'SrcUri');
		_chkFS_Optionalcallback(errorCB,'errorCB');
							
	
		var connectId = fileSystemService.addEventListener('searchSignal(int , QVariant)', searchCB );	
	
		//returning trans id
	    return fileSystemService.search(connectId, matchPattern, SrcUri);
	
		function searchCB(tid, output)
		{
			if (tid == connectId) 
			{
				if ( output.ErrorCode )//if some error
				{
					fileSystemService.removeEventListener(tid);
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}				
				}
				else
				{
					if(output.data.state == 1)//Search COMPLETE == 1
					{
						fileSystemService.removeEventListener(tid);
					}							
					successCB(output.data);
				}
			}		
		}
	},//end of search
	
	/**
	 * Calls getMountPoints method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function getMountPoints
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	getMountPoints : function(successCB, errorCB){
				
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');
		
		
		var connectId = -1;
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('getMountPointsSignal(int, QVariant)', getMountPointsCB );
		}
		
	    var retValue = fileSystemService.getMountPoints(connectId);
		if (connectId != -1) {//async call
			return retValue;//returning trans id
		}
		else{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);			
			else
			    return retValue.data;
		}
		
		function getMountPointsCB(tid, output)
		{		
		if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB(output.data);
						}
				}
			}
		}
	},//end of getMountPoints
	

	/**
	 * Calls getDefaultPath method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function getDefaultPath
	 * @param {string} contentType - content type
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	getDefaultPath : function(contentType,successCB, errorCB){
	
		_chkFS_OptionalString(contentType, 'content Type');
		_chkFS_Optionalcallback(successCB, 'success callback');
		_chkFS_Optionalcallback(errorCB, 'error Callback');
		
		
		if(contentType == "")
		{
			throw new DeviceException(error.INVALID_ARG_ERR,'contentType');
		}
	
		if( (contentType == null) || (typeof contentType == 'undefined') )		    
			contentType = "";
					
		var connectId = -1;
		if( (typeof successCB == 'function') || (typeof errorCB == 'function') )
		{
			connectId = fileSystemService.addEventListener('getDefaultPathSignal(int, QVariant)', getDefaultPathCB );
		}
		
	    var retValue = fileSystemService.getDefaultPath(contentType, connectId);
		if (connectId != -1) {//async call
			return retValue;//returning trans id
		}
		else{//sync call
			if (retValue.ErrorCode)
				throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);
			else
				return retValue.data;
		}
		
		function getDefaultPathCB(tid, output)
		{		
			if(tid == connectId)
			{
				fileSystemService.removeEventListener(tid);
				if( output.ErrorCode )
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}
				}
				else
				{
					//if success callback is defined call it, else ignore the result
					if( typeof successCB == 'function' )
						{
							successCB(output.data);
						}
				}
			}
		}
	},//end of getDefaultPath
	
	/**
	 * Calls notifyMountEvents method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function notifyMountEvents
	 * @param {function} successCB - success call back function
	 * @param {function} errorCB - error call back function
	 * @returns trans Id
	 */
	notifyMountEvents : function(successCB, errorCB){

		_chkFS_Mandatorycallback(successCB,'success callback');			
		_chkFS_Optionalcallback(errorCB,'error callback');
		
		var connectId = -1;
		connectId = fileSystemService.addEventListener('notifyMountEventsSignal(int, QVariant)', notifyMountEventsCB );
	
	    var retValue = fileSystemService.notifyMountEvents(connectId);
		if(retValue.ErrorCode) //if request already gng on
		{
			fileSystemService.removeEventListener(connectId);
			throw new DeviceException(retValue.ErrorCode,retValue.ErrorMessage);
		}
		else
		{
			//dont return the dummy trans id.
			//return retValue.data;
		}
	
		function notifyMountEventsCB(tid, output)
		{	
			if ( output.code )//if some error
			{
				//if error callback is defined call it, else ignore the result
				if( typeof errorCB == 'function' )
				{
					errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
				}
			}
			else
			{
				successCB(output.data);
			}		
		}
	},//end of notifyMountEvents
	
	/**
	 * Calls cacelNotify method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function cacelNotify
	 * @returns None
	 */
	cancelNotify : function(){
	
	    var retValue =  fileSystemService.cancelNotify();
		
		if (retValue.ErrorCode) {
			throw new DeviceException(retValue.ErrorCode, retValue.ErrorMessage);
		}
		else 
		{
			fileSystemService.removeEventListener(retValue.data);
		}
	},//end of cancelNotify
	
	/**
	 * Calls search_matchObject method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function searchMatchObject
	 * @param {function} successCB - success call back function
	 * @param {string} matchObject - items to be matched.
	 * @param {function} errorCB - error call back function
	 * @returns trans Id in async mode
	 */
	searchMatchObject : function(successCB, matchObject, errorCB){
		_chkFS_Mandatorycallback(successCB,'callback');
		_chkFS_MandatoryObject(matchObject, 'matchObject');
		_chkFS_Optionalcallback(errorCB,'errorCB');

		var connectId = -1;
		connectId = fileSystemService.addEventListener('searchMatchObjectSignal(int , QVariant)', searchMatchObjectCB);
		
		//returning trans id
	    return fileSystemService.searchMatchObject(connectId, matchObject);
	
		function searchMatchObjectCB(tid, output)
		{
			if (tid == connectId) 
			{
				fileSystemService.removeEventListener(tid);
				if ( output.ErrorCode )//if some error
				{
					//if error callback is defined call it, else ignore the result
					if( typeof errorCB == 'function' )
						{
							errorCB(new DeviceException(output.ErrorCode, output.ErrorMessage));
						}				
				}
				else
				{
					successCB(output.data);
				}
			}		
		}
	},//end of search	
	
	/**
	 * Calls mount method of fileSystemService that implements IFileSystem interface 
	 * of FileSystem Plugin
	 *
	 * @function mount
	 * @param {string} virtualPath - virtual path to be mounted
	 * @param {string} defaultPathKey - specifies the key for which CJSE default path	 
	 * should be obtained and mapped to the specified virtualPath
	 * @returns 
	 */
	mount : function(virtualPath,defaultPathKey){
		fileSystemService.mount(virtualPath,defaultPathKey);
	}//end of mount

	};//end of filesystem object return
}//end of filesystem class