/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 * @class
 */ 

nokia.device.Calendar = function PSCalendarInterface(){
    
	var DeviceException = nokia.device.DeviceException;
	var DeviceAPIError = nokia.device.DeviceAPIError;
	var notifyTid = null;
	
	var MISSING_ARG_ERR = 1;
	var INVALID_ARG_ERR = 2;
	var NOT_SUPPORTED_ERR = 3;
	
	
	var KErrArgument = -6;
	var KErrGeneral = -2;
	var KErrNotSupported = -5;
	
	var /*const causes rhino to fail */ CALENDAR_APP_ID = 0x10005901;
	
	//multiply the error codes by -1 before using this map
	
	var MapErrorCode = { 
	    33: 100, //KErrTimedOut
	    1: 101,//KErrNotFound
	    11: 102, //KErrAlreadyExists
	    16: 103,//    KErrServerBusy
	    14: 104, //KErrInUse    
	    21: 106, //KErrAccessDenied
	    4: 107, //KErrNoMemory
	    2: 2, // KErrGeneral - mapped to default ,
	    5: 3, //KErrNotSupported
	    6: 2, //KErrArgument
	    105: 105 //KDataRangeErr 
	}

    var error = new DeviceException(0,'dummy');
	
	function filterUndefined(obj){
        if(typeof obj === "object"){
            for(var key in obj) {
                if(typeof obj[key] === "undefined"){
                    delete obj[key];
                } 
                if(typeof obj[key] === "object"){
                    filterUndefined(obj[key]);
                }
            }	
        }
        return obj;
    }
	
	var __sp_calendar_entry_types = "MeetingReminderToDoAnniversaryDayEvent";
	
	function __sp_calendar_handle_error(errorCallback, errorCode){

	    if ((errorCallback != undefined) && (typeof errorCallback == 'function') && (errorCallback != null)) {
	       
	       
		    var retPosErr;
	        
	        switch (errorCode) {
	            case 100: //KErrTimedOut
	                retPosErr = new DeviceAPIError(error.TIMEOUT_ERR, "The implementation can not give a response within the given time");
	                break;
	                
	            case 101://KErrNotFound
	                retPosErr = new DeviceAPIError(error.DATA_NOT_FOUND_ERR, "Referred data entry was not found");
	                break;
	                
	            case 102: //KErrAlreadyExists
	                retPosErr = new DeviceAPIError( error.DATA_ALREADY_EXISTS_ERR,"Referred data entry already exists");
	                break;
	                
	            case 103: //    KErrServerBusy
	                retPosErr = new DeviceAPIError(error.SERVICE_BUSY_ERR , "The service is busy.Unable to process the request now, try again later");
	                break;
	                
	            case 104: //KErrInUse
	                retPosErr = new DeviceAPIError(error.SERVICE_IN_USE_ERR, "Registering for notifications is supported only once");
	                break;
	                
	            case 106: //KErrAccessDenied
	                retPosErr = new DeviceAPIError( error.NOT_ALLOWED_ERR, "No permission to access the specified content");
	                break;
	                
	            case 107: //KErrNoMemory
	                retPosErr = new DeviceAPIError(error.SIZE_EXCEEDED_ERR, "Data size exceeded the limits set by the platform or by the configuration");
	                break;
	            
		   		case 105: //DATA_OUT_OF_RANGE
					retPosErr = new DeviceAPIError(error.DATA_OUT_OF_RANGE_ERR, "Data supplied is out of range");
	                break;
					
	            default: 
	                retPosErr = new DeviceAPIError(-2 ,"General error");
	                break;
	        }
	        
	        
	        //Invoke user supplied error callback
	        errorCallback(retPosErr);
			return;
	        
	    }
	    //if errorCallabck is not supplied or it's not a valid one just ignore the error
	}

	/*
	 * Validate the data type of input params
	 * returns "true" if all data types are valid, else "false"
	 */
	function _isInputValid(__matchCrit){
	
        /* 
         * getlist/delete match criteria
         */
	
		if (__matchCrit != null) {
			if (__matchCrit.id != null) {
				if (typeof(__matchCrit.id) != "string") {
		
					return false;
				}
			}
			
			if (__matchCrit.type != null) {

				if ((typeof(__matchCrit.type) != "string") || !__sp_calendar_entry_types.match(__matchCrit.type)) {
					return false;
				}
			}
			
			if (__matchCrit.text != null) {
				if (typeof(__matchCrit.text) != "string") {
	
					return false;
				}
			}
			
			
			if (__matchCrit.range != null) {
				if (__matchCrit.range && typeof(__matchCrit.range) != "object") {
	
					return false;
				}
				
				if (__matchCrit.range.begin != null) {
					if (typeof(__matchCrit.range.begin) != "object") {
						return false;
					}
				}
				
				if (__matchCrit.range.end != null) {
					if (typeof(__matchCrit.range.end) != "object") {
					
						return false;
					}
				}
				
			}
			
			
			/*
			 * addEntry/updateEntry input ,  id and type checked above
			 */
			if (__matchCrit.summary != null) {
				if (typeof(__matchCrit.summary) != "string") {
					return false;
				}
			}
			
			if (__matchCrit.description != null) {
				if (typeof(__matchCrit.description) != "string") {
					return false;
				}
			}
			
			
			if (__matchCrit.status != null) {
			
				if (typeof(__matchCrit.status) != "string") {
					return false;
				}
				
			}
			
			
			if (__matchCrit.location != null) {
				if (typeof(__matchCrit.location) != "string") {
					return false;
				}
			}
			
			if ( __matchCrit.priority != null) {
				if (typeof(__matchCrit.priority) != "number") {
					return false;
				}
			}
			
			if (__matchCrit.instanceStartTime != null) {
				if (typeof(__matchCrit.instanceStartTime) != "object") {
					return false;
				}
			}
			
			if (__matchCrit.exceptionDates != null) {
				if (typeof(__matchCrit.exceptionDates) != "object") {
					return false;
				}
			}
			
			if (__matchCrit.time != null) {
				if(typeof __matchCrit.time != 'object')
				{
					return false;
				}
					var validDate = new Date("January 1, 1970 00:01"); 
					 
			
				if (__matchCrit.time.begin != null) {
				
					if (typeof(__matchCrit.time.begin) != "object") {
					
						return false;
					}
					if ((__matchCrit.time.begin) < validDate) {
						return false;
					}
				}
				
				if (__matchCrit.time.end != null) {
				
					if (typeof(__matchCrit.time.end) != "object") {
					
						return false;
					}
					var validDate = new Date("January 01, 1970 00:01");
					if((__matchCrit.time.end) < validDate ){
						return false;
					}
				}
				
				if (__matchCrit.time.alarm != null) {
				
					if (typeof(__matchCrit.time.alarm) != "object") {
					
						return false;
					}
				}
				if((__matchCrit.time.end) && (__matchCrit.time.alarm)){
					if((__matchCrit.time.end) < (__matchCrit.time.alarm)){
						return false;
					}
				}
			}
			
			if (__matchCrit.repeatRule != null) {
				if(typeof __matchCrit.repeatRule != 'object')
				{
					return false;
				}
				
				if ( __matchCrit.repeatRule.frequency != null )
				{
					if (typeof(__matchCrit.repeatRule.frequency) != "string") {
						return false;
					}
				}
			
				if ( __matchCrit.repeatRule.startDate != null )
				{
			
					if (typeof(__matchCrit.repeatRule.startDate) != "object") {
						return false;
					}
				}
				
				if ( __matchCrit.repeatRule.untilDate != null )
				{
					if (typeof(__matchCrit.repeatRule.untilDate) != "object") {
						return false;
					}
				}
				if ( __matchCrit.repeatRule.interval != null )
				{
					if (typeof(__matchCrit.repeatRule.interval) != "number") {
						return false;
					}
				}
				
				if ( __matchCrit.repeatRule.month != null )
				{
					if (typeof(__matchCrit.repeatRule.month) != "number") {
						return false;
					}
				}
				
				if ( __matchCrit.repeatRule.weekDays != null )
				{
					if (typeof(__matchCrit.repeatRule.weekDays) != "object") {
						return false;
					}
				}
				
				if ( __matchCrit.repeatRule.daysOfMonth != null )
				{
					if (typeof(__matchCrit.repeatRule.daysOfMonth) != "object") {
						return false;
					}
				}
				if ( __matchCrit.repeatRule.monthDates != null )
				{
					if (typeof(__matchCrit.repeatRule.monthDates) != "object") {
						return false;
					}
				}
				
			}
		}
    
    	return true;
	}
	
	

	
    /*
    * Internal
    * qt calender interface object.
    */

    var qtCalendarIf = nokia._services.load("nokia.device.calendar", "com.nokia.ICalendar", "1.0");
        
    qtCalendarIf.addEventListener("getListCallback(int,int,QObject*)", getList_calendar_cb);
    qtCalendarIf.addEventListener("notifyCallback(int,int,QObject*)", setNotification_cb);

     /*
     * Calendar getList callback handler
     */
    function getList_calendar_cb(errCode, transId, result){
    
        
		var retObj = asyncCallbackMap.get(transId);
		
		asyncCallbackMap.remove(transId);
        
        if (!retObj) {
            alert('asyncReqObj.get error');
            return;
        }
        var callback = retObj.success_cb;
        var errorCallback = retObj.error_cb;
        if (errCode) {
			if(errCode == -6)
			{
				//INVALID_ARG(-6) error should be returned as DATA_OUT_OF_RANGE(105) error for getList
				__sp_calendar_handle_error(errorCallback, 105 );
				return;
			}
            __sp_calendar_handle_error(errorCallback, MapErrorCode[(errCode * -1)]);
            return;
        }
	callback(result);
    }
 
     /*
     * intermediate callback function to handle retrieved event items.
     * @param	errorCode 	is DeviceException errror code.
     * @param	transId	is the trasactionId
     * @param	result is the eventData
     */
    function setNotification_cb(errCode, transId, result){
        var retObj = asyncCallbackMap.get(transId);
        if (retObj) {
            var callback = retObj.success_cb;
            var errorCallback = retObj.error_cb;
            if (errCode) {
                if (errorCallback) {
                    __sp_calendar_handle_error(errorCallback, MapErrorCode[(errCode * -1)]);
                }
            }
            else {
                //Invoke consumer callback with result.
                callback(result);
            }
        }
    }

   
    return {
    /**
    * @constant {String} 
    * interfaceName.
    */		
	interfaceName: "calendar",
    /**
    * @constant {Number} 
    * version.
    */      
    version: nokia.device._interfaces['calendar'],
    /**
     * Gets an iterator of calendar entries matching the supplied pattern.
     * This has both synchronous and asynchronous behaviour.
     * <br><br>
     * The output returned by this method depends on the filter criteria that can optionally
     * be passed with the <i>match</i> parameter, which is an instance of the
     * data scheme {@link MatchPattern}.
     * <br><br>
     * The contents of the {@link MatchPattern} instance will guide <i>getList()</i> as follows:
     * <br><br>
     * A calendar entry has an unique identifier, the id property, for entry identification.
     * If the id property is specified, entries matching this id will be returned,
     * in this case the other filter parameters will be ignored.
     * If id is not specified, the contents of the {@link MatchPattern} instance is interpreted as follows:
     * <ul>
     * <li>If the range property is specified, returns the instances falling within that range.
     *      In case both begin and end range are not specified, it returns all instances present
     *      in the calendar. </li>
     * <li>In case the text propety is specified, returns the instances matched with the summary field of the entry.
     *      The match is not case sensitive.</li>
     * <li>The type property can be used if only instances of a particular type are needed.</li>
     * </ul>
     * In case of instance search i.e. when id is not specified, the begin and end range refers to
     * instances falling within this range. In case of sync, first parameter will be matchPattern
     * <br><br>
     * The <i>matchPattern</i> parameter is optional. If it is not present, then all instances are returned.

     * @param {matchPattern} [match] Specifies a matching filter for the entries to return.The supplied argument may be one of:
     *                 <ul><li><i>null</i> or <i>undefined</i> to match all instances</li>
     *                 <li>an {@link MatchPattern} instace for getting a restricted set of {@link CalendarEntry} instances.</li></ul>
     *
     * @param {function} successCB A callback function with the signature
     *                     <i>function({@link Iterator} of {@link CalendarEntry})</i>
     *                     The iterator will iterate instances of {@link CalendarEntry}, which represent
     *                     calendar entries that match match pattern.
     *
     * @param {function} [errorCB(DeviceError)] This callback function gets invoked with
     * {@link DeviceError} object when error occurs in processing the async request.
     * <i>function(DeviceError)</i>
     * <i>{@link DeviceError}</i> - one of the errors:
     *
     * <br><br>
     * <b>Status Codes</b> are the codes returned in errorCallback function in DeviceError object.<br>
     * <br>
     *              Code 103 (DATA_NOT_FOUND_ERR): id specified does not exist.<br>
     *              Code 106 (DATA_OUT_OF_RANGE_ERR): Invalid value/out of range values passed for any of the parameters.<br>
     *              <br>
     *
     * @return {number/(iterator:calendarentry)} Unique transactionId will be returned.<br> In case of Synchrous invocation the iterator of calendar entries/ instances matching the supplied pattern will be returned instead of TransactionId.<br><br><br>
     *
     *
     * <b> NOTE: </b>{number} <b>nokia.device.calendar.getList (successCB, match, errorCB(DeviceError))</b> signature is also supported at present, but it will deprecated in future.
     * @exception {DeviceException} Code 1 (MISSING_ARG_ERR): A mandatory argument in a method is missing.<br>
     *                              Code 2 (INVALID_ARG_ERR): Invalid value/out of range values passed for any of the parameters.
     *                              <ul>
     *                              <li>successCB is not a function</li>
     *                              <li>match - does not conform to the {@link MatchPattern} data schema</li>
     *                              <li>errorCB is not a function</li>
     *                              </ul>
     *
     *
     *
     */
    getList: function(callback, matchPatternVal, errorCallback){ 
    
                filterUndefined(matchPatternVal);
                
                var isSync = false;
                var matchPattern;
                
                if (!callback && !matchPatternVal) {
                    //Sync
                    isSync = true;
                }
                else 
                    if (callback && (typeof(callback) != 'function') && !matchPatternVal) {
                        //Sync
                        isSync = true;
                    }
                else {
                    //Async
					
                    if (typeof(matchPatternVal) == 'function') {
                        //new style						
			matchPattern = callback;
                        callback = matchPatternVal;                        						
                    }
                    
                    else 
                        if (typeof(callback) == 'function') {
                            //old style                            						
                            matchPattern = matchPatternVal;
                        }
                        
                        else {
                            //Invalid Type error
                            throw new DeviceException(error.INVALID_ARG_ERR, "Calendar : getList : param order is invalid");
                        }
                }
	    if (isSync) {
	        //input validations..
	        matchPattern = callback;
		
	        if (matchPattern != null) {
			    if ((typeof matchPattern) != "object") {
			        throw new DeviceException(error.INVALID_ARG_ERR, "Calendar : getList : matchPattern is invalid");
			    }
			}
	    }
	    else{
                    
	       //input validations..
	        if (callback == "undefined" || callback == null) {
	            throw new DeviceException(error.MISSING_ARG_ERR, "Calendar : getList : callback is missing");
	        }
	        else 
	            if ((typeof callback) != "function") {
	                throw new DeviceException(error.INVALID_ARG_ERR, "Calendar : getList : callback is invalid");
	            }
	 
	        if( matchPattern != null ) {
	            if ((typeof matchPattern) != "object") {
	                throw new DeviceException(error.INVALID_ARG_ERR, "Calendar : getList : matchPattern is invalid");
	            }			
			}
	        
	        if ((typeof errorCallback) == "undefined" || errorCallback == null) {
	            errorCallback = null;
	        }
	        else 
	            if (errorCallback && (typeof errorCallback) != "function") {
	                throw new DeviceException(error.INVALID_ARG_ERR, "Calendar : getList : errorCallback is not a function");
	            }		
		}	
	    var isValid;
	 
	    if (!matchPattern) {    
				
	        isValid = true;
	        matchPattern = {};
	    }
	    
	    else {
	
	        isValid = _isInputValid(matchPattern);
			
		    if (!isValid) {
	            throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: getList: match param is invalid");
	    	}
	    }
	    var returnValue;
	    if (!isSync) {
	        var tid = asyncCallbackMap.add(callback, errorCallback);
			var serviceObject = qtCalendarIf;
	
			returnValue = qtCalendarIf.getList(matchPattern, tid);
			
			if (returnValue.errorCode != 0) {
			
			asyncCallbackMap.remove(tid);
				
			switch (returnValue.errorCode) {
			    case KErrNotSupported:
			    case KErrArgument:
							
					throw new DeviceException(MapErrorCode[(returnValue.errorCode * -1)], "Calendar: getList: Operation Failed");
			        break;
							
			    default:
			        __sp_calendar_handle_error(errorCallback, MapErrorCode[(returnValue.errorCode * -1)]);
			    }
			}
			return returnValue.transactionId;
		}
	    else{
	        returnValue = qtCalendarIf.getList(matchPattern);
			//In case of error returnValue will be null , it will be returned as it is to caller.No exception	            
	            
	        return returnValue;			
		}  
    },

   /**
    * Add a new entry into the calendar database. This is a synchronous method.
    * @param {CalendarEntry} calendarEntry The new calendar entry, that is added. 
    *                        The id property is not needed, it will
    *                         be ignored, as addEntry will generate an id to the new entry.
    * @exception  {DeviceException} Code 1 (MISSING_ARG_ERR): calendarEntry is undefined.<br>
    *                               Code 2 (INVALID_ARG_ERR): calendarEntry does not fit {@link CalendarEntry} data scheme.<br>
    *                               Code 105 (DATA_OUT_OF_RANGE_ERR): Invalid value passed for any of the parameters.
    * @return {string} The id of the added calendar entry.                           
    */
    addEntry : function(calendarEntry){

		filterUndefined(calendarEntry);
	 		
    	var connectId;
	
        if (calendarEntry) {
            if (typeof calendarEntry != 'object') {
                throw new DeviceException(error.INVALID_ARG_ERR, 'Calendar: addEntry: calendarEntry param is invalid');
            }
            
            if (calendarEntry.id) {
                calendarEntry.id = undefined;
            }
            
            if (!calendarEntry.type || !calendarEntry.time) {
                throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: addEntry: mandatory param missing");
            }
            
            if(calendarEntry.time === null) {
	    		//throw missing argument error
	    		throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: addEntry: mandatory param missing");
			}
	
			else if(typeof calendarEntry.time != 'object' || typeof calendarEntry.type != 'string')	{
        		throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: addEntry: calendarEntry param type is invalid");
			}
            
            if ((calendarEntry.type != 'ToDo') && !calendarEntry.time.begin) {
                throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: addEntry: mandatory param StartTime missing");
            }
            
            if (!calendarEntry.time.end && (calendarEntry.type == 'ToDo' || calendarEntry.type == 'Meeting')) {
                throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: addEntry: mandatory param EndTime missing");
            }
            
            if (calendarEntry.repeatRule) {
                if (typeof calendarEntry.repeatRule != 'object') {
                    throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: addEntry: repeatRule param type is invalid");
                }
                
                if (!calendarEntry.repeatRule.frequency) {
                    throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: addEntry: mandatory param Frequency missing");
                }
            }
            
            
            var isValid = _isInputValid(calendarEntry);
            
            if (!isValid) {
                throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: addEntry: calendarEntry param is invalid");
            }
        }
        
        else {
            throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: addEntry: mandatory param missing");
        }
        
        var returnValue;
        
        returnValue = qtCalendarIf.addEntry(calendarEntry);
        
        
        if (returnValue.errorCode != 0) {
            throw new DeviceException(MapErrorCode[(returnValue.errorCode * -1)], "Calendar: addEntry: Operation failed");
        }
        
        return returnValue.uid;
        
    },

    /**
     * Update an existing entry in the calendar database. This is a synchronous method.
     * <br>
     * <br>
     * The calendar entry to be updated must exist and its id must be available in the 
     * {@link CalendarEntry} data scema instance that is passed as a parameter to <i>updateEntry</i>. 
     * In case you wish to modify only a specific instance or occurrence of a repeating calendar entry 
     * then specify start time along with id. If time.begin is not specified, 
     * the whole entry (i.e., all of its instances) will be modified. All the parameters of an 
     * entry can be updated except its type.
     * <br>
     * <br>
     * If an instance of a repeating entry is modified, on child entry supported platforms a new child entry 
     * will be created, on other platforms the new entry will be an independent entry which bears no 
     * relation to the original entry. 
     * @param {CalendarEntry} calendarEntry Calendar entry with at least the id property. The id property must 
     *                        identify an existing calendar entry, otherwise <i>updateEntry</i> will 
     *                        throw an exception. 
     * @return {void}                         
     * @exception {DeviceException} Code 1 (MISSING_ARG_ERR): calendarEntry is undefined.<br>
     *                              Code 2 (INVALID_ARG_ERR): calendarEntry object missed an id field see {@link CalendarEntry}.<br>
     *                              Code 101 (DATA_NOT_FOUND_ERR): <ul><li>id specified does not exist. </li>
     *                                                             <li>instanceStartTime not found.</li>
     *                                                             </ul>
     */
     updateEntry: function(calendarEntry){
        
         var connectId;
            
         filterUndefined(calendarEntry);
            
         if (calendarEntry) {
             if (typeof calendarEntry != 'object') {
                 throw new DeviceException(error.INVALID_ARG_ERR, 'Calendar: updateEntry: calendarEntry param is invalid');
             }
                
             if (!calendarEntry.id) {
                 throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: updateEntry: mandatory param - Id missing");
             }
                
             if (calendarEntry.repeatRule) {
                
                 if (typeof calendarEntry.repeatRule != 'object') {
                     throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: updateEntry: repeatRule param type is invalid");
                 }
                    
             }
                
                
			 if(calendarEntry.time === null) {
				 //throw missing arg
				 throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: updateEntry: mandatory param missing");
			 }
			 var isValid = _isInputValid(calendarEntry);
			        
			 if (!isValid) {
			     throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: updateEntry: calendarEntry param is invalid");
			 }
   		 }
            
	    else {
	        throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: updateEntry: mandatory param missing");
	    }
            
    
	    var returnValue;
	    
	    returnValue = qtCalendarIf.addEntry(calendarEntry);
    
    
	    if (returnValue.errorCode != 0) {
	        throw new DeviceException(MapErrorCode[(returnValue.errorCode * -1)], "Calendar: updateEntry: Operation failed");
	    }
    },
    
    /**
     * Delete an entry or an instance from the calendar database.
     * This is a synchronous API.
     * <br>
     * <br>
     * The {@link DeleteData} instance that is passed as a parameter to <i>deleteEntry</i> 
     * determines whether entry deletion or instance deletion should be performed. 
     * If the {@link DeleteData} instance contains only the id property, then the entry 
     * matching that id will be deleted.  
     * <br>
     * <br>
     * If both id and range properties are present, 
     * then only those instances falling within the specified time range and matching 
     * id specified will be deleted.
     * @param {DeleteData} deleteData A DeleteData instance that specifies what will be deleted.
     * @return {void}
     * @exception {DeviceException} Code 1 (MISSING_ARG_ERR): deleteData is undefined.<br>
     *                              Code 2 (INVALID_ARG_ERR): deleteData does not conform to {@link DeleteData} data scheme.<br>
     */
    deleteEntry : function(deleteData){
		
		filterUndefined(deleteData);
			
	    var connectId;
	
            if (deleteData) {
            
                if (typeof deleteData != 'object') {
                    throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: deleteEntry: delete Data is invalid");
                }
                
                if (!deleteData.id) {
                    throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: deleteEntry: missing Mandatory param");
                }
            }
            
            else {
                throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: deleteEntry: missing delete data");
            }
            
            
            
            var isValid = _isInputValid(deleteData);
            
            if (!isValid) {
                throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: deleteEntry: delete data is invalid");
            }
            
            var returnValue;
            
            returnValue = qtCalendarIf.deleteEntry(deleteData);
            
            if (returnValue.errorCode != 0) {
                throw new DeviceException(MapErrorCode[(returnValue.errorCode * -1)], "Calendar: deleteEntry: Operation failed");
            }
        },

    /**
     * Start the device native calendar editor UI with pre-populated calendar entry data.
     * This is an asynchronous method. 
     * This function can be used to to add a new entry or update an existing entry through
     * the native calendar editor.
     * 
     * @param {function} __success_cb A callback function with the signature
     *                     <i>function(transactionId, status, data)</i>
     * @param {CalendarEntry} calendarEntry A {@link CalendarEntry} data scheme instance,
     *                  whose properties are used to prepopulate the native caledar editor.
     * @param {function} [_error_cb] This callback function gets invoked with
     * {@link DeviceError} object when error occurs in processing the async request.
     * <i>function(DeviceError)</i>
     * {@link DeviceError}
     * @return {number} Returns a transaction id.
     * @exception {DeviceException} Code 1 (MISSING_ARG_ERR): calendarEntry or
     *                callback is undefined.<br>
     *                Code 2 (INVALID_ARG_ERR):
     *                                 <ul>
     *                              <li>callback is not a function</li>
     *                              <li>calendarEntry - does not conform to the specified {@link CalendarEntry} data scheme.</li>
     *                              <li>errorCallback is not a function</li>
     *                              </ul>
     */
    startEditor : function(__success_cb, calendarEntry, _error_cb){
	       
		    if(!__success_cb)
				throw new DeviceException(error.MISSING_ARG_ERR, 'StartEditor:Missing Success Callback');			
	
			if((typeof __success_cb) != "function" )
				throw new DeviceException(error.INVALID_ARG_ERR, 'StartEditor:Invalid Success Callback');
	
			if(_error_cb)
				if((typeof _error_cb) != "function" )
					throw new DeviceException(error.INVALID_ARG_ERR, 'StartEditor:Invalid Error Callback');
			  
		    function startEditor_cb(){   
				
					__success_cb(null);				
		    }
			
			var connId  = qtCalendarIf.addEventListener("asyncCallback()", startEditor_cb);
			
		    try{
			
		    	var ret = qtCalendarIf.startEditor(calendarEntry, connId);
								  
			  	if (ret.errorCode != 0)
	    			throw new DeviceException(MapErrorCode[(ret.errorCode * -1)], ret.errorMessage);
		    }
			
			catch(er){
				//alert('er '+er);
			   	throw(er);
			}

    },
    
    /**
     * Cancels an asynchronous request identified by transactionId.
     * This is synchronous API.
     *
     * @param {number} transactionId
     *           Identifier of the ongoing asynchronous request to be canceled.
     * @return {void}
     * @exception {DeviceException} Code 1 (MISSING_ARG_ERR): transactionId is undefined.<br>
     *                           Code 2 (INVALID_ARG_ERR): transactionId is not a number.<br>
     *                           Code 101 (DATA_NOT_FOUND_ERR): If there are no active asynchronous request corresponds to the specified trasactionid.
     */
    cancel: function(transactionId){
		
        if (!transactionId) {
            throw new DeviceException(error.MISSING_ARG_ERR, "Calendar: cancel: transaction id is missing");
        }
        
        else 
            if (typeof(transactionId) != "number") {
                throw new DeviceException(error.INVALID_ARG_ERR, "Calendar: cancel: transaction id is invalid");
            }
        
        var returnValue = qtCalendarIf.cancel(transactionId);
        
        if (returnValue.errorCode != 0) {
        
            throw new DeviceException(MapErrorCode[(returnValue.errorCode * -1)], "Calendar: cancel: Operation failed");
            
        }
        
	},
	
	/**
     * Subscribe for Calendar entry change Notification.
     * This is an asynchronous method. 
     * 
     * @param {function} succes_cb A callback function with the signature
     *                     <i>function(transactionId, status, data)</i>
     * 
     * @param {function} [fail_cb] This callback function gets invoked with
     * {@link DeviceError} object when error occurs in processing the async request.
     * <i>function(DeviceError)</i>
     * {@link DeviceError}
     * @return {void} 
     * @exception {DeviceException} Code 1 (MISSING_ARG_ERR): calendarEntry or
     *                callback is undefined.<br>
     *                Code 2 (INVALID_ARG_ERR):
     *                                 <ul>
     *                              <li>succes_cb  is not a function</li>
     *                              <li>fail_cb is not a function</li>
     *                              </ul>
     */
	subscribeNotification: function(succes_cb, fail_cb){
	    
	    
		
	    if (!succes_cb) {
	        /*
	         * MissingArgument error "callback not passed";
	         */
	        throw new DeviceException( error.MISSING_ARG_ERR, 'Calendar: subscribeNotification: callback is missing');
	    }
		
	    /*
	     * here we check if the type of callback is function. 
	     * This case is useful when arbitrary values are passed to callback
	     */
	    else 
	        if (typeof succes_cb != "function") {
	            throw new DeviceException(error.INVALID_ARG_ERR, 'Calendar: subscribeNotification: callback is invalid');
	        }
			
		if (fail_cb) {
			if (typeof fail_cb != 'function') {
				throw new DeviceException(error.INVALID_ARG_ERR, 'Calendar: subscribeNotification: error callback is invalid');
			}
		}
 	    //Remove previous callback entry
        if (notifyTid) {
            //asyncCallbackMap.remove(notifyTid);
			throw new DeviceException(error.INVALID_ARG_ERR, 'Calendar: subscribeNotification: multiple requests not supported');
        }
		//Add new callback
        var tid = asyncCallbackMap.add(succes_cb, fail_cb);
		//Store current transactionId			
        notifyTid = tid;
        var serviceObject = qtCalendarIf;
        
        var isValid;
        
        var returnValue = qtCalendarIf.subscribeNotification(tid);
            
        if (returnValue.errorCode != 0) {
			//Error, hence remove callback entry from map
			asyncCallbackMap.remove(notifyTid);
			//Error, hence set to null
			notifyTid = null;
            
            switch (returnValue.errorCode) {
                case KErrNotSupported:
                case KErrArgument:
                    
                    throw new DeviceException(MapErrorCode[(returnValue.errorCode * -1)], "Calendar: subscribeNotification: Operation failed");
                    break;
                    
                default:
                    __sp_calendar_handle_error(fail_cb, MapErrorCode[(returnValue.errorCode * -1)]);
            }
        }
    },
	
	
	/**
     * Cancel a request for notification.
     * This is synchronous API.
     * 
     * @return {void}
     * @exception {DeviceException}  Code 101 (DATA_NOT_FOUND_ERR): If there are no active asynchronous request 
     */
	cancelNotification : function(){
        var returnValue = qtCalendarIf.cancelNotification();
        if (notifyTid) {
            asyncCallbackMap.remove(notifyTid);
            notifyTid = null;
        }
		if(returnValue.errorCode != 0) {
            throw new DeviceException(MapErrorCode[(returnValue.errorCode * -1)], "Calendar: cancelNotification: Operation failed");
        }
    }
        
        
    };
}
