/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <unistd.h>
#include <sys/types.h>

#include <QDebug>
#include <QStringList>

#ifdef Q_OS_MAEMO6
#include <DuiApplication>
#include <DuiApplicationWindow>
#include <DuiLocale>
#endif

#ifdef Q_OS_MAEMO5
#include <QApplication>
#include <QTranslator>
#include <QWidget>
#endif

#include "WidgetManager.h"
#include "widgetinstallerview.h"

// FIXME: there should be no test code in the production build
//
#include "widgetinstallertesthelper.h"

void printExitMessage(int result) {
    qDebug() << "*************************************************";
    qDebug() << "WidgetInstaller exited with " << result;
    qDebug() << "*************************************************";
}

/*This message output will be called
 * when the application tries print to console with qDebug qWarning QCritical or qFatal
 * Since Dui 0.17 there is apparently an another filter which filters everything.
 * This one overwrites it and enables the tests to be executed & verified
 */
void DebugMessageOutput(QtMsgType type, const char *msg) {
    switch (type) {
    case QtDebugMsg:
    case QtWarningMsg:
    case QtCriticalMsg:
    case QtFatalMsg:
        fprintf(stderr, "%s\n", msg);
    }
}

int main(int argc, char *argv[]) {

    bool isRoot = geteuid() == 0;

#ifdef Q_OS_MAEMO6
    DuiApplication app(argc, argv);
    DuiApplicationWindow window;
    if (isRoot) {
        qCritical() << "Running as root! You don't want to do this!";
    return 2;
    }
#endif

#ifdef Q_OS_MAEMO5
    QApplication app(argc, argv);
    QTranslator translator;
    translator.load(":/widgetinstaller.qm");
    app.installTranslator(&translator);

    QWidget window;
    if (isRoot) {
        qCritical() << "Running as root! You don't want to do this!";
        return 2;
    }
#endif

    QString widgetId;

    qInstallMsgHandler(DebugMessageOutput);
    // check for arguments, at least one is required
    if (argc < 2) {
        qDebug("Usage: %s <wgtpackage>", argv[0]);
        return 1;
    }

    // parse the arguments
    QStringList argsList = QApplication::arguments();
    QString wgtName(argsList[1]);

    bool silentflag = argsList.contains("--silent");

    // FIXME: there should be no test code in the production build
    //
    bool testflag = argsList.contains("--test");

    WidgetManager mgr(NULL);

    WidgetInstallerView view(&mgr);
    view.setSilentMode(silentflag || testflag);

#ifndef Q_OS_MAEMO5
    // show main window only if silent flag is not set
    if (!silentflag && !testflag) {
        window.show();
    }
#endif

    QObject::connect(&mgr, SIGNAL(installationError(WidgetInstallError)), &view,
                     SLOT(slotInstallationError(WidgetInstallError)));
    QObject::connect(&mgr, SIGNAL(aboutToReplaceExistingWidget(QString)), &view,
                     SLOT(slotAboutToReplaceExistingWidget()));
    QObject::connect(&mgr, SIGNAL(aboutToInstallWidgetWithFeatures(QList<QString>)), &view,
                     SLOT(slotAboutToInstallWidgetWithFeatures(QList<QString>)));
    QObject::connect(&mgr, SIGNAL(installationSucceed()), &view,
                     SLOT(slotInstallationSucceeded()));
    QObject::connect(&mgr, SIGNAL(aboutToInstallUntrustedWidget(WidgetInstallError)), &view,
                     SLOT(slotAboutToInstallUntrustedWidget(WidgetInstallError)));

    mgr.install(wgtName, widgetId, true);

    // FIXME: there should be no test code in the production build
    //
    if (silentflag || testflag || view.result() == WidgetInstallSuccess) {
        window.close();
        printExitMessage(view.result());
        if (testflag) {
            WRT::Maemo::WidgetInstallerTestHelper::writeOutput(mgr);
        }
        return view.result();
    }

    return app.exec();
}
