/*
 * Copyright (C) 2010 Nokia Corporation.
 *
 * Author:  Flavio Ceolin <flavio.ceolin@profusion.mobi>
 *          Jonas Gastal <jgastal@profusion.mobi>
 *          Raphael Kubo da Costa <kubo@profusion.mobi>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <libxml/xmlreader.h>
#include <libxml/relaxng.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <glib.h>
#include "xmlparser.h"
#include "wgt2deb.h"

#ifdef __ARMEL__
#define TMPDIR "/var/tmp"
#else
#define TMPDIR "/var/cache/apt/archives"
#endif

#define MAEMO_VERSION "maemo5"
#define MAGIC_CHARACTER 'W'
#define NAME_TEMPLATE "widget_%1$d.deb"
#define TEMPLATE_NAME_SIZE (13)

char *configxmlbuff = NULL;

static int get_random_number(const int min, const int max)
{
    static int fd = -1;
    char *next_byte;
    int bytes_to_read;
    unsigned value;

    if (min > max)
        return -1;

    fd = open("/dev/random", O_RDONLY);
    if (fd == -1)
        return -2;

    next_byte = (char *)&value;
    bytes_to_read = sizeof(value);

    do {
        int bytes = read(fd, next_byte, bytes_to_read);
        bytes_to_read -= bytes;
        next_byte += bytes;
    } while (bytes_to_read > 0);

    close(fd);
    return min + (value % (max - min + 1));
}

static char *create_debian_name()
{
    char *debian_name = (char *)malloc(TEMPLATE_NAME_SIZE);

    if (!debian_name)
        return NULL;

    snprintf(debian_name, TEMPLATE_NAME_SIZE, NAME_TEMPLATE,
             get_random_number(10000, 99999));

    return debian_name;
}

static xmlChar *validate_name(xmlChar *string)
{
    if (!string)
        return (xmlChar *)create_debian_name();

    xmlChar *newString = xmlStrdup(string);

    /* The name must be at least two characters and start with alphanumeric */
    if (strlen((char *)newString) < 2) {
        xmlFree(newString);
        return (xmlChar *)create_debian_name();
    }

    if (!isalnum(*newString))
        *newString = MAGIC_CHARACTER;

    int pos = 0;
    while (*(newString + pos) != '\0') {
        *(newString + pos) = (isalnum(*(newString + pos)) || *(newString + pos) == '.')
                              ? *(newString + pos) : '.';
        pos++;
    }

    return newString;
}

static xmlChar *simplify_name(xmlChar *string)
{
    if (!string)
        return string;

    int head = 0;
    int end = 0;
    int size = xmlStrlen(string);
    xmlChar *last = string + size;
    bool quit = true;
    while (1) {
        if (!isalnum(*(string + head))) {
            head++;
            quit = false;
        }
        if (!isalnum(*(last - end))) {
            end++;
            quit = false;
        }

        if (quit)
            break;

        quit = true;
    }

    return xmlStrsub(string, head, size - head - end + 1);
}

static void split_viewmode(xmlChar *string, xmlChar **homescreen, xmlChar **application)
{
    int cont = 0;

    if (!string)
        return;

    gchar **tokens = g_strsplit((gchar *)string, " ", -1);

    if (!tokens)
        return;

    while (tokens[cont] && ((*application == NULL) || (*homescreen == NULL))) {
        if (*application == NULL) {
            if (!g_strcmp0(tokens[cont], "windowed") || !g_strcmp0(tokens[cont], "fullscreen"))
                *application = (xmlChar *)g_strdup(tokens[cont]);

            if (g_strcmp0(tokens[cont], "maximized"))
                *application = (xmlChar *)g_strdup("windowed");
        }

        if ((*homescreen == NULL) && (!g_strcmp0(tokens[cont], "minimized")))
            *homescreen = (xmlChar *)g_strdup(tokens[cont]);
        cont++;
    }

    g_strfreev(tokens);
}

static xmlChar *validate_source(xmlChar *string)
{
    if (!string)
        return NULL;

    int pos = 0;

    if (!isalnum(*string))
        *string = MAGIC_CHARACTER;

    while (*(string + pos) != '\0') {
        *(string + pos) = (isalnum(*(string + pos)) || *(string + pos) == '+' ||
                           *(string + pos) == '.') ? *(string + pos) : '.';
        pos++;
    }

    return string;
}

bool create_debian_package(const char *wgt_path, const char *deb_path, const char *webappid)
{
    xmlRelaxNGCleanupTypes();

    xmlDocPtr doc;
    xmlNodePtr nodeRoot, nodeChildren;
    xmlChar *properties, *width, *height, *version, *name, *validatedName, *author, *license,
        *description, *icon, *viewmode, *viewmode_homescreen, *viewmode_application;
    char *buff = get_config_from_zip(wgt_path);
    bool ret = true;

    if (buff == NULL) {
        ERR("Cannot get config.xml");
        return false;
    }

    viewmode_application = NULL;
    viewmode_homescreen = NULL;
    description = NULL;
    properties = NULL;
    viewmode = NULL;
    version = NULL;
    license = NULL;
    height = NULL;
    author = NULL;
    width = NULL;
    name = NULL;
    validatedName = NULL;
    icon = NULL;

    doc = xmlParseDoc((xmlChar *)buff);

    if (doc == NULL) {
        ERR("File %s does not exist", wgt_path);
        ret = false;
        goto end;
    }

    for (nodeRoot = doc->children; nodeRoot != NULL; nodeRoot = nodeRoot->next) {
        width = xmlGetProp(nodeRoot, (xmlChar *)"width");
        height = xmlGetProp(nodeRoot, (xmlChar *)"height");
        version = xmlGetProp(nodeRoot, (xmlChar *)"version");
        viewmode = xmlGetProp(nodeRoot, (xmlChar *)"viewmodes");

        for (nodeChildren = nodeRoot->children; nodeChildren != NULL; nodeChildren = nodeChildren->next) {
            if (!strncmp((char*)nodeChildren->name, "name", 4))
                name = xmlNodeGetContent(nodeChildren);
            else if (!strncmp((char*)nodeChildren->name, "description", 11))
                description = xmlNodeGetContent(nodeChildren);
            else if (!strncmp((char*)nodeChildren->name, "license", 7))
                license = xmlNodeGetContent(nodeChildren);
            else if (!strncmp((char*)nodeChildren->name, "author", 6))
                author = xmlGetProp(nodeChildren, (xmlChar *)"email");
            else if (!strncmp((char*)nodeChildren->name, "icon", 4))
                icon = xmlGetProp(nodeChildren, (xmlChar *)"src");
        }
    }

    if (!(validatedName = validate_name(name))) {
        ret = false;
        goto end;
    }
    name = simplify_name(name);

    if (!(version = validate_source(version)))
        version = (xmlChar *)strdup("0.0.1");

    split_viewmode(viewmode, &viewmode_homescreen, &viewmode_application);

    if (!create_debian(MAEMO_VERSION, (char *)name, (char *)validatedName, (char *)version, (char *)description,
                       (char **)&icon, (char *)viewmode_homescreen, (char *)viewmode_application,
                       (char *)webappid, (char *)wgt_path, (char *)deb_path, TMPDIR, (char *)license))
        ret = false;

end:
    xmlFree(viewmode_application);
    xmlFree(viewmode_homescreen);
    xmlFree(configxmlbuff);
    xmlFree(description);
    xmlFree(properties);
    xmlFree(viewmode);
    xmlFree(version);
    xmlFree(license);
    xmlFree(height);
    xmlFree(author);
    xmlFree(width);
    xmlFree(name);
    xmlFree(validatedName);
    xmlFree(icon);

    xmlFreeNode(nodeRoot);
    xmlFreeDoc(doc);
    free(buff);

    return ret;
}
