/*
 * Copyright (C) 2010 Nokia Corporation.
 *
 * Author:  Flavio Ceolin <flavio.ceolin@profusion.mobi>
 *          Jonas Gastal <jgastal@profusion.mobi>
 *          Raphael Kubo da Costa <kubo@profusion.mobi>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <stdbool.h>
#include <string.h>
#include <limits.h>
#include <sys/time.h>
#include <errno.h>
#include <archive.h>
#include <archive_entry.h>
#include <glib.h>
#include "wgt2deb.h"

#define ERR_ARCHIVE(err, archive, fmt, ...)                             \
    _err_archive(__FILE__, __LINE__, __FUNCTION__,                      \
                  err, archive, fmt, ##__VA_ARGS__)

#define BUFFSIZE (10240)

#define FOLDER_SIZE (4096)

#define WGT_INSTALL_TEMPLATE "usr/share/wrt/data/widgets_21D_4C7/%1$s"
#define WGT_INSTALL_ZIP_TEMPLATE "usr/share/wrt/data/widgets_21D_4C7/%1$s/%2$s"

#define DESKTOP_PATH_PREFIX   "usr/share/applications/hildon"
#define DESKTOP_HOME_PATH_PREFIX "usr/share/applications"
#define DESKTOP_PATH_TEMPLATE DESKTOP_PATH_PREFIX"/%s.desktop"

#define HILDON_DESKTOP_PATH_PREFIX   DESKTOP_HOME_PATH_PREFIX"/hildon-home"
#define HILDON_DESKTOP_PATH_TEMPLATE HILDON_DESKTOP_PATH_PREFIX"/%s.desktop"

#define DEFAULT_VIEW_MODE "windowed"
#define DEFAULT_ICON "Icon-music"
#define DEFAULT_DESCRIPTION "A web application"

#define DEFAULT_W3C_ICONS "icon.png"
#define ICON_PATH "usr/share/icons/hicolor/scalable/hildon/"

#define DESKTOP_TEMPLATE                                       \
    "[Desktop Entry]\n"                                        \
    "Encoding=UTF-8\n"                                         \
    "Name=%1$s\n"                                              \
    "X-Path=%1$s\n"                                            \
    "Version=%2$s\n"                                           \
    "Comment=%3$s\n"                                           \
    "Icon=%4$s\n"                                              \
    "X-Icon-path=/usr/share/icons/\n"                          \
    "X-Window-Icon=%4$s\n"                                     \
    "X-Window-Icon-Dimmed=%4$s\n"                              \
    "Type=W3CWidget\n"                                         \
    "[WRT]\n"                                                  \
    "Type=W3CWidget\n"                                         \
    "StartHtml=file:///%5$s\n"                                 \
    "ViewMode=%6$s\n"                                          \
    "WebAppId=%7$s"

#define DESKTOP_TEMPLATE_HOME                                                           \
     "[Desktop Entry]\n"                                                                \
     "Encoding=UTF-8\n"                                                                 \
     "Name=%1$s\n"                                                                      \
     "Exec=/usr/bin/webwidgetrunner /usr/share/applications/hildon/%7$s.desktop\n"      \
     "X-Path=%1$s\n"                                                                    \
     "Version=%2$s\n"                                                                   \
     "Comment=%3$s\n"                                                                   \
     "Icon=%4$s\n"                                                                      \
     "Type=Application\n"                                                               \
     "[WRT]\n"                                                                          \
     "StartHtml=file:///%5$s\n"                                                         \
     "ViewMode=%6$s\n"                                                                  \
     "WebAppId=%7$s"

#define CONTROL_TEMPLATE                        \
    "Package: %2$s\n"                           \
    "Source: %2$s\n"                            \
    "Priority: optional\n"                      \
    "Section: user/desktop\n"                   \
    "Maintainer:\n"                             \
    "Depends: qtwrt-experimental\n"             \
    "Version: %3$s\n"                           \
    "Architecture: all\n"                       \
    "Maemo-Display-Name: %1$s\n"                \
    "Description: %4$s\n"                       \
    "Installed-Size: %5$lu\n"

#define PREINST_TEMPLATE                       \
    "#!/bin/sh\n"                              \
    "# preinst script for WRT - Widget\n"     \
    "# see: dh_installdeb(1)\n"                \
    "cat > /tmp/%1$s.txt <<EOF\n"              \
    "%2$s\n"                                   \
    "EOF\n"                                    \
    "maemo-confirm-text /tmp/%1$s.txt\n"       \
    "retval=$?\n"                              \
    "rm /tmp/%1$s.txt\n"                       \
    "exit $retval"

#define POSTINST_TEMPLATE                                                                                      \
    "#!/bin/sh\n"                                                                                              \
    "ulimit > /tmp/ulimit\n"                                                                                   \
    "unzip '/usr/share/wrt/data/widgets_21D_4C7/%1$s/%2$s' -d /usr/share/wrt/data/widgets_21D_4C7/%1$s/\n"     \
    "rm /usr/share/wrt/data/widgets_21D_4C7/%1$s/%2$s\n"                                                       \
    "/usr/bin/webappregisterer /usr/share/wrt/data/widgets_21D_4C7/%1$s %1$s register\n"                       \
    "retval=$?\n"                                                                                              \
    "if [ $retval != 0 ]; then\n"                                                                              \
    "   exit $retval\n"                                                                                        \
    "fi\n"                                                                                                     \
    "if [ ! -d /usr/share/webwidgets/registry ]; then\n"                                                       \
    "   mkdir -p /usr/share/webwidgets/registry\n"                                                             \
    "   mkdir /usr/share/webwidgets/registry/%1$s\n"                                                           \
    "fi\n"                                                                                                     \
    "chmod 777 /usr/share/webwidgets/registry/%1$s\n"                                                          \
    "touch /usr/share/icons/hicolor/\n"                                                                        \
    "exit 0\n"

#define POSTRM_TEMPLATE                                     \
    "#!/bin/sh\n"                                           \
    "rm -rf /usr/share/webwidgets/registry/%1$s\n"          \
    "rm /etc/secure/s/%1$s\n"                               \
    "if [ \"$1\" = \"purge\" -o \"$1\" = \"remove\" ]\n"    \
    "then\n"                                                \
    "rm -rf /usr/share/wrt/data/data/%1$s\n"                \
    "rm -rf /usr/share/wrt/data/data/%2$s\n"                \
    "rm -rf /usr/share/wrt/data/widgets_21D_4C7/%1$s\n"     \
    "fi\n"                                                  \
    "exec /usr/bin/webappregisterer %1$s unregister\n"      \
    "exit 0\n"

static void _err_archive(const char *file, int line, const char *func, int err, struct archive *archive, const char *fmt, ...) GNU_PRINTF(6, 7);
static bool add_file_printf(struct archive *output, const char *uname, const char *gname, mode_t mode, const char *path, const char *fmt, ...) GNU_PRINTF(6, 7);
static void get_security_folder_name(char *deb_path);

static struct timeval now = {0, 0};
static const char *user = NULL;
static const char *group = NULL;
static char *folder_name = NULL;
static int debug = 0;
static FILE *_debug_fp = stderr;
static unsigned long widget_size = 0;

static void _err_archive(const char *file, int line, const char *func, int err, struct archive *archive, const char *fmt, ...)
{
    va_list ap;
    const char *cat;

    switch (err) {
    case ARCHIVE_OK:
        cat = "OK";
        break;
    case ARCHIVE_EOF:
        cat = "EOF";
        break;
    case ARCHIVE_RETRY:
        cat = "RETRY";
        break;
    case ARCHIVE_WARN:
        cat = "WARN";
        break;
    case ARCHIVE_FAILED:
        cat = "FAIL";
        break;
    case ARCHIVE_FATAL:
        cat = "FATAL";
        break;
    default:
        fprintf(_debug_fp, "Unknown error type: %d\n", err);
        cat = "????";
    }

    va_start(ap, fmt);
    fprintf(_debug_fp, "%s: %s:%d %s() ", cat, file, line, func);
    vfprintf(_debug_fp, fmt, ap);
    fprintf(_debug_fp, " (%s)\n", archive_error_string(archive));
    fflush(_debug_fp);
    va_end(ap);
}

void _err(const char *file, int line, const char *func, const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    fprintf(_debug_fp, "ERR: %s:%d %s() ", file, line, func);
    vfprintf(_debug_fp, fmt, ap);
    fputc('\n', _debug_fp);
    fflush(_debug_fp);
    va_end(ap);
}

void _dbg(const char *file, int line, const char *func, const char *fmt, ...)
{
    va_list ap;

    if (!debug)
        return;

    va_start(ap, fmt);
    fprintf(_debug_fp, "DBG: %s:%d %s() ", file, line, func);
    vfprintf(_debug_fp, fmt, ap);
    fputc('\n', _debug_fp);
    fflush(_debug_fp);
    va_end(ap);
}

void dbg_init(void)
{
    const char *env = getenv("DEBUG");
    if (!env)
        return;
    if (strncmp(env, "file:", sizeof("file:") - 1) == 0) {
        const char *fname = env + sizeof("file:") - 1;
        _debug_fp = fopen(fname, "wb+");
        if (!_debug_fp) {
            fprintf(stderr, "ERROR: could not open for writting '%s': %s\n",
                    fname, strerror(errno));
            _debug_fp = stderr;
        }
        debug = 1;
    } else
        debug = atoi(env);
}

/**
 * Formats @p contents so that it can be used in the Description field of a Debian control file.
 *
 * Basically, all lines but the first must start with a space, duplicate empty lines must be removed,
 * and empty lines must have a dot.
 *
 * If @p contents is NULL, this function returns NULL and no processing is done.
 *
 * @param contents The string that will be formatted (it will not be changed).
 * @return A string correctly formatted that must be freed with g_free() later.
 */
static gchar *format_description_for_control(const gchar *contents)
{
    if (!contents)
        return NULL;

    gchar *contents_copy = g_strdup(contents);

    GRegex *regex1 = g_regex_new("\\R(.+)", G_REGEX_RAW, (GRegexMatchFlags)0, NULL);
    GRegex *regex2 = g_regex_new("^$\\R(^$\\R)+", (GRegexCompileFlags)(G_REGEX_RAW | G_REGEX_MULTILINE),
                                 (GRegexMatchFlags)0, NULL);

    gchar *indented_lines = g_regex_replace(regex1, contents_copy, -1, 0, "\n \\1",
                                            (GRegexMatchFlags)0, NULL);
    gchar *empty_lines_with_dot = g_regex_replace(regex2, g_strstrip(indented_lines),
                                                  -1, 0, " .\n", (GRegexMatchFlags)0, NULL);

    g_free(indented_lines);
    g_free(contents_copy);

    g_regex_unref(regex1);
    g_regex_unref(regex2);

    return empty_lines_with_dot;
}

/**
 * Formats @p contents so that it can be used in the Comments field in a .desktop file.
 *
 * Basically, trailing whitespace is removed and then newline characters are escaped.
 *
 * If @p contents is NULL, this function returns NULL and no precessing is done.
 *
 * @param contents The string that will me formatted (it will not be changed).
 * @return A string correctly formatted that must be freed with g_free() later.
 */
static gchar *format_description_for_desktop(const gchar *contents)
{
    if (!contents)
        return NULL;

    gchar *contents_copy = g_strdup(contents);
    gchar *ret;

    ret = g_strescape(g_strstrip(contents_copy), NULL);

    g_free(contents_copy);

    return ret;
}

static struct archive *open_zip(const char *filename)
{
    struct archive *a = archive_read_new();
    int err;

    if (!a) {
        ERR("could not creat read archive.");
        return NULL;
    }

    err = archive_read_support_format_zip(a);
    if (err != ARCHIVE_OK)
        goto error;
    err = archive_read_open_filename(a, filename, BUFFSIZE);
    if (err != ARCHIVE_OK)
        goto error;

    DBG("opened zip '%s': %p", filename, a);
    return a;

  error:
    ERR_ARCHIVE(err, a, "unable to open ZIP for read '%s'.", filename);
    archive_read_finish(a);
    return NULL;
}

static struct archive *create_tgz(const char *filename)
{
    struct archive *archive = archive_write_new();
    int err;

    if (!archive) {
        ERR("could not create write archive.");
        return NULL;
    }

    err = archive_write_set_compression_gzip(archive);
    if (err != ARCHIVE_OK)
        goto error;
    err = archive_write_set_format_pax_restricted(archive);
    if (err != ARCHIVE_OK)
        goto error;
    err = archive_write_open_filename(archive, filename);
    if (err != ARCHIVE_OK)
        goto error;

    DBG("created tgz '%s': %p", filename, archive);
    return archive;

  error:
    ERR_ARCHIVE(err, archive, "unable to open TGZ for write '%s'.", filename);
    archive_write_finish(archive);
    return NULL;
}

static bool add_recursive_dirs(struct archive *output, const char *directory,
                               const char *uname, const char *gname, mode_t mode)
{
    struct archive_entry *entry;
    char dir[PATH_MAX], *p;
    int size;

    if (directory[0] == '/')
        directory++;

    size = strlen(directory);
    if (size >= (int)sizeof(dir)) {
        ERR("directory is too big: '%s'", directory);
        return false;
    }

    entry = archive_entry_new();
    if (!entry) {
        ERR("could not create entry.");
        return false;
    }

    memcpy(dir, directory, size + 1);
    p = dir;
    do {
        int err;

        p = strchr(p, '/');
        if (p)
            *p = '\0';

        gettimeofday(&now, NULL);

        DBG("create 0%o %s:%s '%s'", mode, uname, gname, dir);
        archive_entry_set_pathname(entry, dir);
        archive_entry_set_size(entry, FOLDER_SIZE);
        archive_entry_set_ctime(entry, now.tv_sec, 0);
        archive_entry_set_mtime(entry, now.tv_sec, 0);
        archive_entry_set_atime(entry, now.tv_sec, 0);
        archive_entry_set_birthtime(entry, now.tv_sec, 0);
        archive_entry_set_uname(entry, uname);
        archive_entry_set_gname(entry, gname);
        archive_entry_set_mode(entry, mode);
        archive_entry_set_filetype(entry, AE_IFDIR);

        err = archive_write_header(output, entry);
        if (err != ARCHIVE_OK) {
            ERR_ARCHIVE(err, output, "Unable to write header for '%s'.", dir);
            archive_entry_free(entry);
            return false;
        }

        archive_entry_clear(entry);
        if (p) {
            *p = '/';
            p++;
        }
    } while (p);

    archive_entry_free(entry);
    return true;
}

static bool write_all_data(struct archive *output, const char *buff, size_t size)
{
    size_t done = 0;

    DBG("<<< write all %zd bytes.", size);

    while (done < size) {
        size_t r = archive_write_data(output, buff + done, size - done);
        if (r == (size_t)-1) {
            ERR("Could not write data to file (size=%zd)", size - done);
            return false;
        }
        done += r;
    }

    DBG(">>> wrote all %zd bytes.", done);

    return true;
}

static bool add_file_from_fs(struct archive *output, const char *uname, const char *gname,
                             mode_t mode, const char *src, const char *dst)
{
    struct archive_entry *entry;
    struct stat st;
    int err;
    FILE *input;
    char buff[BUFFSIZE];
    size_t todo;

    if (!dst)
        dst = src;

    DBG("'%s' -> '%s', mode=0%o, owner=%s:%s", src, dst, mode, uname, gname);

    input = fopen(src, "rb");
    if (!input) {
        ERR("Could not open source '%s': %s.", src, strerror(errno));
        return false;
    }

    if (fstat(fileno(input), &st) != 0) {
        ERR("Could not stat source '%s': %s.", src, strerror(errno));
        fclose(input);
        return false;
    }

    st.st_mode = (st.st_mode & ~0777) | mode;

    DBG("'%s' -> '%s', size=%zd, mode=0%o, owner=%s:%s",
        src, dst, (size_t)st.st_size, st.st_mode, uname, gname);

    entry = archive_entry_new();
    if (!entry) {
        ERR("Could not create archive entry for '%s' -> '%s'.", src, dst);
        fclose(input);
        return false;
    }

    archive_entry_copy_stat(entry, &st);
    archive_entry_set_size(entry, st.st_size);
    archive_entry_set_pathname(entry, dst);
    archive_entry_set_uname(entry, uname);
    archive_entry_set_gname(entry, gname);

    err = archive_write_header(output, entry);
    if (err != ARCHIVE_OK) {
        ERR_ARCHIVE(err, output, "Unable to write header for '%s'.", dst);
        archive_entry_free(entry);
        fclose(input);
        return false;
    }

    archive_entry_free(entry);

    todo = st.st_size;
    widget_size += st.st_size;
    while (todo) {
        size_t size = (todo < BUFFSIZE) ? todo : BUFFSIZE;
        size_t r = fread(buff, 1, size, input);

        if (r <= 0) {
            if (feof(input))
                break;
            else {
                ERR("Error reading file '%s': %s.", src, strerror(errno));
                fclose(input);
                return false;
            }
        }

        if (!write_all_data(output, buff, r)) {
            fclose(input);
            return false;
        }

        todo -= r;
    }

    fclose(input);

    err = archive_write_finish_entry(output);
    if (err != ARCHIVE_OK) {
        ERR_ARCHIVE(err, output, "Unable to finish writing entry.");
        return false;
    }

    DBG("done '%s' -> '%s'", src, dst);
    return true;
}

static bool add_file_from_buff(struct archive *output, const char *uname, const char *gname, mode_t mode,
                               const char *path, const char *buff, size_t size)
{
    struct archive_entry *entry = archive_entry_new();
    int err;

    if (!entry) {
        ERR("Could not create archive entry for '%s'.", path);
        return false;
    }

    DBG("'%s', size=%zd, mode=0%o, owner=%s:%s",
        path, size, mode, uname, gname);

    gettimeofday(&now, NULL);

    archive_entry_set_pathname(entry, path);
    archive_entry_set_size(entry, size);
    archive_entry_set_ctime(entry, now.tv_sec, 0);
    archive_entry_set_mtime(entry, now.tv_sec, 0);
    archive_entry_set_atime(entry, now.tv_sec, 0);
    archive_entry_set_birthtime(entry, now.tv_sec, 0);
    archive_entry_set_mode(entry, mode);
    archive_entry_set_filetype(entry, AE_IFREG);
    archive_entry_set_uname(entry, uname);
    archive_entry_set_gname(entry, gname);
    widget_size += size;
    err = archive_write_header(output, entry);
    if (err != ARCHIVE_OK) {
        ERR_ARCHIVE(err, output, "Unable to write header for '%s'.", path);
        archive_entry_free(entry);
        return false;
    }

    archive_entry_free(entry);

    if (!write_all_data(output, buff, size))
        return false;

    err = archive_write_finish_entry(output);
    if (err != ARCHIVE_OK) {
        ERR_ARCHIVE(err, output, "Unable to finish writing entry.");
        return false;
    }

    DBG("done '%s', size=%zd, mode=%o", path, size, mode);
    return true;
}

static bool add_file_printf(struct archive *output, const char *uname, const char *gname,
                            mode_t mode, const char *path, const char *fmt, ...)
{
    va_list ap;
    char *buff;
    int size;
    bool ret;

    va_start(ap, fmt);
    size = vasprintf(&buff, fmt, ap);
    va_end(ap);

    if (size < 0) {
        ERR("could not format file '%s' template:\n%s\n\n", path, fmt);
        return false;
    }

    ret = add_file_from_buff(output, uname, gname, mode, path, buff, size);
    widget_size += size;
    free(buff);
    return ret;
}

static char* validate_icon(char *string)
{
    char *ch = strrchr(string, '.');
    if(ch)
        *ch = '\0';
    return string;
}

static int digits(unsigned int num)
{
    int digits = 0;
    while(num > 0) {
        num /= 10;
        digits++;
    }
    return digits;
}

static char* create_icon_name(const char *webappid, const char *name)
{
    unsigned int idhash = g_str_hash(webappid);
    unsigned int iconhash = g_str_hash(name);
    int icon_sz = digits(idhash) + digits(iconhash) + 1;
    char *dot = strrchr(name, '.');
    icon_sz += dot ? strlen(dot) : 0;
    char *icon = (char*)malloc(icon_sz);
    if(!icon) {
        ERR("Couldn't allocate memory: %d", icon_sz);
        return 0;
    }
    if(dot)
        snprintf(icon, icon_sz, "%u%u%s", idhash, iconhash, dot);
    else
        snprintf(icon, icon_sz, "%u%u", idhash, iconhash);
    return icon;
}

static char* get_zipfile_name(char *string)
{
    char *msg;
    char *token = NULL;
    char *string_copy = strdup(string);

    if (!strrchr(string, '/'))
        return string_copy;

    token = strtok(string_copy,"/");
    while (token != NULL){
        msg = token;
        token = strtok(NULL,"/");
    }
    free(string_copy);
    return strdup(msg);
}

static bool copy_files(struct archive *input, struct archive *output, const char *prefix,
                       const char *webappid, char **icon, const char *uname, const char *gname)
{
    struct archive_entry *entry;
    char path[PATH_MAX], icon_path[PATH_MAX] = ICON_PATH;
    size_t len;

    if (!input) {
        ERR("no input handle.");
        return false;
    }
    if (!output) {
        ERR("no output handle.");
        return false;
    }

    len = (prefix) ? strlen(prefix) : 0;
    while (len > 0 && prefix[len - 1] == '/')
        len--;

    if (len > 0) {
        if (len + 2 + NAME_MAX >= sizeof(path)) {
            ERR("prefix is too long: '%s'", prefix);
            return false;
        }

        memcpy(path, prefix, len);
        path[len] = '/';
        len++;
        path[len] = '\0';

        DBG("prefix: '%s'", path);
    }

    while (archive_read_next_header(input, &entry) == ARCHIVE_OK) {
        const char *name = archive_entry_pathname(entry);
        ERR("name: %s", name);
        DBG("Checking for icon");

        if ((*icon && !strcmp(name, *icon)) || strstr(DEFAULT_W3C_ICONS, name)) {
            *icon = create_icon_name(webappid, name);
            if(!*icon)
                return false;
            strcat(icon_path, *icon);

            struct archive_entry *output_entry = archive_entry_clone(entry);
            archive_entry_set_uname(output_entry, uname);
            archive_entry_set_gname(output_entry, gname);
            archive_entry_set_pathname(output_entry, icon_path);

            char buffer[64 * 1024];
            size_t size = archive_read_data(input, buffer, 64 * 1024);
            widget_size += size;
            archive_entry_set_size(output_entry, size);
            archive_write_header(output, output_entry);
            archive_entry_free(output_entry);

            write_all_data(output, buffer, size);
            archive_write_finish_entry(output);

            *icon = validate_icon(*icon);
            DBG("Icon found: %s", *icon);
            break;
        }
    }

    return true;
}

static bool add_debian_data_maemo5(struct archive *output, const char *prefix,
                                   const char *name, const char *version, const char *description,
                                   const char *icon, const char *viewmode_homescreen,
                                   const char *viewmode_application, const char *webappid)
{
    char desktop_path[PATH_MAX];
    gchar *formatted_desc = NULL;
    bool ret = false;

    if (!add_recursive_dirs
        (output, HILDON_DESKTOP_PATH_PREFIX, "root", "root", 0755))
        goto error;

    formatted_desc = format_description_for_desktop(description);

    if (viewmode_application) {
        snprintf(desktop_path, sizeof(desktop_path), DESKTOP_PATH_TEMPLATE, webappid);
        if (!add_file_printf
            (output, "root", "root", 0644, desktop_path, DESKTOP_TEMPLATE_HOME,
             name, version, formatted_desc, icon, prefix, viewmode_application, webappid))
            goto error;
    }

    if (viewmode_homescreen) {
        snprintf(desktop_path, sizeof(desktop_path), HILDON_DESKTOP_PATH_TEMPLATE,
                webappid);
        if (!add_file_printf
            (output, "root", "root", 0644, desktop_path, DESKTOP_TEMPLATE,
             name, version, formatted_desc, icon, prefix, viewmode_homescreen, webappid))
            goto error;
    }

    ret = true;

error:
    g_free(formatted_desc);
    return ret;
}

static bool create_debian_data_zip(const char *distro, const char *name, const char *version,
                                   const char *description, char **icon, const char *viewmode_homescreen,
                                   const char *viewmode_application, const char *webappid, const char *wgt_path,
                                   const char *data_path)
{
    struct archive *input, *output;
    char prefix[PATH_MAX];
    char widget_path_and_name[PATH_MAX];
    char *zipfile;
    bool ret;

    DBG("create data.tar.gz");
    output = create_tgz(data_path);
    if (!output) {
        return false;
    }

    DBG("target distro: %s", distro);
    snprintf(prefix, sizeof(prefix), WGT_INSTALL_TEMPLATE, webappid);
    if (!add_recursive_dirs(output, prefix, user, group, 0755)) {
        archive_write_finish(output);
        return false;
    }

    DBG("copy wgt -> data.tar.gz");
    zipfile = get_zipfile_name((char *)wgt_path);
    snprintf(widget_path_and_name, sizeof(widget_path_and_name), WGT_INSTALL_ZIP_TEMPLATE, webappid, zipfile);
    if (!add_file_from_fs
        (output, "root", "root", 0644, wgt_path, widget_path_and_name)) {
        archive_write_close(output);
        archive_write_finish(output);
        unlink(data_path);
        free(zipfile);
        return false;
    }
    free(zipfile);

    DBG("copy icon -> data.tar.gz");
    input = open_zip(wgt_path);
    if (!input) {
        archive_write_close(output);
        archive_write_finish(output);
        unlink(data_path);
        return false;
    }

    if (!copy_files(input, output, prefix, webappid ? webappid : name, icon, user, group)) {
        archive_read_close(input);
        archive_read_finish(input);
        archive_write_close(output);
        archive_write_finish(output);
        unlink(data_path);
        return false;
    }

    if(!*icon)
        *icon = strdup(DEFAULT_ICON);

    /* dispatch creation of system specific files (.desktop & the rest) */
    if (strcmp(distro, "maemo5") == 0) {
        ret = add_debian_data_maemo5
            (output, prefix,
             name, version, description, *icon, viewmode_homescreen, viewmode_application, webappid);
    } else {
        ERR("unknown distro: %s", distro);
        ret = false;
    }

    if (ret)
        DBG("done data.tar.gz");
    else
        DBG("failed to create data.tar.gz for %s", distro);
    archive_write_close(output);
    archive_write_finish(output);
    return ret;
}

static bool create_debian_control(const char *name, const char *validatedName, const char *version, const char *description,
                                  const char *license, const char *webappid, const char *control_path, const char *zipfile)
{
    gchar *formatted_desc = NULL;
    char *id = NULL;

    struct archive *output = create_tgz(control_path);
    if (!output)
        return false;

    DBG("create control.tar.gz");

    formatted_desc = format_description_for_control(description);

    if (!add_file_printf
        (output, "root", "root", 0644, "control",
         CONTROL_TEMPLATE, name, validatedName, version, formatted_desc, widget_size / 1024))
        goto error;
    if (!add_file_printf
        (output, "root", "root", 0755, "postinst",
         POSTINST_TEMPLATE, webappid, zipfile))
        goto error;
    if (!add_file_printf
        (output, "root", "root", 0755, "postrm",
         POSTRM_TEMPLATE, webappid, folder_name))
        goto error;

    if (license){
        if (!add_file_printf
            (output, "root", "root", 0755, "preinst",
             PREINST_TEMPLATE, validatedName, license))
            goto error;
    }

    DBG("done control.tar.gz");
    archive_write_close(output);
    archive_write_finish(output);
    g_free(formatted_desc);
    free(folder_name);
    free(id);
    return true;

  error:
    DBG("failed to create control.tar.gz");
    archive_write_close(output);
    archive_write_finish(output);
    g_free(formatted_desc);
    free(folder_name);
    free(id);
    return false;
}

static bool create_debian_deb(const char *data_path, const char *control_path, const char *deb_path)
{
    struct archive *output = archive_write_new();
    int err;

    if (!output) {
        ERR("could not create write archive.");
        return NULL;
    }

    DBG("create %s", deb_path);
    err = archive_write_set_format_ar_bsd(output);
    if (err != ARCHIVE_OK)
        goto error;
    err = archive_write_open_filename(output, deb_path);
    if (err != ARCHIVE_OK)
        goto error;

    if (!add_file_from_buff
        (output, "root", "root", 0644, "debian-binary", "2.0\n", 4))
        goto add_error;
    if (!add_file_from_fs
        (output, "root", "root", 0644, data_path, "data.tar.gz"))
        goto add_error;
    if (!add_file_from_fs
        (output, "root", "root", 0644, control_path, "control.tar.gz"))
        goto add_error;

    DBG("done %s", deb_path);
    archive_write_close(output);
    archive_write_finish(output);

    return true;

  add_error:
    DBG("failed to create %s", deb_path);
    archive_write_close(output);
    archive_write_finish(output);
    return false;

  error:
    ERR_ARCHIVE(err, output, "unable to open DEB for write '%s'.", deb_path);
    archive_write_finish(output);
    return false;
}

bool create_debian(const char *distro, const char *name, const char *validatedName, const char *version,
                   const char *description, char **icon, const char *viewmode_homescreen,
                   const char *viewmode_application, const char *webappid, const char *wgt_path,
                   const char *deb_path, const char *tmpdir, const char *license)
{
    char data[PATH_MAX], control[PATH_MAX];
    char m_description[PATH_MAX], m_viewmode_application[PATH_MAX];
    char *zipfile;

    DBG("create debian");

    get_security_folder_name((char *)deb_path);

    strcpy(m_description, description ? description : DEFAULT_DESCRIPTION);

    if (!viewmode_application) {
        if (!viewmode_homescreen)
            strcpy(m_viewmode_application, DEFAULT_VIEW_MODE);
        else
            m_viewmode_application[0] = '\0';

    } else {
        strcpy(m_viewmode_application, viewmode_application);
    }

    snprintf(data, sizeof(data), "%s/data.tar.gz", tmpdir);
    if (strlen(m_viewmode_application)) {
        if (!create_debian_data_zip
            (distro, name, version, m_description, icon, viewmode_homescreen,
             m_viewmode_application, webappid, wgt_path, data)) {
            unlink(control);
            return false;
        }
    } else {
        if (!create_debian_data_zip
            (distro, name, version, m_description, icon, viewmode_homescreen,
             NULL, webappid, wgt_path, data)) {
            unlink(control);
            return false;
        }
    }
    zipfile = get_zipfile_name((char *)wgt_path);
    snprintf(control, sizeof(control), "%s/control.tar.gz", tmpdir);
    if (webappid) {
        if (!create_debian_control(name, validatedName, version, description, license, webappid, control, zipfile)) {
            free(zipfile);
            return false;
        }
    } else {
        if (!create_debian_control(name, validatedName, version, description, license, name, control, zipfile)) {
            free(zipfile);
            return false;
        }
    }
    free(zipfile);

    if (!create_debian_deb(data, control, deb_path)) {
        unlink(data);
        unlink(control);
        return false;
    }

    DBG("done debian");
    unlink(data);
    unlink(control);

    return true;
}

char *get_config_from_zip(const char *zipfile)
{
    char *contents = NULL;
    struct archive_entry *entry;
    struct archive *m_archive = open_zip(zipfile);

    if (!m_archive) {
        ERR("Cannot open %s", zipfile);
        return NULL;
    }

    while (archive_read_next_header(m_archive, &entry) == ARCHIVE_OK) {
        if (!strcmp(archive_entry_pathname(entry), "config.xml")) {
            size_t size = 0;

            // FIXME: this is ugly, but we can't rely on the entry size returned by archive_entry_size()
            contents = (char*)malloc(64 * 1024);
            if (!contents) {
                ERR("Couldn't allocate memory: %d", 64 * 1024);
                return 0;
            }

            size = archive_read_data(m_archive, contents, 64 * 1024);
            contents[size] = '\0';
            break;
        }
    }

    archive_read_finish(m_archive);
    if (contents)
        return contents;

    return NULL;
}

static void get_security_folder_name(char *deb_path)
{
    if (!deb_path)
        return;
    char *token = NULL;
    char *deb_name = NULL;
    char *copy = strdup(deb_path);

    token = strtok(copy, "/");
    while (token != NULL) {
        deb_name = token;
        token = strtok(NULL, "/");
    }

    deb_name = strdup(deb_name);
    free(copy);
    token = strtok(deb_name, "-");

    if (token) {
        free(folder_name);
        folder_name = strdup(token);
    }

    free(deb_name);
}
