/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "standaloneview.h"

#include "exitbutton.h"
#include "jscontextmenu.h"
#include "webkineticscroller.h"
#include "wrtpage.h"
#include "wrtwebinspector.h"

#include <QGraphicsWebView>
#include <QMenu>
#include <QMenuBar>
#include <QWebFrame>

namespace WRT {
namespace Maemo {

StandaloneView::StandaloneView(QGraphicsWebView* webView, QWidget* parent, bool enableInspector)
    : QGraphicsView(parent)
    , m_webView(webView)
    , m_inspector(0)
    , m_actWebInspector(0)
{
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setFrameShape(QFrame::NoFrame);
    setAttribute(Qt::WA_DeleteOnClose);
    setAttribute(Qt::WA_Maemo5AutoOrientation);
    setWindowFlags(Qt::Window);

    m_scene = new QGraphicsScene;
    setScene(m_scene);

    m_scroller = new WebKineticScroller(m_webView);
    m_scene->addItem(m_scroller);

    setWindowTitle(m_webView->title());

    m_menuBar = new QMenuBar(this);

    m_exitButton = new ExitButton(this);
    m_exitButton->setVisible(false);
    connect(m_exitButton, SIGNAL(clicked()),
            this, SIGNAL(exitRequested()));

    if( enableInspector) {
        m_inspector = new WrtWebInspector();
        m_inspector->hide(); // initially hidden
        m_scene->addItem( m_inspector);

        m_actWebInspector = new QAction("Web inspector", m_menuBar);
        m_actWebInspector->setCheckable(true);
        connect( m_actWebInspector, SIGNAL(toggled(bool)), this, SLOT( showInspector(bool)));
        m_menuBar->addAction( m_actWebInspector);
    }
}

StandaloneView::~StandaloneView()
{
    m_scene->removeItem(m_webView);

    delete m_scene;
}

void StandaloneView::resizeEvent(QResizeEvent* event)
{
    QGraphicsView::resizeEvent(event);

    setUpdatesEnabled(false);

    QRectF rect(QPoint(0, 0), event->size());
    m_scene->setSceneRect(rect);
    m_scroller->setGeometry(rect);
    m_webView->setGeometry(rect);

    setUpdatesEnabled(true);

    if( m_inspector){
        m_inspector->resize( event->size());
    }

    update();
}

void StandaloneView::contextMenuEvent(QContextMenuEvent *event)
{
    QMenu menu(this);
    m_contextMenu->setMenuWidget(&menu);
    m_contextMenu->showContextMenu(event->pos());
    menu.exec(event->globalPos());
}

void StandaloneView::closeEvent(QCloseEvent* event)
{
    QGraphicsView::closeEvent(event);
}

void StandaloneView::setPage(WrtPage* page)
{
    if (!page)
        return;

    page->setViewportSize(geometry().size());

    connect(page, SIGNAL(windowCloseRequested()),
            this, SLOT(close()));

    m_webView->setPage(page);

    if( m_inspector){
        m_inspector->setPage( page);
    }
}

void StandaloneView::setMenu(QMenu* menu)
{
    if (!menu)
        return;

    m_menuBar->addMenu(menu);
}

void StandaloneView::setContextMenu(WRT::JSContextMenu* contextMenu)
{
    if (!contextMenu)
        return;

    m_contextMenu = contextMenu;
}

void StandaloneView::restore()
{
    // XXX:
    // activateScrollingForView() method sets m_webView as a child of
    // m_scroller. For some unknown reason, it crashes if m_scroller
    // is already part of the scene, so we first remove it from
    // the scene, call the method, and then add it to the scene again.
    m_scene->removeItem(m_scroller);
    m_scroller->activateScrollingForView();
    m_scene->addItem(m_scroller);

    QRect rect(QPoint(0, 0), size());
    m_webView->setGeometry(rect);
}

void StandaloneView::changeEvent(QEvent* event)
{
    // Fires visibilityChanged when app goes to background/foreground
    if (event->type() == QEvent::ActivationChange) {
        if (isActiveWindow())
            emit visibilityChanged(true);
        else
            emit visibilityChanged(false);
    }

    QGraphicsView::changeEvent(event);
}

void StandaloneView::showFullScreen(bool windowedSupported)
{
    if (windowedSupported)
        m_exitButton->setIcon(QIcon::fromTheme(QLatin1String("general_fullsize")));
    else
        m_exitButton->setIcon(QIcon::fromTheme(QLatin1String("general_close")));

    QGraphicsView::showFullScreen();
}

void StandaloneView::showInspector(bool show)
{
    if( m_inspector && m_actWebInspector){
        if( show){
            m_actWebInspector->setText("Hide web inspector");
            m_webView->hide();
            m_inspector->show();
        } else {
            m_actWebInspector->setText("Web inspector");
            m_inspector->hide();
            m_webView->show();
        }
    }
}

}
}
