/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#ifndef WGT_INSTALLER_P_MAEMO_H
#define WGT_INSTALLER_P_MAEMO_H
#include "SuperWidget.h"

#include <QObject>
#include <QString>
#include <QDir>
#include <QMutex>

namespace WRT {
    class Storage;
}

#if defined(Q_OS_MAEMO6)
class PackageManagerPendingCallWatcher;
#endif

class WidgetInstallerPrivate : public QObject
{
    Q_OBJECT

public:

    // Widget types
    enum InstallationStatus
    {
        Success = WidgetInstallSuccess,
        DebianPackageStructureCreationFailed,
        DebianPackageCreationFailed,
        DebianPackageInstallationFailed,
        WebWidgetSecureStoringFailed
    };

   explicit WidgetInstallerPrivate(SuperWidget * webWidget, WRT::Storage * storage, QObject * parent=0);
   ~WidgetInstallerPrivate();

   InstallationStatus install(QString source, QString target,
                              QString appId, QString &desktopfilePath);
   static bool uninstall(QString appId);

private:

   //initializes the widget attributes from config.xml
   void readWidgetAttributes();

    /** Creates a unique debian Package name and set package name to member variable m_packageField.
    * @return unique debian package name.
    */
   QString createPackageName();


   //writes a changelog file to maintain versioning of the deb packages
   int writeChangeLog();

   //copyright file!
   int writeCopyRightFile();

   //writes a control file in the Debian structure required for conversion
   void writeControlFile();

   /**
    * Adds icon into debian package structure. Icon must exist before calling this method.
    * @param iconInstallTarget install directory for the icon. For example in Maemo5 "/usr/share/icons/hicolor/64x64/hildon/".
    */
   bool addIcon(QString iconInstallTarget);

   /**
    * Sets m_icon path and m_iconSymLink. Goes through custom and default icons.
    *
    * After calling this method, postInst script must be called.
    * @see writePostinstFile
    */
   void setIcon();

   /*
    * Writes a desktop file in the Debian structure
    * @widgetRoot - path to widget root directory
    * @return path to a desktopfile, that can be used as parameter to
    *         webapprunner
    */
   QString writeDesktopFile(QString widgetRoot);

   /**
     * Writes common entries into [DUI] section.
     * @desktopFile - input/output argument for writing desktop file. Must be a valid reference.
     * @widgetRoot - path to widget root directory
     * @viewMode - view mode for the desktop file (application|floating|mini)
     */
    void writeCommonDuiDesktopEntries(QByteArray & desktopFile, QString widgetRoot, QString viewMode);

   //creates a registry sub directory in Debian structure
   bool createDebianStructure(QString packagename,QString widget);

   /**
    * Writes a post installation file in the Debian structure required for conversion.
    * For an icon there is a symbolic link written to the DUI images folder that points to m_icon.
    * Symlink will have a hashed icon name to avoid collision with icons (default icons).
    */
   int writePostinstFile();

   //writes a post remove file in the Debian structure required for conversion
   int writePostrmFile();

   //creates a Debian package which is ready for installation
   bool createDebianPackage(QString debname);

   //installs the debian package
   bool installDebianPackage(QString debname);

   /** Adds the widget files to secure storage after widget installation.
    * Checks if the secure storage already exits if Yes removes the existing secstore and creates the new
    * Checks the signature of the widget if it fails then clean's up the secure storage and removes the widget package
    * @param packagename name of the .wgt widget package used as the name of the secure storage
    * @return zero if all OK.
    */
   bool secureStoreWidget(QString packagename, QString pathToSecureStore);

   /** Calls dpkg with given parameters.
    * @param params dpkg command line parameters as string list
    * @param maxWaitTime the max. time to be waited for dpkg process to be finished
    * @return zero if all OK. The return value of QProcess::exitCode()
    */
   int dpkg( const QStringList& params, int maxWaitTime = 100000);

   /**
    * If needed, formats the given string so that it can be used as Debian package version.
    * @param originalVersion arbitary version string
    * @return version string that can be used as Debian version. If originalVersion is valid, it is returned without modifications.
    */
   QString generateValidDebianVersion(const QString& originalVersion);

   /**
    * Checks if the given string can be used as Debian package version.
    * @param originalVersion arbitary version string
    */
   bool isValidDebianVersion(const QString& version);

   /**
    * Checks if the given string has valid debian version Epoch field.
    * @param version arbitary version string
    */
   bool hasValidDebianEpoch(const QString& version);

   /**
    * Checks if the given string has valid debian version upstream version field.
    * @param version arbitary version string
    */
   bool hasValidDebianUpstreamVersion(const QString& version);

   /**
    * Checks if the given string has valid debian version revision field.
    * @param version arbitary version string
    */
   bool hasValidDebianRevision(const QString& version);

   /**
    * Checks if the given string and returns string which contains only alphanumeric chars of the original one.
    * @param originaln arbitary string
    */
   QString convertToAlphaNumeric(const QString& original);

#if defined(Q_OS_MAEMO6)
public slots:
    void pkgmgrOperationComplete(const QString &operation, const QString &packagename, const QString &result);
    void pkgmgrDbusError(PackageManagerPendingCallWatcher*);
    void pkgmgrOperationProgress(const QString &operation, const QString &packagename, int percentage);
#endif

private:
    //Private implementations

    SuperWidget * m_webWidget;
    WRT::Storage * m_storage;
    QString m_appId;
#if defined(Q_OS_MAEMO6)
    int pkgmgr(QString debname);
    QMutex m_pkgmgrInstallationMutex;
    bool m_pkgmgrInstallationSuccess;
    int m_pkgmgrInstallationTime;
#endif

    //installers tmp directory
    QString m_tempDir;

    //.wgt package extracted here m_tempDir + import
    QString m_importDir;


    //defines a debian directory
    QString m_debianRootDir;


    //defines a DEBIAN directory for control, change, log
    QString m_debianDir;

    //points to /usr/share/dui/applets
    QString m_desktopDir;

    //points to /usr/share/webwidgets/applications
    QString m_widgetappdesktopDir;

    //points to /usr/share/applications
    QString m_appdesktopDir;

    QString m_tmpInstallDir;

    QString m_InstallerVersion;

    //defaults which are written to .desktop file if they are missing from config.xml
    static const QString S_DEFAULT_WIDTH;
    static const QString S_DEFAULT_HEIGHT;
    static const QString S_DEFAULT_ICON;
    static const QString S_DEFAULT_VERSION;
    static const QString S_DEFAULT_VIEW_MODES;
    static const QString S_DEFAULT_DESCRIPTION;
    static const QString S_INSTALLER_VERSION;
    static const QString S_COPYRIGHT;

    QString m_copyRight;
    QString m_widgetName;
    QString m_widgetVersion;
    QString m_widgetDescription;
    QString m_authorEmail;
    QString m_authorName;
    QString m_packageField;

    /**
     * Icon desktop entry
     */
    QString m_icon;

    /**
     * Real file inside widget package with fullpath.
     */
    QString m_iconPath;

    /**
     * Hashed name for symlink for debian package. Filename to be written in desktop file.
     */
    QString m_iconSymLink;

    //attribute in config.xml possible values are application,floating or mini
    QString m_viewmodes;

    Q_DISABLE_COPY(WidgetInstallerPrivate)
};



#endif // WGT_INSTALLER_P_MAEMO_H
