/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#ifndef _WIDGET_MANAGER_H_
#define _WIDGET_MANAGER_H_

#include "WidgetUtils.h"
#include "SuperWidget.h"

class WidgetProperties;
class W3cXmlPlugin;

namespace WRT {
    namespace Maemo {
        class WidgetInstallerTestHelper;
    }
}

/* Widget managers own widget installation and launching:
 *  Manage widget storage
 *  Own widget preferences
 *  Use widget factory at install and launch
 */

/* Use the widget engine and package management components */
class WIDGETUTILS_EXPORT WidgetManager : public QObject
{
    Q_OBJECT

public:
    WidgetManager(QWidget *parent, bool isChromeless=false);
    ~WidgetManager();

    QString launcherPath(const QString &pkgPath);
    bool isRegistered(const QString &pkgPath);
    WidgetInstallError install(const QString &pkgPath, QString &widgetId,  bool silent=false,  bool update=false, const QString& rootDirectory = "");
    void asyncInstall(const QString &pkgPath, bool silent, const QString& rootDirectory = "", bool update=false);
    WidgetUninstallError uninstall(const QString &uniqueIdentifier, bool silent=false);
    WidgetProperties* getProperties(const QString &pkgPath);
    WidgetProperties* getProperties();

    bool isValidWidgetType(const QString& widgetPath, const QString& contentType);
    bool isValidWidgetType(const QString& widgetPath);
    void setContentType(const QString& contentType);
    bool getContentType(QString& contentType);

    //Backup-restore widget install.
    bool backupRestoreInstall(const QString &pkgPath, bool disableUnsigWidgetSignCheck);
#ifdef __SYMBIAN32__
    QString installSisx(const QString &pkgPath, const QString& rootDirectory = "");
    WidgetUninstallError uninstallSisx(const QString &uniqueIdentifier, bool aRemoveFromList);
    QHash<QString,uint> readSisxWidgetList();
    void writeSisxWidgetList(QHash<QString,uint> aWidgetList);
    QString sisxWidgetFile();
#endif  // __SYMBIAN32__
Q_SIGNALS:
    void installComplete(QString widgetId, WidgetInstallError status);
    void installProgress(int);

    /**
     * This signal is emitted whenever an existing widget is about to be replaced during installation
     * phase.
     *
     * If you intend to replace the widget when this message is signaled, you should
     * tryContinueInstallation() method when you're handling this signal.
     */
    void aboutToReplaceExistingWidget(QString widgetTitle);

     /**
     * This signal is emitted whenever a widget with features is about to be installed.
     *
     *
     * If you intend to install the widget when this message is signaled, you should
     * tryContinueInstallation() method when you're handling this signal.
     */
    void aboutToInstallWidgetWithFeatures(QList<QString> capList);

    /**
     * This signal is emitted whenever an error occurs during installation phase.
     *
     * If you intend to continue installation when error is signaled, you should
     * invoke tryContinueInstallation() method when you're handling installation error.
     *
     */
    void installationError(WidgetInstallError error);

    /**
     * This signal is emitted when a not signed widget is being installed.
     *
     * If you intend to install the widget when this message is signaled, you should invoke
     * tryContinueInstallation() method when you're handling this signal.
     *
     * @param errorCode tells reasons for installed widget being untrusted
     */
    void aboutToInstallUntrustedWidget(WidgetInstallError errorCode);

    /**
     * This signal is emitted whenever an existing widget is about to be replaced during installation
     * phase.
     */
    void installationSucceed();

public Q_SLOTS:
    /**
     * Try to continue installation process when and installation is about to do something.
     *
     * @see aboutToReplaceExistingWidget(QString)
     * @see aboutToInstallUntrustedWidget(WidgetInstallError)
     * @see aboutToInstallWidgetWithFeatures(QList<QString> capList)
     */
    void tryContinueInstallation();

private Q_SLOTS:
    void handleInstallError(WidgetInstallError error);
    void handleWidgetReplacement(QString title);
    void handleFeatureInstallation(QList<QString> capList);
    void handleUntrustedWidgetInstallation(WidgetInstallError errorCode);
    void handleInstallationSuccess();

private:

    WidgetInstallError install(const QString& pkgPath, QString &widgetId, const QString& contentType, const QString& rootDirectory = "",  bool silent=false,  bool update=false, const QString &sigId="");
    WidgetProperties* getWidgetProperties(const QString &pkgPath);
    SuperWidget* createWidget(const QString &pkgPath);
    QString setDefaultRootDirectory(const QString& rootDirectory);
#ifdef __SYMBIAN32__
    QString setS60RootDirectory(const QChar& rootDirectory = NULL);
#endif  // __SYMBIAN32__
    bool validateDriveLetter(const QString& rootDirectory = NULL);
    bool checkCallerPermissions();

protected:
    SuperWidget* getWidgetFromCache(QString pkgPath);
    void setWidgetIntoCache(QString pkgPath, SuperWidget* widget);
    QString getWidgetInstallPath(const QString& rootDirectory);
    bool createDir(const QString& path);


public slots:

    //void widgetFileUnsecure(QString&);

private:
    void * m_data;
    W3cXmlPlugin *m_w3property;
    WidgetType m_widgetType;
    QString m_contentType;
    SuperWidget* m_installingWidget;
    bool m_silent;
    QString m_rootDirectory;
    bool m_disableUnsignWidgetSignCheck;
    QHash<QString, SuperWidget*> m_widgetCache;

#ifdef __SYMBIAN32__
    QString m_uninstallUid;
#endif

    friend class WRT::Maemo::WidgetInstallerTestHelper;
};

#endif // _WIDGET_MANAGER_H_

