/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef __WEBAPPREGISTRY_H__
#define __WEBAPPREGISTRY_H__

#include <QString>
#include <QVariant>
#include "WidgetUtils.h"
#include "AttributeMap.h"
#include "PropertyNames.h"
#include "W3CXmlParser/w3celement.h"

#ifdef Q_OS_SYMBIAN
class WidgetRegistrationS60;
#endif

enum DatabaseVersionInfo {
    CurrentDatabaseVersion = 1
};

class WIDGETUTILS_EXPORT WebAppInfo
{
public:
    enum WidgetType {
    WidgetUnknown,
    WidgetNokia,
    WidgetW3C,
    WidgetSharedLibraryWidget,
    WidgetJIL
    };

    QString appId () const {return m_appId;}
    QString appTitle () const {return m_appTitle;}
    QString appPath () const {return m_appPath;}
    QString iconPath () const {return m_iconPath;}
    QString type () const {return m_widgetType;}
    QString startPath () const {return m_startPath;}
    QString dataPath () const {return m_dataPath;}
    bool capabilityCheck() const {return m_capabilityCheck;}
    QString certificateAki() const {return m_certificateAki;}
    int widgetType() const;
    const AttributeMap& attributes() {return m_data;}
    QList<W3CElement*> getElement(const QString &aKey, const W3CElement *parent = 0, const QString &aLang = QString(""));
    QVariant value (const QString& key) const {return m_data[key];}
    bool operator==(const WebAppInfo &other) {
        return (m_appId == other.appId())? true : false;}

    QString m_appId;
    QString m_appTitle;
    QString m_appPath;
    QString m_iconPath;
    AttributeMap m_data;
    QString m_widgetType;
    QString m_startPath;
    QString m_dataPath;
    QString m_certificateAki;
    bool m_capabilityCheck;
    int m_uid;
    QString m_nativeId;
};


class WebAppRegistryInterface
{
    virtual bool registerApp(const QString& appId,
                             const QString& appTitle,
                             const QString& appPath,
                             const QString& iconPath,
                             const AttributeMap& attributes,
                             const QString& type,
                             unsigned long size,
                             const QString& startPath) = 0;
    virtual bool unregister (const QString& appID, bool update = false) = 0;
    virtual bool isRegistered (const QString& appID) const = 0;
    virtual bool isRegistered(const QString& appID, WebAppInfo& info) = 0;
    virtual QList<WebAppInfo>* registered() = 0;
    virtual bool setWebAppAttribute(const QString& appID,
                                    const QString& attribute,
                                    const QVariant& value) = 0;
    virtual QVariant getAttribute(const QString& appID,
                                        const QString& attribute,
                                        const QVariant& defaultValue = QVariant() ) = 0;

    virtual bool setWebAppVersion(const QString& appID,
                                  const QVariant& value,
                                  const QString& newIconPath) = 0;
    virtual bool setNativeId(const QString& appID,
                             const QString& nativeID) = 0;
    virtual QString nativeIdToAppId(const QString& nativeID) = 0;
    virtual int nextAvailableUid() = 0;
    virtual bool setUid(const QString& appID,
                        int uid) = 0;
    virtual bool setCapabilityCheck(const QString& appID, bool value) = 0;
    virtual bool setCertificateAki(const QString& appID, const QString& value) = 0;
};

class WIDGETUTILS_EXPORT WebAppRegistry : public WebAppRegistryInterface
{
public:
    // singleton
    static WebAppRegistry* instance();
    // FIXME singleton cleanup

    virtual bool registerApp(const QString& appId, const QString& appTitle,
                             const QString& appPath, const QString& iconPath,
                             const AttributeMap& attributes,
                             const QString& type,
                             unsigned long size,
                             const QString& startPath);
    virtual bool unregister(const QString& appID, bool update = false);
    virtual bool isRegistered(const QString& appID) const;
    virtual bool isRegistered(const QString& appID, WebAppInfo& info);
    virtual QList<WebAppInfo>* registered();
    virtual bool setWebAppAttribute(const QString& appID,
                                    const QString& attribute,
                                    const QVariant& value);
    virtual QVariant getAttribute(const QString& appID,
                                        const QString& attribute,
                                        const QVariant& defaultValue = QVariant() );
    virtual bool setWebAppVersion(const QString& appID,
                                  const QVariant& value,
                                  const QString& newIconPath);
    virtual bool setNativeId(const QString& appID,
                             const QString& nativeID);
    virtual QString nativeIdToAppId(const QString& nativeID);
    virtual int nextAvailableUid();
    virtual bool setUid(const QString& appID,
                              int uid);
    virtual bool setCapabilityCheck(const QString& appID, bool value);
    virtual bool setCertificateAki(const QString& appID, const QString& value);

    WebAppRegistry();
    virtual ~WebAppRegistry();

private:
    static WebAppRegistry* m_webappRegistry;
    bool updateAttribute(const QString& appID, QString& attribute, QVariant value);

#ifdef Q_OS_SYMBIAN
    WidgetRegistrationS60* m_s60Instance;
#endif
};

#endif
