/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "SuperWidget.h"

#include <QDebug>
#include <QDir>
#include <QFileInfo>
#include <QFile>
#include <QDesktopServices>
#include <QDirIterator>
#include <QCoreApplication>
#include <QHash>
#include <QTextStream>
#include <QRegExp>

#include "proprietarysettingskeys.h"
#include "w3csettingskeys.h"
#include "widgetinstaller.h"
#include "private/WidgetUtilsLogs.h"

#ifdef Q_OS_SYMBIAN
#include "platform/s60/WidgetUnzipUtilityS60.h"
#else
#include "unzip.h"
#include "zzip.h"
#endif

#ifdef Q_OS_LINUX
#include <sys/vfs.h> // for checking free disk space
#endif

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
#include "storage.h"
#endif

static QHash<QString, int> m_widgetTypeCache;
static QHash<QString, bool> m_widgetExistsCache;
static QHash<QString, QString> m_widgetStartFileCache;

SuperWidget::~SuperWidget()
{
    delete m_manifest;
    delete m_widgetMap;
    delete m_widgetProperties;
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
    delete m_storage;
#endif
    m_widgetTypeCache.clear();
    m_widgetExistsCache.clear();
    m_widgetStartFileCache.clear();
    delete m_widgetInstaller;
}

// initializes the widget to some defaults
// set up widget map
// create secure storage
//
void SuperWidget::initialize(QString& rootDirectory)
{
    LOG("SuperWidget::initialize");
    QString tempLocation = getTempLocation();
    LOG("SuperWidget::initialize.. tempLocation : "<<tempLocation);
    if (tempLocation.isEmpty())
      return;

    QString path = QDir::toNativeSeparators(tempLocation);
    setWidgetBundlePath(path);
    if (path.endsWith(QDir::separator()))
        setWidgetUnZipPath(path+"widgetUnZipPath");
    else
        setWidgetUnZipPath(path+QDir::separator()+"widgetUnZipPath");

    LOG("SuperWidget::initialize() - Widget install path" << rootDirectory);

    setWidgetInstallPath(rootDirectory);
    QString certPath = getCertificatePath();
    // FIXME myWidgetCert.pem hard-coded for now
    certPath = certPath + QDir::separator()+"myWidgetCert.pem";
    setWidgetCertificatePath(certPath);
    setWidgetRootPath(widgetUnZipPath());

    m_manifest = 0;
    m_widgetProperties = 0;

    m_widgetMap = new W3c2PlistMap();
    m_widgetMap->insert(Identifier, W3CSettingsKey::WIDGET_ID);
    m_widgetMap->insert(DisplayName, W3CSettingsKey::WIDGET_NAME);
    m_widgetMap->insert(MainHTML, W3CSettingsKey::WIDGET_CONTENT);
    m_widgetMap->insert(Version, W3CSettingsKey::WIDGET_VERSION);
    m_widgetMap->insert(DeltaVersion, W3CSettingsKey::WIDGET_DELTA);
    m_widgetMap->insert(MiniViewEnabled, "");
    m_widgetMap->insert(AllowNetworkAccess, "");

    m_widgetMap->insert(SharedLibraryFolder, ProprietarySettingsKey::WIDGET_SHARED_LIBRARY_FOLDER);
    m_widgetMap->insert(SharedLibraryWidget, ProprietarySettingsKey::WIDGET_SHARED_LIBRARY_WIDGET);

    m_size = 0;

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
    m_storage = WRT::Storage::createInstance(WIDGET_STORAGE);
    m_widgetInstaller = new WidgetInstaller(this, m_storage);
#else
    m_widgetInstaller = new WidgetInstaller(this, NULL);
#endif
}

bool SuperWidget::createDir(const QString& path)
{

  QDir dir(path);

  if (!dir.exists()){
    if (!dir.mkdir(dir.absolutePath())) {
      LOG("Could not create dir : "<<path);
      return false;
      }
    }
    return true;
}

bool SuperWidget::checkDiskSpaceForInstallation(const QString& widgetPath) const
{
#ifdef Q_OS_SYMBIAN
    // Symbian has different zip handling.
    // TODO: Implement size check for symbian, if needed
    return true;
#else

    // check if there's enough disk space
    // check that uncompressed widget fits to file system
    // get uncompressed zip size...
    unsigned long uncompressedSize = Zzip::uncompressedSize(widgetPath);
    qDebug() << "Widget uncompressed size is:" << uncompressedSize;

    unsigned long long availableSize = 0;

#ifdef Q_OS_LINUX
    // get available disk space...
    // there's no Qt way to do this, so this is platform dependent implementation
    struct statfs stats;
    statfs( widgetPath.toLocal8Bit(), &stats);
    availableSize = ((unsigned long long)stats.f_bavail) *
        ((unsigned long long)stats.f_bsize);
#else
    // Free size check not implemented for other platforms
    // TODO: Implement disk size check for other platforms
    return true;
#endif

    // Approx. 4 x uncompressed widget size should be enough in maemo;
    // The package is unzipped twice, and also widget & debian files are copied to
    // several places during the installation process
#ifdef __MAEMO__
    if (( availableSize/4) < uncompressedSize)
#else
    // 2 x widget size is enough for other platforms?
    if (( availableSize/2) < uncompressedSize)
#endif
    {
        qCritical() << "Widget package cannot be unzipped. Not enough free space";
        return false;
    }
    return true;
#endif
}

QString SuperWidget::getTempLocation()
  {
  QString tempLocation("");

#ifdef Q_OS_SYMBIAN
  // private path of the process
  tempLocation = QCoreApplication::applicationDirPath();
  // append 'Cache'
  tempLocation.append("\\Cache\\");
  LOG("SuperWidget::getTempLocation - Path after appending cache: " << tempLocation);
#else
  tempLocation =  QDesktopServices::storageLocation(QDesktopServices::CacheLocation);
#endif

  return tempLocation;
  }

QString SuperWidget::getCertificatePath()
{
  QString certPath = "";
#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
#ifdef Q_OS_SYMBIAN
  certPath = DEFAULT_ROOT_DIRECTORY;
#else
  QString dataLocation =  QDesktopServices::storageLocation(QDesktopServices::DataLocation);
  certPath = QDir::toNativeSeparators(dataLocation);
#endif
  certPath = certPath+QDir::separator()+WIDGET_FOLDER;
  certPath = QDir::toNativeSeparators(certPath);
  if (! createDir(certPath)) {
    LOG("SuperWidget::getWidgetInstallPath - Could not create certificate path dir");
    return QString();
    }
  certPath = certPath+QDir::separator()+"widgetCertificates";
  certPath = QDir::toNativeSeparators(certPath);
  if (! createDir(certPath)) {
    LOG("SuperWidget::getWidgetInstallPath - Could not create certificate path dir");
    return QString();
    }
#else
  certPath = CERTIFICATE_PATH;
#endif

  return certPath;
}

// function to determine widget type
// parameters:
//     widgetPath    path to widget bundle
//     contentType   contentType from server
// return
//     WidgetType    type of widet
//
// Currently we are relaxing the contentType check in order to
// support modes where contentType is not known.
//
WidgetType SuperWidget::getWidgetType(const QString& widgetPath,
                                      const QString& contentType)
{
    LOG("SuperWidget::getWidgetType. widgetPath : "<<widgetPath<<" contentType : "<<contentType);

    if (widgetPath.isEmpty())
        return WidgetTypeUnknown;

    if (contentType.isEmpty() && QFileInfo(widgetPath).isDir())
        return getWidgetType(widgetPath);

#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
    if ((contentType.isEmpty() || (0 == QString::compare(CONTENT_TYPE_WGZ, contentType, Qt::CaseInsensitive)))
        && (widgetPath.endsWith(EXTENSION_WGZ, Qt::CaseInsensitive))) {
        return WidgetTypeWgz;
    }
#endif

    if (!contentType.isEmpty() && 0 != QString::compare(CONTENT_TYPE_WGT, contentType, Qt::CaseInsensitive))
        return WidgetTypeUnknown;

    QFile file(widgetPath);
    file.open(QIODevice::ReadOnly);
    if (0 == qstrncmp(WIDGET_PACKAGE_MAGIC_NUMBER_WGT, file.read(4).data(), 4))
        return WidgetTypeW3c;

    return WidgetTypeUnknown;
}

// function to unzip widget bundle
// parameters:
//     path    path to widget
// return
//     bool   true is good
//            false is bad
// widet will be unzipped to widgetUnZipPath()
// Currently using zlib to unzip widgets
// This method can be overrided to provide platform specific
// implementation.
//
bool SuperWidget::unZipBundle(const QString& path)
{
    unsigned long size = 0;
    QString widgetPath(path);

    // clean up unzip path
    if (!rmDir(widgetUnZipPath())) return false;
    // unzip bundle
    if (widgetPath.isEmpty())
        widgetPath = widgetBundlePath();

#ifdef Q_OS_SYMBIAN
    if (WidgetUnzipUtilityS60::unzip(widgetPath, widgetUnZipPath(), size)) {
#else
    if (Zzip::unzip(widgetPath, widgetUnZipPath(), size) == 0) {
#endif
        m_size = size;
        return true;
    } else {
        return false;
    }
}

// function to delete all files in a directory
// parameters:
//     path    path to directory that needs to be deleted
//
// Used to delete unzip directory
//
bool SuperWidget::rmDir(const QString& path)
{
    // get all files and folders list in the current folder
    QDirIterator it(path, QDir::Files|QDir::AllDirs|QDir::Hidden|QDir::NoDotAndDotDot);
    QDir currentDir(path);

    while (it.hasNext()) {
        it.next();
        // get file/dir name
        QString cleanitem = QDir::cleanPath(it.fileInfo().absoluteFilePath());

        if ( it.fileInfo().isFile() ) {
            // delete file
            currentDir.remove(cleanitem);
        }
        else {
            // recursevly keep deleting the content
            rmDir(cleanitem);
            // now delete the empty folder
            currentDir.rmdir(cleanitem);
        }
    }
    return true;
}

// function to check if widget type is valid
// return:
//     bool    true if widget type is valid
//             false if widget is invalid
//
bool SuperWidget::isValidWidgetType()
{
    return (getWidgetType()!=WidgetTypeUnknown
            && getContentType() != CONTENT_TYPE_UNKNOWN);
}

// overloaded function to check if widget type is valid
// parameters:
//    widgetPath    path to widget
//    contentType   contentType provided by HTTP or obtained from MIME type
// return:
//     bool    true if widget type is valid
//             false if widget is invalid
//
bool SuperWidget::isValidWidgetType(const QString& widgetPath,
                                    const QString& contentType)
{
    LOG("SuperWidget::isValidWidgetType. widgetPath : "<<widgetPath<<" contentType : "<<contentType);
    return (getWidgetType(widgetPath,contentType) != WidgetTypeUnknown);
}

void SuperWidget::tryContinueInstallation() {
    m_continueInstallation = true;
}

// function to register a widget with WebAppRegistry
//
// return
//    bool   true if successful
bool SuperWidget::registerWidget(const QString& startFile)
{
    // FIXME registerApp needs a better interface to pass values and
    // the widget processing should produce a struct with all required
    // values ready to pass like old code did
    WidgetProperties* widgetProps = widgetProperties();
    bool status = false;
    if (widgetProps) {
            QString startFileFullPath = widgetProps->installPath()+QDir::separator()+startFile;
            qDebug() << "The title's direction attribute(ltr/rtl/lro/rlo):" << widgetProps->titleDir();
            // Add Text Direction attribute as arguments when platform provides support.
            // Following elements will provide support: Name, Author, Description & License
            status
                = (WebAppRegistry::instance()->registerApp(widgetProps->id(),
                                                           widgetProps->title(),
                                                           widgetProps->installPath(),
                                                           widgetProps->iconPath(),
                                                           widgetProps->plist(),
                                                           widgetProps->type(),
                                                           widgetProps->size(),
                                                           startFileFullPath));

            // Save SecSession
            if (status) {
                QString resourcesDir = widgetProps->resourcePath();
                if (!resourcesDir.isNull() && !resourcesDir.isEmpty()) {
                    QDir dir;
                    dir.mkpath(resourcesDir);
                    QString secSessionFileName(QDir::toNativeSeparators(resourcesDir));
                    secSessionFileName += QDir::separator();
                    secSessionFileName += SECSESSION_FILE;
                    QFile secSessionFile(secSessionFileName);
                    if (secSessionFile.open(QIODevice::WriteOnly)) {
                        QTextStream stream(&secSessionFile);
                        stream << widgetProps->secureSessionString(); //codescanner::leave
                        secSessionFile.close();
                    }

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
                    // TODO:
                    // Add to secure storage
                    m_storage->add(secSessionFile);
#endif
                }
            }
    }
    return status;
}

// function to uninstall the widget
// parameters:
//     widgetProps    widget properties
//     removeData     remove wigdet data flag
// return:
//     bool           true if successful
//
// removes widget from registry and deletes all widget files
//
WidgetUninstallError SuperWidget::uninstall(const QString &uniqueIdentifier, const bool removeData)
{
    WebAppInfo info;
    if (WebAppRegistry::instance()->isRegistered(uniqueIdentifier, info)) {
        if (!WidgetInstaller::uninstall(info.appPath(), uniqueIdentifier)) {
            return WidgetUninstallFailed;
        }

        // Remove SecSession
        QString resourcesDir = resourcePath(info.appPath());
        if (!resourcesDir.isNull() && !resourcesDir.isEmpty()) {
            if (removeData) {
                // Remove all widget data
                rmDir(resourcesDir);
            }
            else {
                // Leave widget data, remove only secsession
                QString secSessionFileName(QDir::toNativeSeparators(resourcesDir));
                secSessionFileName += QDir::separator();
                secSessionFileName += SECSESSION_FILE;
                QFile::remove(secSessionFileName);
            }
            QDir dir;
            dir.rmpath(resourcesDir);
        }
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
        // TODO:
        // Remove from secure storage
#endif

        LOG("widget uninstall " << info.appPath());
        if (WebAppRegistry::instance()->unregister(uniqueIdentifier))
            return WidgetUninstallSuccess;
    }
    return WidgetUninstallFailed;
}

QString SuperWidget::resourcePath(const QString& installPath) {
    QDir dir(installPath);
    QString resourcePath;
#if defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
    resourcePath = DATA_PATH;
#else
    if (installPath.section(QDir::separator(),4,4) == SHARED_LIBRARY_FOLDER)
        resourcePath = installPath.left(installPath.indexOf(WIDGET_FOLDER, Qt::CaseInsensitive))+ QString(DATA_FOLDER) + QDir::separator() + SHARED_LIBRARY_FOLDER + QDir::separator();
    else
        resourcePath = installPath.left(installPath.indexOf(WIDGET_FOLDER, Qt::CaseInsensitive))+ QString(DATA_FOLDER) + QDir::separator();
#endif
    QString resourcesDir = resourcePath + dir.dirName();
    resourcesDir = QDir::toNativeSeparators(resourcesDir);
    resourcesDir += QDir::separator();
    return resourcesDir;
}

// function to get widget properties
// parameters:
//     pkgPath             path to widget files
// return:
//     WidgetProperties*   pointer to widget properties
//
// parses widget manifest and returns widget properties
//
WidgetProperties* SuperWidget::getProperties(const QString& pkgPath)
{

    LOG("SuperWidget::getProperties path="<<pkgPath);

    setWidgetRootPath(pkgPath);
    if (!parseManifest(pkgPath))
    {
        LOG("getProperties : failed");
        return 0;
    }

    WidgetProperties* props = widgetProperties();

    return props;


}

// function to clean up widget files
// parameters:
//    path   path to cleanup (optional)
// return:
//    bool   true if successful
//
// removes all files from unzip path
//
bool SuperWidget::cleanup(const QString& path)
{
    QString rmPath(path);
    if (rmPath.isEmpty())
        rmPath = widgetUnZipPath();

    if (!rmDir(rmPath)) {
        LOG("Widget cleanup failed");
        return false;
    }

    LOG("Widget cleanup done");
    return true;
}


bool SuperWidget::parseManifest( const QString&, const bool )
{
    return false;
}

// function to retrieve localized version of a file
// parameters:
//    file            path to file that needs a localized version
//    userAgentLang   language setting of user agent
//    absolute        if true provide absolute path, if false relative path
//    root            path to root of widget
// return:
//    QString         path of localized file
//
QString SuperWidget::getLocalizedFile(const QString& file, QString& userAgentLang,
    const bool absolute, const QString& root)
{
    // if file has userAgentLang, we are done
    //

    if (userAgentLang.isEmpty() )
        return file;



    // Default, nothing found
    QString result("");

    // List of languages
    QStringList langParts = userAgentLang.split("-");
    int langCount(langParts.count());

    QChar sep = QDir::separator();
    QString locales = "locales";
    QString nFile = QDir::toNativeSeparators(file);

    if (file.contains(QString("locales/" + userAgentLang),
          Qt::CaseInsensitive))
    {
        QFileInfo fInfo(nFile);
        nFile = fInfo.fileName();
    }

    // Construct a root path
    QString nRoot = QDir::toNativeSeparators(root);
    if (!nRoot.endsWith(sep)) {
        nRoot += sep;
    }

    // Take apart the path, handle absolute and relative paths
    //
    QString relPath("");
    if ( absolute ) {
        // Convert to relative path
        int len = nFile.length() - nRoot.length();
        if ( len >= 0 ) {
            relPath = nFile.right(len);
        }
        else {
            relPath = nFile;
        }
    }
    else {
        relPath = nFile;
    }

    // Split up the relative path, can be used for smarter algorithm
    //
    QStringList relPathParts = relPath.split(sep);
    int relCount = relPathParts.count() - 1; // last item is the file name
    QString fileName("");
    if ( relCount >=0 ) {
        fileName = relPathParts[relCount];
        relPathParts.removeLast();
    }

    // Lets try to open from best fit to worse fit, it we can open, we've found our answer!
    // Try to open locales folder + relative folder + file name if fail, move on
    //

    // Construct the relative path
    QString path(nRoot);
    path += locales;
    path += sep;

    // Check cache first for faster file access
    bool dirExists(false);
    QHash<QString, bool>::const_iterator it = m_widgetExistsCache.find(path);
    if (it != m_widgetExistsCache.end()) {
        dirExists = (bool) it.value();
    }
    else {
        // "locale" folder is case sensitive,
        // but the actual locale subfolder is not
        QDir dir(nRoot);
        QStringList localesList;
        localesList.append(locales);
        QStringList dirList = dir.entryList(localesList, QDir::CaseSensitive | QDir::Dirs);
        dirExists = (dirList.count() > 0);
        m_widgetExistsCache.insert(path, dirExists);
    }

    // Locales folder exists, so we can do the more expensive file checks
    //
    if (dirExists) {

        // Look for a locales + lang folder
        for (int i = langCount - 1; i >= 0; --i) {

            // build up the proposed localized path
            QString localized(path);

            // Chop extra '-', add separator
            for (int j = 0; j <= i; ++j) {
                localized += langParts[j];
                localized += "-";
            }

            localized.chop(1);

#ifndef Q_OS_SYMBIAN
            QDir langDir(path);
            QStringList langEntries = langDir.entryList(QDir::Dirs|QDir::NoDotAndDotDot);
            foreach(QString locName, langEntries) {
                QString fullLocName = path+locName;
                if(!fullLocName.compare(localized,Qt::CaseInsensitive)) {
                    localized = fullLocName;
                    break;
                }
            }
#endif
            localized += sep;

            // Add relative path, add file
            for (int l = 0; l < relCount; ++l) {
                localized += relPathParts[l];
                localized += sep;
            }

            // Check if we have looked for the folder before, can be cached
            // This will speed up cases where there are many files in the same folder
            //
            QHash<QString, bool>::const_iterator it = m_widgetExistsCache.find(localized);
            if (it != m_widgetExistsCache.end()) {
                dirExists = (bool) it.value();
            }
            else {
                dirExists = QFile::exists(localized);
                m_widgetExistsCache.insert(localized, dirExists);
            }

            // Up to the folder exists, check if file exists
            if (dirExists) {
                localized += fileName;

                // Lets look for the file. Optional file level cache
                if ( QFile::exists(localized) ) {
                    // We found something
                    if ( absolute ) {
                        result = localized;
                    }
                    else {
                        // relative path, just in case length is negative
                        int len(localized.length()-nRoot.length());
                        if ( len >=0 ) {
                            result = localized.right(len);
                        }
                        else {
                            result = localized;
                        }
                    }
                    break;
                }
            }
        } // for lang
    } // dir exists

    // By default, return the original file
    if ( result.length() == 0 ) {
        result = file;
    }
    return result;
}

WidgetInstallError SuperWidget::install( const QString&, const bool )
{
    return WidgetInstallFailed;
}

WidgetInstallError SuperWidget::install( const bool )
{
    return WidgetInstallSuccess;
}

void SuperWidget::writeManifest( const QString& )
{
}

bool SuperWidget::findDigitalSignatures( QStringList&, const QString& )
{
    return false;
}

bool SuperWidget::isValidDigitalSignature( WidgetInstallError &,
                                           const QString&,
                                           bool,
                                           bool )
{
    return false;
}

void SuperWidget::saveStartFile(QString& startFile, const QString& path)
{
    QString uniqueName;
    if (getUniqueNameFromPath(path, uniqueName)) {
        if (!startFile.isEmpty() && !uniqueName.isEmpty() &&
            !m_widgetStartFileCache.contains(uniqueName)) {
            m_widgetStartFileCache.insert(uniqueName, startFile);
        }
    }
}

bool SuperWidget::getStartFile(QString& startFile, const QString& path)
{
    QString uniqueName;
    if (getUniqueNameFromPath(path, uniqueName)) {
        if (m_widgetStartFileCache.contains(uniqueName)) {
            startFile = m_widgetStartFileCache.value(uniqueName);
            return true;
        }
    }
    return false;
}

// getUniqueNameFromPath
// The widget's folder name is unique, we use this to id the widget
bool SuperWidget::getUniqueNameFromPath(const QString pkgPath, QString &uniqueName)
{
    // Convert the separators, remove trailing separator, and assign pkgpath
    // to local variable, since we will modify it
    QString path = QDir::convertSeparators(pkgPath);
    if (path.endsWith(QDir::separator())) {
        path = path.remove(path.length()-1, path.length()-1);
    }

    int i = path.length() - path.lastIndexOf(QDir::separator()) - 1;
    if (i > 0) {
        uniqueName = path.right(i);
        return true;
    }
    return false;
}

bool SuperWidget::findStartFile( QString&, const QString& )
{
    return false;
}

bool SuperWidget::findIcons( QStringList&, const QString& )
{
    return false;
}

bool SuperWidget::findFeatures( WidgetFeatures&, const QString& )
{
    return false;
}

bool SuperWidget::getCertificateCn( QString&, const QString&, bool, bool )
{
    return false;
}


//function to get widget type from path
// parameters:
//    path         path to unzipped widget files
// return:
//    WidgetType   type of widget
//
WIDGETUTILS_EXPORT WidgetType SuperWidget::getWidgetType(const QString& path)
{
    // This ensures that all paths in the cache end with a separator
    // in order to prevent duplicate entries.
    QString sep = QDir::separator();
    QString sPath(QDir::toNativeSeparators(path));
    if (!path.endsWith(sep)) {
        sPath += sep;
    }

    // Check cache first for faster file access
    QHash<QString,int>::const_iterator it = m_widgetTypeCache.find(sPath);
    if (it != m_widgetTypeCache.end())
    {
        return (WidgetType)it.value();
    }

    WidgetType type(WidgetTypeUnknown);

    // W3C packaging spec: "A widget package has exactly one
    // configuration document located at the root of the widget
    // package."  Also: "A valid configuration document file name is
    // the string 'config.xml'." Only lowercase is legal.
    if (QFile::exists(sPath+"config.xml")) {
        type = WidgetTypeW3c;
    }
    // S60 WRT: Doc at http://library.forum.nokia.com/ "Web
    // Developer's Library 1.7 / Web Runtime Widgets / Developing
    // widgets / Widget component files / Creating the info.plist
    // file" says: "Save the XML document under the widget project
    // root folder with the name info.plist."
#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
    else if (QFile::exists(sPath+"info.plist") ||
             QFile::exists(sPath+"Info.plist") ||
             QFile::exists(sPath+"INFO.plist")) {
        // FIXME: The Forum Nokia documentation may not agree with the
        // existing S60 WRT code which is both case insensitive and
        // maybe doesn't require info.plist at root.
        type = WidgetTypeWgz;
    }
#endif
    m_widgetTypeCache.insert(sPath, type);
    return type;
}

// function to split parts of widget file path
// parameters:
//    rootParts     has two parts, first is root of widget,
//                  second is relative path
//    path          path to file in widget
// return:
//    bool          true if successful
//
bool SuperWidget::getWidgetPathParts(QStringList& rootParts, const QString& path)
{

    if (path.isEmpty()) return false;
    QRegExp part(WIDGET_FOLDER);
    QString part2 = path.section(part,1,1);
    QString urlSeparator("/");
    if (part2.isEmpty())
    {
        // If part2 is empty, it means we are not coming from networkaccessmanager.
        // We have to just send back the path.
        rootParts.append(path.section(part,0,0));
    }
    else {
        // We spit the absolute path into two parts: <widget-root> and <widget-relative> path
        // absolute path: <widget-root>/<widget-relative>
        // widget-root  : <some-base-path>/<widget-folder>/<widget-id>

        if (part2.startsWith(urlSeparator + SHARED_LIBRARY_FOLDER)) {
            // part 1 is <sharedlib-root>
            rootParts.append(path.section(part,0,0)+WIDGET_FOLDER+urlSeparator+part2.section(urlSeparator,0,1,QString::SectionSkipEmpty));
            // part 2 is <sharedlib-relative>
            rootParts.append(part2.section(urlSeparator,2,-1,QString::SectionSkipEmpty));
        }
        else {
            // part 1 is <widget-root>
            rootParts.append(path.section(part,0,0)+WIDGET_FOLDER+urlSeparator+part2.section(urlSeparator,0,0,QString::SectionSkipEmpty));
            // part 2 is <widget-relative>
            rootParts.append(part2.section(urlSeparator,1,-1,QString::SectionSkipEmpty));
        }
    }
    LOG("full path = " << path);
    LOG("relative file = " << part2.section(urlSeparator,1,-1,QString::SectionSkipEmpty));
    LOG("root = " << rootParts.at(0));
    return (rootParts.count()>0);
}
