/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QGraphicsWidget>
#include <QWidget>
#include <QSignalMapper>
#include <QAction>

#include "action.h"
#include "actionadapter.h"

namespace WRT {

/*!
    \class WRT::ActionAdapter
    \brief The WRT::ActionAdapter contains menu items created in JavaScript.

    \inmodule WidgetCore

    WRT::ActionAdapter is an internal class for JSContextMenu and JSMenu. This class handles
    adding, removing, and inserting of menu items from native menu widget.
*/

/*!
    Constructs an instance of ActionAdapter with view widget \a widget
*/
ActionAdapter::ActionAdapter(QGraphicsWidget* widget)
    : m_widget(0)
    , m_graphicsWidget(widget)
    , m_actionMapper(new QSignalMapper())
{
}

/*!
    Constructs an instance of ActionAdapter with view widget \a widget
*/
ActionAdapter::ActionAdapter(QWidget* widget)
    : m_widget(widget)
    , m_graphicsWidget(0)
    , m_actionMapper(new QSignalMapper())
{
}

ActionAdapter::~ActionAdapter()
{
    QList<QAction*> list;
    if (m_widget) {
        list = m_widget->actions();
    } else {
        list = m_graphicsWidget->actions();
    }
    QListIterator<QAction*> iterator(list);
    while (iterator.hasNext()) {
        WRT::Action* action = static_cast<WRT::Action*>(iterator.next());
        removeAction(action);
    }
    m_actionMapper->deleteLater();
}

/*!
    Returns a list of WRT::Action that currently in  \a location.
*/
QList<WRT::Action*> ActionAdapter::actions(Action::Location location) const {
    QList<WRT::Action*> actionsByLocation;

    QList<QAction*> list;
    if (m_widget) {
        list = m_widget->actions();
    } else {
        list = m_graphicsWidget->actions();
    }

    QListIterator<QAction*> iterator(list);
    while (iterator.hasNext()) {
        WRT::Action* action = static_cast<WRT::Action*>(iterator.next());
        if (action->location() == location) {
            actionsByLocation.append(action);
        }
    }
    return actionsByLocation;
}

/*!
    Adds \a action to the adapter. The \a action is appended to the view widget.
*/
void ActionAdapter::addAction(Action* action) {
    if (m_widget) {
        m_widget->addAction(action);
    } else {
        m_graphicsWidget->addAction(action);
    }

    addActionMappings(action);
}

/*!
    Inserts \a action to the adapter. The \a action is inserted to the view widget before \a before action.
*/
void ActionAdapter::insertAction(Action* before, Action* action) {
    if (m_widget) {
        m_widget->insertAction(before, action);
    } else {
        m_graphicsWidget->insertAction(before, action);
    }

    addActionMappings(action);
}

/*!
    Removes \a action from the adapter. The \a action is removed to the view widget.
*/
void ActionAdapter::removeAction(Action* action) {
    QList<QAction*> actions;
    gatherActions(actions, action);

    QListIterator<QAction*> iterator(actions);
    while (iterator.hasNext()) {
        QAction* actionToRemove = iterator.next();
        m_actionMapper->removeMappings(actionToRemove);
        if (m_widget) {
            m_widget->removeAction(actionToRemove);
        } else {
            m_graphicsWidget->removeAction(actionToRemove);
        }
        action->disconnect();
    }
}

/*!
    Removes \a actions from the adapter.
*/
void ActionAdapter::removeActions(QList<Action*> actions) {
    QListIterator<WRT::Action*> iterator(actions);
     while (iterator.hasNext()) {
         WRT::Action* action = iterator.next();
         removeAction(action);
     }
}

/*!
    Adds \a actions to the adapter.
*/
void ActionAdapter::addActions(QList<Action*> actions) {
    QListIterator<WRT::Action*> iterator(actions);
    while (iterator.hasNext()) {
        WRT::Action* action = iterator.next();
        addAction(action);
    }
}

/*!
    Returns the signal mapper that is used by the action adapter.
*/
QSignalMapper* ActionAdapter::signalMapper() const {
    return m_actionMapper;
}

/*!
    Gathers a list of \a actions from \a action. Sub-items are gathered recursively.
*/
void ActionAdapter::gatherActions(QList<QAction*>& actions, QAction* action) const {
    if (action) {
        actions.append(action);
        Action* act = static_cast<Action*>(action);
        if (act) {
            QList<QAction*> childActions = act->subActions();
            while (childActions.length() > 0) {
                QAction* child = childActions.takeFirst();
                gatherActions(actions, child);
            }
        }

    }
}

/*!
    Adds action mappings to \a action.

    Following signals from \a action are connected:
    subActionsAdded
    subActionsRemoved
    triggered
*/
void ActionAdapter::addActionMappings(Action* action) {
    QList<QAction*> actions;
    gatherActions(actions, action);

    QListIterator<QAction*> iterator(actions);
    while (iterator.hasNext()) {
        QAction* actionToAdd = iterator.next();
        Action* act = static_cast<Action*>(actionToAdd);
        if (act) {
            m_actionMapper->setMapping(act, act->id());
#if !defined(SUPPORT_ONLY_FLAT_MENUS)
            connect(act, SIGNAL(subActionsAdded()), this, SLOT(updateSubActions()));
            connect(act, SIGNAL(subActionsRemoved()), this, SLOT(removeSubActions()));
#endif
            connect(act, SIGNAL(triggered()), m_actionMapper, SLOT(map()));
        }
    }
}


#if !defined(SUPPORT_ONLY_FLAT_MENUS)

/*!
    Updates sub actions that are added to the action adapter.
*/
void ActionAdapter::updateSubActions() {
    QObject* obj = sender();
    if (obj) {
        Action* action = static_cast<Action*>(obj);

        QList<Action*> actionList = actions(action->location());
        removeAction(action);
        int beforeIndex = actionList.indexOf(action) + 1;

        if (beforeIndex >= actionList.length()) {
            insertAction(0, action);
        } else {
            Action* beforeItem = actionList.value(beforeIndex);
            insertAction(beforeItem, action);
        }
    }
}

/*!
    Removes sub action that are added to the action adapter.
*/
void ActionAdapter::removeSubActions() {
    QObject* obj = sender();
    if (obj) {
        Action* action = static_cast<Action*>(obj);
        QList<QAction*> list;
        gatherActions(list, action);
    }
}
#endif

}
