/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef WEBWIDGET_H_
#define WEBWIDGET_H_

#include <QString>
#include <QStringList>
#include <QObject>
#include <QSize>
#include <QMap>

class WgtWidget;
class WidgetProperties;


namespace WRT {
namespace Maemo {
class DesktopFileReader;

/** Represents one web widget.
 This class represents single web widget that is shown
 on the screen. Class encapsulates metadata that is used
 when rendering the widget.
 */
class WebWidget: public QObject {
        Q_OBJECT
    public:

        /**
         * WebWidget Type
         */

        enum WebWidgetErrorCode {
             NoError = 0,
             GeneralError,
             MetadataError,
             DesktopFileError,
             RegistrationError,
             CapabilityValidationError
         };

        /** Represents the location of the widget.
         *  CANVAS: Located to canvas, e.g. Home Screen. Supported viewmodes
         *  are MINIMIZED and FLOATING.
         *  EMBEDDED: Embedded to a native application. Supported viewmode is
         *  FLOATING.
         *  WEBAPP: Standalone application. Supported viewmodes are FULLSCREEN / MAXIMIZED.
         */
        enum Location {
            CANVAS, EMBEDDED, WEBAPP
        };

        enum Orientation {
            Portrait,
            Landscape
        };

        /**
         * WebWidget Type
         */
        static const QString W3CWIDGET;
        static const QString PLAINWIDGET;

        /** Overloaded widget constructor.
         * Widget constructor taking desktop file path as an input.
         * Widget language is (in the order of preference) system language
         * from LANG environment variable, WebWidget default language or English.
         * @param path Absolute path to widget's desktop file.
         */
        WebWidget(const QString path, QObject* parent = 0);

        virtual ~WebWidget();

        /** Checks if the widget is valid. This function should be called after
         *  creating a new WebWidget.
         *  @return false, if widget is invalid.
         */
        bool isValid();

        /** Reloads widget's language version specific config.xml.
         *  @param language Indicates which config.xml to load.
         *  @return true, if config.xml was found.
         */
        bool reloadWidget(QString language);

        /** Returns the value of given widget property.
         *  @param property Name of the widget property. Possible values:
         *                  name, description, version, license, path,
         *                  browsingcapacity, widgetresource, secstore,
         *                  id, authorname, authoremail, authoruri, icon
         *                  size, type, desktopfilename,
         *                  viewmodes
         * @return Value of requested property. Types of properties:
         *         QString: name, description, version, license, path,
         *                  browsingcapacity, widgetresource, secstore, id,
         *                  desktopfilename
         *         QStringList: viewmodes
         *         QSize: size
         *         If parameter "property" contained unsupported property name, returns
         *         invalid QVariant (QVariant()).
         */

        QVariant getPropertyValue(QString property) const;

        /** sets the desktop icon
         * @param uri fullpath to icon
         * @param width desired width of icon
         * @param height, desired height if icon
         *
         */
        void setIcon(QString uri, int width, int height) const;
    public:
        // getters

        QString getName() const;
        QString getDescription() const;
        QString getVersion() const;
        QString getLocale() const;
        QString getLicense() const;
        QString getUrl() const;
        QString getBrowsingCapacity() const;
        QString getWidgetResource() const;
        QString webAppId() const;
        WebWidgetErrorCode errorCode();

        /**
         * Splash screen path.
         */
        QString splashScreen() const;

        QSize minSize(const WRT::Maemo::WebWidget::Orientation & orientation) const;
        QSize maxSize(const WRT::Maemo::WebWidget::Orientation & orientation) const;
        QSize preferredSize(const WRT::Maemo::WebWidget::Orientation & orientation) const;

        /** Gets the value of X-DUIApplet-WRT-Protected-Store from the .desktop file.
         * @return String representation of the key or an empty string if not found.
         */
        QString protectedStorageName() const;
        // FIXME: Remove this function and use widget registry
        // to get this info
        WidgetProperties* widgetProperties();

        /** Gets the widget size.
         * Gets the size of this widget.
         * @return Size of this widget.
         */
        QSize size() const;

        /** Gets the widget type.
         * Gets the widget type as a string. Possible values are:
         * "W3CWidget" for W3C widget
         * "Plain" for desktop file based widgets
         * "Other" for other types of widgets
         *
         * @return Type of the widget as string.
         */
        QString type() const;

        /** Gets widget's current viewmode.
         *  @return Viewmode: "windowed", "floating", "fullscreen", "maximized", or "minimized"
         */
        QString viewmode() const;
        QStringList supportedViewModes() const;

        /* sets the viewmode to a one specified by the mode
         * Currently the viewmode is purely based on configxml/desktop
         * If the widget is launched to fullscreen mode this needs
         * to be overwriten.
         */
        void setViewMode(QString mode);

    private:
        /** Represents the viewmode and interaction model of the widget.
         *  MINIMIZED: CANVAS widget, finger tap opens an APPLICATION viewmode
         *  of the same widget.
         *  FLOATING: Default mode, used for both CANVAS and EMBEDDED widgets.
         *  WINDOWED: WEBAPP widget, launched from application menu or by
         *  finger tapping a MINI widgets that has application mode defined.
         *  Similar with native applications.
         *
         */
        enum ViewMode {
            MINIMIZED,
            FLOATING,
            WINDOWED,
            FULLSCREEN
        };

        // TODO : Raine: Fix with run-time private implementation of WebWidget, so that there ain't anymore need for playing with friend classes.
        // These methods shouldn't be part of public interface.
        QString fileName();
        QSize widgetSize(const QSize & landspace, const QSize & portrait, const WRT::Maemo::WebWidget::Orientation & orientation) const;

        // Private accessory method for W3C Widget fields.
        bool contains(const QString & key, const QString & attribute = QString("")) const;
        QString value(const QString & key, const QString & attribute = QString("")) const;

        friend class WebWidgetController;
        friend class WebWidgetGraphicsView;
        friend class WebWidgetView;
        friend class WebAppRunner;
        friend class WidgetInstance;
        friend class WebWidgetTest;
        friend class WebWidgetViewPrivate;

    private:
        // methods
        void init(const DesktopFileReader & metadata);
        void initSizeAttbutes(const DesktopFileReader & metadata);
        bool readConfigfile(const DesktopFileReader & metadata);
        void setViewMode(const DesktopFileReader & metadata);

    private:
        // data

        QMap<QString, QVariant> m_properties;

        // widget metadata path
        QString m_metadataPath;
        QString m_locale;
        QString m_webAppId;
        QString m_widgetRoot;

        bool m_valid;

        // Size attributes in different orientation modes
        QSize m_minLandscape;
        QSize m_maxLandscape;
        QSize m_preferredLandscape;

        QSize m_minPortrait;
        QSize m_maxPortrait;
        QSize m_preferredPortrait;
        QStringList m_appParameters;

        WgtWidget * m_widget;

        ViewMode m_mode;

        WebWidgetErrorCode m_errorCode;
};
}
}

#endif /* WEBWIDGET_H_ */
