/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef SIGNATUREPARSER_H
#define SIGNATUREPARSER_H
#include <QDomDocument>
#include <QStringList>

/**
  * Class for parsing W3C Widet's signature.xml
  *
  */
class SignatureParser
{
public:

    /**
     * Constuctor
     *
     * @param workdir Directory where widget has been unzipped
     */
    SignatureParser(QString workdir, QString widgetpath );


    /**
     * Opens signature.xml and tries to parse it
     *
     * @return true if signature.xml exist and can be parsed
     */
    bool parse();

    /**
     * Cheks that Signature and SignedInfo element exist
     *
     * @return true if elements exist
     */
    bool requiredElementsExist();

    /**
     * Checks that for each file there exist a refrence uri in the signature file
     *
     * @return true if all files have equivivalent refrence
     */
    bool checkReferencesExist( QStringList files);

    /**
     * Validates signature
     *
     * @param signature_file path to signature file
     * @param certificate certificate used to encrypt SignedInfo (to generate SignatureValue)
     * @return true if signature is valid
     */
    bool validateSignature(QString signature_file, QByteArray cert);

    /**
     * Validates file hashes against stored hashes
     *
     * @return true if all file hashes matches stored hashes
     */
    bool validateReferences(QStringList files);

    /**
     * endEntityCert is filled with the certificate used for decrypting the widget
     *   signature if the certificate chain is valid
     * @return true if certificate is valid
     */
    bool validateCertificates(QByteArray& endEntityCert);

    /**
     * @return subject/name of the certificate or an empty string "".
     */
    QString getSubject();

    /**
     * @return authority key identifier of certificate or an empty string
     */
    QString getAKI();

private:
    QString cleanQuotes(QString str);

    int createDigest(QByteArray& digest, QString uri, QByteArray data);

    bool dsNamespace(QDomNode node);
    bool dspNamespace(QDomNode node);

    QDomDocument m_doc;
    QString m_workdir;
    QString m_widgetpath;
    QString m_aki;
    bool m_parsed;

};

#endif // SIGNATUREPARSER_H
