/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <openssl/sha.h>
#include <openssl/bio.h>
#include <openssl/dsa.h>
#include <openssl/rsa.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/ossl_typ.h>
#include <openssl/evp.h>
#include <openssl/err.h>


#ifndef LIBXML_C14N_ENABLED
#ifndef LIBXML_OUTPUT_ENABLED
#define LIBXML_C14N_ENABLED
#define LIBXML_OUTPUT_ENABLED
#endif /* LIBXML_OUTPUT_ENABLED */
#endif /* LIBXML_C14N_ENABLED */

#ifdef __SYMBIAN32__
#include <libxml2_parser.h>
#include <libxml2_c14n.h>
#include <libxml2_tree.h>
#include <xmlengxestd.h>
#else
#include <libxml/c14n.h>
#include <libxml/tree.h>
#endif

#include <QtCore>
#include <QDebug>
#include "SignatureValidator.h"

SignatureValidator::SignatureValidator() {
  OpenSSL_add_all_algorithms();
}

SignatureValidator::~SignatureValidator() {
}

QString SignatureValidator::getSubject(QByteArray certificate)
{
  BIO * b = BIO_new_mem_buf(certificate.data(),certificate.size());
  X509 * cert = PEM_read_bio_X509(b, NULL, 0, NULL);

  if (!cert)
      return QString();

  X509_NAME *subject = X509_get_subject_name(cert);
  // Set up a BIO to put the subject line into.
  BIO *subjectBio = BIO_new(BIO_s_mem());
  // Now, put the subject line into the BIO.
  X509_NAME_print_ex(subjectBio, subject, 0, XN_FLAG_RFC2253);
  //X509_NAME_print_ex(subjectBio, subject, 0, XN_FLAG_ONELINE);
  // Obtain a reference to the data and copy out
  // just the length of the data.
  // TODO: use QString for this
  char *dataStart = NULL;
  char *subjectString = NULL;
  long nameLength = BIO_get_mem_data(subjectBio, &dataStart);
  subjectString = new char[nameLength + 1];
  memset(subjectString, 0x00, nameLength + 1);
  memcpy(subjectString, dataStart, nameLength);
  QString result(subjectString);
  qDebug() << "Certificate subject name: " << result;
  return result;
}

int SignatureValidator::rsaVerify(QByteArray data, QByteArray signature, QByteArray certificate,
                                  int digestMethod, int c14nMode, int c14nComments) {

    BIO * b = BIO_new_mem_buf(certificate.data(),certificate.size());
    X509 * x509 = PEM_read_bio_X509(b, NULL, 0, NULL);

    if (!x509)
      return SignatureValidator::Verification_Failure;;

    EVP_PKEY *k = X509_get_pubkey(x509);

    if (!k) {
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        return SignatureValidator::Bad_Certificate;
    }

    if (EVP_PKEY_RSA != EVP_PKEY_type(k->type)) {
        if (NULL != k) EVP_PKEY_free(k);
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        return SignatureValidator::Bad_Certificate;
    }

    RSA *rsa = EVP_PKEY_get1_RSA(k);
    int rsize = RSA_size(rsa);

    QByteArray sig = QByteArray::fromBase64(signature);

    if (sig.size() != rsize || rsize < 11)
    {
        qDebug() << "Decryption error! Signature size != expected bitlength from RSA key";
        qDebug() << "Expected signature size: " << rsize << "; Actual: " << sig.size();
        return SignatureValidator::Verification_Failure;
    }

#ifdef __SYMBIAN32__
    TBuf8<256> decryptedBuf;
    unsigned char* decrypted = (unsigned char*)decryptedBuf.Ptr();
#elif WIN32
    unsigned char decrypted[256];
#else
    unsigned char decrypted[rsize];
#endif
    int len = RSA_public_decrypt(sig.size(), (const unsigned char*) sig.data(), decrypted, rsa, RSA_PKCS1_PADDING);

    if (len < 0) {
        char errBuf[120];
        ERR_error_string(ERR_get_error(), errBuf);
        QByteArray errMsg(errBuf);
        qDebug() << "openssl RSA_public_decrypt error:\n" << errMsg;

        if (NULL != k) EVP_PKEY_free(k);
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        if (NULL != rsa) RSA_free(rsa);
        return SignatureValidator::Verification_Failure;
    }

    // we now have decrypted the base-64 encoded digest of all data
    QByteArray decryptedByteArray = QByteArray::fromRawData((const char*)decrypted, len);

    // check the contents for the proper DigestInfo structure in hex
    QByteArray decryptedBaHex = decryptedByteArray.toHex();
    if ( (digestMethod == this->sha_1 && decryptedBaHex.indexOf(W3C::RSA_digestInfo_SHA1.toUtf8()) != 0)
        || ( digestMethod == this->sha_256 && decryptedBaHex.indexOf(W3C::RSA_digestInfo_SHA256.toUtf8()) != 0) ) {

        QString method;
        if (digestMethod == this->sha_1) {
            method = "SHA1";
        } else if (digestMethod == this->sha_256) {
            method = "SHA256";
        }
        qDebug() << "DigestInfo structure has invalid algorithmIdentifier, expected " << method << " prefix.";
        if (NULL != k) EVP_PKEY_free(k);
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        if (NULL != rsa) RSA_free(rsa);
        return SignatureValidator::Verification_Failure;
    } else {
        if (digestMethod == this->sha_1) {
            decryptedByteArray = decryptedByteArray.right(20); // 20 bytes for SHA1
        } else if (digestMethod == this->sha_256) {
            decryptedByteArray = decryptedByteArray.right(32); // 32 bytes for SHA256
        }
    }

    // perform our own digest on canonicalized data to verify they match
    QByteArray digest;
    QString signedInfo(data);

    QString canSignedInfo("");
    if (!this->canonicalize(signedInfo,canSignedInfo, c14nMode, c14nComments)) {
        qDebug() << "Canonicalization failed!";
        return SignatureValidator::Verification_Failure;
    }

    int rc = this->createDigest(&digest, digestMethod, canSignedInfo.toAscii());
    if (rc != 1) return SignatureValidator::Verification_Failure;

    // clean up OpenSSL structures
    if (NULL != k) EVP_PKEY_free(k);
    if (NULL != b) BIO_free(b);
    if (NULL != x509) X509_free(x509);
    if (NULL != rsa) RSA_free(rsa);

    if (digest != decryptedByteArray) {
        qDebug() << "Error validating signature!\n";
        qDebug() << "decyrptedByteArray:" << decryptedByteArray.length() << "\n" << decryptedByteArray.toHex();
        qDebug() << "digest:" << digest.length() << "\n" << digest.toHex();
        return SignatureValidator::Verification_Failure;
    }

    return SignatureValidator::Verification_Success;
}

int SignatureValidator::dsaVerify(QByteArray data, QByteArray signature, QByteArray certificate,
                                  int digestMethod, int c14nMode, int c14nComments) {
    // perform our own digest on canonicalized data to verify they match
    QString signedInfo(data);
    QString canSignedInfo("");
    if (!this->canonicalize(signedInfo,canSignedInfo, c14nMode, c14nComments)) {
        qDebug() << "Canonicalization failed!";
        return SignatureValidator::Verification_Failure;
    }
    QByteArray digest;
    int rc = this->createDigest(&digest, digestMethod, canSignedInfo.toAscii());
    if (rc != 1) return SignatureValidator::Verification_Failure;

    BIO * b = BIO_new_mem_buf(certificate.data(),certificate.size());
    X509 * x509 = PEM_read_bio_X509(b, NULL, 0, NULL);

    EVP_PKEY *k = X509_get_pubkey(x509);

    if (!k) {
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        return SignatureValidator::Bad_Certificate;
    }

    int keyType = EVP_PKEY_type(k->type);
    if (!(keyType == EVP_PKEY_DSA || keyType == EVP_PKEY_DSA3)) {
        if (NULL != k) EVP_PKEY_free(k);
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        return SignatureValidator::Bad_Certificate;
    }

    DSA *dsa = EVP_PKEY_get1_DSA(k);
    QByteArray sig = QByteArray::fromBase64(signature);

    // TODO: verify that DSA_verify is in fact independent of the message digest algorithm used
    // TODO: verify that the decrypted data from sig is NOT base64, otherwise we're passing bad data for verify() to compare against
    rc = DSA_verify(0, (const unsigned char*) digest.data(), digest.size(), (const unsigned char*) sig.data(), sig.size(), dsa);

    if (NULL != k) EVP_PKEY_free(k);
    if (NULL != b) BIO_free(b);
    if (NULL != x509) X509_free(x509);
    if (NULL != dsa) DSA_free(dsa);

    if (rc < 0) {
        char errBuf[120];
        ERR_error_string(ERR_get_error(), errBuf);
        QByteArray errMsg(errBuf);
        qDebug() << "OpenSSL DSA_verify error:\n" << errMsg;
        return SignatureValidator::Verification_Failure;
    }

    if (rc == 0 ) {
        qDebug() << "DSA verification failed.  Incorrect signature.";
        return SignatureValidator::Verification_Failure;
    }

    return SignatureValidator::Verification_Success;
}

int SignatureValidator::createDigest(QByteArray* digest, int digestMethod, QByteArray data) {
    if ( digestMethod == this->sha_1 ) {
        *digest = QCryptographicHash::hash(data, QCryptographicHash::Sha1);
        return 1;
    #ifdef SHA256_SUPPORT
    } else if ( digestMethod == this->sha_256 ) {
        unsigned char hashed[SHA256_DIGEST_LENGTH];
        SHA256((const unsigned char*)data.data(), data.size(), hashed);
        for (int i = 0; i < SHA256_DIGEST_LENGTH; i++) {
            (*digest)[i] = hashed[i];
        }
        return 1;
    #endif
    } else {
        qDebug() << "Hash algorithm not specified or implemented.";
        return 0;
    }
}

bool SignatureValidator::canonicalize(const QString& inStr, QString& outStr, int c14nMode, int c14nComments)
{
#ifdef __SYMBIAN32__
    TRAP_IGNORE(XmlEngineAttachL());
#endif

    // parse inStr into xmlDoc
    QByteArray inBa = inStr.toUtf8();
    xmlChar* cur = (unsigned char*) inBa.data();
    xmlDocPtr doc = xmlParseDoc(cur);

    QByteArray outBuf(inBa.size(),' '); // get an empty buffer
    xmlChar* doc_txt_ptr = (unsigned char*) outBuf.data();
    int len = xmlC14NDocDumpMemory( doc, NULL, c14nMode, NULL, c14nComments, &doc_txt_ptr);

    if (len<0) {
        qDebug() << "Error: Canonicalization failed";
        return false;
    }
    outStr = QString::fromUtf8((char *)doc_txt_ptr,len);

#ifdef __SYMBIAN32__
    XmlEngineCleanup();
#endif
    return true;
}
