/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QDir>
#include <QFileInfo>
#include <QFile>
#include <QDesktopServices>
#include <QDateTime>
#include "CertificateManager.h"
#include "widgetmanagerconstants.h"
#include <openssl/bio.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/ossl_typ.h>
#include <openssl/objects.h>

namespace WRT {

    CertificateManager::CertificateManager()
    {
        OpenSSL_add_all_algorithms();
    }
    CertificateManager::~CertificateManager()
    {
    }

    bool CertificateManager::retrieveDeviceCertificate(QByteArray topLevel, QByteArray& cert)
    {

        QString aki = getAuthorityKeyIdentifier(topLevel);

        // hard-coded path to public cert store for linux
        QString widgetCertPath = "";
        QString dataLocation =  QDesktopServices::storageLocation(QDesktopServices::DataLocation);
        widgetCertPath = QDir::toNativeSeparators(dataLocation);
        widgetCertPath = widgetCertPath+QDir::separator()+WIDGET_FOLDER;
        widgetCertPath = QDir::toNativeSeparators(widgetCertPath);
        if (! createDir(widgetCertPath)) {
            qDebug() << "CertificateManger::retrieveDeviceCertificate  - Could not create certificate path dir";
            return false;
        }
        widgetCertPath = widgetCertPath+QDir::separator()+"widgetCertificates";
        widgetCertPath = QDir::toNativeSeparators(widgetCertPath);
        if (! createDir(widgetCertPath)) {
            qDebug() << "CertificateManger::retrieveDeviceCertificate - Could not create certificate path dir";
            return false;
        }

        // go through all certificates in widgetCertPath
        // the call to getSubjectKeyIdentifier assumes cert will be in .pem format
        QDir certDir = QDir(widgetCertPath);
        qDebug() << "widgetCertPath: " << widgetCertPath;
        QStringList certList = certDir.entryList(QDir::Files | QDir::NoDotAndDotDot);
        qDebug() << "certList size: " << certList.size();
        while (!certList.isEmpty()) {
            QString certPath = certList.takeFirst();
            certPath.prepend(widgetCertPath + QDir::separator());
            QFile f(certPath);
            qDebug() << "Testing certificate: "+ certPath << " Size= "<< f.size();
            f.open(QIODevice::ReadOnly);
            QByteArray certByteArray(f.readAll());

            QString ski = getSubjectKeyIdentifier(certByteArray);
            // pick a cert whose ski matches aki, convert to bytearray and return
            if (ski == aki) {
                cert = certByteArray;
                return true;
            }
        }

        return false;
    }

    /**
      * Takes a certificate from the widget's signature.xml x509 element, gets the AKI from the cert,
      * and chain validates the certificate stored locally against the trust root.
      *
      * Qt implementation does not perform chain validation - there is only 1 cert in the chain
      */
    bool CertificateManager::validateCertificates(QList<QByteArray> certs, QByteArray endEntity, QByteArray topLevel)
    {
        QByteArray rootCert;
        QMap<QString,QString> extValues;
        if (!retrieveDeviceCertificate(topLevel, rootCert)) return false;

        // get AKI from OpenSSL
        QString aki = getAuthorityKeyIdentifier(topLevel);

        for ( int i=0; i < certs.size(); i++) {
            // check validity dates for all certs in chain
            if (!checkCertificateValidityDates(certs.at(i))) {
                qDebug() << "Error: certificate's Validitiy dates do not include the system date";
                return false;
            }
        }

        // check validity dates for root cert
        if (!checkCertificateValidityDates(rootCert)) {
            qDebug() << "Error: certificate's Validitiy dates do not include the system date";
            return false;
        }

        // TODO: retrieve these AKI values from a file so we can modify trust domains after WRT ships
        if (aki == "6E:3D:26:D3:BF:67:68:B1:1A:F2:32:72:9F:D1:09:2A:D4:6F:D2:D3" ||  // NOCA
            aki == "C0:1D:9D:9A:9F:E8:4E:91:A9:14:48:C5:DC:D7:03:69:03:6C:69:A4") { // NOCA I
            extValues.insert(KEY_USAGE,"Digital Signature");
            extValues.insert(EXT_KEY_USAGE,"1.3.6.1.4.1.94.1.49.1.2.2.15");
            extValues.insert(CERT_POLICY,"Policy: 1.3.6.1.4.1.94.1.49.1.1.3.17");
            if (!checkExtensionValues(endEntity, extValues)) return false;
        }
        else if (aki == "9C:43:CE:FF:80:0B:75:69:7E:17:EF:3E:0F:FC:5E:E2:40:A1:03:3E:18:A9:A0:CE:86:0B:95:7F:B6:23:85:A5") //vf
            {
            extValues.insert(KEY_USAGE,"Digital Signature");
            extValues.insert(EXT_KEY_USAGE,"1.3.6.1.5.5.7.3.3");
            if (!checkExtensionValues(endEntity, extValues)) return false;
            }

        // build the root certificate into an X509
        BIO * root_b = BIO_new_mem_buf(rootCert.data(),rootCert.size());
        X509 * x509_rootCert = PEM_read_bio_X509(root_b, NULL, 0, NULL);

        // build the end entity cert to validate
        BIO * end_b =  BIO_new_mem_buf(endEntity.data(),endEntity.size());
        X509 * x509_endEntity = PEM_read_bio_X509(end_b, NULL, 0, NULL);

        X509_STORE * CAcerts;
        X509_STORE_CTX ca_ctx;

        int error = 0;

        if (!(CAcerts = X509_STORE_new())) {
            error = 1;
        }

        STACK_OF(X509) *untrustedChain;
        if (! (untrustedChain = sk_X509_new_null())) {
            qDebug() << "memory allocation error for untrusted cert chain!";
            error = 2;
        }

        if (!error) {
            for (int i = 0; i < certs.length(); i++) {
                QByteArray cert = certs.at(i);
                BIO * b = BIO_new_mem_buf(cert.data(),cert.size());
                X509 * untrusted_x509_link = PEM_read_bio_X509(b, NULL, 0, NULL);
                sk_X509_push(untrustedChain, untrusted_x509_link);
            }
        }
        /*int X509_STORE_CTX_init(X509_STORE_CTX *ctx, X509_STORE *store,
                         X509 *x509, STACK_OF(X509) *chain);
        */
        if (!error) {
            if (X509_STORE_CTX_init(&ca_ctx, NULL, x509_endEntity, untrustedChain) != 1) {
                error = 3;
            }
        }

        STACK_OF(X509) *trustedChain;
        if (! (trustedChain = sk_X509_new_null())) {
            qDebug() << "memory allocation error for trusted cert chain!";
            error = 4;
        }
        if (! error) {
            sk_X509_push(trustedChain, x509_rootCert);
            X509_STORE_CTX_trusted_stack(&ca_ctx, trustedChain);
        }

        if (!error) {
            int rc = X509_verify_cert(&ca_ctx);
            if (rc != 1) {
                qDebug() << "return code for verify cert: " << rc;
                int err = X509_STORE_CTX_get_error(&ca_ctx);
                const char* errmsg = X509_verify_cert_error_string(err);
                qDebug() << "error in X509_verify_cert: " << errmsg;
                error = 5;
            }
        }

        if (NULL != untrustedChain) sk_X509_free(untrustedChain);
        if (NULL != trustedChain) sk_X509_free(trustedChain);

        if (error) {
            qDebug() << "Error:" << "CertificateManager::validateCertificate" << error;
            return false;
        }

        return true;
    }

    bool CertificateManager::buildCertMapping(QList<QByteArray> certs, QByteArray& endEntityCert, QByteArray& topLevel) {
        // associate AKI and SKI for each cert
        QMap<QString, QString> akiToski;

        // index a key to the cert that it came from
        QMap<QString, int> skiToCertNum;
        QMap<QString, int> akiToCertNum;


        // build a mapping of SKI to AKI
        for (int i = 0; i < certs.length(); i++) {
            QByteArray cert = certs.at(i);
            QString l_aki = getAuthorityKeyIdentifier(cert);
            QString l_ski = getSubjectKeyIdentifier(cert);
            if (l_aki.isEmpty() || l_aki == l_ski) {
                topLevel = cert;
                continue; // we don't need to include this cert in our mapping, we've already figured out that it's the root certificate
            }
            akiToski.insert(l_aki, l_ski);
            skiToCertNum.insert(l_ski, i);
            akiToCertNum.insert(l_aki, i);
        }

        QList<QString> certSKIs = akiToski.values();
        QList<QString> certAKIs = akiToski.keys();

        // find an AKI that does not match an SKI of a cert in the widget package
        if (topLevel.isEmpty()) {
            QMapIterator<QString, QString> i(akiToski);
            while (i.hasNext()) {
                i.next();
                if (! certSKIs.contains(i.key()) ) {
                    topLevel = certs[akiToCertNum.value(i.key())];
                    break;
                }
            }
        }

        // find a cert whose SKI is not referenced as the AKI of another cert
        for (int j = 0; j < certSKIs.length(); j++) {
            QString l_ski = certSKIs.at(j);
            if (! certAKIs.contains( l_ski ) ) {
                endEntityCert = certs[skiToCertNum.value(l_ski)];
                break;
            }
        }

        // if we have an AKI but not an end-entity, we're looking at a widget with a self-signed cert
        if (endEntityCert.isNull() && !topLevel.isEmpty()) {
            endEntityCert = certs[skiToCertNum.value(topLevel)];
        }

        if (!endEntityCert.isNull() && !topLevel.isNull() )
            return true;
        return false;
    }


    QString CertificateManager::getAuthorityKeyIdentifier(QByteArray certificate) {
        QString retVal = QString("");
        BIO * b = BIO_new_mem_buf(certificate.data(),certificate.size());
        X509 * cert = PEM_read_bio_X509(b, NULL, 0, NULL);
        if (!cert) {
            BIO_vfree(b);
            return retVal;
        }
        int extensions = X509_get_ext_count(cert);
        ASN1_OBJECT* aki_obj = OBJ_txt2obj(LN_authority_key_identifier,0);
        for (int i = 0; i < extensions; i++) {
            X509_EXTENSION *ext = X509_get_ext(cert, i);
            ASN1_OBJECT* ext_obj = ext->object;
            if (OBJ_cmp(aki_obj,ext_obj) == 0) {
                BIO *bio = BIO_new(BIO_s_mem());
                if (!X509V3_EXT_print(bio, ext, 0, 0))
                    M_ASN1_OCTET_STRING_print(bio,ext->value);
                BUF_MEM *buf;
                BIO_get_mem_ptr(bio, &buf);
                int bufLength = buf->length;

                retVal.resize(bufLength);
                for (int i = 0; i < bufLength; i++) {
                    retVal[i] = buf->data[i];
                }
                BIO_vfree(bio);
                break;
            }
        }
        ASN1_OBJECT_free(aki_obj);
        BIO_vfree(b);
        X509_free(cert);

        if (!retVal.isEmpty()) {
            int keyidloc = retVal.indexOf("keyid:");
            if (keyidloc != 0)
                return QString();
            int endkeyid = retVal.indexOf("\n");
            if (endkeyid != -1)
                retVal.truncate(endkeyid);
            retVal.remove(0,6); // 6 characters for "keyid:"
            retVal = retVal.simplified();
        }

        return retVal;
    }

    QString CertificateManager::getSubjectKeyIdentifier(QByteArray certificate) {
        QString retVal = QString("");
        BIO * b = BIO_new_mem_buf(certificate.data(),certificate.size());
        X509 * cert = PEM_read_bio_X509(b, NULL, 0, NULL);
        if (!cert) {
            BIO_vfree(b);
            return retVal;
        }
        int extensions = X509_get_ext_count(cert);
        ASN1_OBJECT* ski_obj = OBJ_txt2obj(LN_subject_key_identifier,0);
        for (int i = 0; i < extensions; i++) {
            X509_EXTENSION *ext = X509_get_ext(cert, i);
            ASN1_OBJECT* ext_obj = ext->object;
            if (OBJ_cmp(ski_obj,ext_obj) == 0) {
                BIO *bio = BIO_new(BIO_s_mem());
                if (!X509V3_EXT_print(bio, ext, 0, 0))
                    M_ASN1_OCTET_STRING_print(bio,ext->value);
                BUF_MEM *buf;
                BIO_get_mem_ptr(bio, &buf);
                int bufLength = buf->length;

                retVal.resize(bufLength);
                for (int i = 0; i < bufLength; i++) {
                    retVal[i] = buf->data[i];
                }
                BIO_vfree(bio);
                break;
            }
        }
        ASN1_OBJECT_free(ski_obj);
        BIO_vfree(b);
        X509_free(cert);

        return retVal;
    }

    bool CertificateManager::checkCertificateValidityDates(QByteArray certificate) {
        BIO * b = BIO_new_mem_buf(certificate.data(),certificate.size());
        X509 * cert = PEM_read_bio_X509(b, NULL, 0, NULL);
        if (!cert) {
            BIO_vfree(b);
            return false;
        }

        // CAs adhering to the Certificate and Certificate Revocation List (CRL) Profile MUST represent dates this way:
        // in UTCTime, using 13 characters, for dates between 1950-2049 (YYMMDDHHMMSSZ)
        // in GeneralizedTime, using 15 characters, for dates > 2050 (YYYYMMDDHHMMSSZ)
        ASN1_TIME* before_t = X509_get_notBefore(cert);
        ASN1_TIME* after_t = X509_get_notAfter(cert);
        QString before = (const char*) before_t->data;
        QString after = (const char*) after_t->data;

        if (before.right(1) != "Z" || after.right(1) != "Z") return false;

        //qDebug() << "x509 not valid before:" << before;
        //qDebug() << "x509 not valid after:" << after;

        // setup NOW variable
        QDateTime nowDT = QDateTime::currentDateTime();
        nowDT = nowDT.toUTC();

        // in general, we will parse and accept all dates (including GeneralizedTime for a date < 2050)
        // set up Validity start dates: not-valid-before:
        int before_year;
        if (before.length() == 13) {
            // handle UTCTIME (1950-2049)
            bool ok;
            before_year = before.left(2).toInt(&ok);
            if (!ok) return false; // could not convert the first 2 characters of "before" to an int!
            before.remove(0,2);
            if (before_year < 50) before_year += 2000;
            else before_year += 1900;
        } else if (before.length() == 15 ) {
            // handle GeneralizedTime
            bool ok;
            before_year = before.left(4).toInt(&ok);
            if (!ok) return false;
            before.remove(0,4);
        } else {
            return false; // invalid size of validity start date
        }

        // set up Validity end dates: not-valid-after:
        int after_year;
        if (after.length() == 13) {
            // handle UTCTIME (1950-2049)
            bool ok;
            after_year = after.left(2).toInt(&ok);
            if (!ok) return false; // could not convert the first 2 characters of "before" to an int!
            after.remove(0,2);
            if (after_year < 50) after_year += 2000;
            else after_year += 1900;
        } else if (after.length() == 15 ) {
            // handle GeneralizedTime
            bool ok;
            after_year = after.left(4).toInt(&ok);
            if (!ok) return false;
            after.remove(0,4);
        } else {
            return false; // invalid size of validity start date
        }

        QDate beforeDate = QDate(before_year, before.left(2).toInt(), before.mid(2,2).toInt());
        QDate afterDate = QDate(after_year, after.left(2).toInt(), after.mid(2,2).toInt());

        QTime beforeTime = QTime(before.mid(4,2).toInt(), before.mid(6,2).toInt(), before.mid(8,2).toInt(), 0);
        QTime afterTime = QTime(after.mid(4,2).toInt(), after.mid(6,2).toInt(), after.mid(8,2).toInt(), 0);

        QDateTime beforeDT = QDateTime(beforeDate, beforeTime, Qt::UTC);
        QDateTime afterDT = QDateTime(afterDate, afterTime, Qt::UTC);

        BIO_vfree(b);
        X509_free(cert);

        if (nowDT >= beforeDT && nowDT < afterDT)
            return true;

        return false;
    }

    /**
      * A method for checking the values of Mandatory x509 Extensions for the Nokia Online CA
      */
    bool CertificateManager::checkExtensionValues(QByteArray certificate, QMap<QString,QString>& expectedExtValues) {
        BIO * b = BIO_new_mem_buf(certificate.data(),certificate.size());
        X509 * cert = PEM_read_bio_X509(b, NULL, 0, NULL);
        if (!cert) {
            BIO_vfree(b);
            return false;
        }

        int extensions = X509_get_ext_count(cert);
        ASN1_OBJECT* keyUsage_obj = OBJ_txt2obj(LN_key_usage,0);
        ASN1_OBJECT* extendedKeyUsage_obj = OBJ_txt2obj(LN_ext_key_usage,0);
        ASN1_OBJECT* certPolicies_obj = OBJ_txt2obj(LN_certificate_policies,0);

        int extensionsFound = 0;
        for (int i = 0; i < extensions; i++) {
            X509_EXTENSION *ext = X509_get_ext(cert, i);
            ASN1_OBJECT* ext_obj = ext->object;

            BIO *bio;
            BUF_MEM *buf;
            QString val;
            if ( OBJ_cmp(keyUsage_obj,ext_obj) == 0 ||
                 OBJ_cmp(extendedKeyUsage_obj,ext_obj) == 0 ||
                 OBJ_cmp(certPolicies_obj,ext_obj) == 0 ) {
                bio = BIO_new(BIO_s_mem());
                if (!X509V3_EXT_print(bio, ext, 0, 0))
                    M_ASN1_OCTET_STRING_print(bio,ext->value);
                BIO_get_mem_ptr(bio, &buf);
                val = QString(buf->data);
                val.resize(buf->length);
                val = val.simplified();

                if ((OBJ_cmp(keyUsage_obj,ext_obj) == 0) && (expectedExtValues.value(KEY_USAGE) != "")) {
                    // is this: X509v3 Key Usage
                    if (val != expectedExtValues.value(KEY_USAGE)) {
                        qDebug() << "Key Usage extension != 'digitalSignature'";
                        BIO_vfree(bio);
                        ASN1_OBJECT_free(ext_obj);
                        return false;
                    }
                    extensionsFound++;
                } else if ((OBJ_cmp(extendedKeyUsage_obj,ext_obj) == 0) && (expectedExtValues.value(EXT_KEY_USAGE) != "")) {
                    // is this: X509v3 Extended Key Usage
                    if (val != expectedExtValues.value(EXT_KEY_USAGE)) {
                        qDebug() << "Extended Key Usage extension != expected result";
                        BIO_vfree(bio);
                        ASN1_OBJECT_free(ext_obj);
                        return false;
                    }
                    extensionsFound++;
                } else if ((OBJ_cmp(certPolicies_obj,ext_obj) == 0) && (expectedExtValues.value(CERT_POLICY) != "")) {
                    // is this: X509v3 Certificate Policies
                    if (val != expectedExtValues.value(CERT_POLICY)) {
                        qDebug() << "Certificate Policies Policy Identifier != expected result";
                        BIO_vfree(bio);
                        ASN1_OBJECT_free(ext_obj);
                        return false;
                    }
                    extensionsFound++;
                }

                BIO_vfree(bio);
            }
            ASN1_OBJECT_free(ext_obj);
        }

        if (extensionsFound != expectedExtValues.size()) {
            qDebug() << "All required extensions not found. Returning an error.";
            return false;
        }

        return true;
    }

    bool CertificateManager::createDir(const QString& path) {
        QDir dir(path);
        if (!dir.exists()){
            if (!dir.mkdir(dir.absolutePath())) {
                qDebug() << "Could not create dir : " << path;
                return false;
            }
        }
        return true;
    }

}
