/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include <QtDBus/QtDBus>
#include <QDebug>
#include <sys/statfs.h>
#include <mntent.h>
#include "listdriveinfo.h"

//---------------------------------------
//default constructor
//---------------------------------------
ListDriveInfo::ListDriveInfo()
{
    m_mediaTypes.clear();
    m_mediaTypes.insert(HALMediaDisk, MediaHardDisk);
    m_mediaTypes.insert(HALMediaCdrom, MediaCdRom);
    m_mediaTypes.insert(HALMediaFloppy, MediaFloppyDisk);
    m_mediaTypes.insert(HALMediaTape, MediaNotPresent);
    m_mediaTypes.insert(HALMediaCompactFlash, MediaFlash);
    m_mediaTypes.insert(HALMediaMemoryStick, MediaFlash);
    m_mediaTypes.insert(HALMediaSmartMedia, MediaFlash);
    m_mediaTypes.insert(HALMediaSD_MMC, MediaFlash);
}

//---------------------------------------
//destructor
//---------------------------------------
ListDriveInfo::~ListDriveInfo()
{

}

//---------------------------------------
//initialization
//---------------------------------------
void ListDriveInfo::init()
{
    char driveName='A';
    QStringList mountedFS(getMountPoints());
    QStringList volumes(getVolumes());

    for (int i=0; i<mountedFS.count(); i++)
    {
        QMap<QString, QVariant> driveInfo;
        driveInfo.insert(DriveUrl, volumeProperty(mountedFS.at(i), VolPropDriveUrl));
        driveInfo.insert(DriveName, QVariant(QString(driveName)));
        driveInfo.insert(TotalMemory, volumeProperty(mountedFS.at(i), VolPropTotalMemory));
        driveInfo.insert(FreeMemory, volumeProperty(mountedFS.at(i), VolPropFreeMemory));
        driveInfo.insert(CritcalMemory, volumeProperty(mountedFS.at(i), VolPropCritcalMemory));
        driveInfo.insert(IsRemovableMedia, volumeProperty(mountedFS.at(i), VolPropIsRemovableMedia));
        driveInfo.insert(MediaType, volumeProperty(mountedFS.at(i), VolPropMediaType));
        m_fullInfo.push_back(driveInfo);
        driveName++;
    }

    //here trying to recieve not mounted devices
    for (int i=0; i<volumes.count(); i++)
    {
        QString tmpVolume(volumeHalProperty(volumes.at(i), HALArgVolume_MountPoint).toString());
        if (tmpVolume==QString("") || tmpVolume.isEmpty() || mountedFS.contains(tmpVolume))
        {
            continue;
        }

        QMap<QString, QVariant> driveInfo;
        driveInfo.insert(DriveUrl, volumeProperty(tmpVolume, VolPropDriveUrl));
        driveInfo.insert(DriveName, QVariant(VolPropDriveName));
        driveInfo.insert(TotalMemory, volumeProperty(tmpVolume, VolPropTotalMemoryNonMounted));
        driveInfo.insert(FreeMemory, QVariant((qint64)0));
        driveInfo.insert(CritcalMemory, volumeProperty(tmpVolume, VolPropTotalMemoryNonMounted));
        driveInfo.insert(IsRemovableMedia, volumeProperty(tmpVolume, VolPropIsRemovableMedia));
        driveInfo.insert(MediaType, QVariant(MediaNotPresent));
        m_fullInfo.push_back(driveInfo);
        driveName++;
    }
}

//---------------------------------------
//get info
//---------------------------------------
QMap<QString, QVariant > ListDriveInfo::getInfo(const QString &name)
{
    QMap<QString, QVariant>  info;
    for (int i=0; i<m_fullInfo.count(); i++)
    {
        if (m_fullInfo.at(i).value(DriveName).toString()==name)
        {
            info = m_fullInfo.at(i);
            return info;
        }
    }
    return info;
}

//---------------------------------------
//get names
//---------------------------------------
QList <QString > ListDriveInfo::getNames()
{
    QList< QString > names;
    for (int i=0; i<m_fullInfo.count(); i++)
    {
        names.append(m_fullInfo.at(i).value(DriveName).toString());
    }
    return names;
}

//---------------------------------------
//get volumes
//---------------------------------------
QStringList ListDriveInfo::getVolumes()
{
    QList<QVariant> args;
    QDBusReply<QStringList> reply;
    QStringList volumesAvaliable;

    QDBusInterface halIface(HAL_SERVICE, HAL_PATH, HAL_INTERFACE, QDBusConnection::systemBus());
    if (!halIface.isValid())
    {
        qDebug ("Could not establish a DBus connection to Hal");
        return volumesAvaliable;
    }

    args.append(HALArgVolume);
    reply=halIface.callWithArgumentList(QDBus::Block, HALCallFindDeviceByCapability, args);

    for (int i=0; i<reply.value().count(); i++)
    {
        QString device(reply.value().at(i));
        if (device.isEmpty())
        {
            continue;
        }

        volumesAvaliable.append(device.split('/').last());
    }

    return volumesAvaliable;
}

//---------------------------------------
//get volume property
//---------------------------------------
QVariant ListDriveInfo::volumeProperty(const QString& mountPoint, const QString& prop)
{
    if (prop==VolPropTotalMemory)
    {
        return getStatfsProperty(mountPoint, eStatfsFullSpaceProp);
    }

    if (prop==VolPropTotalMemoryNonMounted)
    {
        if (blockDevice(mountPoint)!=QString(""))
        {
            return volumeHalProperty(blockDevice(mountPoint),HALArgVolume_Size);
        }

        return QVariant((qint64)0);
    }

    if (prop==VolPropFreeMemory)
    {
        return getStatfsProperty(mountPoint, eStatfsFreeSpaceProp);
    }

    if (prop==VolPropCritcalMemory)
    {
        return getStatfsProperty(mountPoint, eStatfsCriticalSpaceProp);
    }

    if (prop==VolPropDriveUrl)
    {
        return QString("file:///%1").arg(QString(mountPoint).remove(0,1));
    }

    if (prop==VolPropDriveName)
    {
        if (blockDevice(mountPoint)!=QString(""))
        {
            return volumeHalProperty(blockDevice(mountPoint), HALArgVolume_UUID);
        }

        return getDriveName(mountPoint);
    }

    if (prop==VolPropIsRemovableMedia)
    {
        if (blockDevice(mountPoint)==QString(""))
        {
            return QVariant(false);
        }

        return storageHalProperty(volumeHalProperty(blockDevice(mountPoint),
                                  HALArgBlock_StorageDevice).toString(),
                                  HALArgStorage_Removable_MediaAvailable);
    }

    if (prop==VolPropMediaType)
    {
        if (blockDevice(mountPoint)!=QString(""))
        {
            QString halMediaName;
            halMediaName=storageHalProperty(volumeHalProperty(blockDevice(mountPoint),
                                                              HALArgBlock_StorageDevice).toString(),
                                                              HALArgStorage_DriveType).toString();
            QMap<QString, QString>::const_iterator found=m_mediaTypes.find(halMediaName);
            if (found==m_mediaTypes.end())
            {
                return QVariant(MediaUnknown);
            }

            return QVariant(found.value());
        }
        else
        {
            //other way;
            switch (getStatfsProperty(mountPoint, eStatfsFSTypeProp).toUInt())
            {
            case CIFS_MAGIC_NUMBER:
            case SMB_SUPER_MAGIC:
                return QVariant(MediaRemote);
            case OPENPROM_SUPER_MAGIC:
                return QVariant(MediaRom);
            case CRAMFS_MAGIC:
            case UBIFS_SUPER_MAGIC:
                return QVariant(MediaRam);
            }
        }
        return QVariant(MediaUnknown);
    }
    //add new here if need
    return QVariant();
}

//---------------------------------------
//get hal volume property
//---------------------------------------
QVariant ListDriveInfo::volumeHalProperty(const QString& drive, const QString& realName)
{
    QList<QVariant> args;

    QDBusInterface halIface(HAL_SERVICE, QString("/org/freedesktop/Hal/devices/%1").arg(drive), HAL_DEVICE_INTERFACE, QDBusConnection::systemBus());
    if (!halIface.isValid())
    {
        qDebug()<<QString("/org/freedesktop/Hal/devices/%1").arg(drive);
        qDebug ("Could not establish a DBus connection to Hal");
        return QVariant();
    }

    args.append(realName);
    QDBusMessage reply=halIface.callWithArgumentList(QDBus::Block, HALCallGetProperty, args);
    if (reply.type()==QDBusMessage::ErrorMessage || reply.arguments().isEmpty())
    {
        return QVariant();
    }

    return QVariant(reply.arguments().at(0));
}

//---------------------------------------
//get hal storage property
//---------------------------------------
QVariant ListDriveInfo::storageHalProperty(const QString& drive, const QString& realName)
{
    QList<QVariant> args;

    QDBusInterface halIface(HAL_SERVICE, drive, HAL_DEVICE_INTERFACE, QDBusConnection::systemBus());
    if (!halIface.isValid())
    {
        qDebug()<<QString("%1").arg(drive);
        qDebug ("Could not establish a DBus connection to Hal");
        return QVariant();
    }

    args.append(realName);
    QDBusMessage reply=halIface.callWithArgumentList(QDBus::Block, HALCallGetProperty, args);
    if (reply.type()==QDBusMessage::ErrorMessage || reply.arguments().isEmpty())
    {
        return QVariant();
    }

    return QVariant(reply.arguments().at(0));
}

//---------------------------------------
//get statfs property
//---------------------------------------
QVariant ListDriveInfo::getStatfsProperty(const QString& mountPoint, statfsprops_t prop)
{
    struct statfs stat;
    if (statfs(mountPoint.toAscii().data(), &stat)!=0)
    {
        return 0;
    }

    qlonglong fullSize;
    qlonglong blocksize;
    switch (prop)
    {
    case eStatfsFreeSpaceProp:
        fullSize=stat.f_bfree;
        blocksize=stat.f_bsize;
        return QVariant(fullSize * blocksize);

    case eStatfsCriticalSpaceProp:
        fullSize=stat.f_bfree - stat.f_bavail;
        blocksize=stat.f_bsize ;
        return QVariant(fullSize * blocksize);

    case  eStatfsFullSpaceProp:
        fullSize=stat.f_blocks;
        blocksize=stat.f_bsize;
        return QVariant(fullSize * blocksize);

    case eStatfsFSTypeProp:
        //qDebug()<<"MountPoint:"<<mountPoint<<"FSType hex:"<<QString("0x%1").arg(stat.f_type, 8, 16, QChar('0'));
        return stat.f_type;

    }

    return 0;
}

//---------------------------------------
//get mount points from mtab
//---------------------------------------
QStringList ListDriveInfo::getMountPoints()
{
    QStringList mountPoints;
    struct mntent *mnt;
    FILE *fp;

    fp = setmntent (_PATH_MOUNTED, "r");
    if (fp == NULL)
    {
        qDebug()<<"Error while open "<<_PATH_MOUNTED;
        return mountPoints;
    }

    while ((mnt = getmntent (fp)))
    {
        //qDebug()<<"fsname: "<<mnt->mnt_fsname;
        //qDebug()<<"mnt_dir: "<<mnt->mnt_dir;
        QString tmp(mnt->mnt_dir);
        int fstype=getStatfsProperty(tmp, eStatfsFSTypeProp).toInt();

        if (mountPoints.contains(tmp))
        {
            continue;
        }

        if (QString(mnt->mnt_fsname)==NoneFileSystem)
        {
            continue;
        }

        if (fstype==SYSFS_MAGIC
           || fstype==DEVFS_SUPER_MAGIC
           || fstype==PROC_SUPER_MAGIC
           || fstype==DEVPTS_SUPER_MAGIC
           || fstype==S_MAGIC_SECURITYFS
           )
        {
            continue;
        }

        mountPoints.push_back(tmp);
    }

    if (endmntent (fp) == 0) {
        qDebug()<<"Error while close "<<_PATH_MOUNTED;
    }

    return mountPoints;
}

//---------------------------------------
//the name of block device
//---------------------------------------
QString ListDriveInfo::blockDevice(const QString& mountPoint)
{
    QString volumeName;
    QList<QVariant> args;
    QDBusReply<QStringList> reply;

    QDBusInterface halIface(HAL_SERVICE, HAL_PATH, HAL_INTERFACE,QDBusConnection::systemBus());
    if (!halIface.isValid())
    {
        qDebug ("Could not establish a DBus connection to Hal");
        return volumeName;
    }


    args.append(HALArgVolume_MountPoint);
    args.append(mountPoint);
    reply=halIface.callWithArgumentList(QDBus::Block, HALCallFindDeviceStringMatch, args);

    for (int i=0; i<reply.value().count(); i++)
    {
        QString device(reply.value().at(i));
        volumeName=device.split('/').last();
    }

    return volumeName;
}

//---------------------------------------
//get specific linux drive names
//---------------------------------------
QString ListDriveInfo::getDriveName(const QString& mountPoint)
{
    struct mntent *mnt;
    FILE *fp;

    fp = setmntent (_PATH_MOUNTED, "r");
    if (fp == NULL)
    {
        qDebug()<<"Error while open "<<_PATH_MOUNTED;
        return QString("");
    }

    while ((mnt = getmntent (fp)))
    {
        if (mountPoint==QString(mnt->mnt_dir))
        {
            return QString(mnt->mnt_fsname);
        }
    }

    if (endmntent (fp) == 0)
    {
        qDebug()<<"Error while close "<<_PATH_MOUNTED;
    }

    return QString("");
}
