/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */



#ifndef QMESSAGINGSERVICEINTERFACE_H
#define QMESSAGINGSERVICEINTERFACE_H

//QT Include
#include <QVariant>
#include <QString>
#include <QMap>

//Service Framework Include
#include "servicebase.h"

static const char* KIMessagingId = "com.nokia.IMessaging/1.0";

// ITypeTest interface id.For Future use
static const ServiceInterfaceInfo KIMessagingInfo =
{
    KIMessagingId, 0
};

class IMessaging : public IServiceBase
{
public:
    //public interfaces

    /**
     * QMessagingService::startEditor()
     * This method launches the message editor.
     * This is a synchronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param  aMessageData       Represents message data that need to be populated when the message editor is launched.
     *                            type - type of message SMS or MMS to be sent
     *                            to - Array of recepient addresses.
     *                            body - message to be sent
     *                            subject - subject for the MMS
     *                            cc - Array of cc addresses.
     *                            attachments - Array of attachments to be sent in a MMS
     */
    virtual QVariant startEditor(const QMap<QString, QVariant>& aMessageData) = 0;

    /**
     * QMessagingService::getList()
     * This method is used to send a message. The message is sent to outbox.This is an Asynchronous API
     * --------------------------------------------------------------------------------------------------------------------
     * @param  messageData        Represents the message data required for sending a Message. This includes,
     *                            type - type of message SMS or MMS to be sent
     *                            to - Array of recepient addresses.
     *                            body - message to be sent
     *                            subject - subject for the MMS
     *                            cc - Array of cc addresses.
     *                            attachments - Array of attachments to be sent in a MMS
     *
     * @param  id                 Message id of another message for populating the data. Usefull in "Forward" operation.
     * --------------------------------------------------------------------------------------------------------------------
     * @return aTransactionId
     * @emits  void asyncCallback(int status, int transactionId, QObject* contIter)
     */
    virtual QVariant getList(const QMap<QString, QVariant>& aMatch,
            const int aSortKey, const int aSortOrder, const int aTransactionId) = 0;


    /**
     * QMessagingService::getListSync()
     * This method is used to get the list of  message ID's synchronously.
     * --------------------------------------------------------------------------------------------------------------------
     * @param  aMatch             Represents the filter criteria data required for getting a Message.
     *                            This includes,
     *                            folder - Name of the folder inside of which the message to be retrieved.
     *                            type - type of the message to be retrieved.
     *                            senders - message senders as filter param.
     *                            subject - subject as a filter param.
     * @param  aSortKey           sort key.
     * @Param  aSortOrder         sort order.
     * --------------------------------------------------------------------------------------------------------------------
     * @return aTransactionId
     * @emits  void asyncCallback(int status, int transactionId, QObject* contIter)
     */
    virtual QObject* getListSync(const QMap<QString, QVariant>& aMatch,
                                 const int aSortKey,
                                 const int aSortOrder)=0;
    /**
     * QMessagingService::getEmailAccounts()
     * This method gets all the mail boxes configured on the device.
     * --------------------------------------------------------------------------------------------------------------------
     * @return QVariant        List of maps with account ID and account Name.
     */
    virtual QVariant getEmailAccounts()=0;

    /**
    * QMessagingService::getCurrentEmailAccount()
    * This method gets the current email account set.
    * --------------------------------------------------------------------------------------------------------------------
    * @return Account        Map with account ID and account Name.
    */
   virtual QVariant getCurrentEmailAccount()=0;

   /**
   * QMessagingService::setCurrentEmailAccount()
   * This method gets the current email account set.
   * --------------------------------------------------------------------------------------------------------------------
   * @param aAccountId     Account Id of the account to be set as current email account.
   * @return Account       void
   */

  virtual QVariant setCurrentEmailAccount(QString aAccountId)=0;

    /**
     * QMessagingService::send()
     * This method is used to send a message. The message is sent to outbox.This is an Asynchronous API
     * --------------------------------------------------------------------------------------------------------------------
     * @param  aMessageData       Represents the message data required for sending a Message. This includes,
     *                            type - type of message SMS or MMS to be sent
     *                            to - Array of recepient addresses.
     *                            body - message to be sent
     *                            subject - subject for the MMS
     *                            cc - Array of cc addresses.
     *                            attachments - Array of attachments to be sent in a MMS
     *
     * @param  aId                Message id of another message for populating the data. Usefull in "Forward" operation.
     * --------------------------------------------------------------------------------------------------------------------
     * @return returnValue        Map with error code , error message and transaction id
     * @emits  void asyncCallback(int aStatus, int aTransactionId, int aId)
     */
    virtual QVariant send(const QMap<QString, QVariant>& aMessageData, const QString& aId, const int aTransactionId) = 0;

    /**
     * QMessagingService::setNotifier()
     * Registers a notifier for received messages. This is an Asynchronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @return  aTransactionId    transaction id of the new message arrived
     * @emits  void asyncCallback(int aStatus, int aTransactionId, int aId)
     */
    virtual QVariant setNotifier(const int aTransactionId) = 0;

    /**
     * QMessagingService::cancelNotifier()
     * Unregisters the currently set message notifier, ending message notifications. Synchronous API.
     */
    virtual QVariant cancelNotifier() = 0;

    /**
     * QMessagingService::getMessage()
     * Gets a message by id from the message store. It's a Synchoronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param aMessageId          The id of an existing message.
     * --------------------------------------------------------------------------------------------------------------------
     * @return MessageInfo        Message data for a particular message id with read-only information describing
     *                            a message retrieved from the message store.
     */
    virtual QVariant getMessage(const QString& aMessageId, const QString aFolderName = NULL ) = 0;

    /**
     * QMessagingService::deleteMessage()
     * Deletes a message from the message store. It's a Synchoronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param aId                  The id of an existing message.
     */
    virtual QVariant deleteMessage(const QMap<QString, QVariant>& aMessageData) = 0;

    /**
     * QMessagingService::setStatus()
     * Sets the read/unread status of a message in the message store. This is a Synchronous API
     * --------------------------------------------------------------------------------------------------------------------
     * @param aId                  The id of an existing message
     * @param aStatus              Either STATUS_UNREAD(0) or STATUS_READ(1)
     */
    virtual QVariant setStatus(const QString& aId,const int aStatus) = 0;

    /**
     * QMessagingService::cancel()
     * Cancels any asynchronous request identified uniquely by trasactionid. This is Synchronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param aTransactionId       Identifier of the ongoing asynchronous request to be canceled.
     */
    virtual QVariant cancel(const int aTransactionId) = 0;

};
Q_DECLARE_INTERFACE(IMessaging, KIMessagingId);

#endif //QMESSAGINGSERVICEINTERFACE_H
