/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef QMESSAGINGSERVICE_H_
#define QMESSAGINGSERVICE_H_

//QT Includes
#include <QObject>
#include <QVariant>
#include <QMap>
#include <QString>

//Project includes
#include "servicesecurityinterface.h"
#include "qmessagingserviceinterface.h"
#include "qmessagingconstants.h"
#include "sendmessage.h"


//QtMobility includes
#include <QMessageManager>
#include <QMessageId>
QTM_USE_NAMESPACE

class GetMsgListTask;
class MsgGetListPattern;

class QMessagingService : public QObject,                //For QT
                          public IMessaging,             //Messaging Interface Definition
                          public IServiceSecurity       //Security
{
    Q_OBJECT
    Q_INTERFACES(IMessaging)

public:
    /**
    * constructor
    * @since x.y
    * @param
    * @return
    */
    QMessagingService();

    /**
     * destructor
     * 1)Delete core class instance
     * @since x.y
     * @param
     * @return
     */
    virtual ~QMessagingService();

    /**
     * Implemented from IServiceSecurity.Needed to handle capabilities.
     * @since x.y
     * @param WRT::SecSession
     * @return
     */
    void setSecuritySession(WRT::SecSession *aSecSession);


public slots:
    void notifyResult(const int aErrorCode, int aTransactionId);
    void notifySMSResult(const int aErrorCode, int aTransactionId, QString aRecipient, bool aMoreRecipients);

    void getListCallback(GetMsgListTask*, qint32, int, QVariantList);


    /**
     * QMessagingService::startEditor()
     * This method launches the message editor.
     * This is a synchronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param  aMessageData       Represents message data that need to be populated when the message editor is launched.
     *                            type - type of message SMS or MMS to be sent
     *                            to - Array of recepient addresses.
     *                            body - message to be sent
     *                            subject - subject for the MMS
     *                            cc - Array of cc addresses.
     *                            attachments - Array of attachments to be sent in a MMS
     */
    QVariant startEditor(const QMap<QString, QVariant>& aMessageData);

    /**
     * QMessagingService::getList()
     * This method is used to send a message. The message is sent to outbox.This is an Asynchronous API
     * --------------------------------------------------------------------------------------------------------------------
     * @param  messageData        Represents the message data required for sending a Message. This includes,
     *                            type - type of message SMS or MMS to be sent
     *                            to - Array of recepient addresses.
     *                            body - message to be sent
     *                            subject - subject for the MMS
     *                            cc - Array of cc addresses.
     *                            attachments - Array of attachments to be sent in a MMS
     *
     * @param  id                 Message id of another message for populating the data. Usefull in �Forward� operation.
     * --------------------------------------------------------------------------------------------------------------------
     * @return aTransactionId
     * @emits  void asyncCallback(int status, int transactionId, QObject* contIter)
     */
    QVariant getList(const QMap<QString, QVariant>& aMatch,
            const int aSortKey, const int aSortOrder, const int aTransactionId);

    /**
     * QMessagingService::getListSync()
     * This method is used to get the list of  message ID's synchronously.
     * --------------------------------------------------------------------------------------------------------------------
     * @param  aMatch             Represents the filter criteria data required for getting a Message.
     *                            This includes,
     *                            folder - Name of the folder inside of which the message to be retrieved.
     *                            type - type of the message to be retrieved.
     *                            senders - message senders as filter param.
     *                            subject - subject as a filter param.
     * @param  aSortKey           sort key.
     * @Param  aSortOrder         sort order.
     * --------------------------------------------------------------------------------------------------------------------
     * @return aTransactionId
     * @emits  void asyncCallback(int status, int transactionId, QObject* contIter)
     */
    virtual QObject* getListSync(const QMap<QString, QVariant>& aMatch,
                                 const int aSortKey,
                                 const int aSortOrder);

    /**
     * QMessagingService::getEmailAccounts()
     * This method gets all the mail boxes configured on the device.
     * --------------------------------------------------------------------------------------------------------------------
     * @return QVariant        List of maps with account ID and account Name.
     */
    virtual QVariant getEmailAccounts();

    /**
     * QMessagingService::getCurrentEmailAccount()
     * This method gets the current email account set.
     * --------------------------------------------------------------------------------------------------------------------
     * @return Account        Map with account ID and account Name.
     */
    virtual QVariant getCurrentEmailAccount();

    /**
     * QMessagingService::setCurrentEmailAccount()
     * This method gets the current email account set.
     * --------------------------------------------------------------------------------------------------------------------
     * @param aAccountId     Account Id of the account to be set as current email account.
     * @return Account       void
     */
    virtual QVariant setCurrentEmailAccount(QString aAccountId);

    /**
     * QMessagingService::send()
     * This method is used to send a message. The message is sent to outbox.This is an Asynchronous API
     * --------------------------------------------------------------------------------------------------------------------
     * @param  aMessageData       Represents the message data required for sending a Message. This includes,
     *                            type - type of message SMS or MMS to be sent
     *                            to - Array of recepient addresses.
     *                            body - message to be sent
     *                            subject - subject for the MMS
     *                            cc - Array of cc addresses.
     *                            attachments - Array of attachments to be sent in a MMS
     *
     * @param  aId                Message id of another message for populating the data. Usefull in "Forward" operation.
     * --------------------------------------------------------------------------------------------------------------------
     * @return returnValue        Map with error code , error message and transaction id
     * @emits  void asyncCallback(int aStatus, int aTransactionId, int aId)
     */
    QVariant send(const QMap<QString, QVariant>& aMessageData, const QString& aId, const int aTransactionId);

    /**
     * QMessagingService::setNotifier()
     * Registers a notifier for received messages. This is an Asynchronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @return  aTransactionId    transaction id of the new message arrived
     * @emits  void asyncCallback(int aStatus, int aTransactionId, int aId)
     */
    QVariant setNotifier(const int aTransactionId);

    /**
     * QMessagingService::cancelNotifier()
     * Unregisters the currently set message notifier, ending message notifications. Synchronous API.
     */
    QVariant cancelNotifier();

    /**
     * QMessagingService::getMessage()
     * Gets a message by id from the message store. It�s a Synchoronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param aMessageId          The id of an existing message.
     * --------------------------------------------------------------------------------------------------------------------
     * @return MessageInfo        Message data for a particular message id with read-only information describing
     *                            a message retrieved from the message store.
     */
    QVariant getMessage(const QString& aMessageId, const QString aFolderName = NULL );

    /**
     * QMessagingService::deleteMessage()
     * Deletes a message from the message store. It�s a Synchoronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param aId                  The id of an existing message.
     */
    QVariant deleteMessage(const QMap<QString, QVariant>& aMessageData);

    /**
     * QMessagingService::setStatus()
     * Sets the read/unread status of a message in the message store. This is a Synchronous API
     * --------------------------------------------------------------------------------------------------------------------
     * @param aId                  The id of an existing message
     * @param aStatus              Either STATUS_UNREAD(0) or STATUS_READ(1)
     */
    QVariant setStatus(const QString& aId, const int aStatus);

    /**
     * QMessagingService::cancel()
     * Cancels any asynchronous request identified uniquely by trasactionid. This is Synchronous API.
     * --------------------------------------------------------------------------------------------------------------------
     * @param aTransactionId       Identifier of the ongoing asynchronous request to be canceled.
     */
    QVariant cancel(const int aTransctionId);

signals:
    /**
     * The signal emitted for the all the Async functions which doesnt output Iterator.
     * @since x.y
     * @param aStatus             0 or Error code
     * @param aTransactionId      TransactionID of the Async operation
     * @param aId                 The actual response data (in case of setnotifier and send)
     * @return
     */
    void asyncCallback(int aStatus, int aTransactionId, int aId);

    /**
     * The signal emitted for the all the Async functions which has the output as Iterator.
     * @since x.y
     * @param status              0 or Error code
     * @param aTransId            TransactionID of the Async operation
     * @param iterator            The actual response data
     * @return
     */
    void asyncCallback(int status, int transactionId, QObject* aContIter);

    /**
     * The signal emitted for the send
     * @since x.y
     * @param status               0 or Error code
     * @param transactionId        TransactionID of the Async operation
     * @param recipient            Message recipient
     * @return
     */
    void asyncCallback(int status, int transactionId, QString recipient);

public slots:
    void sendSMSMessageFinished(QSendMessage* message, int aErrorCode, int aTransactionId, QString recipient);

    void newMessageCallback(const QMessageId& eventId, const QMessageManager::NotificationFilterIdSet& filterSet);

protected:
    WRT::SecSession *iSecSession;

signals:
    void asyncComplete(int aErrorCode, int aTransactionId);
    void asyncGetListComplete(int aErrorCode, int aTransactionId);

private slots:
    void notifyGetListError(const int aErrorCode, int aTransactionId);

private:
    QList<QSendMessage*> iPendingMessages;
    QList<qint32> iGoingAsyncTransactions;
    QList<qint32> iGoingSendTransactions;

    QMessageManager m_manager;

    qint32 iNotifierTransactionId;
    bool iNotifierSet;
    };

#endif // QMESSAGINGSERVICE_H_

