/*
* ============================================================================
*  Name        : qmediaserviceinterface.h
*  Part of     : serviceframework / WRT
*  Description : Interface for media services
*  Version     : %version: 1 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef qmediaserviceinterface_h
#define qmediaserviceinterface_h

//User include
#include "servicebase.h"

static const char* KIMediaId = "com.nokia.IMedia/1.0";

// IMedia interface id
static const ServiceInterfaceInfo KIMedia =
{
    KIMediaId, 0
};

/**
 * Base class for Media service.
 * @lib mediaplugin.lib
 * @since x.y
 */
class IMedia : public IServiceBase
{
public:
    /**
     * Get the list of media files present in the device
     * --------------------------------------------------------------------------------------
     *  |Key                | Data type|Description
     *  --------------------------------------------------------------------------------------
     *  |transactionId      | int      |Unique transaction Id for async calls.
     *  |                   |          |
     *  --------------------------------------------------------------------------------------
     *  |matchPattern       | QMap     |Input filter criteria for getList.
     *  --------------------------------------------------------------------------------------
     *  |sortOrder          | int      |Sorting order: 0 - Ascending
     *  |                   |          |               1 - Descending
     *  ----------------------------------------------------------------------------------------
     * @param transactionId Unique transaction Id for async calls.
     * @param matchPattern  Input filter criteria.
     * @param sortOrder     Sorting order.
     * @return transaction ID in case of success or non-zero error code/message in case of failure.
     * @emits getListCallback it emits getListCallback signal to notify the result.
     * @since x.y
     */
    virtual QVariant getList(const int transactionId,
                                    const QMap<QString,QVariant>& matchPattern,
                                    const int sortOrder) = 0;

    /**
        * Creates the thumbnail of image or video media file.
        * @param transactionId Unique transaction id to identify async request
        * @param thumbnailInfo Thumbnail info
        * thumbnailInfo is a Map described as below:
        * -------------------------------------------------------------------------------------
        *  |Key                | Data type|Description
        *  --------------------------------------------------------------------------------------
        *  |uri                | QString  |uri of the image or video file.
        *  |                   |          |E.g file://C:/Data/Images/image.jpg.
        *  --------------------------------------------------------------------------------------
        *  |size               | QMap     |Map containing information about the required resolution
        *  |                   |          |of thumbnail to be generated. height and width are defined in this map
        * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * |transactionId       | int      |Unique Id identifying this request in case of success.
        * ---------------------------------------------------------------------------------
        * @emits getListCallback it emits getListCallback signal to notify the result.
        * @since x.y
        */
    virtual QVariant getThumbnail(const int transactionId,
                                    const QMap<QString,QVariant>& thumbnailInfo) = 0;

    /**
        * Adds a music file to the music collection database.
        * @param transactionId Unique transaction id to identify async request
        * @param uri Full path of the music file
        * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * |transactionId       | int      |Unique Id identifying this request in case of success.
        * ---------------------------------------------------------------------------------
        * @emits addMusicToCollectionCallback it emits addMusicToCollectionCallback signal to notify the result.
        * @since x.y
        */
    virtual QVariant addMusicToCollection(const int transactionId,
                                    const QString& uri) = 0;

    /**
        * Adds a streaming Uri.
        * @param uri Full path of the stram uri
        * @return  Map containing errorCode,errorMessage.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * @since x.y
        */
    virtual QVariant addStreamUri(const QString& uri) = 0;

    /**
        * Deletes a streaming Uri.
        * @param uri Full path of the stram uri
        * @return  Map containing errorCode,errorMessage.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * @since x.y
        */
    virtual QVariant deleteStreamUri(const QString& uri) = 0;

    /**
        * Refreshes database.
        * @param uri Full path of database file
        * @return  Map containing errorCode,errorMessage.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * @since x.y
        */
    virtual QVariant refreshMediaDb(const QString& uri) = 0;

    /**
     * Resizes the user provided image to given dimensions. Returns the filepath of the temporary
     * resized image file.
     * @param fileUri Full path of the image file to be resized
     * @param resizeOptions
     * @param height Height of the resized image
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |width               | int      |Specifies the width of the resized image in pixels.
     * ---------------------------------------------------------------------------------
     * |height              | int      |Specifies the width of the resized image in pixels.
     * --------------------------------------------------------------------------------
     * |[aspectRatioOption ]  | int      |Specifies if aspect ratio should be maintained.
     *                               |||If 0, the image is scaled freely, the aspect ratio is not preserved.
     *                               |||If 1, the image is scaled to a rectangle as large as possible inside the size
     *                               |||as given, preserving the aspect ratio.
     *                               |||If 2, the image is scaled to a rectangle as small as possible outside the size
     *                               |||as given, preserving the aspect ratio.
     *                               |||The default value is 0.
     * ---------------------------------------------------------------------------------
     * @param destFileUri Path of the resized image provided by user. If empty string is passed then resized image is stored
     * in temporary directory.
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      |error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------
     * |Uri                 | string   |The URI of the temporary resized image.
     * ---------------------------------------------------------------------------------
     * @since x.y
     */
    virtual QVariant resizeImage(QString fileUri, const QVariantMap& resizeOptions, QString destFileUri) = 0;

    /**
     * Cancel the subscribed async request.
     * @param transactionId unique number used to identify a previously invoked
     * async call.
     * @since x.y
     * @return 0 in case of success ; non-zero error code otherwise.
     */
    virtual QVariant cancel(const int transactionId) = 0;

};
Q_DECLARE_INTERFACE(IMedia, KIMediaId);

#endif //qmediaserviceinterface_h
