/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#ifndef qmediaservice_h
#define qmediaservice_h

#include <QObject>
#include <QMap>
#include <QVariant>
#include <QUrl>

#include "servicesecurityinterface.h"
#include "qmediaserviceinterface.h"


class AddMusicHandler;
class GetMediaListTask;
class GetThumbnailTask;
//Time conversion factor from milliseconds to Microseconds
const int KTimeConversion = 1000;


/**
 * Media provider class. Provides methods to get list of media objects present in
 * device as well as thumbnail associated with media objects.
 * @lib libmediaserviceplugin.so
 * @since x.y
 */
class MediaProvider : public QObject,
                         public IMedia,
                         public IServiceSecurity
{
    Q_OBJECT
    Q_INTERFACES(IMedia)

public:
    /**
     * Default constructor
     */
    MediaProvider();

    /**
     * Destructor
     */
    virtual ~MediaProvider();

    /**
     * setSecuritySession Setting a security session
     * @param secSession handle to the security session
     * @since x.y
     */
    void setSecuritySession(WRT::SecSession *secSession);

public slots:
   /**
     * Callback method for getList async request.
     * @param aTask task that emitted appropriate signalGetListCb. This task must be finished using aTask->finishTask() after emitting of signalGetListCb
     * @param aTransactionId Connect Id to match async request with the callback
     * @param aErrorCode Async request completion code
     * @param aList Array of returned media meta data
     * @emits getListCallback if transaction identified by aTransactionId is not cancelled
     * @since x.y
     */
    void handleGetListCallback(GetMediaListTask* aTask, qint32 aTransactionId, int aErrorCode,
                                            QVariantList aList);

    /**
     * Callback method for getThumbnail async request.
     * @param aUri Full path of the thumbnail
     * @param aErrorCode Async request completion code.
     * @param aTransactionId Connect Id to match async request with the callback
     * @emits getThumbnailCallback if transaction identified by aTransactionId is not cancelled.
     * @since x.y
     */
    void handleGetThumbnailCallback(GetThumbnailTask * aTask,QUrl aUri, int aErrorCode, int aTransactionId);

    /**
     * Callback method for addMusicToCollection async request.
     * @param aTransactionId Connect Id to match async request with the callback
     * @param aErrorCode Async request completion code.
     * @emits addMusicToCollectionCallback if transaction identified by aTransactionId is not cancelled.
     * @since x.y
     */
    void handleAddMusicToCollectionCallback(AddMusicHandler* aHandler,
                                            const int aTransactionId,
                                            const int aErrorCode);

private:

signals:
    /**
     * Emitted as signal whenever getList completes
     * data is obtained or an error occurs during retrieval.
     * @param transactionId uniquely identifies a getList async request.
     * @param errorCode error that has occured
     * @param mediaIdIter Media objects iterator in case error param is 0. The
     * media object contanis full information about the media object based on its type.
     * @return none
     * @since x.y
     */
    void getListCallback(int transactionId, int errorCode, QObject* mediaIdIter);

    /**
     * Emitted as signal whenever getThumbnail completes
     * data is obtained or an error occurs during retrieval.
     * @param transactionId uniquely identifies a getThumbnail async request.
     * @param errorCode error that has occured
     * @param thumbnailPath Full path of the thumbnail
     * @return none
     * @since x.y
     */
    void getThumbnailCallback(int transactionId, int errorCode, const QString thumbnailPath );

    /**
     * Emitted as signal whenever addMusicToCollection completes
     * data is obtained or an error occurs during retrieval.
     * @param transactionId uniquely identifies a addMusicToCollection async request.
     * @param errorCode error that has occured
     * @return none
     * @since x.y
     */
    void addMusicToCollectionCallback(int transactionId, int errorCode);

public slots:
    //Methods inherited from IMedia

    /**
     * Retrieves the Media objects iterator asynchronously. Emits getListCallback when request is completed
     * @param transactionId Unique transaction id to identify async request
     * @param matchPattern Input match pattern
     * @param sortOrder Input sort order.
     * matchPattern is a Map described as below:
     * -------------------------------------------------------------------------------------
     *  |Key                | Data type|Description
     *  --------------------------------------------------------------------------------------
     *  |type               | QString  |Media type the user is interested. E.g image,
     *  |                   |          |video, audio, streamUri or all.
     *  --------------------------------------------------------------------------------------
     *  |field              | QMap     |Map containing information on what field to perform
     *  |                   |          |filter and the value for that filter.
     *  --------------------------------------------------------------------------------------
     *  |dateRange          | QMap     |Map containing information of start and end date.
     *  |                   |          |Media object which are created within this range are
     *  |                   |GetPropertyDefObject          |returned.
     *  --------------------------------------------------------------------------------------
     *  |sortBy             | QString  |Field on which to sort the media objects.
     * -------------------------------------------------------------------------------------------
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      |error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------
     * |transactionId       | int      |Unique Id identifying this request in case of success.
     * ---------------------------------------------------------------------------------
     * @emits getListCallback it emits getListCallback signal to notify the result.
     * @since x.y
     */
    QVariant getList(const int transactionId,
                                    const QVariantMap& matchPattern,
                                    const int sortOrder);
    /**
        * Creates the thumbnail of image or video media file.
        * @param transactionId Unique transaction id to identify async request
        * @param thumbnailInfo Thumbnail info
        * thumbnailInfo is a Map described as below:
        * -------------------------------------------------------------------------------------
        *  |Key                | Data type|Description
        *  --------------------------------------------------------------------------------------
        *  |uri                | QString  |uri of the image or video file.
        *  |                   |          |E.g file:///home/user/MyDocs/image.jpg.
        *  --------------------------------------------------------------------------------------
        *  |size               | QMap     |Map containing information about the required resolution
        *  |                   |          |of thumbnail to be generated. height and width are defined in this map
        * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * |transactionId       | int      |Unique Id identifying this request in case of success.
        * ---------------------------------------------------------------------------------
        * @emits getListCallback it emits getListCallback signal to notify the result.
        * @since x.y
        */
    QVariant getThumbnail(const int transactionId,
                                    const QVariantMap& thumbnailInfo);

    /**
        * Adds a music file to the music collection database..
        * @param transactionId Unique transaction id to identify async request
        * @param uri Full path of the music file
        * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * |transactionId       | int      |Unique Id identifying this request in case of success.
        * ---------------------------------------------------------------------------------
        * @emits addMusicToCollectionCallback it emits addMusicToCollectionCallback signal to notify the result.
        * @since x.y
        */

    QVariant addMusicToCollection(const int transactionId,
                                    const QString& uri);

    /**
        * Adds a streaming Uri synchronously
        * @param uri Full path of the stram uri
        * @return  Map containing errorCode,errorMessage.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * @since x.y
        */
    QVariant addStreamUri(const QString& uri);

    /**
        * Deletes a streaming Uri synchronously
        * @param uri Full path of the stram uri
        * @return  Map containing errorCode,errorMessage.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * @since x.y
        */
    QVariant deleteStreamUri(const QString& uri);

    /**
        * Refreshes database. Currently empty implementation for Maemo platform
        * @param uri Full path of database file
        * @return  Map containing errorCode,errorMessage.Content of this map is given
        * below :
        * --------------------------------------------------------------------------------
        * |Key                 |DataType  |String
        * --------------------------------------------------------------------------------
        * |errorCode           | int      |error code in case of error, 0 otherwise.
        * ---------------------------------------------------------------------------------
        * |errorMessage        | string   |error description in engineering english.
        * --------------------------------------------------------------------------------
        * @since x.y
        */
    QVariant refreshMediaDb(const QString& uri);

    /**
     * Resizes the user provided image to given dimensions. Returns the filepath of the temporary
     * resized image file.
     * @param transactionId Unique transaction id to identify async request
     * @param fileUri Full path of the image file to be resized
     * @param height Height of the resized image
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |width               | int      |Specifies the width of the resized image in pixels.
     * ---------------------------------------------------------------------------------
     * |height              | int      |Specifies the width of the resized image in pixels.
     * --------------------------------------------------------------------------------
     * |aspectRatioOption   | int      |Specifies if aspect ratio should be maintained.
     *                               |||If 0, the image is scaled freely, the aspect ratio is not preserved.
     *                               |||If 1, the image is scaled to a rectangle as large as possible inside the size
     *                               |||as given, preserving the aspect ratio.
     *                               |||If 2, the image is scaled to a rectangle as small as possible outside the size
     *                               |||as given, preserving the aspect ratio.
     *                               |||The default value is 0.
     * ---------------------------------------------------------------------------------
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      |error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------
     * |transactionId       | int      |Unique Id identifying this request in case of success.
     * ---------------------------------------------------------------------------------
     * @since x.y
     */
    QVariant resizeImage(QString fileUri, const QVariantMap& resizeOptions, QString destFileUri);

    /**
     * Clear the previously subscribed async request.
     * @parama transcationId unique ID which corresponds to one active
     * async request.
     * @return  Map containing errorCode,errorMessage.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      |error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * ---------------------------------------------------------------------------------
     * @since x.y
     */
    QVariant cancel(const int transcationId);


private:
    //List of ongoing asynchronous transactions
    QList<qint32> iGoingAsyncTransactions;

    //Handle to the security session
    WRT::SecSession *securitySession;

};
#endif //qmediaservice_h
