/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

//#define LOG_TO_SYSLOG

#include <QtDebug>
#include <QObject>
#include <QMap>
#include <QVariant>
#include <QPixmap>
#include <QThreadPool>
#include <QEventLoop>

#include <glib-object.h>
#include <dbus/dbus-glib.h>

#include "secsession.h"
#include "addmusichandler.h"
#include "getthumbnailtask.h"
#include "msselect.h"
#include "qmediaservice.h"
#include "qmediakeyconstants.h"
#include "qmediaerrorcodes.h"
#include "qmediagetlistiterator.h"
#include "getmedialisttask.h"
#include "resizeimagetask.h"

#ifdef LOG_TO_SYSLOG
#include <syslog.h>
#endif


#ifdef LOG_TO_SYSLOG
void QMessageOutput(QtMsgType type, const char *msg)
{
    switch (type) {
        case QtDebugMsg:
            syslog(LOG_LOCAL6 | LOG_DEBUG, "MediaProvider (QDebug): %s", msg);
            break;
        case QtWarningMsg:
            syslog(LOG_LOCAL6 | LOG_WARNING, "MediaProvider (QWarning): %s", msg);
            break;
        case QtCriticalMsg:
            syslog(LOG_LOCAL6 | LOG_ERR, "MediaProvider (QCritical): %s", msg);
            break;
        case QtFatalMsg:
            syslog(LOG_LOCAL6 | LOG_CRIT, "MediaProvider (QFatal): %s", msg);
            abort();
    }
}
#endif

/**
 * class MediaProvider
 * Contains methods for getting media objects present in device
 *
 */

/**
 * Default constructor
 */
MediaProvider::MediaProvider()
{
    g_type_init();
    
    if (!g_thread_supported ())
    {
        g_thread_init (NULL);
    }
    
    dbus_g_thread_init ();
#ifdef LOG_TO_SYSLOG
    openlog("", LOG_PID|LOG_CONS, LOG_LOCAL6);
    qInstallMsgHandler(QMessageOutput);
#endif
    qDebug(__FUNCTION__);
}

/**
 * Destructor
 */
MediaProvider::~MediaProvider()
{
    qDebug(__FUNCTION__);
#ifdef LOG_TO_SYSLOG
    closelog();
#endif
}

void MediaProvider::setSecuritySession(WRT::SecSession *secSession)
{
    qDebug(__FUNCTION__);
    securitySession = secSession;
}

void MediaProvider::handleGetListCallback(GetMediaListTask* aTask, qint32 aTransactionId, int aErrorCode, QVariantList aList)
{

    if (iGoingAsyncTransactions.contains(aTransactionId))
    {

        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(aTransactionId));
        QMediaGetListIterator* mediaObjIterator = new QMediaGetListIterator(aList);
        QObject* retIter = static_cast<QObject*>(mediaObjIterator);
        emit getListCallback(aTransactionId, aErrorCode, retIter);
    }
    if(aTask)
    {
        aTask->finishTask();
    }
}

void MediaProvider::handleGetThumbnailCallback(GetThumbnailTask * aTask,QUrl aUri, int aErrorCode, int aTransactionId)
{
    qDebug() << Q_FUNC_INFO<<" id = "<<aTransactionId;
    if (iGoingAsyncTransactions.contains(aTransactionId))
    {
        qDebug() << "Emitting result signal with uri: " << aUri.toString();       

        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(aTransactionId));
        emit getThumbnailCallback(aTransactionId, aErrorCode, aUri.toString());
    }
    if(aTask)
    {
        aTask->finishTask();
    }
}

void MediaProvider::handleAddMusicToCollectionCallback(AddMusicHandler* aHandler, const int aTransactionId, const int aErrorCode)
{
    qDebug() << __PRETTY_FUNCTION__ << "TransactionId=" << aTransactionId << " Error=" << aErrorCode;

    if (iGoingAsyncTransactions.contains(aTransactionId))
    {
        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(aTransactionId));
        emit addMusicToCollectionCallback(aTransactionId, aErrorCode);
    }
    if(aHandler)
    {
        aHandler->finish();
    }
}


QVariant MediaProvider::getList(const int transactionId,
                                const QVariantMap& matchPattern,
                                const int sortOrder )
{
    qDebug() << __PRETTY_FUNCTION__ << transactionId;

    MediaFilter pattern;
    int errCode = NO_ERROR;
    QString errMsg;
    pattern.fillFilterData(matchPattern, sortOrder, errCode, errMsg);
    qDebug() << "Match Pattern = " << matchPattern;
    qDebug() << "Pattern data = " << pattern.startTime();
    iGoingAsyncTransactions.append(transactionId);
    qDebug() << "MediaProvider::getList -> starting task";
    // Run getList in a thread
    GetMediaListTask * task = new GetMediaListTask(transactionId, pattern);
    connect(task, SIGNAL(signalGetListCb(GetMediaListTask*, qint32, int, QVariantList)), this,
            SLOT(handleGetListCallback(GetMediaListTask*, qint32, int, QVariantList)));

    qDebug() << "MediaProvider::getList -> real starting";
    QThreadPool::globalInstance()->start(task);

    QVariantMap retVal;
    retVal[KErrCode] = errCode;
    retVal[KErrMsg] = errMsg;
    retVal[KTransId] = transactionId;

    return retVal;
}

QVariant MediaProvider::getThumbnail(const int transactionId,
                                const QVariantMap& thumbnailInfo)
{
    qDebug(__PRETTY_FUNCTION__);
    QVariantMap retVal;
    int errCode = NO_ERROR;
    QString errMsg;

    if(thumbnailInfo.contains(KUri))
    {
        // Check if is there any range for height and width
        int height = 0;
        int width = 0;
        QString quri = thumbnailInfo[KUri].toString();
        if(!quri.startsWith("file://"))
        {
            quri = "file://" + quri;
        }

        // Validating size map
        if(thumbnailInfo.contains(KSize))
        {
           QVariantMap sizeMap = thumbnailInfo[KSize].toMap();
           if(sizeMap.contains(KWidth))
                width = sizeMap[KWidth].toInt();
           if(sizeMap.contains(KHeight))
                height = sizeMap[KHeight].toInt();
        }
        iGoingAsyncTransactions.append(transactionId);

        // Run getThumbnail in a thread
        GetThumbnailTask * thumbtask = new GetThumbnailTask(transactionId, QUrl(quri), width, height);

#ifdef THUMBNAIL_THREAD_ENABLED
        connect(thumbtask, SIGNAL(signalGetThumbnailCb(QUrl, int, int)),
                this, SLOT(handleGetThumbnailCallback(QUrl, int, int)));
        QThreadPool::globalInstance()->start(thumbtask);
#else
        connect(thumbtask, SIGNAL(signalGetThumbnailCb(GetThumbnailTask *,QUrl, int, int)),
                this, SLOT(handleGetThumbnailCallback(GetThumbnailTask *,QUrl, int, int)), Qt::QueuedConnection);
        thumbtask->asyncRun();
#endif
    }
    else
    {
        errCode = MISSING_ARG_ERR;
        errMsg = "Media:getThumbnail:Missing input uri";
    }

    retVal[KTransId] = transactionId;
    retVal[KErrCode] = errCode;
    retVal[KErrMsg] = errMsg;
    return retVal;
}

QVariant MediaProvider::addMusicToCollection(const int transactionId, const QString& uri)
    {
        QVariantMap retVal;
        QStringList capList;
        capList.append("media.write");
        if (!securitySession->isAllowed(capList)) {
            retVal[KErrCode] = NOT_ALLOWED_ERR;
            retVal[KErrMsg] = "Permission Denied";
            return retVal;
        }

        int errCode = NO_ERROR;
        QString errMsg="";
        AddMusicHandler* handler = new AddMusicHandler(transactionId, uri);
        connect(handler, SIGNAL(musicAdded(AddMusicHandler*,int,int)), this, SLOT(handleAddMusicToCollectionCallback(AddMusicHandler*,int,int)),
                 Qt::QueuedConnection);
        handler->start(errCode, errMsg);

        if (errCode == NO_ERROR)
        {
            iGoingAsyncTransactions.append(transactionId);
        }
        else
        {
            handler->finish();
        }
        retVal[KErrCode] = errCode;
        retVal[KErrMsg] = errMsg;
        retVal[KTransId] = transactionId;
        return retVal;
    }

QVariant MediaProvider::addStreamUri(const QString& uri)
{
    QVariantMap retVal;
    QUrl mediaStream(uri);
    if (mediaStream.isValid() && (mediaStream.scheme() == "rtsp" || mediaStream.scheme() == "rtspu"))
    {
         MSselect* select=new MSselect();
         select->addStream(uri);
          delete select;

        retVal[KErrCode] = NO_ERROR;
        retVal[KErrMsg] = "Success";
        qDebug() << "MediaProvider::addStreamUri. Success";
    }
    else
    {
        retVal[KErrCode] = INVALID_ARG_ERR;
        retVal[KErrMsg] = "Operation failed";
        qDebug() << "MediaProvider::addStreamUri. Failed";
    }
    return retVal;
}

QVariant MediaProvider::deleteStreamUri(const QString& uri)
{
    QVariantMap retVal;
    QUrl mediaStream(uri);
    if (mediaStream.isValid() && (mediaStream.scheme() == "rtsp" || mediaStream.scheme() == "rtspu"))
    {
        MSselect* select=new MSselect();
        if(select->checkExistsStream(uri))
        {
            qDebug() << "MediaProvider::deleteStreamUri: deleting";
            select->deleteStream(uri);
            retVal[KErrCode] = NO_ERROR;
            retVal[KErrMsg] = "Success";
        }
        else
        {
            qDebug() << "MediaProvider::deleteStreamUri: URI not found";
            retVal[KErrCode] = DATA_NOT_FOUND_ERR;
            retVal[KErrMsg] = "Operation failed";
        }
        delete select;
    }
    else
    {

        retVal[KErrCode] = INVALID_ARG_ERR;
        retVal[KErrMsg] = "Operation failed";
    }
    return retVal;
}

//on maemo functionality of refreshMediaDb is realized by tracker
QVariant MediaProvider::refreshMediaDb(const QString& /*uri*/)
    {
    qDebug(__PRETTY_FUNCTION__);
    QVariantMap retVal;    
    retVal[KErrCode] = NO_ERROR;
    retVal[KErrMsg] = "Success";
    return retVal;
    }

QVariant MediaProvider::resizeImage(QString fileUri, const QVariantMap& resizeOptions, QString destFileUri)
{
    QVariantMap retVal;
    int err = NO_ERROR;
    int width = resizeOptions.value(KWidth).toInt();
    int height = resizeOptions.value(KHeight).toInt();
    int aspectRatioMode = Qt::IgnoreAspectRatio;
    if (resizeOptions.contains(KAspectRatioOption))
    {
        int aspectRatio = resizeOptions.value(KAspectRatioOption).toInt();
        if (aspectRatio == 1)
        {
            aspectRatioMode = Qt::KeepAspectRatio;
        }
        else if (aspectRatio == 2)
        {
            aspectRatioMode = Qt::KeepAspectRatioByExpanding;
        }
    }

    qDebug() << "MediaProvider::resizeImage. path:" << fileUri << "Width:" <<width <<"Height:" <<height << "Keep aspect ratio:" <<aspectRatioMode;
    QUrl resultUrl;
    if (err == NO_ERROR)
    {
        ResizeImageTask * resizetask = new ResizeImageTask(0, QUrl(fileUri), width, height, aspectRatioMode, destFileUri);
        resizetask->run(true);
        resultUrl = resizetask->getResultUri();
        err = resizetask->getResultError();
        resizetask->finishTask();
    }
    retVal[KErrCode] = err;
    retVal[KReturnValue] = resultUrl;
    return retVal;
}

QVariant MediaProvider::cancel(const int transactionId)
{

    QVariantMap retVal;
    retVal[KTransId] = transactionId;

    if (iGoingAsyncTransactions.contains(transactionId))
    {
        // Remove request from the array
        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(transactionId));
        retVal[KErrCode] = NO_ERROR;
        retVal[KErrMsg] = "Success";
    }
    else
    {
        retVal[KErrCode] = DATA_NOT_FOUND_ERR;
        retVal[KErrMsg] = "Not found";
    }
    return retVal;
}



