/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef __MEDIA_FILTER_H__
#define __MEDIA_FILTER_H__

#include <QObject>
#include <QVariant>
#include <QMap>
#include <QString>
#include <QDateTime>


typedef unsigned long long int64;

class MediaFilter
{
public:
    MediaFilter(): iFilterFlags (0), iSortFlags(0), iSortMethod(0), iMediaType(0) {}
    MediaFilter (const MediaFilter& a): iFilterFlags(a.iFilterFlags), iFilterValue(a.iFilterValue),
        iSortFlags (a.iSortFlags), iSortMethod(a.iSortMethod), iMediaType(a.iMediaType), iStartTime(a.iStartTime), iEndTime(a.iEndTime) {}

   /**
     * Fills the matchObj with the data provided by matchPattern
     * @param aMatchPattern Structure representing match pattern data
     * @param aSortOrder Sort order (KSortAscending or KSortDescending)
     * @param aErrorCode on return, contains parsing error code
     * @param aErrorMessage On return, contains error message
     * @return true if match pattern is valid
     * @since x.y
     */
    bool fillFilterData(const QVariantMap& aMatchPattern,
            const int aSortOrder, int& aErrorCode, QString& aErrorMessage);

public:
    int64 filterFlags() const {return iFilterFlags;}
    QString filterValue() const {return iFilterValue;}
    int64 sortFlags() const {return iSortFlags;}
    int sortMethod() const {return iSortMethod;}
    QDateTime startTime() const {return iStartTime;}
    QDateTime endTime() const {return iEndTime;}
    int64 mediaType() const {return iMediaType;}

private:
   /**
     * Checks if 'filter' field is valid. Fills the matchObj with the 'filter' data if valid
     * @param aFilterField Data representing QVariantMap with 'filter' pattern data.
     * @param aErrorCode on return, contains parsing error code
     * @param aErrorMessage On return, contains error message
     * @return true if 'filter' field is valid
     * @since x.y
     */
    bool validateFilterField(const QVariant& aFilterField,
                                            int& aErrorCode,
                                            QString& aErrorMessage);

   /**
     * Checks if 'sort by' field is valid. Fills the matchObj with the 'sort by' data if valid
     * @param aSortBy Data representing QString with 'sort' pattern data.
     * @param aErrorCode on return, contains parsing error code
     * @param aErrorMessage On return, contains error message
     * @return true if 'sort by' field is valid
     * @since x.y
     */
    bool validateSortbyField(const QVariant& aSortBy,
                                            int& aErrorCode,
                                            QString& aErrorMessage);

   /**
     * Checks if 'date range' field is valid. Fills the matchObj with the 'date range' data if valid
     * @param aDateRange Data representing QString with 'date range' pattern data.
     * @param aErrorCode on return, contains parsing error code
     * @param aErrorMessage On return, contains error message
     * @return true if 'sort by' field is valid
     * @since x.y
     */
    bool validateDateRangeField(const QVariant& aDateRange,
                                            int& aErrorCode,
                                            QString& aErrorMessage);

   /**
     * Checks if 'type' field is valid. Fills the matchObj with the 'type' data if valid
     * @param mediaType Data representing QString with 'media type' pattern data.
     * @param aErrorCode on return, contains parsing error code
     * @param aErrorMessage On return, contains error message
     * @return true if 'media type' field is valid
     * @since x.y
     */
    bool validateMediaType(const QVariant& mediaType,
                                            int& aErrorCode,
                                            QString& aErrorMessage);


   /**
     * Gets the value flags by string representation using mediaFieldList data
     * @param aStringValue String value representation
     * @since x.y
     */
    static int64 getHashValue(const QString& aStringValue);

private:
    int64 iFilterFlags;
    QString iFilterValue;

    int64 iSortFlags;
    int iSortMethod; //KSortAscending or KSortDescending

    int64 iMediaType;

    QDateTime iStartTime;
    QDateTime iEndTime;
};

#endif
