/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QUrl>
#include <QFile>
#include <QDir>

#include <gio/gio.h>

#include "mediacontenthelper.h"
#include "qmediaerrorcodes.h"
#include "qmediakeyconstants.h"

int MediaContentHelper::getMimeType(const QString& aFileUri, QString& aMimeTypeByContent, QString& aMimeTypeByExtension)
{
    qDebug() << Q_FUNC_INFO << aFileUri;
    qDebug() <<"MediaContentHelper::getMimeType 111";
    QUrl mediaUri(aFileUri);

    QString uriFile = mediaUri.toLocalFile();
    if (!QFile::exists(uriFile))
    {
        qDebug() << "File does not exist " << uriFile;
        return URI_NOT_FOUND_ERR;
    }
    QDir dir (uriFile);
    if (dir.exists())
    {
        qDebug() << "URL is a directory" << uriFile;
        return NOT_SUPPORTED_ERR;
    }

    //get mime type
    QString content_type_by_content;
    QString mime_type_by_content;

    QString content_type_by_ext;
    QString mime_type_by_ext;

    QFile file (uriFile);
    file.open(QFile::ReadOnly);
    QByteArray beginning = file.read(MAX_LENGHT_DET); //the maximum length needed to determine content type
    file.close();
    gboolean uncertain = FALSE;
    char* cntType = g_content_type_guess(NULL, (const guchar*) beginning.constData(), beginning.size(), &uncertain);
    qDebug() << QString(cntType) << " Uncertain:" << uncertain;
    content_type_by_content = cntType;
    g_free (cntType);

    //Try to get standard content type
    GFile* gfl = g_file_new_for_uri(mediaUri.toString().toLocal8Bit().constData());
    GError* typeError = NULL;

    if (!gfl)
    {
        return NOT_SUPPORTED_ERR;
    }

    GFileInfo* gfi = g_file_query_info(gfl, G_FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE, G_FILE_QUERY_INFO_NONE, NULL, &typeError);

    if (gfi)
    {
        if (typeError == NULL)
        {
            content_type_by_ext = g_file_info_get_content_type(gfi);
        }
        else
        {
            g_error_free(typeError);
        }
        g_object_unref(gfi);
    }
    else
    {
        return NOT_SUPPORTED_ERR;
    }
    g_object_unref(gfl);

    if (!content_type_by_content.isEmpty())
    {
        char* mt = NULL;
        mt = g_content_type_get_mime_type (content_type_by_content.toLocal8Bit().constData());
        if (mt)
        {
            mime_type_by_content = mt;
            g_free (mt);
        }
        else
        {
            mime_type_by_content = content_type_by_content;
        }
    }
    if (!content_type_by_ext.isEmpty())
    {
        char* mt = NULL;
        mt = g_content_type_get_mime_type (content_type_by_ext.toLocal8Bit().constData());
        if (mt)
        {
            mime_type_by_ext = mt;
            g_free (mt);
        }
        else
        {
            mime_type_by_ext = content_type_by_ext;
        }
    }

    qDebug() << Q_FUNC_INFO << "mime_type_by_content=" << mime_type_by_content << "; mime_type_by_ext=" << mime_type_by_ext << uncertain;
    aMimeTypeByExtension = mime_type_by_ext;
    if (uncertain)
    {
        aMimeTypeByContent = mime_type_by_ext;
    }
    else
    {
        aMimeTypeByContent = mime_type_by_content;
    }
    return NO_ERROR;
}


