/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QMapIterator>
#include <QMap>
#include <QVariant>
#include <QFile>
#include <QDir>
#include <QPixmap>
#include <QByteArray>

#include <gio/gio.h>



#include "getthumbnailtask.h"
#include "qmediaerrorcodes.h"
#include "resizeimagetask.h"

GetThumbnailTask::GetThumbnailTask(int aTransactionId, QUrl aUrl, int aWidth, int aHeight)
    : iTransactionId(aTransactionId),
    iSourceUri(aUrl),
    iWidth(aWidth),
    iHeight(aHeight)
{
    connect(this, SIGNAL(asyncRunSignal()), this, SLOT(doRunSlot()), Qt::QueuedConnection);
    iThumbnailer = new ThumbnailerFetch();
    connect(iThumbnailer, SIGNAL(thumbnail(QUrl, QUrl,  QString)),
            this, SLOT(thumbnailReady(QUrl, QUrl,  QString)), Qt::QueuedConnection);
    connect(iThumbnailer, SIGNAL(error(QString, QUrl)),
            this, SLOT(handleThumbnailError(QString, QUrl)), Qt::QueuedConnection);
}

GetThumbnailTask::~GetThumbnailTask()
{
    qDebug() << Q_FUNC_INFO;
    if(iThumbnailer)
    {
        delete iThumbnailer;
        iThumbnailer = NULL;
    }

}
void GetThumbnailTask::finishTask()
{
    delete this;
}

void GetThumbnailTask::asyncRun()
{
    emit asyncRunSignal();
}

void GetThumbnailTask::doRunSlot()
{
    run();
}

void GetThumbnailTask::run()
{
    g_type_init();
    qDebug() << "GetThumbnailTask::run(): " << iSourceUri;


    if (!iSourceUri.isValid())
    {
        qDebug() << "____GetThumbnailTask::run() iSourceUri is not valid";
        emit signalGetThumbnailCb(this,QUrl(), INVALID_URI_ERR, iTransactionId);
        return;
    }
 qDebug() << "____ after if (!iSourceUri.isValid())";
    QString uriFile = iSourceUri.toLocalFile();
    if (!QFile::exists(uriFile))
    {
        qDebug() << "File does not exist " << uriFile;
        emit signalGetThumbnailCb(this,QUrl(), URI_NOT_FOUND_ERR, iTransactionId);
        return;
    }
    qDebug() << "____ after if (!QFile::exists(uriFile))";
    QDir dir (uriFile);
    if (dir.exists())
    {
        qDebug() << "URL is a directory" << uriFile;
        emit signalGetThumbnailCb(this,QUrl(), NOT_SUPPORTED_ERR, iTransactionId);
        return;
    }

    qDebug() << "____ after if (dir.exists())";
    //get mime type
    QString content_type;
    QString mime_type;

    QFile file (uriFile);
    file.open(QFile::ReadOnly);
    qDebug() << "____ after file open";
    QByteArray beginning = file.read(4096); //the maximum length needed to determine content type
    //    qDebug() << beginning.constData();
    file.close();
    qDebug() << "____ after file close";
    gboolean uncertain = FALSE;
    char* cntType = g_content_type_guess(NULL, (const guchar*) beginning.constData(), beginning.size(), &uncertain);
    qDebug() << QString(cntType) << " Uncertain:" << uncertain;
    if (uncertain)
    {
        //result is uncertain. Try to get standard content type
        GFile* gfl = g_file_new_for_uri(iSourceUri.toString().toLocal8Bit().constData());
        GError* typeError = NULL;
        GFileInfo* gfi = g_file_query_info(gfl, G_FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE, G_FILE_QUERY_INFO_NONE, NULL, &typeError);

        if (gfi)
        {
            if (typeError == NULL)
            {
                content_type = g_file_info_get_content_type(gfi);
            }
            else
            {
                g_error_free(typeError);
            }
            g_object_unref(gfi);
        }

        g_object_unref(gfl);
    }
    else
    {
        content_type = cntType;
    }
    g_free (cntType);

    if (!content_type.isEmpty())
    {
        char* mt = NULL;
        mt = g_content_type_get_mime_type (content_type.toLocal8Bit().constData());
        if (mt)
        {
            mime_type = mt;
            g_free (mt);
        }
        else
        {
            mime_type = content_type;
        }
    }

    //    content_type="image/bmp";
    qDebug() << "Content type is:" << content_type << ". Mime type is:" << mime_type;

    if (mime_type.isEmpty())
    {
        emit signalGetThumbnailCb(this,QUrl(), INVALID_URI_ERR, iTransactionId);
        return;
    }

    if (!(mime_type.startsWith("image") || mime_type.startsWith("video")))
    {
        emit signalGetThumbnailCb(this,QUrl(), NOT_SUPPORTED_ERR, iTransactionId);
        return;
    }
    QList<QUrl> urlList;
    urlList.append(iSourceUri);

    QStringList mimetypes;
    mimetypes.append(mime_type);
    qDebug() << "requesting thumbnail";


#ifdef THUMBNAIL_THREAD_ENABLED
    iMutex2.lock(); //this is for use cases when 'thumbnail' is emitted during iThumbnailer->request execution
    iThumbnailer->request(urlList, mimetypes, true);
    qDebug() << "locking mutex";
    iMutex.lock();
    iMutex2.unlock();
    if (!iWait.wait(&iMutex, 10000)) //wait 3 seconds
    {
        qDebug() << "GetThumbnailTask::run() - TIMEOUT: " << iSourceUri;
        handleThumbnailError("Error", iSourceUri);
    }

    iMutex.unlock();
#else

    iThumbnailer->request(urlList, mimetypes,iWidth,iHeight, needScaling());

#endif
    qDebug() << "END GetThumbnailTask::run(): " << iSourceUri;
}

void GetThumbnailTask::thumbnailReady(QUrl resourceUri, QUrl thumbnailUri,  QString /*flavor*/)
{
    qDebug() << Q_FUNC_INFO;
    if (resourceUri == iSourceUri)
    {

#ifdef THUMBNAIL_THREAD_ENABLED
        iMutex2.lock();
#endif
        QFileInfo fi(thumbnailUri.path());
        if(!fi.exists())
        {
            qDebug() << "GetThumbnailTask::thumbnailReady file: "<<thumbnailUri.path()<< "not exists";
        }

        if (thumbnailUri.isValid() && fi.exists())
        {
            //rescale image now
            int ratioMode = Qt::KeepAspectRatio;
            if (!needScaling())
            {
                qDebug() << "width and height are 0 - returning default thumbnail";
                emit signalGetThumbnailCb(this,thumbnailUri, NO_ERROR, iTransactionId);
                return;
            }
            else if (iWidth == 0)
            {
                qDebug() << "width is 0 - expanding thumbnail";
                ratioMode = Qt::KeepAspectRatioByExpanding;
                iWidth = 1; //just allow resizing because 'iWidth = 0' will return empty pixmap
            }
            else if (iHeight == 0)
            {
                qDebug() << "height is 0 - expanding thumbnail";
                ratioMode = Qt::KeepAspectRatioByExpanding;
                iHeight = 1; //just allow resizing because 'iHeight = 0' will return empty pixmap
            }

            ResizeImageTask * resizetask = new ResizeImageTask(iTransactionId, thumbnailUri, iWidth, iHeight, ratioMode, "");
            connect(resizetask, SIGNAL(signalResizeImageCb(ResizeImageTask*, QString, int, int)),
                    this, SLOT(slotResizeImage(ResizeImageTask*, QString, int, int)));
            QPixmap pixmap;
            pixmap.load(thumbnailUri.path());
            if(!pixmap.isNull())
            {
                qDebug() << "GetThumbnailTask::thumbnailReady runFromScaling";
                resizetask->runFromScaling(pixmap);
            }
            else
            {
                qDebug() << "GetThumbnailTask::thumbnailReady pixmap is null";
            }

        }
        else
        {
            // do we need to handle this possibility...

            emit signalGetThumbnailCb(this,QUrl(), INVALID_URI_ERR, iTransactionId);
        }

#ifdef THUMBNAIL_THREAD_ENABLED
        iMutex2.unlock();
        iWait.wakeAll();
#endif
    }
}

void GetThumbnailTask::slotResizeImage(ResizeImageTask* aTask, QString aUri, int aErrorCode, int aTransactionId)
{
    qDebug() << "GetThumbnailTask::slotResizeImage" << aUri << aErrorCode << aTransactionId;
    emit signalGetThumbnailCb(this,aUri, aErrorCode, aTransactionId);
    if(aTask)
    {
        aTask->finishTask();
    }
    qDebug() << "End GetThumbnailTask::slotResizeImage";
}


void GetThumbnailTask::handleThumbnailError(QString error, QUrl resourceUri)
{
    qDebug() << Q_FUNC_INFO;
    if (resourceUri == iSourceUri)
    {
        qDebug() << Q_FUNC_INFO << " " + error;

#ifdef THUMBNAIL_THREAD_ENABLED
        iMutex2.lock();
        emit signalGetThumbnailCb(this,QUrl(), NOT_SUPPORTED_ERR, iTransactionId);

        iMutex2.unlock();
        iWait.wakeAll();
#else
        emit signalGetThumbnailCb(this,QUrl(), NOT_SUPPORTED_ERR, iTransactionId);
#endif
    }
}

bool GetThumbnailTask::needScaling() const
{
    return iWidth != 0 || iHeight != 0;
}

