/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


//INCLUDES
#include <QDebug>
#include <QFSFileEngine>
#include "secsession.h"
#include "filesystemservice.h"
#include "fserrorcodes.h"
#include "fsutilities.h"
#include "fsconstants.h"

#include "fsworker.h"
#include "fsioworker.h"

// --------------------------------------------------------------------
// FileSystemService::FileSystemService()
// Constructor.
// --------------------------------------------------------------------
FileSystemService::FileSystemService()\
: m_NotifyingWorker(NULL), _sec_session(NULL)
                {
                LOG("FileSystemService::Constructor\n");
                m_openFileList.clear();
                }

// --------------------------------------------------------------------
// FileSystemService::~FileSystemService()
// Destructor.
// --------------------------------------------------------------------
FileSystemService::~FileSystemService()
    {
    LOG("FileSystemService::Destructor\n");

    LOG("Checking for open file handles\n");
    QFile* file;
    for (int i = 0; i < m_openFileList.count(); i++)
        {
        file = m_openFileList[i].fileHandle;
        if ( file!= NULL)
            {
            LOG(QString("Handle not closed: %1\n").arg(i));
            file = m_openFileList[i].fileHandle;
            file->close();
            file = NULL;
            }
        }
    m_openFileList.clear();

    if (m_NotifyingWorker)
        {
        LOG("~FileSystemService, notificaion worker is active, deleting it\n");
        delete m_NotifyingWorker;
        }

    //we need to chk for threads which are alive in m_threadList
    //and then terminate & delete them.
    //but even if something is alive if we try to terminate it
    //then its craashing on hardware..
    //for that reason we have not done it currently. :-)

    LOG("End FileSystemService::Destructor\n");
    }

// --------------------------------------------------------------------
// FileSystemService::setSecuritySession()
// Provides implementation for setSecuritySession derived
// from IServiceSecurity. Checks if the consumer is a Trusted
// or untrusted widget and sets the mountTable entry accordingly
// --------------------------------------------------------------------
void FileSystemService::setSecuritySession(WRT::SecSession *secSession)
    {
    LOG("FileSystemService::setSecuritySession\n");
    _sec_session = secSession;
    QStringList capsList;
    capsList.clear();

    capsList.append("io.file.unrestricted");

    if (!_sec_session->isAllowed(capsList))
        {
        //Untrusted, 3rd party widget
        LOG("Third party widget, partial access\n");
        mountTable.insert((FILE_PROTOCOL + NOKIA_USER), "false");
        mount("Image");
        mount("Audio");
        mount("Video");
        }
    else
        {
        //Trusted widget
        LOG("Trusted widget, full access\n");
        mountTable.insert((FILE_PROTOCOL + NOKIA_USER), "true");
        }
    }

// --------------------------------------------------------------------
// FileSystemService::mount()
// Used to set the mapping of virtual mount points to actual paths on
// the file system. These virtual mount points can be used by all the
// API calls of Filesystem.
// --------------------------------------------------------------------
QVariant FileSystemService::mount(QString defaultKey)
    {
    LOG("FileSystemService::mount\n");
    int dummyTId = -1;
    QMap<QString, QVariant> retMap;
    QList<QVariant> defaultPaths;

    //Call getDefaultPaths to obtain the mapping to specified virtual path
    QVariant ret = (getDefaultPath(defaultKey, dummyTId)).toMap().value("data");
    LOG("After getting default paths\n");
    defaultPaths = ret.toList();
    int len = defaultPaths.length();
    LOG(QString("Number of default paths: %1 \n")
            .arg(len));

    for (int i = 0; i < len; i++)
        {
        QMap<QString, QVariant> defaultPath = defaultPaths[i].toMap();
        LOG(QString("Obtained Default path: %1 \n").arg(defaultPath[MOUNTPOINT_URI].toString()));
        if (defaultPath[MOUNTPOINT_ISDEFAULT].toBool() == true)
            {
			//It is the default one out of the returned paths
            QString finalDefaultPath= defaultPath[MOUNTPOINT_URI].toString();
            mountTable.insert(defaultKey.toLower(), finalDefaultPath.toLower());
            LOG(QString("Appended Default path: %1 \n")
                                .arg(finalDefaultPath));
            }
        }
    return retMap;
    }


// --------------------------------------------------------------------
// FileSystemService::ExecFileSystemCall()
// Calls the FileSytem worker in Sync or Async mode
// --------------------------------------------------------------------
void FileSystemService::ExecFileSystemCall(FileSystemWorker* thread, int callBackId)
    {
    LOG("Enter ExecFileSystemCall\n");

    bool ret = connect(thread, SIGNAL(FileSystemWorkerSignal(int , int , QMap<QString,QVariant>)),
            this,  SLOT(ServiceSlot(int , int , QMap<QString,QVariant>))
    );
    LOG(QString("connect returned: %1\n").arg(ret));

    // asynchronous call
    if (callBackId != SYNC_CALL)
        {
        LOG("ASYNC Call\n");
        m_threadList.insert(callBackId,thread);
        thread->start();
        LOG("After calling thread start\n");
        }
    // synchronous call
    else
        {
        LOG("SYNC Call\n");
        thread->run();
        LOG("Sync call execution done\n");
        delete thread;
        }
    LOG("Exit ExecFileSystemCall\n");
    }

// --------------------------------------------------------------------
// FileSystemService::ExecFileIOCall()
// Calls the File IO worker in Sync or Async mode
// --------------------------------------------------------------------
void FileSystemService::ExecFileIOCall(FileIOWorker* thread, int callBackId)
    {
    LOG("Enter ExecFileIOCall\n");

    connect(thread, SIGNAL(FileIOWorkerSignal(int , int , QMap<QString,QVariant>)),
            this,  SLOT(ServiceSlot(int , int , QMap<QString,QVariant>))
    );

    // asynchronous call
    if (callBackId != SYNC_CALL)
        {
        LOG("ASYNC Call\n");
        m_threadList.insert(callBackId,thread);
        thread->start();
        LOG("After calling thread start\n");
        }
    // synchronous call
    else
        {
        LOG("SYNC Call\n");
        thread->run();
        LOG("Sync call execution done\n");
        delete thread;
        }
    LOG("Exit ExecFileIOCall\n");
    }

// --------------------------------------------------------------------
// FileSystemService::ServiceSlot()
// Slot to receive from worker & notify JS the outcome
// --------------------------------------------------------------------
void FileSystemService::ServiceSlot(int task, int cbId, QMap<QString,QVariant> data)
    {
    LOG(QString("ServiceSlot: %1\n").arg(cbId));
    m_data.clear();

    if (cbId == SYNC_CALL)
        {
        LOG("Sync call, setting data\n");
        m_data = data;
        }
    else
        {
        LOG(QString("ASync call: %1\n").arg(cbId));
        //we need special handling of this bcoz
        //this will not be present in thread list
        if ( (task == NOTIFYMOUNTEVENTS_METHOD) && m_NotifyingWorker)
            {
            LOG("Emitting signal for drive notification\n");
            emit notifyMountEventsSignal(NOTIFICATION_DUMMY_ID, data);
            return;
            }

        QThread* thread = m_threadList.value(cbId,NULL);
        if ( thread != NULL )
            {
            LOG(QString("cbId found in thread list, emitting signal: %1\n").arg(cbId));

            switch (task)
                {
                case CREATEDIR_METHOD:
                    emit createDirSignal(cbId, data);
                    break;
                case REMOVE_METHOD:
                    emit removeSignal(cbId, data);
                    break;
                case RENAME_METHOD:
                    emit renameSignal(cbId, data);
                    break;
                case GETDIRCONTENTS_METHOD:
                    emit getDirContentsSignal(cbId, data);
                    break;
                case COPY_METHOD:
                    emit copySignal(cbId, data);
                    break;
                case MOVE_METHOD:
                    emit moveSignal(cbId, data);
                    break;
                case OPENFILE_METHOD:
                    emit openFileSignal(cbId, data);
                    break;
                case CLOSE_METHOD:
                    emit closeSignal(cbId, data);
                    break;
                case READ_METHOD:
                    emit readSignal(cbId, data);
                    break;
                case READLINE_METHOD:
                    emit readLineSignal(cbId, data);
                    break;
                case READBASE64_METHOD:
                    emit readBase64Signal(cbId, data);
                    break;
                case WRITE_METHOD:
                    emit writeSignal(cbId, data);
                    break;
                case WRITELINE_METHOD:
                    emit writeLineSignal(cbId, data);
                    break;
                case WRITEBASE64_METHOD:
                    emit writeBase64Signal(cbId, data);
                    break;
                case FLUSH_METHOD:
                    emit flushSignal(cbId, data);
                    break;
                case GETMOUNTPOINTS_METHOD:
                    emit getMountPointsSignal(cbId, data);
                    break;
                case GETDEFAULTPATH_METHOD:
                    emit getDefaultPathSignal(cbId, data);
                    break;
                case GETELEMENTINFO_METHOD:
                    emit getElementInfoSignal(cbId, data);
                    break;
                case SEARCHMATCHOBJECT_METHOD:
                    emit searchMatchObjectSignal(cbId, data);
                    break;
                default:
                    LOG("Invalid signal");
                    //do nothing
                }


            //Begin Search handling
            if (task == SEARCH_METHOD)
                {
                emit searchSignal(cbId, data);

                QMap<QString, QVariant> searchMap = data[DATA_STR].toMap();
                if ( (data[ERRORCODE_STR].toInt() == SUCCESS) &&
                        ((searchMap.value(STATE_STR)).toInt() == PARTIAL)
                )
                    {
                    LOG("search is successful, but not completed, so dont delete the thread\n");
                    return;
                    }
                }
            //End Search
            deleteThreadPointer(cbId);
            }
        else
            {
            LOG(QString("Did not emit signal to JS as it is not in list: %1\n").arg(cbId));
            }
        }
    }

// --------------------------------------------------------------------
// FileSystemService::deleteThreadPointer()
// deletes the thread point stored
// --------------------------------------------------------------------
void FileSystemService::deleteThreadPointer(int cbId)
    {
    LOG(QString("Enter deleteThreadPointer: %1\n").arg(cbId));
    if ( m_threadList.value(cbId, NULL) != NULL )
        {
        LOG(QString("FOUND thread pointer in the map: %1\n").arg(cbId));
        QThread* thread = m_threadList.take(cbId);
        thread->disconnect();
        if (thread->isRunning())
            {
            LOG(QString("thread is still running, so terminating: %1\n").arg(cbId));
            thread->quit();
            LOG(QString("After thread quit: %1\n").arg(cbId));
            bool waitStatus = thread->wait(1000);            
            LOG(QString("Wait status: %1\n").arg(waitStatus));
            }
        LOG(QString("deleting thread pointer: %1\n").arg(cbId));
        delete thread;
        }
    else
        {
        LOG(QString("NOT FOUND thread pointer in the map: %1\n").arg(cbId));
        }
    LOG(QString("Exit deleteThreadPointer: %1\n").arg(cbId));
    }

// --------------------------------------------------------------------
// FileSystemService::createDir()
// Creates the user sepecified path
// This method supports both synchronous & asynchronous.
// --------------------------------------------------------------------
QVariant FileSystemService::createDir(const QString& path, int callBackId)
    {
    LOG("*************ENTER createDir******************\n");
    LOG(QString("Uri : %1\n").arg(path));
    LOG(QString("Call back id: %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(CREATEDIR_METHOD, path,
            callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT createDir******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::remove()
// deletes the user sepecified path
// This method supports both synchronous & asynchronous.
// --------------------------------------------------------------------
QVariant FileSystemService::remove(const QString& path, bool recursive, int callBackId)
    {
    LOG("*************ENTER remove******************\n");
    LOG(QString("path : %1\n").arg(path));
    LOG(QString("recursive : %1\n").arg(recursive));
    LOG(QString("Call back id: %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(REMOVE_METHOD, path,
            recursive, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT remove******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }

// --------------------------------------------------------------------
// FileSystemService::remove()
// removes the user provided path
// --------------------------------------------------------------------
QVariant FileSystemService::rename(const QString& srcUri, const QString& newName, int callBackId)
    {
    LOG("*************ENTER rename******************\n");
    LOG(QString("srcUri : %1\n").arg(srcUri));
    LOG(QString("newName : %1\n").arg(newName));
    LOG(QString("Call back id: %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(RENAME_METHOD, srcUri,
            newName, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT rename******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::getDirContents()
// gets the contents of a directory
// --------------------------------------------------------------------
QVariant FileSystemService::getDirContents(const QString& srcUri, const QString& matchPattern, int callBackId)
    {
    LOG("*************ENTER getDirContents******************\n");
    LOG(QString("srcUri : %1\n").arg(srcUri));
    LOG(QString("matchPattern : %1\n").arg(matchPattern));
    LOG(QString("Call back id: %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(GETDIRCONTENTS_METHOD,
            srcUri, matchPattern, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT getDirContents******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }

// --------------------------------------------------------------------
// FileSystemService::copy()
// copies a file or directory from source to destination
// --------------------------------------------------------------------
QVariant FileSystemService::copy(const QString& srcUri, const QString& destUri, QString newName,\
        bool overWrite, int callBackId)
    {
    LOG("*************ENTER copy******************\n");
    LOG(QString("srcUri : %1\n").arg(srcUri));
    LOG(QString("destUri : %1\n").arg(destUri));
    LOG(QString("newName : %1\n").arg(newName));
    LOG(QString("overWrite : %1\n").arg(overWrite));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(COPY_METHOD, srcUri,
            destUri, newName, overWrite, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT copy******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }

// --------------------------------------------------------------------
// FileSystemService::move()
// moves a file or directory from source to destination
// --------------------------------------------------------------------
QVariant FileSystemService::move(const QString& srcUri, const QString& destUri, QString newName,\
        bool overWrite, int callBackId)
    {
    LOG("*************ENTER move******************\n");
    LOG(QString("srcUri : %1\n").arg(srcUri));
    LOG(QString("destUri : %1\n").arg(destUri));
    LOG(QString("newName : %1\n").arg(newName));
    LOG(QString("overWrite : %1\n").arg(overWrite));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(MOVE_METHOD, srcUri,
            destUri, newName, overWrite, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT move******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::openFile()
// created a new file or opens an existing file
// --------------------------------------------------------------------
QVariant FileSystemService::openFile(const QString& uri, const QString& mode, const QString& encoding, int callBackId)
    {
    LOG("*************ENTER openFile******************\n");
    LOG(QString("uri : %1\n").arg(uri));
    LOG(QString("mode : %1\n").arg(mode));
    LOG(QString("encoding : %1\n").arg(encoding));

    FileSystemWorker* worker = new FileSystemWorker(OPENFILE_METHOD, uri,
            mode, encoding, &m_openFileList, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT openFile******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::close()
// closes an open file handle.
// --------------------------------------------------------------------
QVariant FileSystemService::close(int handleId, int callBackId)
    {
    LOG("*************ENTER close******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(CLOSE_METHOD, file, callBackId);

    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT close******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::read()
// reads data using an open file handle.
// --------------------------------------------------------------------
QVariant FileSystemService::read(int handleId, int maxLength, int pos, int callBackId)
    {
    LOG("*************ENTER read******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("maxLength: %1\n").arg(maxLength));
    LOG(QString("position: %1\n").arg(pos));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(READ_METHOD, file, callBackId, maxLength,pos);
    if (file != NULL)
        worker->setEncoding(m_openFileList[handleId].encoding);

    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT read******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::readLine()
// reads data till a new line encoutners using an open file handle.
// --------------------------------------------------------------------
QVariant FileSystemService::readLine(int handleId, int maxLength, int callBackId)
    {
    LOG("*************ENTER readLine******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("maxLength: %1\n").arg(maxLength));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(READLINE_METHOD, file, callBackId, maxLength);
    if (file != NULL)
        worker->setEncoding(m_openFileList[handleId].encoding);

    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT readLine******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::readBase64()
// reads data in base 64 encoded format.
// --------------------------------------------------------------------
QVariant FileSystemService::readBase64(int handleId, int maxLength, int callBackId)
    {
    LOG("*************ENTER readBase64******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("maxLength: %1\n").arg(maxLength));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(READBASE64_METHOD, file, callBackId, maxLength);
    if (file != NULL)
        worker->setEncoding(m_openFileList[handleId].encoding);

    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT readBase64******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::write()
// writes data into an open file.
// --------------------------------------------------------------------
QVariant FileSystemService::write(int handleId, const QString& data, int pos, int callBackId)
    {
    LOG("*************ENTER write******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("data: %1\n").arg(data));
    LOG(QString("position: %1\n").arg(pos));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(WRITE_METHOD, file, callBackId, 0 , pos);
    worker->setData(data);
    if (file != NULL)
        worker->setEncoding(m_openFileList[handleId].encoding);

    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT write******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::writeLine()
// writes data alogn with a new line into an open file.
// --------------------------------------------------------------------
QVariant FileSystemService::writeLine(int handleId, const QString& data, int callBackId)
    {
    LOG("*************ENTER writeLine******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("data: %1\n").arg(data));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(WRITELINE_METHOD, file, callBackId);
    worker->setData(data);
    if (file != NULL)
        worker->setEncoding(m_openFileList[handleId].encoding);

    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT writeLine******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::writeBase64()
// writes base 64 encoded data into an open file
// --------------------------------------------------------------------
QVariant FileSystemService::writeBase64(int handleId, const QString& data, int callBackId)
    {
    LOG("*************ENTER writeBase64******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("data: %1\n").arg(data));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(WRITEBASE64_METHOD, file, callBackId);
    if (file != NULL)
        {
        worker->setData(data);
        worker->setEncoding(m_openFileList[handleId].encoding);
        }


    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT writeBase64******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }

// --------------------------------------------------------------------
// FileSystemService::cancel()
// cancels an ongoing aysnc transaction.
// --------------------------------------------------------------------
QVariant FileSystemService::cancel(int cbId)
    {
    LOG("***********Enter Cancel************\n");
    LOG(QString("Cancel, tid received: %1\n").arg(cbId));
    int errCode = SUCCESS;
    QString errMsg = "";
    if (m_threadList.value(cbId,NULL) != NULL)
        {
        LOG("On going Async FOUND for this trans Id\n");
        deleteThreadPointer(cbId);
        }
    else
        {
        LOG("On going Async NOT FOUND for this trans Id, DATA_NOT_FOUND_ERR\n");
        errCode = DATA_NOT_FOUND_ERR;
        errMsg = TRANS_ID_NOT_FOUND_ERR_STR;
        }
    LOG("***********Exit Cancel************\n");
    return Utilities::ReturnMap(errCode,errMsg);
    }
// --------------------------------------------------------------------
// FileSystemService::search()
// searches the user given path using match pattern provided
// --------------------------------------------------------------------
QVariant FileSystemService::search(int callBackId, const QString& matchPattern, const QString& srcUri)
    {
    LOG("*************ENTER search******************\n");
    LOG(QString("Call back id: %1\n").arg(callBackId));
    LOG(QString("matchPattern : %1\n").arg(matchPattern));
    LOG(QString("srcUri : %1\n").arg(srcUri));

    FileSystemWorker* worker = new FileSystemWorker(SEARCH_METHOD, srcUri,
            matchPattern, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT search******************\n");
    return callBackId;
    }
// --------------------------------------------------------------------
// FileSystemService::flush()
// preforms flush on a open file handle
// --------------------------------------------------------------------
QVariant FileSystemService::flush(int handleId, int callBackId)
    {
    LOG("*************ENTER flush******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("callBackId : %1\n").arg(callBackId));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(FLUSH_METHOD, file, callBackId);
    ExecFileIOCall(worker,callBackId);

    LOG("*************EXIT flush******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }
// --------------------------------------------------------------------
// FileSystemService::flush()
// preforms seek on a open file handle
// --------------------------------------------------------------------
QVariant FileSystemService::seek(int handleId, int seekOption, int position)
    {
    LOG("*************ENTER seek******************\n");
    LOG(QString("handleId : %1\n").arg(handleId));
    LOG(QString("seekOption : %1\n").arg(seekOption));
    LOG(QString("position : %1\n").arg(position));

    QFile** file = NULL;

    if (handleId > m_openFileList.count()-1 || handleId < 0)
        {
        file = NULL;
        }
    else
        {
        file = &(m_openFileList[handleId].fileHandle);
        }

    FileIOWorker* worker = new FileIOWorker(SEEK_METHOD, file, SYNC_CALL, seekOption, position);
    ExecFileIOCall(worker, SYNC_CALL);//SYNC call

    LOG("*************EXIT seek******************\n");
    return m_data;
    }
// --------------------------------------------------------------------
// FileSystemService::getMountPoints()
// lists all tha available mount points available on device
// --------------------------------------------------------------------
QVariant FileSystemService::getMountPoints(int callBackId)
    {
    LOG("*************ENTER getMountPoints******************\n");
    LOG(QString("Call back id: %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(GETMOUNTPOINTS_METHOD,
            "", callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT getMountPoints******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }

// --------------------------------------------------------------------
// FileSystemService::getDefaultPath()
// lists all tha available default paths available on device
// --------------------------------------------------------------------
QVariant FileSystemService::getDefaultPath(const QString& contentType, int callBackId)
    {
    LOG("*************ENTER getDefaultPath******************\n");
    LOG(QString("contentType: %1\n").arg(contentType));
    LOG(QString("Call back id: %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(GETDEFAULTPATH_METHOD,
            contentType, callBackId, mountTable);
    ExecFileSystemCall(worker, callBackId);

    LOG("*************EXIT getDefaultPath******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }

// --------------------------------------------------------------------
// FileSystemService::notifyMountEvents()
// Notifies about the newly available mount point.
// --------------------------------------------------------------------
QVariant  FileSystemService::notifyMountEvents(int callBackId)
    {
    LOG("*************ENTER notifyMountEvents******************\n");
    LOG(QString("Call back id: %1\n").arg(callBackId));

    //if notification is already active
    if (m_NotifyingWorker)
        {
        LOG("NOTIFCATION_ALREADY_ACTIVE, so can't make another rquest\n");
        return Utilities::ReturnMap(NOT_SUPPORTED_ERR, NOTIFICATION_ALREADY_ACTIVE_STR);
        }
    //this should go as a SYNC call
    m_NotifyingWorker = new FileSystemWorker(NOTIFYMOUNTEVENTS_METHOD, "", SYNC_CALL, mountTable);
    m_NotificationId = callBackId;

    bool ret = connect(m_NotifyingWorker, SIGNAL(FileSystemWorkerSignal(int , int , QMap<QString,QVariant>)),
            this,  SLOT(ServiceSlot(int , int , QMap<QString,QVariant>))
    );
    LOG(QString("m_NotifyingWorker connect returned: %1\n").arg(ret));

    m_NotifyingWorker->run();

    LOG("*************EXIT notifyMountEvents******************\n");
    return m_data;
    }
// --------------------------------------------------------------------
// FileSystemService::cancelNotify()
// cancels the notification of mount events.
// --------------------------------------------------------------------
QVariant FileSystemService::cancelNotify()
    {
    LOG("*************ENTER cancelNotify******************\n");

    //if notification is not active
    if (!m_NotifyingWorker)
        {
        LOG("NOTIFCATION_NOT_ACTIVE, so can't CANCEL it\n");
        return Utilities::ReturnMap(NOT_SUPPORTED_ERR, SERVICE_NOT_IN_USE_STR);
        }
    else
        {
        m_NotifyingWorker->run_cancelNotify();
        LOG(QString("Notification Id: %1\n").arg(m_NotificationId));
        delete m_NotifyingWorker;
        m_NotifyingWorker = NULL;
        }

    LOG("*************EXIT cancelNotify******************\n");
    return Utilities::ReturnMap(SUCCESS, SUCCESS_STR, m_NotificationId);
    }
// --------------------------------------------------------------------
// FileSystemService::getElementInfo()
// gets the element information
// --------------------------------------------------------------------
QVariant FileSystemService::getElementInfo(const QString& srcUri, int callBackId)
    {
    LOG("*************ENTER getElementInfo******************\n");
    LOG(QString("srcUri : %1\n").arg(srcUri));
    LOG(QString("Call back id: %1\n").arg(callBackId));

    FileSystemWorker* worker = new FileSystemWorker(GETELEMENTINFO_METHOD, srcUri, callBackId,mountTable);
    ExecFileSystemCall(worker,callBackId);

    LOG("*************EXIT getElementInfo******************\n");
    return (callBackId == SYNC_CALL ? m_data : callBackId);
    }

// --------------------------------------------------------------------
// FileSystemService::searchMatchObject()
// searches the filesystem using match pattern object provided
// --------------------------------------------------------------------
QVariant FileSystemService::searchMatchObject(int callBackId, const QMap<QString,QVariant>& matchObject)
    {
    LOG("*************ENTER searchMatchObject******************\n");
    LOG(QString("Call back id: %1\n").arg(callBackId));
    //LOG(QString("matchObject : %1\n").arg(matchObject));

    FileSystemWorker* worker = new FileSystemWorker(SEARCHMATCHOBJECT_METHOD, matchObject, callBackId,mountTable);
    ExecFileSystemCall(worker,callBackId);

    LOG("*************EXIT searchMatchObject******************\n");
    return callBackId;
    }
// End of file

