/*
* ============================================================================
*  Name        : qcontactinterface.h
*  Part of     : serviceproviders / WRT
*  Description : Interface for contact service
*  Version     : %version: 1 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef QCONTACTSERVICEINTERFACE_H
#define QCONTACTSERVICEINTERFACE_H

//QT Include
#include <QVariant>
#include <QString>
#include <QMap>

//Service Framework Include
#include "servicebase.h"

static const char* KIContactId = "com.nokia.IContacts/1.0";

// ITypeTest interface id.For Future use
static const ServiceInterfaceInfo KIContactInfo =
{
    KIContactId, 0
};

class IContacts : public IServiceBase
{
public:
    //public interfaces

     /**
      * This method is used to get the list of contacts from default database.
      * This is an Asynchronous API
      * @since  x.y
      * @param  match         Specifies a matching filter for the contacts to return.
      *                       This string will be searched for only in the default fields.
      *                       All the contacts which start with the search string in the default fields will be listed.
      *                       If the match string is NULL or Empty then all the contacts will be returned
      *
      * @param  sortOrder     [0] - Ascending,1 - Descending
      * @return transactionId
      * @emits  void asyncCallback(QString status,int transactionId,QVariant iterator )
      */
     virtual QVariant getContactsAsync (  const QString& match , const int sortOrder , const int transId) = 0;

     /**
      * This method is used to get the list of ContactIDs from default database.(Asynchronous)
      * @since x.y
      * @param match  Specifies a matching filter for the contacts to return.
      *               This string will be searched for only in the default fields
      * @param sortOrder  [0] - Ascending,1 - Descending
      * @return transactionId
      * @emits  void asyncCallback(QString status,int transactionId,QVariant iterator )
      */
     virtual QVariant getContactIdsAsync ( const QString& match , const int sortOrder , const int transId ) = 0;

     /**
       * This method is used to get the list of ContactIDs from default database.
       * @since x.y
       * @param match  Specifies a matching filter for the contacts to return.
       *               This string will be searched for only in the default fields
       * @param sortOrder  [0] - Ascending,1 - Descending
       * @return transactionId       *
       */
      virtual QVariant getContactIds ( const QString& match , const int sortOrder , const int transId ) = 0;
     /**
      * This method is used to retrieve details of a single contact.
      * This is a Synchronous API
      * @since  x.y
      *
      * @param  contactId     Specifies the contactId whose details has to be retrieved.
      * @return transactionId and contactInfo as a Map.
      *         For more details on contactInfo, please see the contactData structure definition below.
      */
     virtual QVariant getContactInfo ( const QString& contactId )  =  0;

     /**
      * This method is used to add a new contact to the default database.
      * This is a Synchronous API
      * @since  x.y
      *
      * @param  contactData   This QMap contains the details need to be added for a new contact.
      *                       Please see the contactData structure definition below for more details.
      * @return contactID(ID of the newly added contact) and transactionId as a Map
      */
     virtual QVariant addContact ( const QMap<QString, QVariant>& contactData,const int transId )  =  0;


     /**
      * This method is used to update an existing contact in the default database.
      * This is a Synchronous API
      * @since  x.y
      *
      * @param  contactData    This contains the details needed to modify an existing contact.
      *                        Please see the contactData structure definition below for more details.
      *                        Id Field in contactData is mandatory
      * @return 0 or ErrorCode
      */
     virtual QVariant updateContact ( const QMap<QString , QVariant>& contactData, const int transId)  =  0;

     /**
      * This method is used to delete an existing contact from the default database.
      * This is a Synchronous API
      * It can delete multiple contacts also.
      * @since x.y
      *
      * @param  contactIds      An Array of ContactID strings
      * @return transactionId
      */
     virtual QVariant deleteContacts ( const QVariant& contactIdsvar )  =  0;

     /**
      * This method is used to add a new group to default database.
      * This is a Synchronous API
      * @since  x.y
      *
      * @param  groupName     The name of the new group.
      * @return groupId(ID of the newly added group) and transactionId as a Map.
      */
     virtual QVariant addGroup ( const QString& groupName )  =  0;

     /**
      * This method is used to delete an existing group from the default database.
      * This is a Synchronous API
      * It can delete multiple groups also.
      * @since x.y
      *
      * @param  groupIds       An Array of groupID strings
      * @return transactionId
      */
     virtual QVariant deleteGroups ( const QVariant& groupIds )  =  0;

     /**
      * This method is used to update an existing group in the default database.
      * This is a Synchronous API
      * @since  x.y
      *
      * @param  groupData      This contains the details need to modify an existing group.
      *                        Please see the groupData structure definition below for more details.
      *                        Id Field in contactData is mandatory
      * @return transactionId
      */
     virtual QVariant updateGroup ( const QMap<QString, QVariant>& groupData )  =  0;

     /**
      * This method is used to associate contacts to a Group.
      * This is a Synchronous API
      * @since x.y
      *
      * @param  groupId        The group to which a list contacts have to be associated.
      * @param  contactIDs     The list of contacts
      * @return transactionId
      */
     virtual QVariant addContactsToGroup ( const QString& groupId, const QVariant& contactIDs )  =  0;

     /**
      * This method is used to disassociate contacts from a Group.
      * This is a Synchronous API
      * @since x.y
      *
      * @param  groupId         The group from which a list contacts have to be disassociated
      * @param  contactIDs      The list of contacts
      * @return transactionId
      */
     virtual QVariant removeContactsFromGroup ( const QString& groupId, const QVariant& contactIDs )  =  0;

     /**
      * This method is used to get the list of groups from default database.
      * This is an Asynchronous API
      * @since  x.y
      * @param
      * @return transactionId
      * @emits  void asyncCallback( QString status,int transactionId,QVariant iterator )
      */
     virtual QVariant getGroupsAsync (const int transId) = 0;

     /**
      * This method is used to get the list of GroupIDs from default database.
      * This is an Asynchronous API
      * @since  x.y
      *
      * @param
      * @return transactionId
      * @emits  void asyncCallback( QString status, int transactionId, QVariant iterator )
      */
     virtual QVariant getGroupIdsAsync (const int transId)  =  0;

     /**
      * This method is used to get the list of GroupIDs from default database.
      * This is an Synchronous API
      * @since  x.y
      *
      * @param
      * @return Array of GroupIds
      */
     virtual QVariant getGroupIds ()  =  0;

     /**
      * This method is used to retrieve details of a single group.
      * This is a Synchronous API
      * @since  x.y
      *
      * @param  GroupId       Specifies the groupId whose details has to be retrieved.
      * @return transactionId and groupInfo as a Map.
      *         For more details on groupInfo, please see the groupData structure definition below.
      */
     virtual QVariant getGroupInfo ( const QString& agroupId )  =  0;

     /**
       * Cancels any asynchronous request identified uniquely by trasactionid.
       * This is a Synchronous API
       * @since x.y
       *
       * @param transactionID    Identifier of the ongoing asynchronous request to be canceled.
       */
      virtual QVariant cancel( const int transactionID )  =  0;

      /**
           * This method is used to start the notification for any operation on default database.
           * This is an Asynchronous API
           * @return transactionId
           * @emits  void asyncCallback( int status, int transactionId, QMap<QString,QVariant> notifyData )
      */
      virtual QVariant startNotification (const int transId) = 0;


      /**
           * This method is used to stop the notification started by calling startNotification
           * This is an Synchronous API
           * @return Error Map
      */
      virtual QVariant stopNotification() = 0;

      /**
       * This method launches the native contacts editor application
             * @param contactData Input contact data(optional,platform specific)
             * @return  An error map with the error code and error description
       */
      virtual QVariant startEditor(const QMap<QString, QVariant>& contactData) = 0;
};
Q_DECLARE_INTERFACE(IContacts, KIContactId);

#endif //QCONTACTSERVICEINTERFACE_H
