/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef QCONTACTSERVICE_H_
#define QCONTACTSERVICE_H_

#include <QObject>
#include <QList>
// Qt Mobility
#include <qtcontacts.h>
#ifndef __MAEMO5__
#include <contactsinterface.h>
#endif

#include "qcontactshelpers.h"
#include "qcontactserviceinterface.h"
#include "servicesecurityinterface.h"
#include "qgetcontactidstask.h"
#include "qgetcontactstask.h"
#include "qgetgroupidstask.h"
#include "qgetgroupstask.h"

using namespace QTM_NAMESPACE;

/**
* Status codes returned by APIs
* See CJSEStatusCodes qcontacthelpers.h
*/
class QContactService: public QObject,             //Need for QT
                         public IContacts,         //The Contact Interface Definition
                         public IServiceSecurity   //Security stuff
{
    Q_OBJECT
    Q_INTERFACES(IContacts)

public:

    /**
    * constructor
    * @since x.y
    * @param
    * @return
    */
    QContactService();

    /**
     * destructor
     *     1 ) Loops thru Transaction Map and deletes all iterator Objects
     *     2 ) Delete core class instance
     * @since x.y
     * @param
     * @return
     */
    virtual ~QContactService();

    /**
     * Implemented from IServiceSecurity.Needed to handle capabilities.
     * @since x.y
     * @param WRT::SecSession
     * @return
     */
    void setSecuritySession(WRT::SecSession *aSecSession);

public slots:
    /** Retrieves a list of contacts from database. Result is
     *  delivered in callback function.
     *  @param match Filter for contacts to return. The string is
     *  only searched from default fields (FirstName and/or LastName)
     *  If string is null or empty, all contacts will be returned.
     *  @param sortOrder Specifies the sort order. Value can be
     *  SORT_ASCENDING or SORT_DESCENDING. If no value is
     *  specified the default value will be used (SORT_ASCENDING). Sort is done based
     *  on LastName, FirstName fields in that order of priority.
     *  @param transId Transaction id to be used as the callback parameter
     *  @return 0 or Error Code.
     */
    QVariant getContactsAsync (const QString& match, const int sortOrder=0 , const int transId=-1);

    /**
     * This method is used to get the list of ContactIDs from default database.(Asynchronous)
     * @since   x.y
     * @param   match  Specifies a matching filter for the contacts to return.
     *                   This string will be searched for only in the default fields
     * @param   sortOrder  [0] - Ascending,1 - Descending
     * @return  0 or Error Code
     * @emits  void asyncCallback(QString status,int transactionId,QVariant iterator )
     */
    QVariant getContactIdsAsync ( const QString& match , const int sortOrder=0 , const int transId=-1 );

     /**
       * This method is used to get the list of ContactIDs from default database.
       * @since x.y
       * @param match  Specifies a matching filter for the contacts to return.
       *               This string will be searched for only in the default fields
       * @param sortOrder  [0] - Ascending,1 - Descending
       * @return transactionId       *
       */
    QVariant getContactIds ( const QString& match , const int sortOrder , const int transId );

    /**
     * This method is used to retrieve details of a single contact.
     * This is a Synchronous API
     * @since  x.y
     *
     * @param  contactId     Specifies the contactId whose details has to be retrieved.
     * @return transactionId and contactInfo as a Map.
     *         For more details on contactInfo, please see the contactData structure definition below.
     */
    QVariant getContactInfo ( const QString& contactId );

    /**
     * This method is used to add a new contact to the default database.
     * This is synchronous API.
     * @since  x.y
     *
     * @param  contactData   This QMap contains the details need to be added for a new contact.
     *                       Please see the contactData structure definition below for more details.
     * @return contactID(ID of the newly added contact) and transactionId as a Map
     */
    QVariant addContact ( const QMap<QString, QVariant>& contactData,const int transId = -1);

    /**
     * This method is used to update an existing contact in the default database.
     * This is a Synchronous API
     * @since  x.y
     *
     * @param  contactData    This contains the details needed to modify an existing contact.
     *                        Please see the contactData structure definition below for more details.
     *                        Id Field in contactData is mandatory
     * @return 0 or Error Code
     */
    QVariant updateContact ( const QMap<QString , QVariant>& contactData,const int transId = -1 );

    /**
     * This method is used to delete an existing contact from the default database.
     * This is a Synchronous API
     * It can delete multiple contacts also.
     * @since x.y
     *
     * @param  contactIds      An Array of ContactID strings
     * @return transactionId
     */
    QVariant deleteContacts ( const QVariant& contactIdsvar );

    /**
     * This method is used to add a new group to default database.
     * This is a Synchronous API
     * @since  x.y
     *
     * @param  groupName     The name of the new group.
     * @return groupId(ID of the newly added group) and transactionId as a Map.
     */
    QVariant addGroup ( const QString& groupName );

    /**
     * This method is used to delete an existing group from the default database.
     * This is a Synchronous API
     * It can delete multiple groups also.
     * @since x.y
     *
     * @param  groupIds       An Array of groupID strings
     * @return transactionId
     */
    QVariant deleteGroups ( const QVariant& groupIds );

    /**
     * This method is used to update an existing group in the default database.
     * This is a Synchronous API
     * @since  x.y
     *
     * @param  groupData      This contains the details need to modify an existing group.
     *                        Please see the groupData structure definition below for more details.
     *                        Id Field in contactData is mandatory
     * @return transactionId
     */
    QVariant updateGroup ( const QMap<QString, QVariant>& groupData );

    /**
     * This method is used to associate contacts to a Group.
     * This is a Synchronous API
     * @since x.y
     *
     * @param  groupId        The group to which a list contacts have to be associated.
     * @param  contactIDs     The list of contacts
     * @return transactionId
     */
    QVariant addContactsToGroup ( const QString& groupId, const QVariant& contactIDs );

    /**
     * This method is used to disassociate contacts from a Group.
     * This is a Synchronous API
     * @since x.y
     *
     * @param  groupId         The group from which a list contacts have to be disassociated
     * @param  contactIDs      The list of contacts
     * @return transactionId
     */
    QVariant removeContactsFromGroup ( const QString& groupId, const QVariant& contactIDs );

    /**
     * This method is used to get the list of groups from default database.
     * This is an Asynchronous API
     * @since  x.y
     * @param
     * @return transactionId
     * @emits  void asyncCallback( QString status,int transactionId,QVariant iterator )
     */
    QVariant getGroupsAsync (const int transId);

    /**
     * This method is used to get the list of GroupIDs from default database.
     * This is an Asynchronous API
     * @since  x.y
     *
     * @param
     * @return transactionId
     * @emits  void asyncCallback( QString status, int transactionId, QVariant iterator )
     */
    QVariant getGroupIdsAsync (const int transId);

     /**
      * This method is used to get the list of GroupIDs from default database.
      * This is an Synchronous API
      * @since  x.y
      *
      * @param
      * @return Array of GroupIds
      */
     QVariant getGroupIds ();

    /**
     * This method is used to retrieve details of a single group.
     * This is a Synchronous API
     * @since  x.y
     *
     * @param  GroupId       Specifies the groupId whose details has to be retrieved.
     * @return transactionId and groupInfo as a Map.
     *         For more details on groupInfo, please see the groupData structure definition below.
     */
    QVariant getGroupInfo ( const QString& agroupId );

    /**
      * Cancels any asynchronous request identified uniquely by trasactionid.
      * This is a Synchronous API
      * @since x.y
      *
      * @param transactionID    Identifier of the ongoing asynchronous request to be canceled.
      */
    QVariant cancel( const int transactionID );

    /**
    * This method is used to start the notification for any operation on default database.
    * This is an Asynchronous API
    * @return transactionId
    * @emits  void asyncCallback( int status, int transactionId, QMap<QString,QVariant> notifyData )
    */
    QVariant startNotification (const int transId);

    /**
       * This method is used to stop the notification started by calling startNotification
       * This is an Synchronous API
       * @return Error Map
    */
    QVariant stopNotification();

    /**
    * This method launches the native contacts editor application
         * @param contactData Input contact data(optional,platform specific)
         * @return  An error map with the error code and error description
    */
    QVariant startEditor(const QMap<QString, QVariant>& contactData);

private slots:

    /** Calls JavaScript callback that signals transactionId and status.
     *  @param qint32 Transaction id of the handled request.
     *  @param qint32 Status of the handled request.
     */
    void fireJavaScriptCallback(qint32 trId, qint32 status);

    /** Calls JavaScript callback that gets an iterator as parameter.
     *  @param QStringList List of JSON formatted strings to be iterated
     *  over.
     *  @param qint32 Transaction id of the handled request.
     *  @param qint32 Status of the handled request.
     */
    void fireJavaScriptCallback(QStringList* json, qint32 trId, qint32 status);

    /** Calls JavaScript callback that gets an array of contacts as
     *  parameter.
     *  @param QList<QContact> List of contacts.
     *  @param qint32 Transaction id of the handled request.
     *  @param qint32 Status of the handled request.
     */
    void fireJavaScriptCallback(GetContactsTask* aTask,
                QList<QContact> json, qint32 trId, qint32 status);

    /** Calls JavaScript callback that gets an array of groups as
     *  parameter.
     *  @param QList<QContactGroup> List of groups.
     *  @param qint32 Transaction id of the handled request.
     *  @param qint32 Status of the handled request.
     */
    void fireGroupsCallback(GetGroupsTask* aTask,
                QList<QContact> json, qint32 trId, qint32 status);

    /** Calls JavaScript callback that gets an array of ids as
     *  parameter.
     *  @param QList<QUniqueId> List of unique ids.
     *  @param qint32 Transaction id of the handled request.
     *  @param qint32 Status of the handled request.
     */
    void fireJavaScriptStringListCallback(GetContactIdsTask* aTask,
                QList<QContactLocalId> json, qint32 trId, qint32 status);

    /** Calls JavaScript callback that gets an array of ids as
     *  parameter.
     *  @param QList<QUniqueId> List of unique ids.
     *  @param qint32 Transaction id of the handled request.
     *  @param qint32 Status of the handled request.
     */
    void fireJavaScriptStringListCallback(GetGroupIdsTask* aTask,
                QList<QContactLocalId> json, qint32 trId, qint32 status);

    /** Is called from QContactManager if contacts are added to datastore.
     *  @param QList<quint32> List of contact ids.
     */
    void handleContactsAdded(const QList<QContactLocalId>& contactIds);

    /** Is called from QContactManager if contacts are changed to datastore.
     *  @param QList<quint32> List of contact ids.
     */
    void handleContactsChanged(const QList<QContactLocalId>& contactIds);

    /** Is called from QContactManager if contacts are removed from datastore.
     *  @param QList<quint32> List of contact ids.
     */
    void handleContactsRemoved(const QList<QContactLocalId>& contactIds);

signals:

    /**
    * The signal emitted for the all the Async functions.
    * @since x.y
    * @param status   - 0 or Error code
    * @param aTransId - TransactionID of the Async operation
    * @param iterator - The actual response data
    * @return
    */
    void asyncCallbackC( int status, int transactionId, QObject *contIter );

    /**
    * The signal emitted for the all the Async functions.
    * @since x.y
    * @param status   - 0 or Error code
    * @param aTransId - TransactionID of the Async operation
    * @param iterator - The actual response data
    * @return
    */
    void asyncCallbackG( int status, int transactionId, QObject *contIter );

    /**
    * The signal emitted for the all the Async functions which doest need Iterator in output
    * @since x.y
    * @param status   - 0 or Error code
    * @param aTransId - TransactionID of the Async operation
    * @param iterator - The actual response data
    * @return
    */
    void asyncCallback( int status, int transactionId, QVariant contData );

    /**
      * The signal emitted for the all the Async functions.
      * This signal is emitted in case of occurance of any event to notify the user about any change to the contacts database
      * @since x.y
      * @param status   - 0 or Error code
      * @param aTransId - TransactionID of the Async operation
      * @param notifyData - The actual response data
      * @return
      */
    void asyncCallbackN( int status, int transactionId, QMap<QString,QVariant> notifyData );

protected:
    WRT::SecSession *iSecSession;

private:
    QList<qint32> m_onGoingAsyncTransctions;
    bool m_isNotifyInProgress;
    qint32 m_notifyTrId;

};

#endif

