/*
 * ============================================================================
 *  Name        : qcommlogevent.h
 *  Part of     : serviceframework / WRT
 *  Description : Event class for CommLog service provider
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef EVENT_H
#define EVENT_H
#include <QDateTime>
#include <QVariant>
#include <QString>

class Event
{
public:
    enum ColumnId
    {
        Id               = 0,
        ServiceId,
        TypeId,
        StorageTime,
        StartTime,
        EndTime          = 5,
        IsRead,
        Flags,
        BytesSent,
        BytesReceived,
        LocalUid         = 10,
        LocalName,
        RemoteUid,
        RemoteName,
        RemoteEbookUid,
        Channel          = 15,
        FreeText,
        GroupUid,
        Service,
        EventType,
        AdditionalText   = 20,
        IconName,
        PangoMarkup,
        Direction, //This field now in RTComElEvent structure
        Status,
        ContactId        = 25,
        TpMessageId,
        EventCount, // <-- CallEvent related
        NumberOfColumns
    };

    enum EventTypeId
    {
        UnknownType = 0,
        CallInbound,
        CallOutbound,
        CallMissed,
        CallVoicemail,
        ChatInbound,
        ChatOutbound,
        SmsInbound,
        SmsOutbound
    };

    enum EventServiceId
    {
        UnknownService = 0,
        CallService,
        IMService,
        SMSService
    };

    enum EventDirection
    {
        Inbound = 0,
        Outbound,
        UnknownDirection
    };

    enum EventStatus {
        UnknownStatus = 0,
        SendingStatus,
        SentStatus,
        DeliveredStatus,
        FailedStatus
    };

    enum EventFlags
    {
        NoFlags             = 0,
        SMS_Pending         = 1 << 0,
        SMS_Temporary_Error = 1 << 1,
        CALL_Missed         = 1 << 2,
        Draft               = 1 << 3
    };

    Event();
    Event(const Event &other);
    ~Event();

    Event &operator=(const Event &other);

    bool operator==(const Event &other) const;

    bool isValid() const;

    //Accesses column based on database index
    virtual QVariant data(int column);

    //Gets name of a database column.
    static QString columnName(int column);

    //////////////////////////// G E T - A C C E S S O R S ////////////////////

    int getId() const;

    int getServiceId() const;

    Event::EventTypeId getTypeId() const;

    QTime getStorageTime() const;

    QDateTime getStartTime() const;

    QDateTime getEndTime() const;

    bool getIsRead() const;

    Event::EventFlags getFlags() const;

    int getBytesSent() const;

    int getBytesReceived() const;

    QString getLocalUid() const;  /* telepathy account */

    QString getLocalName() const;

    QString getRemoteUid() const;

    QString getRemoteName() const;

    QString getRemoteEbookUid() const;

    QString getChannel() const;

    QString getFreeText() const;

    QString getGroupUid() const; /* communication channel */

    QString getService() const;

    QString getEventType() const;

    QString getAdditionalText() const;

    QString getIconName() const;

    QString getPangoMarkup() const;

    Event::EventDirection getDirection() const;

    Event::EventStatus getStatus() const;

    int getContactId() const;

    uint getTpMessageId() const;

    int getEventCount() const;

    //////////////////////////////////// S E T - A C C E S S O R S ///////////////////////////
    void setId(int id);

    void setServiceId(Event::EventServiceId aserviceId);

    void setTypeId(Event::EventTypeId atypeId);

    void setStorageTime(const QTime &astorageTime);

    void setStartTime(const QDateTime &astartTime);

    void setEndTime(const QDateTime &aendTime);

    void setIsRead(bool aisRead);

    void setFlags(Event::EventFlags aflags);

    void setBytesSent(int bytes);

    void setBytesReceived(int bytes);

    void setLocalUid(const QString &uid);

    void setLocalName(const QString &name);

    void setRemoteUid(const QString &uid);

    void setRemoteName(const QString &name);

    void setRemoteEbookUid(const QString &uid);

    void setChannel(const QString &achannel);

    void setFreeText(const QString &text);

    void setGroupUid(const QString &uid);

    void setService(const QString &aservice);

    void setEventType(const QString &eventtype);

    void setAdditionalText(const QString &token);

    void setIconName(const QString &iname);

    void setPangoMarkup(const QString &pango);

    void setDirection(Event::EventDirection adirection);

    void setStatus( Event::EventStatus astatus );

    void setContactId(int id);

    void setTpMessageId(uint id);

    void setEventCount( int count );

private:
    /*Mapping RTComElEvent structure to EventPrivate*/
    int id;                                 // gint fld_id;
    Event::EventServiceId serviceId;        // gint fld_service_id
    Event::EventTypeId typeId;              // gint fld_event_type_id;
    QTime storageTime;                      // time_t fld_storage_time;
    QDateTime startTime;                    // time_t fld_start_time
    QDateTime endTime;                      // time_t fld_end_time
    bool isRead;                            // gboolean fld_is_read;
    Event::EventFlags flags;                // gint fld_flags;
    int bytesSent;                          // gint fld_bytes_sent;
    int bytesReceived;                      // gint fld_bytes_received;
    QString localUid;                       // fld_local_uid;
    QString localName;                      // gchar * fld_local_name;
    QString remoteUid;                      // gchar * fld_remote_uid;
    QString remoteName;                     // gchar * fld_remote_name;
    QString remoteEbookUid;                 // gchar * fld_remote_ebook_uid;
    QString channel;                        // gchar * fld_channel;
    QString freeText;                       // gchar * fld_free_text;
    QString groupUid;                       // gchar * fld_group_uid;

    QString service;                        // gchar * fld_service
    QString eventType;                      // gchar * fld_event_type;
    QString additionalText;                 // gchar * fld_additional_text;
    QString iconName;                       // gchar * fld_icon_name;
    QString pangoMarkup;                    // gchar * fld_pango_markup;
    Event::EventDirection direction;        // gboolean fld_outgoing
    /*Mapping RTComElEvent structure to EventPrivate*/


    Event::EventStatus status;
    int contactId;
    uint tpMessageId;
    int eventCount;
};
#endif // EVENT_H
