/*
 * ============================================================================
 *  Name        : qcalendarservice.cpp
 *  Part of     : serviceframework / WRT
 *  Description : Qt class for Calendar service
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QMapIterator>
#include <QDebug>
#include <QThreadPool>

#include "secsession.h"
#include "qcalendarservice.h"
#include "qgetlisttask.h"
#include "qcalendarhelper.h"
#include "qcalendarconstants.h"

#define DBUS_PATH              "/com/nokia/calendar"
#define DBUS_INTERFACE         "com.nokia.calendar"
#define DBUS_EVENT             "dbChange"

const QString E_ADDED = ":EVENT:ADDED:";
const QString E_MODIFIED = ":EVENT:MODIFIED:";
const QString E_DELETED = ":EVENT:DELETED:";
const QString T_ADDED = ":TODO:ADDED:";
const QString T_MODIFIED = ":TODO:MODIFIED:";
const QString T_DELETED = ":TODO:DELETED:";

using namespace CalendarHelpers;

QCalendarService::QCalendarService()
{
    qDebug() << "QCalendarService()";
    multiCalendar = CMulticalendar::MCInstance();
    calendar = multiCalendar->getSynchronizedCalendar();
    calendarId = calendar->getCalendarId();
    m_isNotifyInProgress = false;
    qDebug() << "* QCalendarService(): multiCalendar:" << multiCalendar;
    qDebug() << "* QCalendarService(): calendar:" << calendar;
    qDebug() << "* QCalendarService(): calendarId:" << calendarId;
}

QCalendarService::~QCalendarService()
{
    qDebug() << "~QCalendarService destructor";
    calendar = NULL;
    delete multiCalendar;
    multiCalendar = NULL;
    qDebug() << "~QCalendarService()";
}

void QCalendarService::setSecuritySession(WRT::SecSession *secSession)
{
    securitySession = secSession;
}

QVariant QCalendarService::startEditor(const QMap<QString,QVariant>& calendarEntry, const int transactionId)
{
    QMap<QString,QVariant> map;
    map.insert(KErrMessage, "startEditor not supported");
    map.insert(KErrCode, JSEStatusCodes::NOT_SUPPORTED_ERR);
    map.insert(KTransId, transactionId);
    return map;
}

QVariant QCalendarService::addEntry(const QMap<QString,QVariant>& calendarEntry)
{
    QMap<QString,QVariant> rMap;
    QVariant retVar;
    qint32 statusCode;

    qDebug() << "addEntry()";
    QString qUID = calendarEntry.value(KCalendarEntryId).toString();
    if (!qUID.isEmpty())
    {
        QMap<QString,QVariant> targetEntry;
        if (!CalendarHelpers::getCalendarEntryById(multiCalendar, targetEntry, qUID))
        {
            rMap.insert(KErrCode,JSEStatusCodes::DATA_NOT_FOUND_ERR);
            rMap.insert(KErrMessage,"Entry with given id not found");
            retVar = rMap;
            qDebug() << "* addEntry(): returning retVar:" << retVar;
            return retVar;
        }
        qDebug() << "* addEntry(): targetEntry.Id:" << targetEntry.value(KCalendarEntryId);
        qDebug() << "* addEntry(): targetEntry.time:" << targetEntry.value(KCalendarEntryTime);
        if (calendarEntry.contains(KCalendarEntryType) &&
            calendarEntry.value(KCalendarEntryType).toString().compare(targetEntry.value(KCalendarEntryType).toString()) != 0)
        {
            qDebug() << "* addEntry(): type of the entry cannot be changed!:";
            qDebug() << "* addEntry(): existing entry type:" << targetEntry.value(KCalendarEntryType).toString();
            qDebug() << "* addEntry(): new entry type:" << calendarEntry.value(KCalendarEntryType).toString();
            rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
            rMap.insert(KErrMessage,"Entry validation failed");
            retVar = rMap;
            qDebug() << "* addEntry(): returning retVar:" << retVar;
            return retVar;
        }
        statusCode = _parseEntry(_mergeQmap(targetEntry, calendarEntry), TRUE);
        targetEntry.clear();
        if (statusCode != 0)
        {
            parsedEntry.clear();
            rMap.insert(KErrCode,statusCode);
            rMap.insert(KErrMessage,"Entry validation failed");
            retVar = rMap;
            qDebug() << "* addEntry(): returning retVar:" << retVar;
            return retVar;
        }
        retVar = _updateEntry(qUID);
    }
    else
    {
        statusCode = _parseEntry(calendarEntry, FALSE);
        if (statusCode != 0)
        {
            parsedEntry.clear();
            rMap.insert(KErrCode,statusCode);
            rMap.insert(KErrMessage,"Entry validation failed");
            retVar = rMap;
            qDebug() << "* addEntry(): returning retVar:" << retVar;
            return retVar;
        }
        retVar = _createEntry();
    }
    parsedEntry.clear();
    qDebug() << "* addEntry(): returning retVar:" << retVar;
    return retVar;
}

QVariant QCalendarService::deleteEntry(const QMap<QString,QVariant>& data)
{
    int errorCode;
    bool result = FALSE;
    QVariant retVar = NULL;
    QMap<QString,QVariant> rMap;
    QDateTime newBeginDate;
    QDateTime newEndDate;
    QDateTime newUntilDate;


    qDebug() << "deleteEntry()";
    QString qUID = data.value(KDeleteDataId).toString();
    qDebug() << "* deleteEntry(): id:" << qUID;
    if (qUID.isEmpty())
    {
        rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
        rMap.insert(KErrMessage,"Mandatory id field is missing");
        retVar = rMap;
        qDebug() << "* deleteEntry(): returning retVar:" << retVar;
        return retVar;
    }
    qDebug() << "* deleteEntry(): deleting entry...";
//Only id is specified, deleting by Id
    if (!data.contains(KDeleteDataRange) ||
        data.value(KDeleteDataRange) == KUndefined ||
        !data.value(KDeleteDataRange).canConvert<QVariantMap>())
    {
        qDebug() << "* deleteEntry(): Only id is specified, deleting by Id";
        vector <string> idList;
        idList.push_back(qUID.toStdString());
        multiCalendar->deleteComponents(idList, calendarId, errorCode);
        if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
        {
            rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
            rMap.insert(KErrMessage,"Error deleting entry");
        }
        else
        {
            rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
            rMap.insert(KErrMessage,"SUCCESS");
        }
        retVar = rMap;
        qDebug() << "* deleteEntry(): returning retVar:" << retVar;
        return retVar;
    }
//Deleting by range
//Validating range-------------------------------------------------------
    qDebug() << "* deleteEntry(): validating passed range";
    QVariantMap range = data.value(KDeleteDataRange).toMap();
    QDateTime rangeBegin = MIN_DATE_TIME;
    QDateTime rangeEnd = MAX_DATE_TIME;

    if (range.contains(KRangeDataBegin) &&
        range.value(KRangeDataBegin) != KUndefined &&
        range.value(KRangeDataBegin).canConvert<QDateTime>())
    {
        rangeBegin = range.value(KRangeDataBegin).toDateTime();
    }

    if (range.contains(KRangeDataEnd) &&
        range.value(KRangeDataEnd) != KUndefined &&
        range.value(KRangeDataEnd).canConvert<QDateTime>())
    {
        rangeEnd = range.value(KRangeDataEnd).toDateTime();
    }
    if ((!rangeBegin.isValid() && !rangeEnd.isValid()) ||
        (rangeBegin.isValid() && rangeEnd.isValid() && rangeEnd < rangeBegin))
    {
        rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
        rMap.insert(KErrMessage,"range dates are invalid");
        retVar = rMap;
        qDebug() << "* deleteEntry(): returning retVar:" << retVar;
        return retVar;
    }
    qDebug() << "* deleteEntry(): range.Begin:" << rangeBegin;
    qDebug() << "* deleteEntry(): range.End:" << rangeEnd;
//Validating range End----------------------------------------------------
    CEvent * pEvent = NULL;
    CTodo * pTodo = NULL;
    pEvent = calendar->getEvent(qUID.toStdString(), errorCode);
    if (errorCode != CALENDAR_OPERATION_SUCCESSFUL || pEvent == NULL)
    {
        rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
        rMap.insert(KErrMessage,"Error deleting entry");
        retVar = rMap;
        qDebug() << "* deleteEntry(): returning retVar:" << retVar;
        return retVar;
    }
    CComponent * component = NULL;
    component = static_cast<CComponent*>(pEvent);
    int type = component->getType();
    switch (type)
    {
        case KTypeEvent:
            qDebug() << "* deleteEntry(): type is event";
            break;
        case KTypeTodo:
            delete pEvent;
            pEvent = NULL;
            pTodo = calendar->getTodo(qUID.toStdString(), errorCode);
            if (errorCode != CALENDAR_OPERATION_SUCCESSFUL || pTodo == NULL)
            {
                rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
                rMap.insert(KErrMessage,"Error deleting entry");
                retVar = rMap;
                qDebug() << "* deleteEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* deleteEntry(): type is todo";
            component = static_cast<CComponent*>(pTodo);
            break;
        default:
            rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
            rMap.insert(KErrMessage,"Entry type is not valid");
            retVar = rMap;
            delete component;
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            return retVar;
    }

    QMap<QString,QVariant> entry;
    result = CalendarHelpers::getCalendarEntryById(multiCalendar, entry, qUID);
    if (!result)
    {
        rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
        rMap.insert(KErrMessage,"Error deleting entry");
        retVar = rMap;
        delete component;
        qDebug() << "* deleteEntry(): returning retVar:" << retVar;
        return retVar;
    }
    if (type == KTypeTodo)
    {
        QMap<QString,QVariant> qTime = entry.value(KCalendarEntryTime).toMap();
        if (qTime.value(KTimeDataEnd).toDateTime() > rangeBegin && qTime.value(KTimeDataEnd).toDateTime() < rangeEnd)
        {
            qDebug() <<  "* deleteEntry(): removing todo";
            vector <string> idList;
            idList.push_back(qUID.toStdString());
            multiCalendar->deleteComponents(idList, calendarId, errorCode);
            if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
            {
                rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
                rMap.insert(KErrMessage,"Error deleting entry");
            }
            else
            {
                rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
                rMap.insert(KErrMessage,"SUCCESS");
            }
            retVar = rMap;
            delete component;
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            return retVar;
        }
        rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
        rMap.insert(KErrMessage,"SUCCESS");
        retVar = rMap;
        delete component;
        qDebug() << "* deleteEntry(): returning retVar:" << retVar;
        return retVar;
    }
    QDateTime currentTime;
    if (!entry.contains(KCalendarEntryRepeatRule))
    {
        currentTime = entry.value(KCalendarEntryTime).toMap().value(KTimeDataStart).toDateTime();
//if entry doesn't have repeat rule - simply delete it
        if (currentTime >= rangeBegin &&
            currentTime <= rangeEnd)
        {
            qDebug() << "* deleteEntry(): entry doesn't have repeat rule, deleting...";
            vector <string> idList;
            idList.push_back(qUID.toStdString());
            multiCalendar->deleteComponents(idList, calendarId, errorCode);
            if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
            {
                rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
                rMap.insert(KErrMessage,"Error deleting entry");
            }
            else
            {
                rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
                rMap.insert(KErrMessage,"SUCCESS");
            }
            retVar = rMap;
            delete component;
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            return retVar;
        }
        else
        {
            rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
            rMap.insert(KErrMessage,"SUCCESS");
            retVar = rMap;
            delete component;
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            return retVar;
        }
    }
//if does have - generate instances time, perform validation
    vector<time_t> times;
    component->generateInstanceTimes(MIN_DATE_TIME.toTime_t(), MAX_DATE_TIME.toTime_t(), times);
    qDebug() << "* deleteEntry(): instance count:" << times.size();
    QDateTime firstInstTime = QDateTime::fromTime_t(times.front());
    QDateTime lastInstTime = QDateTime::fromTime_t(times.back());
    qDebug() << "* deleteEntry(): entry does have repeat rule, validating..";
    qDebug() << "* deleteEntry(): firstInstTime:" << firstInstTime;
    qDebug() << "* deleteEntry(): lastInstTime:" << lastInstTime;
    if (firstInstTime == lastInstTime &&
        firstInstTime >= rangeBegin &&
        firstInstTime <= rangeEnd )
    {
        qDebug() << "* deleteEntry(): only one instance, deleting..";
        vector <string> idList;
        idList.push_back(qUID.toStdString());
        multiCalendar->deleteComponents(idList, calendarId, errorCode);
        if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
        {
            rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
            rMap.insert(KErrMessage,"Error deleting entry");
        }
        else
        {
            rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
            rMap.insert(KErrMessage,"SUCCESS");
        }
        retVar = rMap;
        delete component;
        qDebug() << "* deleteEntry(): returning retVar:" << retVar;
        return retVar;
    }
    for (int i = 0; i < times.size(); i++)
    {
        currentTime = QDateTime::fromTime_t(times.at(i));
        qDebug() << "* deleteEntry(): instances:" << currentTime;
    }
//since Fremantle calendar doesn't support "instance" concept, the deletion of instances can result in
//updating or even creating new entries
    if (firstInstTime >= rangeBegin)
    {
/*
------[range.Begin]---[startDate]---[instance2]---[instance3]---[instance4]---[instance5]---
*/
        qDebug() << "* deleteEntry(): firstInstTime >= rangeBegin";
        delete component;
        if (lastInstTime <= rangeEnd)
        {
/*
------[range.Begin]---[startDate]---[instance2]---[instance3]---[instance4]---[instance5]---[range.End]---
*/
            qDebug() << "* deleteEntry(): lastInstTime <= rangeEnd, removing full entry";
            vector <string> idList;
            idList.push_back(qUID.toStdString());
            multiCalendar->deleteComponents(idList, calendarId, errorCode);
            if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
            {
                rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
                rMap.insert(KErrMessage,"Error deleting entry");
            }
            else
            {
                rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
                rMap.insert(KErrMessage,"SUCCESS");
            }
            retVar = rMap;
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            return retVar;
        }
        else //lastInstTime > rangeEnd
        {
/*
------[range.Begin]---[startDate]---[instance2]---[instance3]---[instance4]---[range.End]---[instance5]---
searching for the 1st instance time which > range.end, updating entry - newStartDate will be 1st instance
time > range.end, newEndTime = newStartDate + diff in seconds between oldStartDate/EndDate
*/
            qDebug() << "* deleteEntry(): lastInstTime > rangeEnd";
            for (int instanceCount = 0; instanceCount < times.size(); instanceCount++)
            {
                currentTime = QDateTime::fromTime_t(times.at(instanceCount));
                qDebug() << "* deleteEntry(): instance #:" << instanceCount;
                qDebug() << "* deleteEntry(): currentInstanceTime:" << currentTime;
                qDebug() << "* deleteEntry(): range.Begin:" << rangeBegin;
                if (currentTime > rangeEnd)
                    break;
            }
            newBeginDate = currentTime;
            QMap<QString,QVariant> qTime = entry.value(KCalendarEntryTime).toMap();
            int secondsBeginToEnd = qTime.value(KTimeDataStart).toDateTime().secsTo(qTime.value(KTimeDataEnd).toDateTime());
            newEndDate = newBeginDate.addSecs(secondsBeginToEnd);
            qDebug() << "* deleteEntry(): newBeginDate:" << newBeginDate;
            qDebug() << "* deleteEntry(): newEndDate:" << newEndDate;
            qTime.remove(KTimeDataStart);
            qTime.remove(KTimeDataEnd);
            qTime.insert(KTimeDataStart, newBeginDate);
            qTime.insert(KTimeDataEnd, newEndDate);
            qDebug() << "* deleteEntry(): new time:" << qTime;
            entry.remove(KCalendarEntryTime);
            entry.insert(KCalendarEntryTime, qTime);
            errorCode = _parseEntry(entry, TRUE);
            entry.clear();
            if (errorCode != 0)
            {
                rMap.insert(KErrCode, errorCode);
                rMap.insert(KErrMessage,"Entry validation failed");
                retVar = rMap;
                qDebug() << "* deleteEntry(): returning retVar:" << retVar;
                return retVar;
            }
            retVar = _updateEntry(qUID);
            parsedEntry.clear();
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            return retVar;
        }
    }
    //end of if (firstInstTime >= rangeBegin)---------------------------------------------
    if (firstInstTime < rangeBegin)
    {
/*
------[startDate]---[instance2]---[range.Begin]---[instance3]---[instance4]---[instance5]---
*/
        qDebug() << "* deleteEntry(): firstInstTime < rangeBegin";
        int calId = multiCalendar->getCalendarIdForComponent(qUID.toStdString(), errorCode);
        if (errorCode != CALENDAR_OPERATION_SUCCESSFUL)
        {
            rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
            rMap.insert(KErrMessage,"Error getting calendar Id");
            retVar = rMap;
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            delete component;
            return retVar;
        }
        if (lastInstTime <= rangeEnd)
        {
/*
------[startDate]---[instance2]---[range.Begin]---[instance3]---[instance4]---[instance5]---[range.end]---
searching for the last instance time which < range.begin, updating entry - newUntilDate will be last instance
time < range.begin, everything else will be deleted
*/
            qDebug() << "* deleteEntry(): lastInstTime <= rangeEnd";
            currentTime = QDateTime::fromTime_t(times.at(1));
            qDebug() << "* deleteEntry(): secondInstTime:" << currentTime;
            qDebug() << "* deleteEntry(): range.begin:" << rangeBegin;
            if (currentTime >= rangeBegin)
            {
/*
------[startDate]---[range.Begin]---[instance2]---[instance3]---[instance4]---[instance5]---[range.end]---
if 2nd instance time is already >= range.begin, then simply remove recurrence from entry
*/
                qDebug() << "* deleteEntry(): secondInstTime > rangeBegin, removing recurrence from entry";
                component->removeRecurrence();
                result = multiCalendar->modifyEvent(pEvent, calId, errorCode);
                delete component;
                if (!result || errorCode != CALENDAR_OPERATION_SUCCESSFUL)
                {
                    rMap.insert(KErrCode,JSEStatusCodes::MISSING_ARG_ERR);
                    rMap.insert(KErrMessage,"Error partially deleting entry");
                }
                else
                {
                    rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
                    rMap.insert(KErrMessage,"SUCCESS");
                }
                retVar = rMap;
                qDebug() << "* deleteEntry(): returning retVar:" << retVar;
                return retVar;
            }
            else //2nd instance time < range.begin
            {
/*
------[startDate]---[instance2]---[range.Begin]---[instance3]---[instance4]---[instance5]---[range.end]---
searching for the last instance time which < range.begin, updating entry - newUntilDate will be last instance
time < range.begin, everything else will be deleted
*/
                qDebug() << "* deleteEntry(): searching for the last instance time which < range.begin to update untilDate";
                delete component;
                for (int instanceCount = (times.size() - 1); instanceCount > 0; instanceCount--)
                {
                    currentTime = QDateTime::fromTime_t(times.at(instanceCount));
                    qDebug() << "* deleteEntry(): instance #:" << instanceCount;
                    qDebug() << "* deleteEntry(): currentInstanceTime:" << currentTime;
                    qDebug() << "* deleteEntry(): range.End:" << rangeEnd;
                    if (currentTime < rangeBegin)
                        break;
                }
                newUntilDate = currentTime;
                qDebug() << "* deleteEntry(): newUntilDate:" << newUntilDate;
                QMap<QString,QVariant> repeatRule = entry.value(KCalendarEntryRepeatRule).toMap();
                repeatRule.remove(KRepeatRuleUntilDate);
                repeatRule.insert(KRepeatRuleUntilDate, newUntilDate);
                entry.remove(KCalendarEntryRepeatRule);
                entry.insert(KCalendarEntryRepeatRule, repeatRule);
                qDebug() << "* deleteEntry(): updated untilDate:" << entry.value(KCalendarEntryRepeatRule).toMap().value(KRepeatRuleUntilDate).toDateTime();
                errorCode = _parseEntry(entry, TRUE);
                entry.clear();
                if (errorCode != 0)
                {
                    rMap.insert(KErrCode, errorCode);
                    rMap.insert(KErrMessage,"Entry validation failed");
                    retVar = rMap;
                    qDebug() << "* deleteEntry(): returning retVar:" << retVar;
                    return retVar;
                }
                retVar = _updateEntry(qUID);
                parsedEntry.clear();
                qDebug() << "* deleteEntry(): returning retVar:" << retVar;
                return retVar;

            }
        }
        //end of if (lastInstTime <= rangeEnd)---------------------------------------------
        if (lastInstTime > rangeEnd)
        {
/*
------[startDate]---[instance2]---[range.Begin]---[instance3]---[range.end]---[instance4]---[instance5]---
original entry will be update with newUntilDate. New entry will be created with newStartDate = 1st instance
time > range.end, newEndTime = newStartDate + diff in seconds between oldStartDate/EndDate
*/
            delete component;
            QMap<QString,QVariant> newEntry = entry;
            qDebug() << "* deleteEntry(): lastInstTime > rangeEnd";
            for (int instanceCount = (times.size() - 1); instanceCount > 0; instanceCount--)
            {
                currentTime = QDateTime::fromTime_t(times.at(instanceCount));
                qDebug() << "* deleteEntry(): instance #:" << instanceCount;
                qDebug() << "* deleteEntry(): currentInstanceTime:" << currentTime;
                qDebug() << "* deleteEntry(): range.End:" << rangeEnd;
                if (currentTime < rangeBegin)
                    break;
            }
            newUntilDate = currentTime;
            qDebug() << "* deleteEntry(): newUntilDate:" << newUntilDate;
            QMap<QString,QVariant> repeatRule = entry.value(KCalendarEntryRepeatRule).toMap();
            repeatRule.remove(KRepeatRuleUntilDate);
            repeatRule.insert(KRepeatRuleUntilDate, newUntilDate);
            entry.remove(KCalendarEntryRepeatRule);
            entry.insert(KCalendarEntryRepeatRule, repeatRule);
            qDebug() << "* deleteEntry(): updated untilDate:" << entry.value(KCalendarEntryRepeatRule).toMap().value(KRepeatRuleUntilDate).toDateTime();
            errorCode = _parseEntry(entry, TRUE);
            entry.clear();
            if (errorCode != 0)
            {
                rMap.insert(KErrCode, errorCode);
                rMap.insert(KErrMessage,"Entry validation failed");
                retVar = rMap;
                qDebug() << "* deleteEntry(): returning retVar:" << retVar;
                return retVar;
            }
            retVar = _updateEntry(qUID);
            parsedEntry.clear();
//creating new entry
            for (int instanceCount = 0; instanceCount < times.size(); instanceCount++)
            {
                currentTime = QDateTime::fromTime_t(times.at(instanceCount));
                qDebug() << "* deleteEntry(): instance #:" << instanceCount;
                qDebug() << "* deleteEntry(): currentInstanceTime:" << currentTime;
                qDebug() << "* deleteEntry(): range.End:" << rangeEnd;
                if (currentTime > rangeEnd)
                    break;
            }
            newBeginDate = currentTime;
            QMap<QString,QVariant> qTime = newEntry.value(KCalendarEntryTime).toMap();
            int secondsBeginToEnd = qTime.value(KTimeDataStart).toDateTime().secsTo(qTime.value(KTimeDataEnd).toDateTime());
            newEndDate = newBeginDate.addSecs(secondsBeginToEnd);
            qDebug() << "* deleteEntry(): newBeginDate:" << newBeginDate;
            qDebug() << "* deleteEntry(): newEndDate:" << newEndDate;
            qTime.remove(KTimeDataStart);
            qTime.remove(KTimeDataEnd);
            qTime.insert(KTimeDataStart, newBeginDate);
            qTime.insert(KTimeDataEnd, newEndDate);
            qDebug() << "* deleteEntry(): new time:" << qTime;
            newEntry.remove(KCalendarEntryTime);
            newEntry.insert(KCalendarEntryTime, qTime);
            errorCode = _parseEntry(newEntry, TRUE);
            newEntry.clear();
            if (errorCode != 0)
            {
                rMap.insert(KErrCode, errorCode);
                rMap.insert(KErrMessage,"Entry validation failed");
                retVar = rMap;
                qDebug() << "* deleteEntry(): returning retVar:" << retVar;
                return retVar;
            }
            retVar = _createEntry();
            parsedEntry.clear();
            qDebug() << "* deleteEntry(): returning retVar:" << retVar;
            return retVar;
        }
        //end of if (lastInstTime > rangeEnd)---------------------------------------------
    }
    //end of if (firstInstTime < rangeBegin)---------------------------------------------
}

QVariant QCalendarService::cancel(const int transactionId)
{

    QVariant retVar = NULL;
    QMap<QString,QVariant> rMap;

    if (m_onGoingAsyncTransactions.contains(transactionId)) {
        // Remove request from the array
        m_onGoingAsyncTransactions.removeAt(m_onGoingAsyncTransactions.indexOf(transactionId));
        rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
        rMap.insert(KErrMessage,"SUCCESS");
        retVar = rMap;
    }
    else
    {
        rMap.insert(KErrCode,JSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert(KErrMessage,"Transaction doesn't exist");
        retVar = rMap;
    }
    return retVar;
}

QVariant QCalendarService::getList(const QMap<QString,QVariant>& matchCriteria,int transId)
{
    qDebug() << "* getList() async";
    QMap<QString,QVariant> matchPattern = matchCriteria;
    QMap<QString,QVariant> rMap = _validateMatchPattern(matchPattern);
    if (rMap.value(KErrCode) != JSEStatusCodes::SUCCESS)
    {
        qDebug() << "* getList(): matchpattern is not valid, return error map with transaction id";
        rMap.insert(KTransId,transId);
        return (QVariant) rMap;
    }

    m_onGoingAsyncTransactions.append(transId);
    qDebug() << "* getList(): creating task...";
    GetListTask * task = new GetListTask(multiCalendar, matchPattern,transId);
    qDebug() << "* getList(): done, connecting...";
    connect(task, SIGNAL(listOfCalendarEntries(const QList<QVariantMap>*, qint32, qint32)),
            this, SLOT(fireGetListCallback(const QList<QVariantMap>*, qint32, qint32)));
    qDebug() << "* getList(): done, starting task in separate thread...";
    QThreadPool::globalInstance()->start(task);

    rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
    rMap.insert(KErrMessage,"SUCCESS");
    rMap.insert(KTransId,transId);
    QVariant retVar = rMap;
    qDebug() << "* getList(): returing reVar:" << retVar;
    return retVar;
}

QObject* QCalendarService::getList(const QMap<QString,QVariant>& matchCriteria)
{
    qDebug() << "* getList() sync";
    QObject *iterator = NULL;
    QMap<QString,QVariant> matchPattern = matchCriteria;
    QMap<QString,QVariant> rMap = _validateMatchPattern(matchPattern);
    if (rMap.value(KErrCode) != JSEStatusCodes::SUCCESS)
    {
        qDebug() << "* getList(): matchpattern is not valid, return empty iterator";
        return iterator;
    }

    GetListTask * task = new GetListTask(multiCalendar, matchPattern,-1);
    QList<QVariantMap>* mapList = new QList<QVariantMap>();
    qint32 status = task->getList(mapList);
    if (status == JSEStatusCodes::SUCCESS && (mapList->count() > 0))
    {
        iterator = new QCalendarIterator(*mapList);
        iterator->setParent(this);
    }
    delete task;
    delete mapList;
    qDebug() << "* getList(): returing iterator:" << iterator;
    return iterator;
}

QVariant QCalendarService::subscribeNotification(int transactionId)
{
    qDebug() << "* subscribeNotification(): transactionId:" << transactionId;
    QMap<QString,QVariant> rMap;
    if (m_isNotifyInProgress)
    {
        rMap.insert(KErrMessage,"multiple requests not suppported");
        rMap.insert(KErrCode, JSEStatusCodes::SERVICE_IN_USE_ERR);
        rMap.insert(KTransId,KInvalidTransId);
        qDebug() << "* subscribeNotification(): rMap:" << rMap;
        return rMap;
    }
    qDebug() << "* subscribeNotification(): registering notifications...";
    QDBusConnection::sessionBus().connect("", DBUS_PATH, DBUS_INTERFACE, DBUS_EVENT,
                                          this, SLOT(fireSubscribeCallback(const QDBusMessage&)));
    m_isNotifyInProgress = true;
    m_notifyTrId = transactionId;

    rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
    rMap.insert(KErrMessage,"SUCCESS");
    rMap.insert(KTransId,transactionId);
    qDebug() << "* subscribeNotification(): done, return rMap:" << rMap;
    return rMap;
}

QVariant QCalendarService::cancelNotification()
{
    qDebug() << "* cancelNotification(): m_isNotifyInProgress:" << m_isNotifyInProgress;
    QMap<QString,QVariant> rMap;
    if ( !m_isNotifyInProgress )
    {
        rMap.insert(KErrMessage, "multiple requests not suppported ");
        rMap.insert(KErrCode, JSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert(KTransId, KInvalidTransId);
        qDebug() << "* cancelNotification(): rMap:" << rMap;
        return rMap;
    }
    qDebug() << "* cancelNotification(): unregistering notifications...";
    QDBusConnection::sessionBus().disconnect("", DBUS_PATH, DBUS_INTERFACE, DBUS_EVENT,
                                             this, SLOT(fireSubscribeCallback(const QDBusMessage&)));
    rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
    rMap.insert(KErrMessage,"SUCCESS");
    rMap.insert(KTransId, m_notifyTrId);
    m_isNotifyInProgress = false;
    m_notifyTrId = KInvalidTransId;
    qDebug() << "* cancelNotification(): done return rMap:" << rMap;
    return rMap;
}

void QCalendarService::fireGetListCallback(const QList<QVariantMap>* mapList, qint32 trId, qint32 status)
{
    qDebug() << "* fireGetListCallback(): received singal listOfCalendarEntries";
    if (m_onGoingAsyncTransactions.contains(trId)) {
        qDebug() << "* fireGetListCallback(): received mapList:" << *mapList << ", creating iterator";
        QObject *iterator = new QCalendarIterator(*mapList);
        iterator->setParent(this);
        qDebug() << "* fireGetListCallback(): emmiting getListCallback with status:" << status;
        qDebug() << "* fireGetListCallback(): emmiting getListCallback with trId:" << trId;
        qDebug() << "* fireGetListCallback(): emmiting getListCallback with iterator:" << iterator;
        emit getListCallback(status, trId, iterator);
        m_onGoingAsyncTransactions.removeAt(m_onGoingAsyncTransactions.indexOf(trId));
    }
    qDebug() << "* fireGetListCallback(): deleting mapList";
    delete mapList;
}

void QCalendarService::fireSubscribeCallback(const QDBusMessage& messages)
{
    QList<QVariantMap> mapList;
    qDebug() << "* fireSubscribeCallback(): messages:" << messages;
    QList<QVariant> arguments = messages.arguments();
    QString message = arguments.first().toString();
    int index;
    QString changeType;
    mapList.clear();
    qDebug() << "* fireSubscribeCallback(): message:" << message;
    if ((index = message.indexOf(E_ADDED, 0, Qt::CaseSensitive)) != -1)
    {
        qDebug() << "* fireSubscribeCallback(): event added";
        message.remove(0, (index + E_ADDED.length()));
        changeType = KChangeTypeAdd;
    }
    else if ((index = message.indexOf(T_ADDED, 0, Qt::CaseSensitive)) != -1)
    {
        qDebug() << "* fireSubscribeCallback(): todo added";
        message.remove(0, (index + T_ADDED.length()));
        changeType = KChangeTypeAdd;
    }
    else if ((index = message.indexOf(E_DELETED, 0, Qt::CaseSensitive)) != -1)
    {
        qDebug() << "* fireSubscribeCallback(): event deleted";
        message.remove(0, (index + E_DELETED.length()));
        changeType = KChangeTypeDelete;
    }
    else if ((index = message.indexOf(T_DELETED, 0, Qt::CaseSensitive)) != -1)
    {
        qDebug() << "* fireSubscribeCallback(): todo deleted";
        message.remove(0, (index + T_DELETED.length()));
        changeType = KChangeTypeDelete;
    }
    else if ((index = message.indexOf(E_MODIFIED, 0, Qt::CaseSensitive)) != -1)
    {
        qDebug() << "* fireSubscribeCallback(): event modified";
        message.remove(0, (index + E_MODIFIED.length()));
        changeType = KChangeTypeModify;
    }
    else if ((index = message.indexOf(T_MODIFIED, 0, Qt::CaseSensitive)) != -1)
    {
        qDebug() << "* fireSubscribeCallback(): todo modified";
        message.remove(0, (index + T_MODIFIED.length()));
        changeType = KChangeTypeModify;
    }
    if (index != -1)
    {
        QList<QString> parsedMessage = _parseDbusMessage(message);
        qDebug() << "* fireSubscribeCallback(): parsedMessage.count:" << parsedMessage.count();
        qDebug() << "* fireSubscribeCallback(): parsedMessage:" << parsedMessage;
        QVariantMap map;
        for (int i = 1; i <= parsedMessage.first().toInt(); i++)
        {
            map.insert(KChangeType, changeType);
            map.insert(KId, parsedMessage.at(i));
            mapList.append(map);
        }
    }
    qDebug() << "* fireSubscribeCallback(): mapList:" << mapList;
    if (mapList.count())
    {
        if (m_isNotifyInProgress)
        {
            QObject *iterator = new QCalendarIterator(mapList);
            iterator->setParent(this);
            qDebug() << "* fireSubscribeCallback(): emmitting notifyCallback";
            emit notifyCallback(0, m_notifyTrId, iterator);
        }
    }
}

QList<QString> QCalendarService::_parseDbusMessage(QString message)
{
    QList<QString> parsedMessage;
    int index = message.indexOf(":", 0, Qt::CaseInsensitive);
    QString count = message.left(index);
    QString ids = message.right((message.length() - index - 1));
    parsedMessage.append(count);
    qDebug() << "* _parseDbusMessage(): count:" << count;
    qDebug() << "* _parseDbusMessage(): ids:" << ids;
    for (int i = 0; i < count.toInt(); i++)
    {
        index = ids.indexOf(",", 0, Qt::CaseInsensitive);
        QString id = ids.left((index));
        parsedMessage.append(id);
        ids.remove(0, (index + 1));
    }
    return parsedMessage;
}

int QCalendarService::_parseEntry(const QMap<QString,QVariant>& calendarEntry, bool isUpdate)
{
    qDebug() << "_parseEntry()";
    parsedEntry.clear();
//TYPE--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating type...";
    QString qType = calendarEntry.value(KCalendarEntryType).toString();
    if (qType.isEmpty() ||
        qType.isNull())
    {
        qDebug() << "* _parseEntry(): type field missed";
        return JSEStatusCodes::MISSING_ARG_ERR;
    }
    if (qType.compare(KTypeMeeting, Qt::CaseInsensitive)  &&
        qType.compare(KTypeToDo, Qt::CaseInsensitive) &&
        qType.compare(KTypeReminder, Qt::CaseInsensitive) &&
        qType.compare(KTypeAnniversary, Qt::CaseInsensitive) &&
        qType.compare(KTypeDayEvent, Qt::CaseInsensitive) )
    {
        qDebug() << "* _parseEntry(): type field invalid";
        return JSEStatusCodes::INVALID_ARG_ERR;
    }
    parsedEntry.insert(KCalendarEntryType, qType);
    qDebug() << "* _parseEntry(): OK, type:" << qType;
//SUMMARY--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating summary...";
    QString summary;//= calendarEntry.value(KCalendarEntrySummary).toString();
    if (calendarEntry.value(KCalendarEntrySummary).toString().isEmpty() ||
        (isUpdate == TRUE && calendarEntry.value(KCalendarEntrySummary).isNull()))
    {
        qDebug() << "* _parseEntry(): summary empty, filling with spaces";
        summary = " ";
    }
    else
        summary = calendarEntry.value(KCalendarEntrySummary).toString();
    parsedEntry.insert(KCalendarEntrySummary, summary);
    qDebug() << "* _parseEntry(): OK, summary:" << summary;
//DESCRIPTION--------------------------------------------------------------------
    QString description;// = calendarEntry.value(KCalendarEntryDescription).toString();
    if (isUpdate && calendarEntry.value(KCalendarEntryDescription).isNull())
        description = " ";
    else
        description = calendarEntry.value(KCalendarEntryDescription).toString();
    parsedEntry.insert(KCalendarEntryDescription, description);
    qDebug() << "* _parseEntry(): description:" << description;
//LOCATION--------------------------------------------------------------------
    QString location;
    if (qType.compare(KTypeMeeting, Qt::CaseInsensitive) == 0)
    {
        if (isUpdate && calendarEntry.value(KCalendarEntryLocation).isNull())
            location = " ";
        else
            location = calendarEntry.value(KCalendarEntryLocation).toString();
    }
    else
        location  = " ";
    parsedEntry.insert(KCalendarEntryLocation, location);
    qDebug() << "* _parseEntry(): location:" << location;
//TIME--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating time...";
    QDateTime qStartTime;
    QDateTime qEndTime;
    QMap<QString,QVariant> qTime = calendarEntry.value(KCalendarEntryTime).toMap();
    if (qTime.isEmpty())
    {
        qDebug() << "* _parseEntry(): time field missed";
        return JSEStatusCodes::MISSING_ARG_ERR;
    }
    if (qType.compare(KTypeToDo, Qt::CaseInsensitive))
    {
        if (!qTime.contains(KTimeDataStart) ||
            qTime.value(KTimeDataStart).isNull() ||
            !qTime.value(KTimeDataStart).toDateTime().isValid())
        {
            qDebug() << "* _parseEntry(): begin time invalid";
            return JSEStatusCodes::INVALID_ARG_ERR;
        }
    }
    qStartTime = qTime.value(KTimeDataStart).toDateTime();
//All Fremantle Events shall have start and end time
    if (qType.compare(KTypeAnniversary, Qt::CaseInsensitive) == 0 || qType.compare(KTypeReminder, Qt::CaseInsensitive) == 0)
        qEndTime = qStartTime;
    else
    {
        if (!qTime.contains(KTimeDataEnd) && !qType.compare(KTypeDayEvent, Qt::CaseInsensitive))
        {
            qDebug() << "* _parseEntry(): end time is missing for DayEvent, setting equal to startTime";
            qEndTime = qStartTime;
        }
        else if (!qTime.contains(KTimeDataEnd) ||
             qTime.value(KTimeDataEnd).isNull() ||
            !qTime.value(KTimeDataEnd).toDateTime().isValid())
        {
            qDebug() << "* _parseEntry(): end time invalid";
            return JSEStatusCodes::INVALID_ARG_ERR;
        }
        else
        {
            qEndTime = qTime.value(KTimeDataEnd).toDateTime();
        }
    }

    parsedEntry.insert(KTimeDataStart, qStartTime);
    parsedEntry.insert(KTimeDataEnd, qEndTime);
    qDebug() << "* _parseEntry(): OK, time:" << qTime;
//ALARM--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating alarm time...";
    QDateTime currentTime;
    if (qTime.contains(KTimeDataAlarm) &&
        !qTime.value(KTimeDataAlarm).isNull() &&
        !qTime.value(KTimeDataAlarm).toDateTime().isValid())
    {
        qDebug() << "* _parseEntry(): alarm time is invalid";
        return JSEStatusCodes::INVALID_ARG_ERR;
    }
    if (qTime.contains(KTimeDataStart) &&
        qTime.contains(KTimeDataEnd))
    {
        if (qTime.value(KTimeDataEnd).toDateTime() <= qTime.value(KTimeDataStart).toDateTime() &&
            (!qType.compare(KTypeMeeting, Qt::CaseInsensitive) || !qType.compare(KTypeDayEvent, Qt::CaseInsensitive)))
        {
            qDebug() << "* _parseEntry(): begin time is after end time";
            return JSEStatusCodes::INVALID_ARG_ERR;
        }
        if (qTime.contains(KTimeDataAlarm) &&
            !qTime.value(KTimeDataAlarm).isNull() &&
            qTime.value(KTimeDataAlarm).toDateTime().isValid() &&
            qTime.value(KTimeDataAlarm).toDateTime() >= qTime.value(KTimeDataStart).toDateTime())
        {
            qDebug() << "* _parseEntry(): alarm time is after begin time";
            return JSEStatusCodes::INVALID_ARG_ERR;
        }
    }
    if (!qTime.value(KTimeDataAlarm).isNull() &&
        qTime.value(KTimeDataAlarm).toDateTime() < currentTime.currentDateTime())
    {
        qDebug() << "* _parseEntry(): alarm time is before current time";
        return JSEStatusCodes::INVALID_ARG_ERR;
    }
    QDateTime qAlarmTime = qTime.value(KTimeDataAlarm).toDateTime();
    parsedEntry.insert(KTimeDataAlarm, qAlarmTime);
    qDebug() << "* _parseEntry(): OK, alarm:" << qAlarmTime;

//STATUS--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating status...";
    QString qStatus;
    if ((!qType.compare(KTypeToDo, Qt::CaseInsensitive) || !qType.compare(KTypeMeeting, Qt::CaseInsensitive)) &&
        calendarEntry.contains(KCalendarEntryStatus))
    {
        if (calendarEntry.value(KCalendarEntryStatus).canConvert<QString>())
        {
            //this is needed to reconize when float/int 0 is passed and when null
            qStatus = calendarEntry.value(KCalendarEntryStatus).toString();
            qDebug() << "* _parseEntry(): canConvert<QString>: TRUE, qStatus:" << qStatus;
        }
        if (!qStatus.isEmpty())
        {
            if (!qType.compare(KTypeToDo, Qt::CaseInsensitive) &&
               (qStatus.compare(KStatusNeedsAction) &&
                qStatus.compare(KStatusCompleted)) )
            {
                return JSEStatusCodes::INVALID_ARG_ERR;
            }

            if (!qType.compare(KTypeMeeting, Qt::CaseInsensitive) &&
               (qStatus.compare(KStatusTentative) &&
                qStatus.compare(KStatusConfirmed) &&
                qStatus.compare(KStatusCancelled)))
            {
                return JSEStatusCodes::INVALID_ARG_ERR;
            }

            qDebug() << "* _parseEntry(): status before conversion:" << qStatus;
            if (qStatus.compare(KStatusNeedsAction, Qt::CaseInsensitive) == 0)
                qStatus = "0";
            else if (qStatus.compare(KStatusCompleted, Qt::CaseInsensitive) == 0)
                qStatus = "1";
            else if (qStatus.compare(KStatusTentative, Qt::CaseInsensitive) == 0)
                qStatus = "5";
            else if (qStatus.compare(KStatusConfirmed, Qt::CaseInsensitive) == 0)
                qStatus = "4";
            else if (qStatus.compare(KStatusCancelled, Qt::CaseInsensitive) == 0)
                qStatus = "3";
            else
            {
                if (qType.compare(KTypeToDo, Qt::CaseInsensitive) == 0)
                    qStatus = "0";
                else if (qType.compare(KTypeMeeting, Qt::CaseInsensitive) == 0)
                    qStatus = "5";
            }
        }
        else
        {
            qDebug() << "* _parseEntry(): status is missing, setting default";
            if (qType.compare(KTypeToDo, Qt::CaseInsensitive) == 0)
                qStatus = "0";
            else if (qType.compare(KTypeMeeting, Qt::CaseInsensitive) == 0)
                qStatus = "5";
        }
        parsedEntry.insert(KCalendarEntryStatus, qStatus);
        qDebug() << "* _parseEntry(): OK, status:" << qStatus.toInt();
    }
    else
    {
        if (calendarEntry.contains(KCalendarEntryStatus))
        {
            qDebug() << "* _parseEntry(): status set for non Meeting or Todo - ignoring..";
            qStatus = "-1";
            parsedEntry.insert(KCalendarEntryStatus, qStatus);
        }
    }
//INSTANCESTARTTIME--------------------------------------------------------------------
    if (calendarEntry.contains(KCalendarEntryInstanceStartTime))
    {
       if (!calendarEntry.value(KCalendarEntryInstanceStartTime).isNull() &&
           !calendarEntry.value(KCalendarEntryInstanceStartTime).canConvert<QDateTime>())
        {
            qDebug() << "instanceStartTime is invalid";
            return JSEStatusCodes::INVALID_ARG_ERR;
        }
        else
        {
            qDebug() << "instanceStartTime is valid, return NOT_SUPPORTED_ERR";
            return JSEStatusCodes::NOT_SUPPORTED_ERR;
        }
    }
//EXCEPTIONDATES--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating exceptionDates...";
    QList<QVariant> exceptionDates;
    if (calendarEntry.contains(KCalendarEntryExceptionDates))
    {
        exceptionDates =  calendarEntry.value(KCalendarEntryExceptionDates).toList();
        if (!calendarEntry.value(KCalendarEntryExceptionDates).isNull())
        {
            if (calendarEntry.value(KCalendarEntryExceptionDates).toList().count() < 0)
            {
                qDebug()<< "* _parseEntry(): exceptionDates invalid";
                return JSEStatusCodes::INVALID_ARG_ERR;
            }
            foreach(QVariant qDate, exceptionDates)
            {
                if (!qDate.toDateTime().isValid())
                {
                    qDebug()<< "* _parseEntry(): exceptionDates invalid";
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
            }
        }
        parsedEntry.insert(KCalendarEntryExceptionDates, exceptionDates);
        qDebug() << "* _parseEntry(): OK, exceptionDates:" << exceptionDates;
    }
    else
    {
        qDebug() << "* _parseEntry(): exceptionDates are missing";
    }
//REPEATRULE--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating repeatRule...";
    QMap<QString,QVariant> repeatRule = calendarEntry.value(KCalendarEntryRepeatRule).toMap();
    qDebug() << "* _parseEntry(): repeatRule:" << repeatRule;
    if (calendarEntry.contains(KCalendarEntryRepeatRule))
    {
        QString frequency = repeatRule.value(KRepeatRuleFrequency).toString();
        if (!calendarEntry.value(KCalendarEntryRepeatRule).isNull())
        {
            if (repeatRule.contains(KRepeatRuleFrequency))
            {
                if (repeatRule.value(KRepeatRuleFrequency).isNull())
                {
                    qDebug() << "* _parseEntry(): frequency is null";
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
                if (frequency.compare(KFrequencyDaily, Qt::CaseInsensitive) &&
                   frequency.compare(KFrequencyWeekly, Qt::CaseInsensitive) &&
                   frequency.compare(KFrequencyMonthly, Qt::CaseInsensitive) &&
                   frequency.compare(KFrequencyYearly, Qt::CaseInsensitive))
                {
                    qDebug() << "* _parseEntry(): invalid frequency";
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
                qDebug() << "* _parseEntry(): frequency:" << frequency;
            }
            else
            {
                qDebug() << "* _parseEntry(): frequency missed";
                return JSEStatusCodes::MISSING_ARG_ERR;
            }
            if (repeatRule.contains(KRepeatRuleStartDate))
            {
                if (!repeatRule.value(KRepeatRuleStartDate).isNull())
                {
                    if (!repeatRule.value(KRepeatRuleStartDate).canConvert<QDateTime>())
                    {
                        qDebug() << "* _parseEntry(): Repeat rule startDate invalid";
                        return JSEStatusCodes::INVALID_ARG_ERR;
                    }
                }
                else if (repeatRule.value(KRepeatRuleStartDate).canConvert<QDateTime>() &&
                          !repeatRule.value(KRepeatRuleStartDate).toDateTime().isValid() )
                {
                    qDebug() << "* _parseEntry(): Repeat rule startDate invalid";
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
            }

            QDateTime qStartDate = repeatRule.value(KRepeatRuleStartDate).toDateTime();
            qDebug() << "* _parseEntry(): qStartDate:" << qStartDate;
            if (!qStartDate.isValid())
            {
                qDebug() << "* _parseEntry(): !qStartDate.isValid()";
                qStartDate = qStartTime;
            }

            parsedEntry.insert(KRepeatRuleStartDate, qStartDate);
            qDebug() << "* _parseEntry(): qStartDate:" << qStartDate;
            QDateTime qUntilDate;
            if (repeatRule.contains(KRepeatRuleUntilDate))
            {
                if (!repeatRule.value(KRepeatRuleUntilDate).isNull())
                {
                    if (!repeatRule.value(KRepeatRuleUntilDate).canConvert<QDateTime>())
                    {
                        //qDebug() << "Repeat rule untilDate invalid";
                        return JSEStatusCodes::INVALID_ARG_ERR;
                    }
                }
                if (repeatRule.value(KRepeatRuleUntilDate).toDateTime().isNull() ||
                    !repeatRule.value(KRepeatRuleUntilDate).toDateTime().isValid())
                    qUntilDate = MAX_DATE_TIME;
                else
                    qUntilDate = repeatRule.value(KRepeatRuleUntilDate).toDateTime();
            }
            else
            {
                qUntilDate = MAX_DATE_TIME;
            }
            if (qUntilDate < qStartDate)
            {
                qDebug() << "* _parseEntry(): Repeat rule untilDate before startDate";
                return JSEStatusCodes::INVALID_ARG_ERR;
            }
            parsedEntry.insert(KRepeatRuleUntilDate, qUntilDate);
            qDebug() << "* _parseEntry(): qUntilDate:" << qUntilDate;

            //difference between start and end time in days
            float days = qStartTime.secsTo(qEndTime);
            days /= (3600*24);
            if (!repeatRule.value(KRepeatRuleFrequency).toString().compare(KFrequencyDaily, Qt::CaseInsensitive) &&
                days > 1)
            {
                qDebug() << "* _parseEntry(): 24h event for daily repeat";
                return JSEStatusCodes::INVALID_ARG_ERR;
            }
            if (!repeatRule.value(KRepeatRuleFrequency).toString().compare(KFrequencyWeekly, Qt::CaseInsensitive) &&
                days > 7)
            {
                qDebug() << "* _parseEntry(): 1 week event for weekly repeat";
                return JSEStatusCodes::INVALID_ARG_ERR;
            }
            if (!repeatRule.value(KRepeatRuleFrequency).toString().compare(KFrequencyMonthly, Qt::CaseInsensitive) &&
                days > 31)
            {
                qDebug() << "* _parseEntry(): 1 month event for monthly repeat ";
                return JSEStatusCodes::INVALID_ARG_ERR;
            }
            if (!repeatRule.value(KRepeatRuleFrequency).toString().compare(KFrequencyYearly, Qt::CaseInsensitive) &&
                days > 365)
            {
                qDebug() << "* _parseEntry(): 1 year event for yearly repeat ";
                return JSEStatusCodes::INVALID_ARG_ERR;
            }

            if (repeatRule.contains(KRepeatRuleMonth))
            {
                int month = repeatRule.value(KRepeatRuleMonth).toInt();
                if ( month < 0 || month > 11 )
                {
                    qDebug() << "* _parseEntry(): Repeat Rule Month failed";
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
                parsedEntry.insert(KRepeatRuleMonth, month);
                qDebug() << "* _parseEntry(): Repeat rule month:" << month;
            }
            else
            {
                qDebug() << "* _parseEntry(): Repeat rule month is missing";
            }
            if (repeatRule.contains(KRepeatRuleWeekDays))
            {
                qDebug() << "* _parseEntry(): weekDays are present, validating...";
                if (!repeatRule.value(KRepeatRuleWeekDays).isNull() &&
                    !repeatRule.value(KRepeatRuleWeekDays).canConvert<QVariantList>())
                {
                    //not an array
                    qDebug() << "* _parseEntry(): weekDays are not valid";
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
                qDebug() << "* _parseEntry(): returning weekDays not supported";
                return JSEStatusCodes::NOT_SUPPORTED_ERR;
            }
            if (repeatRule.contains(KRepeatRuleDaysOfMonth))
            {
                qDebug() << "* _parseEntry(): daysOfMonth are present, validating...";
                //return JSEStatusCodes::INVALID_ARG_ERR; // not supported
                if (!repeatRule.value(KRepeatRuleDaysOfMonth).isNull() &&
                    !repeatRule.value(KRepeatRuleDaysOfMonth).canConvert<QVariantList>())
                {
                    //not an array
                    qDebug() << "* _parseEntry(): daysOfMonth are not valid";
                    return JSEStatusCodes::INVALID_ARG_ERR;
                }
                qDebug() << "* _parseEntry(): returning daysOfMonth not supported";
                return JSEStatusCodes::NOT_SUPPORTED_ERR;
            }

            QList<QVariant> monthDays = repeatRule.value(KRepeatRuleMonthDates).toList();
            if (repeatRule.contains(KRepeatRuleMonthDates))
            {
                if (!repeatRule.value(KRepeatRuleMonthDates).isNull())
                {
                    if (!repeatRule.value(KRepeatRuleMonthDates).canConvert<QVariantList>())
                    {
                        qDebug()<< "* _parseEntry(): monthDates is not an QVariantList";
                        return JSEStatusCodes::INVALID_ARG_ERR;
                    }

                    if (monthDays.count() < 0)
                    {
                        qDebug() << "* _parseEntry(): count of monthDays shall >=0";
                        return JSEStatusCodes::INVALID_ARG_ERR;
                    }
                    for (int i = 0; i < monthDays.count() ; i++)
                    {
                        int mthdays = monthDays[i].toInt();
                        if ( mthdays < 1 || mthdays > 31 )
                        {
                            qDebug() << "* _parseEntry(): mthdays shall be from 1 to 31";
                            return JSEStatusCodes::INVALID_ARG_ERR;
                        }
                    }
                    parsedEntry.insert(KRepeatRuleMonthDates, monthDays);
                    qDebug() << "* _parseEntry(): Repeat rule monthDays:" << monthDays;
                }
            }
        }
        if (repeatRule.value(KRepeatRuleInterval).toString().isEmpty())
        {
            qDebug() << "* _parseEntry(): interval is missing, will be used interval=1";
        }
        else
        {
            QString interval = repeatRule.value(KRepeatRuleInterval).toString();
            parsedEntry.insert(KRepeatRuleInterval, interval);
            qDebug() << "* _parseEntry(): interval:" << interval;
        }

        QString repeatFrequency;
        QString repeatType;
        if (frequency.compare(KFrequencyDaily, Qt::CaseInsensitive) == 0)
        {
            repeatFrequency = "0";
            repeatType = "1";
        }
        if (frequency.compare(KFrequencyWeekly, Qt::CaseInsensitive) == 0)
        {
            repeatFrequency = "1";
            repeatType = "3";
        }
        if (frequency.compare(KFrequencyMonthly, Qt::CaseInsensitive) == 0)
        {
            repeatFrequency = "3";
            repeatType = "4";
        }
        if (frequency.compare(KFrequencyYearly, Qt::CaseInsensitive) == 0)
        {
            repeatFrequency = "5";
            repeatType = "5";
        }
        parsedEntry.insert(KRepeatRuleFrequency, repeatFrequency);
        parsedEntry.insert(KRepeatRuleType, repeatType);
        qDebug() << "* _parseEntry(): repeatFrequency:" << repeatFrequency << ", repeatType:" << repeatType;
        qDebug() << "* _parseEntry(): OK, repeatRule:" << repeatRule;
    }
    else
    {
        qDebug() << "* _parseEntry(): repeatRule is missing";
    }

//PRIORITY--------------------------------------------------------------------
    qDebug() << "* _parseEntry(): validating priority...";
    if (calendarEntry.contains(KCalendarEntryPriority))
    {
        int priority = calendarEntry.value(KCalendarEntryPriority).toInt();
        if (priority > 9 || priority < 0)
        {
            qDebug() << "* _parseEntry(): priority validation failed";
            return JSEStatusCodes::DATA_OUT_OF_RANGE_ERR;
        }
        qDebug() << "* _parseEntry(): OK, priority:" << priority;
        parsedEntry.insert(KCalendarEntryPriority, priority);
    }
    else
    {
        qDebug() << "* _parseEntry(): priority is missing, setting default";
        parsedEntry.insert(KCalendarEntryPriority, 0);
    }
//--------------------------------------------------------------------
    qDebug() << "_parseEntry(): success";
    return JSEStatusCodes::SUCCESS;
}

QVariant QCalendarService::_createEntry()
{
    CEvent *pEvent = NULL;
    CTodo *pTodo = NULL;
    CRecurrence *pRecurrence = NULL;
    QMap<QString,QVariant> rMap;
    QVariant retVar;
    bool result = FALSE;
    bool typeIsTodo = FALSE;
    int errorCode = 0;

    qDebug() << "_createEntry()";
    QString entryType = parsedEntry.value(KCalendarEntryType).toString();
    qDebug() << "* _createEntry(): type is:" << entryType;

    if (entryType.compare(KTypeToDo, Qt::CaseInsensitive) == 0)
        typeIsTodo = TRUE;
    if (typeIsTodo)
    {
//Creating Todo-------------------------------------------------------------------------------------
        pTodo = new CTodo(parsedEntry.value(KCalendarEntrySummary).toString().toStdString().data(),
                          parsedEntry.value(KTimeDataEnd).toDateTime().toTime_t(),
                          parsedEntry.value(KCalendarEntryStatus).toInt());
        if (pTodo == NULL)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error creating Todo");
            retVar = rMap;
            qDebug() << "* _createEntry(): returning retVar:" << retVar;
            return retVar;
        }
//Todo: add description-------------------------------------------------------------------------------------
        result = pTodo->setDescription(parsedEntry.value(KCalendarEntryDescription).toString().toStdString().data());
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error setting Description for Todo");
            retVar = rMap;
            delete pTodo;
            qDebug() << "* _createEntry(): returning retVar:" << retVar;
            return retVar;
        }
//Todo: add priority-------------------------------------------------------------------------------------
        if (!parsedEntry.value(KCalendarEntryPriority).toString().isEmpty())
        {
            qDebug() << "* _createEntry(): setting priority...";
            result = pTodo->setPriority(parsedEntry.value(KCalendarEntryPriority).toInt());
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error setting Priority for Todo");
                retVar = rMap;
                delete pTodo;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _createEntry(): priority:" << pTodo->getPriority();
        }
//Todo: create Alarm-------------------------------------------------------------------------------------
        if (!parsedEntry.value(KTimeDataAlarm).isNull())
        {
            qDebug() << "* _createEntry(): creating Alarm for Todo";
            int secondsBeforeAlarm;
            secondsBeforeAlarm = parsedEntry.value(KTimeDataAlarm).toDateTime().toUTC().secsTo(
                                 parsedEntry.value(KTimeDataEnd).toDateTime().toUTC());
            CComponent * component = static_cast<CComponent *>(pTodo);
            result = component->setAlarmBefore(secondsBeforeAlarm);
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error adding Alarm to Todo");
                retVar = rMap;
                delete pTodo;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _createEntry(): done";
        }
    }
    else
    {
//Creating Event-------------------------------------------------------------------------------------
        QDateTime endDate;
        pEvent = new CEvent(parsedEntry.value(KCalendarEntrySummary).toString().toStdString().data(),
                            parsedEntry.value(KCalendarEntryDescription).toString().toStdString().data(),
                            parsedEntry.value(KCalendarEntryLocation).toString().toStdString().data(),
                            parsedEntry.value(KTimeDataStart).toDateTime().toTime_t(),
                            parsedEntry.value(KTimeDataEnd).toDateTime().toTime_t());
        if (pEvent == NULL)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error creating Event");
            retVar = rMap;
            qDebug() << "* _createEntry(): returning retVar:" << retVar;
            return retVar;
        }
        if (entryType.compare(KTypeMeeting, Qt::CaseInsensitive) == 0)
        {
//Event-Meeting: set status-----------------------------------------------------------------------------
            if (!parsedEntry.value(KCalendarEntryStatus).toString().isEmpty())
            {
                qDebug() << "* _createEntry(): setting status...";
                result = pEvent->setStatus(parsedEntry.value(KCalendarEntryStatus).toInt());
                if (!result)
                {
                    rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                    rMap.insert(KErrMessage,"Error setting Status for Event");
                    retVar = rMap;
                    delete pEvent;
                    qDebug() << "* _createEntry(): returning retVar:" << retVar;
                    return retVar;
                }
                qDebug() << "* _createEntry(): done, status=" << pEvent->getStatus();
            }
//Event-Meeting: set priority----------------------------------------------------------------------------
            if (!parsedEntry.value(KCalendarEntryPriority).toString().isEmpty())
            {
                qDebug() << "* _createEntry(): setting priority...";
                result = pEvent->setPriority(parsedEntry.value(KCalendarEntryPriority).toInt());
                if (!result)
                {
                    rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                    rMap.insert(KErrMessage,"Error setting Priority for Component");
                    retVar = rMap;
                    delete pEvent;
                    qDebug() << "* _createEntry(): returning retVar:" << retVar;
                    return retVar;
                }
            }
        }
//Event: create repeat rules----------------------------------------------------------------------------
        qDebug() << "* _createEntry(): create repeat rules...";
        if (parsedEntry.contains(KRepeatRuleFrequency))
        {
            QDateTime untilDate;
            pRecurrence = new CRecurrence();
            if (!pRecurrence)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error creating Recurrence object");
                retVar = rMap;
                delete pEvent;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
            if (!_createRecurrence(pRecurrence))
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error creating Recurrence");
                retVar = rMap;
                delete pRecurrence;
                delete pEvent;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
//Event: set repeat rule -----------------------------------------------------------------
            if (!entryType.compare(KTypeAnniversary))
                untilDate = MAX_DATE_TIME;//MAX ical date
            else
                untilDate = parsedEntry.value(KRepeatRuleUntilDate).toDateTime();
            result = pEvent->setUntil(untilDate.toTime_t());
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error setting Until Time to Event");
                retVar = rMap;
                delete pRecurrence;
                delete pEvent;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _createEntry(): set rRule for Event...";
            result = pEvent->setRecurrence(pRecurrence);
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error setting Recurrence to Event");
                retVar = rMap;
                delete pRecurrence;
                delete pEvent;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
        }
        else
        {
            qDebug() << "* _createEntry(): repeat rules are empty...";
        }
//Event: Create Alarm-------------------------------------------------------------------------------------
        //NOTE: Alarm could be set maximum in 2485 days before event.
        //Duration(alarm time which displayed in ui) could be:
        //None, 0 min, 5 min, 15 min, 30 min, 1 hour, 3 hours, day before
        //and automatically calculated by backend
        CComponent * component = static_cast<CComponent *>(pEvent);
        if (!parsedEntry.value(KTimeDataAlarm).isNull())
        {
            qDebug() << "* _createEntry(): creating Alarm for Event";
            int secondsBeforeAlarm;
            secondsBeforeAlarm = parsedEntry.value(KTimeDataAlarm).toDateTime().toUTC().secsTo(
                                 parsedEntry.value(KTimeDataStart).toDateTime().toUTC());
            result = component->setAlarmBefore(secondsBeforeAlarm);
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error creating Alarm to Event");
                retVar = rMap;
                if (pRecurrence)
                    delete pRecurrence;
                delete pEvent;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
        }
        if (entryType.compare(KTypeDayEvent, Qt::CaseInsensitive) == 0)
        {
            result = component->setAllDay(1);
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error creating DayEvent");
                retVar = rMap;
                if (pRecurrence)
                    delete pRecurrence;
                delete pEvent;
                qDebug() << "* _createEntry(): returning retVar:" << retVar;
                return retVar;
            }
        }
        CProperties *pProp = NULL;
        pProp = new CProperties();
        if (!pProp)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error creating x_properties");
            retVar = rMap;
            if (pRecurrence)
                delete pRecurrence;
            delete pEvent;
            qDebug() << "* _createEntry(): returning retVar:" << retVar;
            return retVar;
        }
        vector <CProperties *> vProps;
        result = pProp->setPropName(X_PROPERTY);
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error creating DayEvent");
            retVar = rMap;
            if (pRecurrence)
                delete pRecurrence;
            delete pEvent;
            delete pProp;
            qDebug() << "* _createEntry(): returning retVar:" << retVar;
            return retVar;
        }
        PropType propType;
        qDebug() << "* _createEntry(): setting x_property to:" << entryType;
        propType.szString = entryType.toStdString();
        result = pProp->setPropValue(propType);
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error creating DayEvent");
            retVar = rMap;
            if (pRecurrence)
                delete pRecurrence;
            delete pEvent;
            delete pProp;
            qDebug() << "* _createEntry(): returning retVar:" << retVar;
            return retVar;
        }
        vProps.push_back(pProp);
        result = pEvent->setXProperties(vProps);
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error creating DayEvent");
            retVar = rMap;
            if (pRecurrence)
                delete pRecurrence;
            delete pEvent;
            delete pProp;
            qDebug() << "* _createEntry(): returning retVar:" << retVar;
            return retVar;
        }
        delete pProp;
        qDebug() << "* _createEntry(): done";
    }

    qDebug() << "* _createEntry(): adding entry to calendar...";
//Add entry to calendar-----------------------------------------------------------------
    qDebug() << "* _createEntry(): typeIsTodo:" << typeIsTodo;
    //do not use calendar->addEvent() or addTodo(), since this way d-bus notification will not send
    if (!typeIsTodo)
        result = multiCalendar->addEvent(pEvent, calendarId, errorCode);
    else
        result = multiCalendar->addTodo(pTodo, calendarId, errorCode);
    qDebug() << "* _createEntry(): result:" << result;
    qDebug() << "* _createEntry(): errorCode:" << errorCode;
    //events with the same Summary, date start, date end, date until are treated by backend as duplicated
    //and will not be created! field for todo: summary and due date
    if (!result)
//    if (!result || errorCode != CALENDAR_OPERATION_SUCCESSFUL)
    {
        rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
        rMap.insert(KErrMessage,"Error adding event to calendar");
        retVar = rMap;
        if (pRecurrence)
            delete pRecurrence;
        delete pEvent;
        qDebug() << "* _createEntry(): returning retVar:" << retVar;
        return retVar;
    }
    QString qUID;
    if (typeIsTodo)
    {
        qUID = QString::fromStdString(pTodo->getId());
        qDebug() << "* _createEntry(): done, Todo id:" << qUID;
        if (pTodo != NULL)
        {
            delete pTodo;
            pTodo = NULL;
        }
    }
    else
    {
        qUID = QString::fromStdString(pEvent->getId());
        qDebug() << "* _createEntry(): done, Event id:" << qUID;
        if (pEvent != NULL)
        {
            delete pEvent;
            pEvent = NULL;
        }
        if (pRecurrence != NULL)
        {
            delete pRecurrence;
            pRecurrence = NULL;
        }
    }
    rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
    rMap.insert(KErrMessage,"SUCCESS");
    rMap.insert(KUid,qUID);
    retVar = rMap;
    qDebug() << "* _createEntry(): returning retVar:" << retVar;
    return retVar;
}

QVariant QCalendarService::_updateEntry(QString uid)
{
    CComponent * component = NULL;
    CEvent *pEvent = NULL;
    CTodo *pTodo = NULL;
    CRecurrence * pRecurrence = NULL;
    QMap<QString,QVariant> rMap;
    QVariant retVar;
    bool isNewRecurrence = FALSE;
    int result;
    qDebug() << "_updateEntry()";

    qDebug() << "* _updateEntry(): parsedEntry:" << parsedEntry;

    if (parsedEntry.value(KCalendarEntryType) == KTypeToDo)
    {
        pTodo = calendar->getTodo(uid.toStdString(), result);
        if (pTodo == NULL)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error getting Todo");
            retVar = rMap;
            qDebug() << "* _updateEntry(): returning retVar:" << retVar;
            return retVar;
        }
        qDebug() << "* _updateEntry(): pTodo:" << pTodo;
        component = static_cast<CComponent*>(pTodo);
    }
    else
    {
        pEvent = calendar->getEvent(uid.toStdString(), result);
        if (pEvent == NULL)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error getting Event");
            retVar = rMap;
            qDebug() << "* _updateEntry(): returning retVar:" << retVar;
            return retVar;
        }
        qDebug() << "* _updateEntry(): pEvent:" << pEvent;
        component = static_cast<CComponent*>(pEvent);
    }

    qDebug() << "* _updateEntry(): component:" << component;
    qDebug() << "* _updateEntry(): uid:" << uid;
    int calId = multiCalendar->getCalendarIdForComponent(component->getId(), result);

    result = component->setSummary(parsedEntry.value(KCalendarEntrySummary).toString().toStdString());
    if (!result)
    {
        rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
        rMap.insert(KErrMessage,"Error updating Summary for Component");
        retVar = rMap;
        delete component;
        qDebug() << "* _updateEntry(): returning retVar:" << retVar;
        return retVar;
    }
    result = component->setDescription(parsedEntry.value(KCalendarEntryDescription).toString().toStdString());
    if (!result)
    {
        rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
        rMap.insert(KErrMessage,"Error updating Description for Component");
        retVar = rMap;
        delete component;
        qDebug() << "* _updateEntry(): returning retVar:" << retVar;
        return retVar;
    }
    qDebug() << "* _updateEntry(): summary, description updated";
    if (!parsedEntry.value(KCalendarEntryType).toString().compare(KTypeMeeting, Qt::CaseInsensitive) ||
        !parsedEntry.value(KCalendarEntryType).toString().compare(KTypeToDo, Qt::CaseInsensitive)      )
    {
        result = component->setStatus(parsedEntry.value(KCalendarEntryStatus).toInt());
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error updating Status for Component");
            retVar = rMap;
            delete component;
            qDebug() << "* _updateEntry(): returning retVar:" << retVar;
            return retVar;
        }
        qDebug() << "* _updateEntry(): status updated";
    }
    qDebug() << "* _updateEntry(): type:" << parsedEntry.value(KCalendarEntryType).toString();
    if (parsedEntry.value(KCalendarEntryType).toString().compare(KTypeToDo, Qt::CaseInsensitive))
    {
        qDebug() << "* _updateEntry(): Event!";
        if (!parsedEntry.value(KCalendarEntryType).toString().compare(KTypeMeeting, Qt::CaseInsensitive))
        {
            result = component->setLocation(parsedEntry.value(KCalendarEntryLocation).toString().toStdString());
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error updating Location for Event");
                retVar = rMap;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _updateEntry(): Location updated";
            if (parsedEntry.contains(KCalendarEntryPriority))
            {
                qDebug() << "* _updateEntry(): setting Event priority:" << parsedEntry.value(KCalendarEntryPriority).toInt();
                result = pEvent->setPriority(parsedEntry.value(KCalendarEntryPriority).toInt());
                if (!result)
                {
                    rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                    rMap.insert(KErrMessage,"Error setting Priority for Event");
                    retVar = rMap;
                    delete component;
                    qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                    return retVar;
                }
                qDebug() << "* _updateEntry(): Priority updated";
            }
        }
        result = component->setDateStart(parsedEntry.value(KTimeDataStart).toDateTime().toTime_t());
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error updating DateStart for Event");
            retVar = rMap;
            delete component;
            qDebug() << "* _updateEntry(): returning retVar:" << retVar;
            return retVar;
        }
        if (parsedEntry.value(KCalendarEntryType).toString().compare(KTypeAnniversary, Qt::CaseInsensitive) &&
            parsedEntry.value(KCalendarEntryType).toString().compare(KTypeReminder, Qt::CaseInsensitive)      )
        {
            result = component->setDateEnd(parsedEntry.value(KTimeDataEnd).toDateTime().toTime_t());
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error updating DateEnd for Event");
                retVar = rMap;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _updateEntry(): dtStart, dtEnd updated";
        }
        qDebug() << "* _updateEntry(): pEvent:" << pEvent;
        if (parsedEntry.contains(KRepeatRuleFrequency))
        {
            qDebug() << "* _updateEntry(): getting recurrence";
            pRecurrence = component->getRecurrence();
            qDebug() << "* _updateEntry(): pRecurrence:" << pRecurrence;
            if (!parsedEntry.value(KCalendarEntryType).toString().compare(KTypeMeeting, Qt::CaseInsensitive) &&
                pRecurrence != NULL)
            {
                vector <string> exDates;
                exDates.clear();
                exDates = pRecurrence->getEDays();
                qDebug() << "* _updateEntry(): existing exDates count:" << exDates.size();
                if (!exDates.empty())
                {
                    QList<QVariant> exceptionDates = createExceptionDatesFromRecurrence(pRecurrence);
                    qDebug() << "* _updateEntry(): exception dates exist:" << exceptionDates;
                    QList<QVariant> list = parsedEntry.value(KCalendarEntryExceptionDates).toList();
                    qDebug() << "* _updateEntry(): exception dates to append:" << list;
                    for (int i = 0; i < exceptionDates.count(); i++)
                    {
                        list.prepend(exceptionDates.at(i));
                    }
                    parsedEntry.insert(KCalendarEntryExceptionDates, list);
                    qDebug() << "* _updateEntry(): exception dates appended:" << parsedEntry.value(KCalendarEntryExceptionDates);
                }
            }

            if (pRecurrence)
            {
                component->removeRecurrence();
                qDebug() << "* _updateEntry(): removed old Recurrence";
            }
            qDebug() << "* _updateEntry(): creating new Recurrence";
            pRecurrence = new CRecurrence();
            if (!pRecurrence)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error creating Recurrence object");
                retVar = rMap;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
            if (!_createRecurrence(pRecurrence))
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error updating Repeat Rule");
                retVar = rMap;
                delete pRecurrence;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _updateEntry(): setting until date";
            result = component->setUntil(parsedEntry.value(KRepeatRuleUntilDate).toDateTime().toTime_t());
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error setting Until Time to Event");
                retVar = rMap;
                delete pRecurrence;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _updateEntry(): set rRule for Event...";
            result = component->setRecurrence(pRecurrence);
            qDebug() << "* _updateEntry(): setRecurrence result:" << result;
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error setting Recurrence to Event");
                retVar = rMap;
                delete pRecurrence;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
        }
        if (parsedEntry.value(KTimeDataAlarm).isNull())
        {
            component->removeAlarm();
            qDebug() << "* _updateEntry(): event alarm removed";
        }
        else
        {
            int secondsBeforeAlarm;
            secondsBeforeAlarm = parsedEntry.value(KTimeDataAlarm).toDateTime().toUTC().secsTo(
                                 parsedEntry.value(KTimeDataStart).toDateTime().toUTC());
            result = component->setAlarmBefore(secondsBeforeAlarm);
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error creating Alarm for Event");
                retVar = rMap;
                delete pRecurrence;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _updateEntry(): alarm updated";
        }
        qDebug() << "* _updateEntry(): modifying Event, pEvent:" << pEvent << ", component:" << component;
        result = multiCalendar->modifyEvent(pEvent, calId, result);
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error modifying Event");
            retVar = rMap;
            delete pRecurrence;
            delete component;
            qDebug() << "* _updateEntry(): returning retVar:" << retVar;
            return retVar;
        }
        qDebug() << "* _updateEntry(): done, isNewRecurrence:" << isNewRecurrence;
        delete pRecurrence;
        pRecurrence = NULL;
    }
    else //Todo
    {
        qDebug() << "* _updateEntry(): Todo!";
        result = component->setDateStart(parsedEntry.value(KTimeDataEnd).toDateTime().toTime_t());
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error updating DateStart for Todo");
            retVar = rMap;
            delete component;
            qDebug() << "* _updateEntry(): returning retVar:" << retVar;
            return retVar;
        }
        CTodo * pTodo = static_cast<CTodo *>(component);
        if (parsedEntry.contains(KCalendarEntryPriority))
        {
            qDebug() << "* _updateEntry(): setting Todo priority:" << parsedEntry.value(KCalendarEntryPriority).toInt();
            result = pTodo->setPriority(parsedEntry.value(KCalendarEntryPriority).toInt());
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error setting Priority for Todo");
                retVar = rMap;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
        }
        qDebug() << "* _updateEntry(): dueDate, priority updated";
        if (parsedEntry.value(KTimeDataAlarm).isNull())
        {
            component->removeAlarm();
            qDebug() << "* _updateEntry(): todo alarm removed";
        }
        else
        {
            int secondsBeforeAlarm;
            secondsBeforeAlarm = parsedEntry.value(KTimeDataAlarm).toDateTime().toUTC().secsTo(
                                 parsedEntry.value(KTimeDataEnd).toDateTime().toUTC());
            result = component->setAlarmBefore(secondsBeforeAlarm);
            if (!result)
            {
                rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
                rMap.insert(KErrMessage,"Error creating Alarm for Todo");
                retVar = rMap;
                delete component;
                qDebug() << "* _updateEntry(): returning retVar:" << retVar;
                return retVar;
            }
            qDebug() << "* _updateEntry(): alarm updated";
        }

        qDebug() << "* _updateEntry(): modifying Todo...";
        result = multiCalendar->modifyTodo(pTodo, calId, result);
        if (!result)
        {
            rMap.insert(KErrCode,JSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert(KErrMessage,"Error modifying Event");
            retVar = rMap;
            delete component;
            qDebug() << "* _updateEntry(): returning retVar:" << retVar;
            return retVar;
        }
        qDebug() << "* _updateEntry(): done";
    }

    qDebug() << "* _updateEntry(): component:" << component;
    rMap.insert(KErrCode,JSEStatusCodes::SUCCESS);
    rMap.insert(KErrMessage,"SUCCESS");
    rMap.insert(KUid,uid);
    retVar = rMap;
    qDebug() << "* _updateEntry(): deleting component:" << component;
    delete component;
    component = NULL;
    qDebug() << "* _updateEntry(): returning retVar:" << retVar;
    return retVar;
}

QMap<QString,QVariant> QCalendarService::_mergeQmap(const QMap<QString,QVariant>&  qTarget, const QMap<QString,QVariant>&  qSource)
{
    typedef QMap<QString,QVariant> StringVariantMap;
    QMap<QString,QVariant> qMergeResult = qTarget;
    QMapIterator<QString,QVariant> i(qSource);
    while (i.hasNext())
    {
        i.next();
        if (qSource.contains(i.key()))
        {
            if (i.value().canConvert<StringVariantMap>())
            {
                if ( qSource.value(i.key()).canConvert<StringVariantMap>())
                {
                    const QMap<QString,QVariant> newTarget = qTarget.value(i.key()).toMap();
                    const QMap<QString,QVariant> newSource = i.value().toMap();
                    QMap<QString,QVariant> newValue = _mergeQmap(newTarget, newSource);
                    qMergeResult.remove(i.key());
                    qMergeResult.insert(i.key(), newValue);
                }
            }
            else
            {
                qMergeResult[i.key()] = qSource.value(i.key());
            }
        }
    }
    qDebug() << "* _mergeQmap(): merged map:" << qMergeResult;
    return qMergeResult;
}

vector <string> QCalendarService::_createRecurrenceRuleFromRepeatRule()
{
    vector <string> rulesVector;
    QString rRuleString;
    struct icaltimetype untilTimeAsICalType;
    QString freqString("FREQ=");
    QString intervalString(";INTERVAL=");
    QString untilString(";UNTIL=");
    QString untilTimeString;

/*sample
    QString freqRule[10] =  { "DAILY", "WEEKLY;BYDAY=MO,TH,FR", "WEEKLY", "MONTHLY", "YEARLY",
                             "MONTHLY;BYMONTHDAY=21,22,23,28", "YEARLY;BYDAY=MO,TH"};
*/
    QString freqRule[8] =   { "DAILY",                  //0
                              "WEEKLY",                 //1
                              "WEEKLY;BYDAY=",          //2
                              "MONTHLY",                //3
                              "MONTHLY;BYMONTHDAY=",    //4
                              "YEARLY",                 //5
                              "YEARLY;BYMONTH=",        //6
                              ";BYMONTHDAY="};          //7

    rulesVector.clear();
    //creating recurrence rule vector
    untilTimeAsICalType = icaltime_from_timet(parsedEntry.value(KRepeatRuleUntilDate).toDateTime().toTime_t(), 0);
    untilTimeString = icaltime_as_ical_string(untilTimeAsICalType);

    if (!parsedEntry.value(KRepeatRuleMonth).toString().isEmpty() && !parsedEntry.value(KRepeatRuleMonthDates).toList().isEmpty())
    {
        QString monthDates;
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): month and monthDates rules are present";
        monthDates.clear();
        int monthDatesCount = parsedEntry.value(KRepeatRuleMonthDates).toList().count();
        for (int i = 0; i < monthDatesCount; i++)
        {
            monthDates = monthDates + parsedEntry.value(KRepeatRuleMonthDates).toList().at(i).toString() +
                          KSeparator;
        }
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): monthDates:" << monthDates;
        monthDates.remove(rRuleString.length() - 1, 1);
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): cutted monthDates:" << monthDates;
        rRuleString = freqString + freqRule[6] + parsedEntry.value(KRepeatRuleMonth).toString() +
                      freqRule[7] + monthDates;
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): rRuleString:" << rRuleString;
    }
    else if (!parsedEntry.value(KRepeatRuleMonth).toString().isEmpty())
    {
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): month rule is present";
        rRuleString = freqString + freqRule[6] + parsedEntry.value(KRepeatRuleMonth).toString();
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): rRuleString:" << rRuleString;
    }
    else if (!parsedEntry.value(KRepeatRuleMonthDates).toList().isEmpty())
    {
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): monthDates rule is present";
        rRuleString = freqString + freqRule[4];
        int monthDatesCount = parsedEntry.value(KRepeatRuleMonthDates).toList().count();
        for (int i = 0; i < monthDatesCount; i++)
        {
            rRuleString = rRuleString +
                          parsedEntry.value(KRepeatRuleMonthDates).toList().at(i).toString() +
                          KSeparator;
        }
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): rRuleString:" << rRuleString;
        rRuleString.remove(rRuleString.length() - 1, 1);
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): cutted rRuleString:" << rRuleString;
    }
    else
    {
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): only frequency rule is present";
        rRuleString = freqString + freqRule[parsedEntry.value(KRepeatRuleFrequency).toInt()];
        qDebug() << "* _createRecurrenceRuleFromRepeatRule(): rRuleString:" << rRuleString;
    }
    if (!parsedEntry.value(KRepeatRuleInterval).toString().isEmpty())
    rRuleString = rRuleString + intervalString + parsedEntry.value(KRepeatRuleInterval).toString();
    rRuleString  = rRuleString + untilString + untilTimeString;
    qDebug() << "* _createRecurrenceRuleFromRepeatRule(): rRuleString:" << rRuleString;
    rulesVector.push_back(rRuleString.toStdString());
    return rulesVector;
}

vector <string> QCalendarService::_createRecurrenceRuleFromExceptionDates()
{
    vector <string> exDatesVector;
    QDateTime exDate;
    QString exDateString;
    QString dateString;
    struct icaltimetype dateAsICalType;

    //creating exception dates vector
    exDatesVector.clear();
    dateString.clear();
    int exDatesCount = parsedEntry.value(KCalendarEntryExceptionDates).toList().count();
    qDebug() << "* _createRecurrenceRuleFromExceptionDates(): exDatesCount:" << exDatesCount;
    for (int i = exDatesCount; i > 0; i--)
    {
        exDate = parsedEntry.value(KCalendarEntryExceptionDates).toList().at(i-1).toDateTime();
        qDebug() << "* _createRecurrenceRuleFromExceptionDates(): exDate:" << exDate;
        dateAsICalType = icaltime_from_timet(exDate.toTime_t(), 0);
        //dateAsICalType = icaltime_from_timet_with_zone(exDate.toTime_t(), 0);
        if (dateString.isEmpty())
        {
            dateString = icaltime_as_ical_string(dateAsICalType);
        }
        else
        {
            dateString = dateString + "," + icaltime_as_ical_string(dateAsICalType);
        }
    }
    qDebug() << "* _createRecurrenceRuleFromExceptionDates(): dateString:" << dateString;
    exDatesVector.push_back(dateString.toStdString());
    return exDatesVector;
}

bool QCalendarService::_createRecurrence(CRecurrence *pRecurrence)
{
    int repeatType;
    int result;

    qDebug() << "_createRecurrence()";

    qDebug() << "* _createRecurrence(): pRecurrence:" << pRecurrence;
    if (pRecurrence == NULL)
    {
        return FALSE;
    }
//Event-Anniversary: create repeat rule Yearly-----------------------------------------------------------
    if (parsedEntry.value(KCalendarEntryType).toString().compare(KTypeAnniversary, Qt::CaseInsensitive) == 0)
    {
    //Anniversary recurrence rule is constant with frequency Yearly and until time = MAX ical time
        vector <string> rule;
        rule.push_back("FREQ=YEARLY;UNTIL=20371230T210000");
    //Recurrence: set repeat rule---------------------------------------------------------------
        result = pRecurrence->setRrule(rule);
        if (!result)
        {
            return FALSE;
        }
        repeatType = 5;//Yearly
        qDebug() << "* _createRecurrence(): Anniversary";
    }
    else
    {
//Event-Other: create repeat rule -----------------------------------------------------------------
    //Recurrence: set repeat rule---------------------------------------------------------------
        qDebug() << "* _createRecurrence(): pRecurrence->setRrule...";
        result = pRecurrence->setRrule(_createRecurrenceRuleFromRepeatRule());
        qDebug() << "* _createRecurrence(): result:" << result;
        if (!result)
        {
            return FALSE;
        }
        qDebug() << "* _createRecurrence(): done";
    //Recurrence: set Exception Days---------------------------------------------------------------
        if (parsedEntry.value(KCalendarEntryType).toString().compare(KTypeMeeting, Qt::CaseInsensitive) == 0 &&
            !parsedEntry.value(KCalendarEntryExceptionDates).toList().isEmpty())
        {
            qDebug() << "* _createRecurrence(): pRecurrence->setEDays...";
            result = pRecurrence->setEDays(_createRecurrenceRuleFromExceptionDates());
            if (!result)
            {
                return FALSE;
            }
            qDebug() << "* _createRecurrence(): done";
        }
        repeatType = parsedEntry.value(KRepeatRuleType).toInt();
    }
//Recurrence: set repeat type (used to display corresponding repeat type in UI)------------------
    qDebug() << "* _createRecurrence(): pRecurrence->setRtype...";
    result = pRecurrence->setRtype(repeatType);
    if (!result)
    {
            return FALSE;
    }
    qDebug() << "_createRecurrence()";
    return TRUE;
}

QMap<QString,QVariant> QCalendarService::_validateMatchPattern(QMap<QString, QVariant> &matchPattern)
{
    qDebug() << "_validateMatchPattern()";
    QMap<QString,QVariant> rMap;
    rMap.insert(KErrCode, JSEStatusCodes::SUCCESS);
//Validating ID------------------------------------------------------------------
    if (matchPattern.contains(KMatchPatternId) &&
         !matchPattern.value(KMatchPatternId).isNull() &&
         (matchPattern.value(KMatchPatternId) != KUndefined) &&
         matchPattern.value(KMatchPatternId).type() != QVariant::String)
    {
        qDebug() << "* _validateMatchPattern(): MatchPatternId type is invalid";
        rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
        rMap.insert(KErrMessage,"MatchPatternId type is invalid");
        return rMap;
    }
//Validating Range------------------------------------------------------------------
    if (matchPattern.contains(KMatchPatternRange) &&
        !matchPattern.value(KMatchPatternRange).isNull() &&
        (matchPattern.value(KMatchPatternRange) != KUndefined) &&
        !matchPattern.value(KMatchPatternRange).canConvert<QVariantMap>())
    {
        qDebug() << "* _validateMatchPattern(): MatchPatternRange type is invalid";
        rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
        rMap.insert(KErrMessage,"MatchPatternRange type is invalid");
        return rMap;
    }
    if (matchPattern.contains(KMatchPatternRange))
    {
        QVariantMap qRange = matchPattern.value(KMatchPatternRange).toMap();
        if (qRange.contains(KRangeDataBegin))
        {
            if (qRange.value(KRangeDataBegin).isNull() || (qRange.value(KRangeDataBegin) == KUndefined))
            {
                qRange.remove(KRangeDataBegin);
                qRange.insert(KRangeDataBegin, MIN_DATE_TIME);
                matchPattern.remove(KMatchPatternRange);
                matchPattern.insert(KMatchPatternRange, qRange);
            }
            else if (qRange.value(KRangeDataBegin).type() != QVariant::DateTime)
            {
                qDebug() << "* _validateMatchPattern(): RangeDataBegin type is invalid";
                rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
                rMap.insert(KErrMessage,"RangeDataBegin type is invalid");
                return rMap;
            }
        }
        if (qRange.contains(KRangeDataEnd))
        {
            if (qRange.value(KRangeDataEnd).isNull() || (qRange.value(KRangeDataEnd) == KUndefined))
            {
                qRange.remove(KRangeDataEnd);
                qRange.insert(KRangeDataEnd, MAX_DATE_TIME);
                matchPattern.remove(KMatchPatternRange);
                matchPattern.insert(KMatchPatternRange, qRange);
            }
            else if (qRange.value(KRangeDataEnd).type() != QVariant::DateTime)
            {
                qDebug() << "* _validateMatchPattern(): RangeDataEnd type is invalid";
                rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
                rMap.insert(KErrMessage,"RangeDataEnd type is invalid");
                return rMap;
            }
        }
    }
//Validating Type------------------------------------------------------------------
    if (matchPattern.contains(KMatchPatternType) &&
        !matchPattern.value(KMatchPatternType).isNull() &&
        (matchPattern.value(KMatchPatternType) != KUndefined) &&
         matchPattern.value(KMatchPatternType).type() != QVariant::String)
    {
        qDebug() << "* _validateMatchPattern(): MatchPatternType type is invalid";
        rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
        rMap.insert(KErrMessage,"MatchPatternType type is invalid");
        return rMap;
    }
//Validating Text------------------------------------------------------------------
    if (matchPattern.contains(KMatchPatternText) &&
        !matchPattern.value(KMatchPatternText).isNull() &&
        (matchPattern.value(KMatchPatternText) != KUndefined) &&
         matchPattern.value(KMatchPatternText).type() != QVariant::String)
    {
        qDebug() << "* _validateMatchPattern(): MatchPatternText type is invalid";
        rMap.insert(KErrCode,JSEStatusCodes::INVALID_ARG_ERR);
        rMap.insert(KErrMessage,"MatchPatternText type is invalid");
        return rMap;
    }
}
