/*
* ============================================================================
*  Name        : qcalendarhelper.cpp
*  Part of     : serviceframework / WRT
*  Description : calendar helper class for calendar service
*  Version     : %version: 1 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QRegExp>
#include <QStringList>
#include <libical/ical.h>

#include "qcalendarhelper.h"
#include "qcalendarconstants.h"


namespace CalendarHelpers {

QVariantMap createCalendarEntry(CComponent *entry, QDateTime instanceTime)
{
    qDebug() << "* createCalendarEntry(instance): creating entry";
    QVariantMap map = createCalendarEntry(entry);
    QVariantMap time = createTimeFromEntry(entry);
    qDebug() << "* createCalendarEntry(instance): updating time";
    createInstanceTime(entry->getType(), &time, instanceTime);
    map.remove(KCalendarEntryTime);
    map.insert(KCalendarEntryTime, time);
    qDebug() << "* createCalendarEntry(instance): done, time:" << map.value(KCalendarEntryTime);
    return map;
}
QVariantMap createCalendarEntry(CComponent *entry)
{
    QVariantMap map;
    qDebug() << "createCalendarEntry()";
    if (entry == NULL)
    {
        qDebug() << "* createCalendarEntry(): component = NULL!";
        return map;
    }

    //type
    QString entryType;
    int type =  entry->getType();
    if (type == KTypeEvent)
    {
        int isAllDay = FALSE;
        isAllDay = entry->getAllDay();
        if (isAllDay)
            entryType = KTypeDayEvent;
        else
        {
            vector <CProperties *> pProp;
            CEvent * pEvent = static_cast<CEvent*>(entry);
            qDebug() << "* createCalendarEntry(): getting x_properties";
            pProp = pEvent->getXProperties();
            PropType propType;
            bool isXPropPresent = FALSE;
            qDebug() << "* createCalendarEntry(): x_properties count: " << pProp.size();
            for (uint i = 0; i < pProp.size(); i++)
            {
                CProperties * property = NULL;
                property = pProp.at(i);
                if (property)
                {
                    QString propName = QString::fromStdString(property->getPropName());
                    qDebug() << "* createCalendarEntry(): x_property number:" << (i + 1);
                    qDebug() << "* createCalendarEntry(): x_property name:" << propName;
                    if (!propName.compare(X_PROPERTY, Qt::CaseSensitive))
                    {
                        qDebug() << "* createCalendarEntry(): it is needed x_property, getting value";
                        propType = property->getPropValue();
                        isXPropPresent = TRUE;
                        break;
                    }
                }
                delete property;
            }
            if (isXPropPresent)
            {
                entryType = QString::fromStdString(propType.szString);
                qDebug() << "* createCalendarEntry(): x_property type:" << entryType;
            }
            else
            {
                entryType = KTypeMeeting;
            }
        }
    }
    else if (type == KTypeTodo)
        entryType = KTypeToDo;
    else
    {
        qDebug() << "* createCalendarEntry(): invalid type!";
        return map;
    }
    map.insert(KCalendarEntryType, entryType);
    qDebug() << "* createCalendarEntry(): entryType" << entryType;

    // id
    map.insert(KCalendarEntryId, entry->getId().c_str());
    qDebug() << "* createCalendarEntry(): Id" << entry->getId().c_str();

    // time
    map.insert(KCalendarEntryTime, createTimeFromEntry(entry));

    // summary
    QString summary = QString::fromStdString(entry->getSummary());
    if (!summary.isEmpty())
    {
        map.insert(KCalendarEntrySummary, summary);
        qDebug() << "* createCalendarEntry(): summary" << summary;
    }

    // description
    QString description = QString::fromStdString(entry->getDescription());
    if (!description.isEmpty())
    {
        map.insert(KCalendarEntryDescription, description);
        qDebug() << "* createCalendarEntry(): description" << description;
    }

    //priority & location
    if (entryType == KTypeMeeting)
    {
        CEvent * pEvent = static_cast<CEvent *>(entry);
        int priority = pEvent->getPriority();
        //-1 is default fremantle priority
        if (priority != -1)
        {
            map.insert(KCalendarEntryPriority, priority);
            qDebug() << "* createCalendarEntry(): Event priority:" << priority;
        }
        QString location = entry->getLocation().c_str();
        if (!location.isEmpty())
        {
            map.insert(KCalendarEntryLocation, location);
            qDebug() << "* createCalendarEntry(): location" << location;
        }
    }
    if (entryType == KTypeToDo)
    {
        CTodo * pTodo = static_cast<CTodo *>(entry);
        int priority = pTodo->getPriority();
        if (priority != 0)
        {
            map.insert(KCalendarEntryPriority, priority);
            qDebug() << "* createCalendarEntry(): Todo priority" << priority;
        }
    }

    // status
    int status = entry->getStatus();
    qDebug() << "* createCalendarEntry(): status:" << status;
    if (entryType == KTypeMeeting)
    {
        switch (status)
        {
            case 4:
                map.insert(KCalendarEntryStatus, KStatusConfirmed);
                break;
            case 5:
                map.insert(KCalendarEntryStatus, KStatusTentative);
                break;
            case 3:
                map.insert(KCalendarEntryStatus, KStatusCancelled);
                break;

            default:
                break;
        }
    }
    else if (entryType.compare(KTypeToDo, Qt::CaseInsensitive) == 0)
    {
        switch (status)
        {
            case 0:
                map.insert(KCalendarEntryStatus, KStatusNeedsAction);
                break;
            case 1:
                map.insert(KCalendarEntryStatus, KStatusCompleted);
                break;
            default:
                break;
        }
    }

    // repeatRule
    CRecurrence *pRecurrence = NULL;
    QVariantMap repeatRule;
    QVariantList exceptionDates;
    //do not delete pRecurrence! it'll be deleted on deletion of component.
    pRecurrence = entry->getRecurrence();
    qDebug() << "* createCalendarEntry(): pRecurrence:" << pRecurrence;
    if (pRecurrence != NULL)
    {
        repeatRule = createRepeatRuleFromRecurrence(pRecurrence);
        exceptionDates = createExceptionDatesFromRecurrence(pRecurrence);
    }
    qDebug() << "* createCalendarEntry(): calendar repeatRule:" << repeatRule;
    if (!repeatRule.isEmpty())
    {
        map.insert(KCalendarEntryRepeatRule, repeatRule);
    }
    // exceptionDates
    qDebug() << "* createCalendarEntry(): calendar exceptionDates:" << exceptionDates;
    if (!exceptionDates.isEmpty())
    {
        map.insert(KCalendarEntryExceptionDates, exceptionDates);
    }

    qDebug() << "* createCalendarEntry(): returning entry=" << map;
    qDebug() << "createCalendarEntry()";
    return map;
}

bool getCalendarEntryById(CMulticalendar* multiCalendar, QVariantMap &entry, const QString uid)
{
    CComponent *component = NULL;
    CCalendar * calendar = NULL;
    int errorCode;
    qDebug() << "getCalendarEntryById()";
    calendar = multiCalendar->getSynchronizedCalendar();
    if (!calendar)
    {
        qDebug() << "* getCalendarEntryById(): error getting calendar";
        return FALSE;
    }
    CEvent *pEvent = NULL;
    pEvent = calendar->getEvent(uid.toStdString(), errorCode);
    if (pEvent == NULL || errorCode != CALENDAR_OPERATION_SUCCESSFUL)
    {
        qDebug() << "* getCalendarEntryById(): error getting event by Id";
        return FALSE;
    }
    component = static_cast<CComponent*>(pEvent);
    qDebug() << "* getCalendarEntryById(): component=" << component << ", errorCode=" << errorCode;
    int type = component->getType();
    qDebug() << "* getCalendarEntryById(): 1 - CEvent, 2 - CTodo, type:" << type;
    if (type == KTypeEvent)
    {
        qDebug() << "* getCalendarEntryById(): This is Event!";
        entry = CalendarHelpers::createCalendarEntry(component);
        qDebug() << "* getCalendarEntryById(): updated entry=" << entry;
        delete component;
        qDebug() << "* getCalendarEntryById(): return TRUE";
        return TRUE;
    }
    else if (type == KTypeTodo)
    {
        if (pEvent)
        {
            delete pEvent;
            component = NULL;
        }
        qDebug() << "* getCalendarEntryById(): This is Todo!";
        CTodo * pTodo = calendar->getTodo(uid.toStdString(), errorCode);
        component = static_cast<CComponent*>(pTodo);
        qDebug() << "* getCalendarEntryById(): component=" << component << ", errorCode=" << errorCode;
        if (component != NULL && errorCode == CALENDAR_OPERATION_SUCCESSFUL)
        {
            entry = CalendarHelpers::createCalendarEntry(component);
            qDebug() << "* getCalendarEntryById(): updated entry=" << entry;
            delete component;
            qDebug() << "* getCalendarEntryById(): return TRUE";
            return TRUE;
        }
        else
        {
            qDebug() << "* getCalendarEntryById(): Error getting todo by Id";
            qDebug() << "* getCalendarEntryById(): return FALSE";
            return FALSE;
        }
    }
    qDebug() << "* getCalendarEntryById(): component type is invalid, return FALSE";
    return FALSE;
}

void createInstanceTime(int type, QVariantMap * map, QDateTime instanceTime)
{
    QVariant oldStartTime = map->value(KTimeDataStart);
    QVariant oldEndTime = map->value(KTimeDataEnd);
    qDebug() << "* createInstanceTime(): received time:" << map;
    qDebug() << "* createInstanceTime(): oldStartTime:" << oldStartTime;
    qDebug() << "* createInstanceTime(): oldEndTime:" << oldEndTime;
    int secondsBeginToEnd = oldStartTime.toDateTime().secsTo(oldEndTime.toDateTime());
    qDebug() << "* createInstanceTime(): secondsBeginToEnd:" << secondsBeginToEnd;
    qDebug() << "* createInstanceTime(): time to update:" << instanceTime;
    if (type == KTypeEvent)
    {
        map->remove(KTimeDataStart);
        map->insert(KTimeDataStart, instanceTime);
        QDateTime newEndTime = instanceTime.addSecs(secondsBeginToEnd);
        map->remove(KTimeDataEnd);
        map->insert(KTimeDataEnd, newEndTime);
        qDebug() << "* createInstanceTime(): updated time:" << map;
    }
    else if (type == KTypeTodo)
    {
        map->remove(KTimeDataEnd);
        map->insert(KTimeDataEnd, instanceTime);
        qDebug() << "* createInstanceTime(): updated time:" << map->value(KTimeDataEnd);
    }
    else
        qDebug() << "* createInstanceTime(): component type is invalid!";
}

QVariantMap createTimeFromEntry(CComponent *entry)
{
    QVariantMap map;
    QDateTime startTime;
    QDateTime endTime;
    QDateTime alarmTime;

    qDebug() << "createTimeFromEntry()";
//Get Start Date-----------------------------------------------------------
    int entryType = entry->getType();
    if (entryType == KTypeEvent)
    {
        time_t stTime = entry->getDateStart();
        startTime = QDateTime::fromTime_t(stTime);
        map.insert(KTimeDataStart, startTime);
        qDebug() << "* createTimeFromEntry(): startTime: " << startTime;

        time_t enTime = entry->getDateEnd();
        endTime = QDateTime::fromTime_t(enTime);
        map.insert(KTimeDataEnd, endTime);
        qDebug() << "* createTimeFromEntry(): endTime: " << endTime;
    }
//Get End Date-----------------------------------------------------------
    if (entryType == KTypeTodo)
    {
        time_t enTime = entry->getDateStart();
        endTime = QDateTime::fromTime_t(enTime);
        map.insert(KTimeDataEnd, endTime);
        qDebug() << "* createTimeFromEntry(): endTime: " << endTime;
    }
//Get Alarm-----------------------------------------------------------
    CAlarm * pAlarm = entry->getAlarm();
    if (pAlarm != NULL)
    {
        qDebug() << "* createTimeFromEntry(): CAlarm: " << pAlarm;
        time_t alTimeAsTime_t = pAlarm->getTrigger();
        qDebug() << "* createTimeFromEntry(): alTimeAsTime_t: " << alTimeAsTime_t;
        alarmTime = QDateTime::fromTime_t(alTimeAsTime_t);
        if (alarmTime.isValid())
        {
            map.insert(KTimeDataAlarm, alarmTime);
            qDebug() << "* createTimeFromEntry(): alarmTime: " << alarmTime;
        }
    }
    else
    {
        qDebug() << "* createTimeFromEntry(): no alarm";
    }
    qDebug() << "createTimeFromEntry()";
    return map;
}

QList<int> getSpecificRuleFromRecurrence(vector <string> rRules, const QString ruleSpecificator)
{
    QString rRule;
    QString date;
    QList<int> result;
    int startIndex;
    int endIndex;
    result.clear();
    qDebug() << "getSpecificRuleFromRecurrence()";

    qDebug() << "* getSpecificRuleFromRecurrence(): rRules.size():" << rRules.size();
    //per JSAPI need to support only one repeat rule
    rRule = QString::fromStdString(rRules.front());
    qDebug() << "* getSpecificRuleFromRecurrence(): rRule:" << rRule;
    startIndex = rRule.indexOf(ruleSpecificator, 0, Qt::CaseInsensitive);
    qDebug() << "* getSpecificRuleFromRecurrence(): startIndex:" << startIndex;
    //rule string doesn't contain needed rule
    if (startIndex == -1)
        return result;
    endIndex = rRule.indexOf(";", startIndex, Qt::CaseInsensitive);
    qDebug() << "* getSpecificRuleFromRecurrence(): endIndex:" << endIndex;
    rRule.remove(endIndex, (rRule.length() - endIndex));
    rRule.remove(0, startIndex + ruleSpecificator.length());
    qDebug() << "* getSpecificRuleFromRecurrence(): rRule:" << rRule;

    //if only one date is present
    if (rRule.length() == 1 || rRule.length() == 2)
    {
        result.append(rRule.toInt());
        qDebug() << "* getSpecificRuleFromRecurrence(): result:" << result;
        return result;
    }
    while (startIndex != -1)
    {
        date = rRule;
        qDebug() << "* getSpecificRuleFromRecurrence(): date = rRule:" << rRule;
        startIndex = rRule.indexOf(",", 0, Qt::CaseInsensitive);
        if (startIndex == -1)
            break;
        date.remove(startIndex, date.length() - startIndex);
        rRule.remove(0, date.length() + 1);
        qDebug() << "* getSpecificRuleFromRecurrence(): date:" << date;
        qDebug() << "* getSpecificRuleFromRecurrence(): rRule:" << rRule;
        result.append(date.toInt());
        qDebug() << "* getSpecificRuleFromRecurrence(): result:" << result;
    }
    result.append(date.toInt());
    qDebug() << "* getSpecificRuleFromRecurrence(): result:" << result;
    qDebug() << "getSpecificRuleFromRecurrence()";
    return result;
}

QVariantMap createRepeatRuleFromRecurrence(CRecurrence *recurrence)
{
    CRecurrenceRule * rRule = NULL;
    QString frequency;
    QVariantMap map;
    FREQUENCY freq;
    bool result;

    qDebug() << "createRepeatRuleFromRecurrence()";
    vector <string> rRuleString = recurrence->getRrule();
    if (rRuleString.empty())
    {
        qDebug() << "* createRepeatRuleFromRecurrence(): recurrence rule is empty";
        return map;
    }
    vector<CRecurrenceRule *> rRules = recurrence->getRecurrenceRule();
    //per JSAPI need to support only one repeat rule
    rRule = rRules.front();
    if (!rRule)
        return map;
    result = rRule->rruleParser(rRuleString.front());
    if (!result)
    {
        qDebug() << "* createRepeatRuleFromRecurrence(): error parsing recurrence rule!";
        return map;
    }
    freq = rRule->getFrequency();
    printf("* createRepeatRuleFromRecurrence(): freq:%d\n", freq);
    QString rule = QString::fromStdString(rRule->getRrule());
    qDebug() << "rule:" << rule;

    switch (freq)
    {
    case DAILY_RECURRENCE:
        frequency = KFrequencyDaily;
        break;
    case WEEKLY_RECURRENCE:
        frequency = KFrequencyWeekly;
        break;
    case MONTHLY_RECURRENCE:
        frequency = KFrequencyMonthly;
        break;
    case YEARLY_RECURRENCE:
        frequency = KFrequencyYearly;
        break;
    case NO_RECURRENCE:
        qDebug() << "* createRepeatRuleFromRecurrence(): No frequency!";
        return map;
    default:
        // recurrence not supported
        qDebug() << "* createRepeatRuleFromRecurrence(): Unsupported frequency: " << freq;
        return map;
    }
    qDebug() << "* createRepeatRuleFromRecurrence(): frequency:" << frequency;
    map.insert(KRepeatRuleFrequency, frequency);

    // untilDate
    QDateTime untilDate;
    untilDate = QDateTime::fromTime_t(rRule->getUntil());
    qDebug() << "* createRepeatRuleFromRecurrence(): untilDate:" << untilDate;
    if (untilDate.isValid())
    {
        map.insert(KRepeatRuleUntilDate, untilDate);
    }
    else
    {
        qDebug() << "* createRepeatRuleFromRecurrence(): untilDate is invalid!";
        return map;
    }

    // interval
    int interval = rRule->getInterval();
    if (interval > 0)
    {
        map.insert(KRepeatRuleInterval, interval);
        qDebug() << "* createRepeatRuleFromRecurrence(): interval:" << interval;
    }
    else
    {
        qDebug() << "* createRepeatRuleFromRecurrence(): interval is invalid, will be set to 1";
    }

    // month
    if (frequency == KFrequencyYearly)
    {
        vector <string> ruleString = recurrence->getRrule();
        QList<int> month = getSpecificRuleFromRecurrence(ruleString, "BYMONTH=");
        if (!month.isEmpty())
            map.insert(KRepeatRuleMonth, month.first());
    }

    // monthDates
    if (frequency == KFrequencyMonthly || frequency == KFrequencyYearly)
    {
        vector <string> ruleString = recurrence->getRrule();
        QList<int> monthDates = getSpecificRuleFromRecurrence(ruleString, "BYMONTHDAY=");
        QList<QVariant> dayVariantList;
            for (int i = 0; i < monthDates.size(); i++)
            {
                dayVariantList.append(monthDates[i]);
            }
            if (!dayVariantList.isEmpty())
            {
                map.insert(KRepeatRuleMonthDates, dayVariantList);
            }
    }

    qDebug() << "* createRepeatRuleFromRecurrence(): return map:" << map;
    return map;
}

QVariantList createExceptionDatesFromRecurrence(CRecurrence *recurrence)
{
    struct icaltimetype icaltime;
    vector <string> exDatesStrings;
    QString exDate;
    QList<QVariant> exceptionDates;
    exDatesStrings.clear();
    exDatesStrings = recurrence->getEDays();
    qDebug() << "createExceptionDatesFromRecurrence()";
    qDebug() << "* createExceptionDatesFromRecurrence(): exDates.count:" << exDatesStrings.size();
    if (exDatesStrings.empty())
    {
        qDebug() << "* createExceptionDatesFromRecurrence(): no exception dates";
        qDebug() << "createExceptionDatesFromRecurrence()";
        return exceptionDates;
    }
    exceptionDates.clear();
    for (uint i = 0; i < exDatesStrings.size(); i++)
    {
        QRegExp rx(",");
        QStringList list;
        QString res;
        QDateTime date;
        int startIndex = 0;
        exDate = QString::fromStdString(exDatesStrings.at(i));

        //extracting date from string
        while (startIndex != -1)
        {
            res = exDate;
            qDebug() << "* createExceptionDatesFromRecurrence(): exDate:" << exDate << "res:" << res;
            startIndex = exDate.indexOf(",", 0, Qt::CaseInsensitive);
            if (startIndex == -1)
                break;
            res.remove(startIndex, res.length() - startIndex);
            exDate.remove(0, res.length() + 1);
            qDebug() << "* createExceptionDatesFromRecurrence(): exDate:" << exDate << "res:" << res;
            //converting from icaltime to QDateTime
            icaltime = icaltime_from_string(res.toStdString().c_str());
            date = QDateTime::fromTime_t(icaltime_as_timet(icaltime));
            exceptionDates.prepend(date);
            qDebug() << "* createExceptionDatesFromRecurrence(): while exceptionDates:" << exceptionDates;
        }
        //converting from icaltime to QDateTime
        icaltime = icaltime_from_string(res.toStdString().c_str());
        date = QDateTime::fromTime_t(icaltime_as_timet(icaltime));
        exceptionDates.prepend(date);
        qDebug() << "* createExceptionDatesFromRecurrence(): exceptionDates:" << exceptionDates;
    }
    qDebug() << "createExceptionDatesFromRecurrence(): return exceptionDates";
    return exceptionDates;
}

}
