/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "servicehandler_p.h"
#include <serviceipc.h>
#include "serviceinfolist.h"
#include <serviceresolutioncriteria.h>
#include "serviceplugininterface.h"
#include "serviceresolver.h"
#include "secmgrdefs.h"
#include "secsession.h"
#include "sfwexception.h"
#include "serviceframeworkdefs.h"

namespace WRT
{

/*!
 \class ServiceHandlerPrivate
 Keeps all service handler's private members and methods.
 A pointer to this class is declared in ServiceHandler class
 */

/*!
 Constructor
 @param parent service handler instance that contains this private object
 */
ServiceHandlerPrivate::ServiceHandlerPrivate(ServiceHandler *parent)
    : m_session(NULL)
    , m_isConnected(false)
{
#if (!defined(__WINSCW__) || defined(__S60_50__))
    m_session = new ServiceFwIPC(parent);
    if (m_session) {
        int retry(2);
        for (;;)
        {
            m_isConnected = m_session->connect(SERVICE_RESOLVER_NAME);
            if (m_isConnected)
            {
                break;
            }
            else
            {
                QString serviceResolverSvr(SERVICE_RESOLVER_SERVER);
                if ( !m_session->startServer(
                        SERVICE_RESOLVER_NAME, serviceResolverSvr ) )
                { // start server failed.
                    break;
                }
            }
            if (0==--retry)
            {
                break;
            }
        }
    }
    if (!m_isConnected) {
        delete m_session;
        m_session = NULL;
    }
#endif
}

/*!
 Destructor
 */
ServiceHandlerPrivate::~ServiceHandlerPrivate()
{
    int count = m_loadedPlugins.count();
    for (qint32 i = 0; i < count; ++i) {
        delete m_loadedPlugins[i];
    }
    m_loadedPlugins.clear();
    if (m_session) {
        m_session->disconnect();
        delete m_session;
    }
    qDeleteAll(m_objects);
    m_objects.clear();
}

/*!
 Creates an instance of a service

 In the current implementation, the server resolver will scan through all XML
 files and return the closest matching service. The service provider name is
 specified as a dot notation in the format of "com.nokia.servicename".
 The service class will be instantiated by first loading the service plugin dll,
 then invoking a specified a factory method to obtain the instance.

 @param aServiceName Name of the service provider
 @param aInterfaceInfo Interface info
 @param aSecSession Security Session (not owned). If NULL then that means
                    no security check will be done.
 @param aRefCounted true the underlying service object is reference counted.
                    false  the underlying service object is not reference counted
                           and the object is owned by service handler
 @return IServiceBase* interface pointer to the service if the service is found,
                       NULL if service is not found
 */
IServiceBase* ServiceHandlerPrivate::getServiceBase(const QString& aServiceName,
                                             ServiceInterfaceInfo& aInterfaceInfo,
                                             bool aRefCounted,
                                             SecSession* aSecSession)
{
    CLEAR_ERRORS(q);
    IServiceBase* serviceBase(NULL);
#if (!defined(__WINSCW__) || defined(__S60_50__))
    //multiple processes not supported on S60 v3.1 v3.2 emulators
    if ( m_isConnected ) {
        ServiceResolutionCriteria* criteria = new ServiceResolutionCriteria();
        criteria->setServiceName( aServiceName );
        QByteArray interfaceId = QByteArray(aInterfaceInfo.m_iid).trimmed();
        QList<QByteArray> idList = interfaceId.split(KSlash);

        bool sent(false);
        if (idList.size()==1 || idList.size()==2) {
            criteria->setInterfaceName(idList[0]);
            if (idList.size()==1) {
                idList.append(KDefaultVersion);
            }
           criteria->setInterfaceVersion(idList[1]);
           QByteArray data = criteria->serialize();
           sent = m_session->sendSync( GETSERVICEREQ, data );
           delete criteria;
        } // the format of interface id is not correct, don't sent request

        if (sent) {
            // data is available
            ServiceInfo service;
            if ((service.deserialize(m_session->readAll()))
                            && (service.isValid()) ) {
                // for default version, set iid to the latest interface version's idd that resolver has found
                if (idList[1] == KDefaultVersion) {
                    m_ba.clear();
                    m_ba.append(idList[0]); // interface name
                    m_ba.append(KSlash);
                    m_ba.append(service.interfaceVersion().toAscii());
                    aInterfaceInfo.m_iid = m_ba.constData();
                }
                // Check if client is allowed to access the service provider
                // only if a security session is given.
                bool allowed( true );
                if ( aSecSession )
                {
                    QStringList capList = service.interfaceCapabilities();
                    allowed = aSecSession->isAllowed( capList );
                }
                if ( allowed )
                {
                    serviceBase = loadPlugin(service.serviceDllFileName(),
                                             aRefCounted,
                                             aSecSession );
                } else {
                    SET_ERROR(q, ServiceHandlerPrivate::SFW_ERROR_CLIENT_NOT_ALLOWED_TO_ACCESS_SERVICE,
                        ServiceHandlerErrDesc[ServiceHandlerPrivate::SFW_ERROR_CLIENT_NOT_ALLOWED_TO_ACCESS_SERVICE]);
                }
            } else {
                SET_ERROR(q, ServiceHandlerPrivate::SFW_ERROR_SERVICE_NOT_FOUND,
                    ServiceHandlerErrDesc[ServiceHandlerPrivate::SFW_ERROR_SERVICE_NOT_FOUND]);
            }
        }
    }
#else //3.1 && 3.2 winscw
    if (aServiceName == "com.nokia.TestServices") {
        // Hard code for now because resolver isn't ready
        QString path = "c:\\resource\\qt\\sfwplugin\\testserviceplugin.qtplugin";
        serviceBase = loadPlugin(path, aRefCounted, aSecSession);
    } else if (aServiceName == "com.nokia.referenceService") {
        // Hard code for now because resolver isn't ready
        QString path = "c:\\resource\\qt\\sfwplugin\\referenceserviceplugin.qtplugin";
        serviceBase = loadPlugin(path, aRefCounted, aSecSession);
    }
    else if ( aServiceName == "com.nokia.SecureTestServices") {
        // Hard code for now because resolver isn't ready
        QString path = "c:\\resource\\qt\\sfwplugin\\secureTestServicePlugin.qtplugin";
        serviceBase  = loadPlugin(path, aRefCounted, aSecSession);
    }
    else if ( aServiceName == "com.nokia.AppMgtService") {
        // Hard code for now because resolver isn't ready
        QString path = "c:\\resource\\qt\\sfwplugin\\appmgtplugin.qtplugin";
        serviceBase  = loadPlugin(path, aRefCounted, aSecSession);
    }

#endif
    return serviceBase;
}

/*!
 Load a Plugin and call a factory method to create service QObject
 @param aDllName plugin dll name
 @param aRefCounted true, reference counted interface pointer
 */
IServiceBase* ServiceHandlerPrivate::loadPlugin(const QString& aDllName,
                                         bool aRefCounted,
                                         SecSession *aSecSession)
{
    CLEAR_ERRORS(q);
    IServiceBase* serviceBase(NULL);

    // Create the current working path
    // Perhaps this should point to the plugin specific dir?
    QString path = aDllName;
    // Attempt to load the plugin
    QPluginLoader* loader = new QPluginLoader(path);

    QObject* plugin(loader->instance());
    if (plugin) {
        IServicePlugin* service(qobject_cast<IServicePlugin*> (plugin));
        if (service) {
            try {
                serviceBase = service->getServiceBase(aRefCounted, aSecSession);
            } catch (SFWException *e) {
                //exception occurred in service provider constructor
                SET_ERROR(q, e->getErrCode(), e->getErrDescription().toLatin1());
                delete e;
                e = NULL;
            }
            // Service Handler holds service object if not reference counted
            if (!aRefCounted && serviceBase) {
                QObject* object = serviceBase->getServiceObject();
                if (-1 == m_objects.indexOf(object)) {
                    m_objects.append(object);
                }
            }
        }
        delete plugin;
        plugin = NULL;
    } else {
        SET_ERROR(q, ServiceHandlerPrivate::SFW_ERROR_LOADING_PLUGIN,
            ServiceHandlerErrDesc[ServiceHandlerPrivate::SFW_ERROR_LOADING_PLUGIN]);
    }

    if (serviceBase) {
        // Ownership scope of this object
        m_loadedPlugins.append(loader);
    } else {
        delete loader;
    }
    return serviceBase;
}
} //WRT namespace
//end of file
