/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "servicehandler.h"
#include "servicehandler_p.h"
#include "serviceinfolist.h"
#include "secsession.h"
#include "serviceipc.h"
#include "serviceresolver.h"

namespace WRT
{
    /*!
     \class ServiceHandler
     Service Handler implementation. This class is used to resolve and
     load service provider plugins. This class also allows listing of
     service providers

     This class is a service handler capable of returning instances of services.\n
     Current assumptions for this class are as follows:
     -Service is registered in the service db\n
     -Client DLL name is known\n
     -Ownership of instantiated service object is returned to the owner\n
     -Service class should make all Javascript accessible methods as public slots
     */

    /*!
     Constructor
     @param parent parent to this QObject
     */
    ServiceHandler::ServiceHandler(QObject* parent)
        : QObject(parent)
    {
        CLEAR_ERRORS(this);
        d = new ServiceHandlerPrivate(this);
        d->q = this;
        if (!d->m_isConnected) {
            SET_ERROR(this, ServiceHandlerPrivate::SFW_ERROR_CONNECTING_TO_RESOLVER,
                ServiceHandlerErrDesc[ServiceHandlerPrivate::SFW_ERROR_CONNECTING_TO_RESOLVER]);
        }
    }

    /*!
     Destructor
     */
    ServiceHandler::~ServiceHandler()
    {
        delete d;
    }


    /*!
     Gets a list of all services from the service resolver
     @return QStringList of all services
     */
    QStringList ServiceHandler::getServiceProviders(SecSession * /*aSecSession=NULL*/)
    {
        CLEAR_ERRORS(this);
        QStringList providers;
#if (!defined(__WINSCW__) || defined(__S60_50__))
        if (d->m_isConnected) {
            QString null;
            bool sent = d->m_session->sendSync(LISTSERVICEREQ, null.toAscii());
            if (sent) {
                // QStringList derives from QList<QString>
                ServiceInfoList services;
                QByteArray serviceData(d->m_session->readAll());
                if ((services.deserialize(serviceData))
                                && (services.count() > 0)) {
                    for (int index = 0; index < services.count(); index++) {
                        ServiceInfo service = services.getService(index);
                        providers.append(services.getService(index).serviceName());
                    }
                } else {
                    if (serviceData.size()!=0){
                        SET_ERROR(this, ServiceHandlerPrivate::SFW_ERROR_GETTING_LIS_OF_SERVICES,
                            ServiceHandlerErrDesc[ServiceHandlerPrivate::SFW_ERROR_GETTING_LIS_OF_SERVICES]);
                    } // else no service providers
                }
            }
        }
#endif
        return providers;
    }

    /*!
     Gets version of a service provider from the service resolver
     @param aServiceName Name of the service provider
     @return QString service version
     */
    QString ServiceHandler::getServiceVersion(const QString &aServiceName)
    {
        CLEAR_ERRORS(this);
        QString serviceVersion = "NotSupport";

#if (!defined(__WINSCW__) || defined(__S60_50__))
        //multiple processes not supported on S60 v3.1 v3.2 emulators
        if (d->m_isConnected) {
            bool sent = d->m_session->sendSync( GETSERVICEVERSIONREQ, aServiceName.toAscii() );

            if (sent) {
                // data is available
                serviceVersion = d->m_session->readAll();
            }
        }
#endif
        return serviceVersion;
    }


    /*!
     Calls getServiceBase in private implementation to create an instance of service

     @param aServiceName Name of the service provider
     @param aInterfaceInfo Interface info
     @param aSecSession Security Session (not owned). If NULL then that means
                        no security check will be done.
     @param aRefCounted true the underlying service object is reference counted.
                        false  the underlying service object is not reference counted
                               and the object is owned by service handler
     @return IServiceBase* interface pointer to the service if the service is found,
                           NULL if service is not found
     */
    IServiceBase* ServiceHandler::getServiceBase(const QString &aServiceName,
                                                 ServiceInterfaceInfo& aInterfaceInfo,
                                                 bool aRefCounted,
                                                 SecSession *aSecSession)
    {
        return d->getServiceBase(aServiceName, aInterfaceInfo, aRefCounted, aSecSession);
    }


} //WRT namespace
// END OF FILE
