/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QtTest/QtTest>
#include <QtCore>
#include <serviceframeworkdefs.h>
#include <servicedatabase.h>
#include <serviceresolutioncriteria.h>
#include "testhelpers.h"

using namespace WRT;

#ifdef __SYMBIAN32__
    static const char KTESTDATAFOLDER[]         = "c:/testdata/";
#else
    static const char KTESTDATAFOLDER[]         = "testdata/";
#endif

#define ADD_TEST_TO_SERVICE_TABLE "ALTER TABLE Service ADD Test Text;"
#define ADD_TEST_TO_INTERFACE_TABLE "ALTER TABLE Interface ADD Test Text;"

class ServiceDatabaseUnitTest: public QObject
{
    Q_OBJECT

private slots:
    void initTestCase();
    void testServiceRegistration();
    void testServiceReRegistration();
    void ErrorTest1();
    void ErrorTest2();
    void ErrorTest3();
    void ErrorTest4();
    void ErrorTest5();
    void testIsOpen();
    void testCheckTable1();
    void testCheckTable2();
    void cleanupTestCase();

private:
    ServiceMetaData* parser;
    QDir dir;

};

void ServiceDatabaseUnitTest::initTestCase()
{
    __countTestsHelper__(this);
#ifndef __SYMBIAN32__
    dir = QDir::currentPath();
#else
    dir = QDir("c:\\Data");
#endif
}

void ServiceDatabaseUnitTest::testServiceRegistration()
{
    if (dir.exists(
        QString(KCwrtSvcProvInstallFolder) + QString(RESOLVERDATABASE))){
        dir.remove(QString(KCwrtSvcProvInstallFolder)
            + QString(RESOLVERDATABASE));

    }
    QDir testdir = QDir(KTESTDATAFOLDER);
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    QCOMPARE(database.open(), (int)ServiceDatabase::SFW_ERROR_DB_RECREATED);
    ServiceMetaData parser(testdir.absoluteFilePath("ServiceTest.xml"));
    if (parser.extractMetadata()){
        QCOMPARE(0,database.registerService(*parser.getServiceMetadata()));
        ServiceResolutionCriteria criteria;
        criteria.setServiceName("TestService");
        criteria.setInterfaceName("com.nokia.IDownloader");
        criteria.setInterfaceVersion("-1");
        QByteArray test = criteria.serialize();

        ServiceResolutionCriteria newCriteria;
        QVERIFY(newCriteria.deserialize(test));
        ServiceInfo serviceInfo;
        QCOMPARE(database.getService(newCriteria,serviceInfo),0);
        QCOMPARE(serviceInfo.serviceName(),QString("TestService"));
#ifdef __SYMBIAN32__
        QCOMPARE(serviceInfo.serviceDllFileName(),QString("testservice"));
#else
        QCOMPARE(serviceInfo.serviceDllFileName(),QString(KTESTDATAFOLDER) + "testservice.dll");
#endif
        QCOMPARE(serviceInfo.interfaceCapabilities().size() , 2);
        QCOMPARE(serviceInfo.interfaceVersion(),QString("1.0"));
        ServiceInfoList list;
        QCOMPARE(database.listServices(list),0);
        QCOMPARE(list.count(),1);
        serviceInfo = list.getService(0);
        QCOMPARE(serviceInfo.serviceName(),QString("TestService"));
#ifdef __SYMBIAN32__
        QCOMPARE(serviceInfo.serviceDllFileName(),QString("testservice"));
#else
        QCOMPARE(serviceInfo.serviceDllFileName(),QString(KTESTDATAFOLDER) + "testservice.dll");
#endif
        QCOMPARE(serviceInfo.serviceVersion(),QString("1.4.56"));
    }
    QCOMPARE(database.close(),0);
    QCOMPARE(database.open(),0);
    QCOMPARE(database.close(),0);
}

void ServiceDatabaseUnitTest::testServiceReRegistration()
{
    QDir testdir = QDir(KTESTDATAFOLDER);
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    QCOMPARE(database.open(),0);
    ServiceMetaData parser(testdir.absoluteFilePath("ServiceTest.xml"));
    if (parser.extractMetadata())
    {
        // Registration of the same service should be okay.
        QCOMPARE(database.registerService(*parser.getServiceMetadata()), 0);
    }
    QCOMPARE(database.close(),0);

}

void ServiceDatabaseUnitTest::ErrorTest1()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    QCOMPARE(database.open(),0);
    ServiceResolutionCriteria criteria;
    criteria.setServiceName("TestService");
    ServiceInfo service;
    QCOMPARE(database.getService(criteria,service),(int)ServiceDatabase::SFW_ERROR_INVALID_SEARCH_CRITERIA);
    QVERIFY(!service.isValid());
    QCOMPARE(database.getService(criteria,service),(int)ServiceDatabase::SFW_ERROR_INVALID_SEARCH_CRITERIA);
    QVERIFY(!service.isValid());
    criteria.setInterfaceName("com.nokia.ILocation");
    QCOMPARE(database.getService(criteria,service),(int)ServiceDatabase::SFW_ERROR_INVALID_SEARCH_CRITERIA);
    QVERIFY(!service.isValid());
    QCOMPARE(database.close(),0);
}

void ServiceDatabaseUnitTest::ErrorTest2()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    QCOMPARE(database.open(),0);
    ServiceResolutionCriteria criteria;
    criteria.setServiceName("TestService");
    criteria.setInterfaceName("com.nokia.ILocation1");
    criteria.setInterfaceVersion("1.0");
    ServiceInfo service;
    QCOMPARE(database.getService(criteria,service),(int)ServiceDatabase::SFW_ERROR_SERVICE_NOT_FOUND);
    QVERIFY(!service.isValid());
    QCOMPARE(database.close(),0);
}

void ServiceDatabaseUnitTest::ErrorTest3()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    ServiceResolutionCriteria criteria;
    criteria.setServiceName("TestService");
    criteria.setInterfaceName("com.nokia.ILocation");
    criteria.setInterfaceVersion("1.0");
    ServiceInfo service;
    QCOMPARE(database.getService(criteria,service),(int)ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN);
    QVERIFY(!service.isValid());
    ServiceInfoList services;
    QCOMPARE(database.listServices(services),(int)ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN);
    ServiceMetaData parser("test");
    QCOMPARE(database.registerService(*parser.getServiceMetadata()),(int)ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN);

}

void ServiceDatabaseUnitTest::ErrorTest4()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    ServiceResolutionCriteria criteria;
    criteria.setServiceName("TestService");
    criteria.setInterfaceName("com.nokia.ILocation");
    criteria.setInterfaceVersion("1.0");
    QCOMPARE(database.open(),0);
    {
        QSqlDatabase db = QSqlDatabase::database();
        QSqlQuery query(db);
        if (query.prepare(RESOLVERDATABASE_DROP_SERVICE_TABLE)) {
            if (query.exec()) {
                ServiceInfo service;
                QCOMPARE(database.getService(criteria,service),1);
                QVERIFY(!service.isValid());
            }
        }
    }
    QCOMPARE(database.close(),0);
}

void ServiceDatabaseUnitTest::ErrorTest5()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    QCOMPARE(database.open(),(int)ServiceDatabase::SFW_ERROR_DB_RECREATED);
    {
        QSqlDatabase db = QSqlDatabase::database();
        QSqlQuery query(db);
        if (query.prepare(RESOLVERDATABASE_DROP_SERVICE_TABLE)) {
            query.exec();
        }
        if (query.prepare(RESOLVERDATABASE_DROP_INTERFACE_TABLE)) {
            if (query.exec()) {
            }
        }
    }
    QCOMPARE(database.close(),0);
    QDir testdir = QDir(KTESTDATAFOLDER);

    QString testServiceName = QString("TestService");
    QString testServiceVersion = QString("1.4.56");

    QCOMPARE(database.deleteService(testServiceName), (int)ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN);
    QCOMPARE(database.open(),(int)ServiceDatabase::SFW_ERROR_DB_RECREATED);
    ServiceMetaData parser(testdir.absoluteFilePath("ServiceTest.xml"));

    QString filePath; //use for getServiceVersion
    if (parser.extractMetadata())
    {
        QCOMPARE(0,database.registerService(*parser.getServiceMetadata()));
        ServiceResolutionCriteria criteria;
        criteria.setServiceName(testServiceName);
        criteria.setInterfaceName("com.nokia.IDownloader");
        criteria.setInterfaceVersion("1.0");
        QByteArray test = criteria.serialize();

        ServiceResolutionCriteria newCriteria;
        QVERIFY(newCriteria.deserialize(test));

        ServiceInfo service;
        QCOMPARE(database.getService(newCriteria,service),0);
        QVERIFY(service.isValid());

        QCOMPARE(service.serviceName(),QString(testServiceName));
#ifdef __SYMBIAN32__
        QCOMPARE(service.serviceDllFileName(),QString("testservice"));
#else
        QCOMPARE(service.serviceDllFileName(),QString(KTESTDATAFOLDER) + "testservice.dll");
#endif
        QCOMPARE(service.interfaceCapabilities().size(),2);
        QCOMPARE(service.interfaceVersion(),QString("1.0"));

        ServiceInfoList list;
        QCOMPARE(database.listServices(list),0);
        QCOMPARE(list.count(),1);
        service = list.getService(0);
        QCOMPARE(service.serviceName(),QString(testServiceName));
#ifdef __SYMBIAN32__
        QCOMPARE(service.serviceDllFileName(),QString("testservice"));
#else
        QCOMPARE(service.serviceDllFileName(),QString(KTESTDATAFOLDER) + "testservice.dll");
#endif
        QCOMPARE(service.serviceVersion(),QString(testServiceVersion));
        QCOMPARE(database.getServiceVersion(testServiceName,testServiceVersion,filePath),0);
    }
    QCOMPARE(database.close(),0);
    QCOMPARE(database.getServiceVersion(testServiceName,testServiceVersion,filePath),(int)ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN);
}

void ServiceDatabaseUnitTest::testIsOpen()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif

    QCOMPARE(database.isOpen(),false);
    QCOMPARE(database.open(),0);
    QCOMPARE(database.isOpen(),true);
    ServiceInfoList list;
    QCOMPARE(database.listServices(list),0);
    QCOMPARE(list.count(),1);
    QCOMPARE(database.close(),0);
    QCOMPARE(database.isOpen(),false);
}

void ServiceDatabaseUnitTest::testCheckTable1()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    QCOMPARE(database.open(), 0);
    QSqlDatabase db = QSqlDatabase::database();
    QSqlQuery query(db);

    if (!query.prepare(ADD_TEST_TO_SERVICE_TABLE)) {
        qDebug() << "\n" << query.lastError() << "\n";
        QFAIL("Failed to prepare query");
    } else {
        if (!query.exec()) {
            QFAIL("Failed to execute query");
        }
    }

    QCOMPARE(database.close(),0);
    QCOMPARE(database.open(),(int)ServiceDatabase::SFW_ERROR_DB_RECREATED);
}

void ServiceDatabaseUnitTest::testCheckTable2()
{
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(QString("c:\\Data\\"));
#endif
    QCOMPARE(database.open(), 0);
    QSqlDatabase db = QSqlDatabase::database();
    QSqlQuery query(db);

    if (!query.prepare(ADD_TEST_TO_INTERFACE_TABLE)) {
        qDebug() << "\n" << query.lastError() << "\n";
        QFAIL("Failed to prepare query");
    } else {
        if (!query.exec()) {
            QFAIL("Failed to execute query");
        }
    }

    QCOMPARE(database.close(),0);
    QCOMPARE(database.open(),(int)ServiceDatabase::SFW_ERROR_DB_RECREATED);
}

void ServiceDatabaseUnitTest::cleanupTestCase()
{
    if (dir.exists(
        QString(KCwrtSvcProvInstallFolder) + QString(RESOLVERDATABASE))){
        dir.remove(QString(KCwrtSvcProvInstallFolder)
            + QString(RESOLVERDATABASE));
    }
}

QTEST_MAIN(ServiceDatabaseUnitTest)
#include "servicedatabasetest.moc"
