/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "sfwlog.h"
#include "npplugin.h"

#if defined(__SYMBIAN32__) && defined(CWRT_S60_WEBKIT)
// CWRT browser in Symbian
#include "npinterface.h"
#endif

using namespace WRT;

#ifdef MSVC
#pragma warning( disable: 4100)
#endif

static const char* KPluginName = "Service Framework Plugin";
static const char* KPluginDescription = "Nokia Service JavaScript Enabler";

/*!
    Creates a new instance of a plug-in

    @param pluginType pointer to the MIME type for new plug-in instance.
    @param instance contains instance-specific private data for the plug-in and the browser.
    @param mode display mode of plug-in. Values:
                \li NP_EMBED: (1) Instance was created by an EMBED tag and shares
                                  the browser window with other content.
                \li NP_FULL: (2) Instance was created by a separate file and is
                                  the primary content in the window
    @param argc Number of HTML arguments in the EMBED tag for an embedded plug-in;
                determines the number of attributes in the argn and argv arrays.
    @param argn Array of attribute names passed to the plug-in from the EMBED tag
    @param argv Array of attribute values passed to the plug-in from the EMBED tag
    @param saved Pointer to data saved by NPP_Destroy for a previous instance of
                 this plug-in at the same URL. If non-null, the browser passes ownership
                 of the NPSavedData object back to the plug-in. The plug-in is
                 responsible for freeing the memory for the NPSavedData and the buffer it contains
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
NPError NPP_New( NPMIMEType /*pluginType*/,  // Not used locally
                 NPP instance,            // Stores pointer to SystemInfo
                 uint16 mode,             //
                 CDesCArray* /*argn*/,    // The number of arguments passed
                 CDesCArray* /*argv*/,    // The values of arguments in the array
                 NPSavedData* saved )     //
#else
NPError NPP_New(NPMIMEType /*pluginType*/,
                NPP instance,
                uint16 mode,
                int16 /*argc*/,
                char* /*argn*/[],
                char* /*argv*/[],
                NPSavedData* saved)
#endif // __SYMBIAN32__
{
    NPError error = NPERR_NO_ERROR;

    if (!instance) {
        error = NPERR_INVALID_INSTANCE_ERROR;
    }
    else if (mode != NP_EMBED) {
        error = NPERR_INVALID_PARAM;
    }
    else {
        if (saved) {
            if (saved->buf) {
                NPN_MemFree(saved->buf);
            }
            NPN_MemFree(saved);
        }

        NPPlugin* pPlugin = new NPPlugin(instance);
        if (pPlugin) {
            instance->pdata = (void *)pPlugin;
            SFW_DEBUG3("NPP_New NPPlugin 0x%08x of instance 0x%08x", pPlugin, instance);
        }
        else {
            error = NPERR_OUT_OF_MEMORY_ERROR;
        }
    }
    return error;
}

/*!
    Deletes a specific instance of a plug-in

    @param instance Pointer to the plug-in instance to delete
    @param save State or other information to save for reuse by a new instance
                of this plug-in at the same URL. Passed to NPP_New
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
NPError NPP_Destroy (NPP instance, NPSavedData** /*save*/)
{
    NPError error = NPERR_NO_ERROR;

    if (instance) {
        NPPlugin* pPlugin = (NPPlugin*) instance->pdata;
        if (pPlugin) {
            SFW_DEBUG3("NPP_Destroy NPPlugin 0x%08x of instance 0x%08x", pPlugin, instance);
            delete pPlugin;
            instance->pdata = NULL;
        }
    }
    else {
        error = NPERR_INVALID_INSTANCE_ERROR;
    }

    return error;
}

/*!
    Tells the plug-in when a window is created, moved, sized, or destroyed.

    @param instance Pointer to the current plug-in instance
    @param pNPWindow Pointer to the window into which the instance draws.
            The window structure contains a window handle and values for
            top left corner, width, height, and clipping rectangle.
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
NPError NPP_SetWindow (NPP instance, NPWindow* /*pNPWindow*/)
{
    return instance ? NPERR_NO_ERROR : NPERR_INVALID_INSTANCE_ERROR;
}

/*!
    Notifies a plug-in instance of a new data stream.

    @param instance Pointer to the current plug-in instance
    @param type Pointer to MIME type of the stream.
    @param stream Pointer to new stream.
    @param seekable Boolean indicating whether the stream is seekable.
    @param stype Requested mode of new stream.
                 NP_NORMAL (Default): Delivers stream data to the instance in a
                                      series of calls to NPP_WriteReady and NPP_Write.
                NP_ASFILEONLY: Saves stream data to a file in the local cache.
                NP_ASFILE: File download. Like NP_ASFILEONLY except that data is
                           delivered to the plug-in as it is saved to the file
                           (as in mode NP_NORMAL).
                NP_SEEK: Stream data randomly accessible by the plug-in as needed,
                         through calls to NPN_RequestRead
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
NPError NPP_NewStream(NPP instance,
                      NPMIMEType /*type*/,
                      NPStream* /*stream*/,
                      NPBool /*seekable*/,
                      uint16* /*stype*/)
{
    return instance ? NPERR_NO_ERROR : NPERR_INVALID_INSTANCE_ERROR;
}

/*!
    Tells the plug-in that a stream is about to be closed or destroyed.

    @param instance Pointer to the current plug-in instance
    @param stream Pointer to current stream.
    @param reason Reason the stream was destroyed.
                  NPRES_DONE (Most common): Completed normally; all data was sent
                                            to the instance.
                  NPRES_USER_BREAK: User canceled stream directly by clicking the
                                    Stop button or indirectly by some action such
                                    as deleting the instance or initiating higher-priority
                                    network operations.
                  NPRES_NETWORK_ERR: Stream failed due to problems with network,
                                     disk I/O, lack of memory, or other problems
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
NPError NPP_DestroyStream(NPP instance, NPStream * /*stream*/, NPError /*reason*/)
{
    return instance ? NPERR_NO_ERROR : NPERR_INVALID_INSTANCE_ERROR;
}

/*!
    Provides a local file name for the data from a stream.

    @param instance Pointer to the current plug-in instance
    @param stream Pointer to current stream.
    @param fname Pointer to full path to a local file. If an error occurs while
                 retrieving the data or writing the file, fname may be null.
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
void NPP_StreamAsFile(NPP /*instance*/, NPStream* /*stream*/, const TDesC& /*fname*/)
#else
void NPP_StreamAsFile(NPP /*instance*/, NPStream* /*stream*/, const char* /*fname*/)
#endif // __SYMBIAN32__
{
    //Do nothing
}

/*!
    Determines maximum number of bytes that the plug-in can consume.

    @param instance Pointer to the current plug-in instance
    @param stream Pointer to current stream.
    @return the maximum number of bytes that an instance is prepared to accept
            from the stream
*/
int32 NPP_WriteReady(NPP instance, NPStream * /*stream*/)
{
    return instance ? 0x0fffffff : static_cast<int32>(NPERR_INVALID_INSTANCE_ERROR);
}

/*!
    Delivers data to a plug-in instance.

    @param instance Pointer to the current plug-in instance
    @param stream Pointer to current stream.
    @param offset Offset in bytes of buf from the beginning of the data in the stream.
                  Can be used to check stream progress or bye range requests
                  from NPN_RequestRead.
    @param len Length in bytes of buf; number of bytes accepted.
    @param buffer Buffer of data, delivered by the stream, that contains len bytes
                  of data offset bytes from the start of the stream. The buffer is
                  allocated by the browser and is deleted after returning from
                  the function, so the plug-in should make a copy of the data it
                  needs to keep.
    @return If successful, the function returns the number of bytes consumed by the instance
            If unsuccessful, the function destroys the stream by returning a negative value.
*/
int32 NPP_Write(NPP instance, NPStream * /*stream*/, int32 /*offset*/,
        int32 len, void * /*buffer*/)
{
    return instance ? len : static_cast<int32>(NPERR_INVALID_INSTANCE_ERROR);
}

/*!
    Requests a platform-specific print operation for an embedded or full-screen plug-in.

    @param instance Pointer to the current plug-in instance
    @param printInfo Pointer to NPPrint structure
*/
void NPP_Print(NPP /*instance*/, NPPrint* /*printInfo*/)
{
    //Do nothing
}

/*!
    Delivers a platform-specific window event to the instance.

    @param instance Pointer to the current plug-in instance
    @param event Platform-specific value representing the event handled by the function.
    @return If the plug-in handles the event, the function should return true
            If the plug-in ignores the event, the function returns false
*/
int16 NPP_HandleEvent(NPP instance, void* event)
{
    (void)instance;
    (void)event;

    return NPBool(false);
}

/*!
    Notifies the instance of the completion of a URL request.

    @param instance Pointer to the current plug-in instance
    @param url URL of the NPN_GetURLNotify or NPN_PostURLNotify request.
    @param reason Reason code for completion of request. Values:
                  NPRES_DONE (most common): Completed normally.
                  NPRES_USER_BREAK: User canceled stream directly by clicking the
                                    Stop button or indirectly by some action such as
                                    deleting the instance or initiating
                                    higher-priority network operations.
                  NPRES_NETWORK_ERR: Stream failed due to problems with network,
                                     disk I/O, lack of memory, or other problems
    @param notifData Plug-in-private value for associating a previous
                     NPN_GetURLNotify or NPN_PostURLNotify request with a subsequent
                     NPP_URLNotify call.
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
void NPP_URLNotify(NPP /*instance*/, const TDesC& /*url*/,
                   NPReason /*reason*/, void* /*notifyData*/)
#else
void NPP_URLNotify(NPP /*instance*/, const char* /*url*/, NPReason /*reason*/, void* /*notifyData*/)
#endif // __SYMBIAN32__
{
    // Do nothing
}

/*!
    Allows the browser to query the plug-in for information.

    @param instance Pointer to the current plug-in instance
    @param variable Plug-in information the call gets.
    @param value Value of the variable
    @return If successful, the function returns the function returns NPERR_NO_ERROR.
            If unsuccessful, the function returns a NPErr error code.
*/
NPError NPP_GetValue(NPP instance, NPPVariable variable, void *value)
{
    NPError error = NPERR_NO_ERROR;

    if (instance) {
        NPPlugin* pPlugin = (NPPlugin*) instance->pdata;
        if (!pPlugin) {
            error = NPERR_GENERIC_ERROR;
        }
        else {
            switch (variable) {
            case NPPVpluginScriptableNPObject:
                *(NPObject**) value = pPlugin->GetFactory();
                break;
            default:
                error = NPERR_GENERIC_ERROR;
            }
        }
    }
    else { //these variables does not require plugin instace
       switch (variable) {
       case NPPVpluginNameString: {
           *((const char **)value) = KPluginName;
           break;
       }
        case NPPVpluginDescriptionString: {
           *((const char **)value) = KPluginDescription;
           break;
        }
        default:
            error = NPERR_GENERIC_ERROR;
        }
    }
    return error;
}

/*!
    Sets information about the plug-in.

    @param instance Pointer to the current plug-in instance
    @param variable Plug-in information the call gets.
    @param value Value of the variable
    @return If successful, the function returns the function returns NPERR_NO_ERROR.
            If unsuccessful, the function returns a NPErr error code.
*/
NPError NPP_SetValue(NPP instance, NPNVariable /*variable*/, void * /*value*/)
{
    return instance ? NPERR_NO_ERROR : NPERR_INVALID_INSTANCE_ERROR;
}
